import React, { useEffect, useMemo, useState } from "react";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Input } from "@/components/ui/input";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Download, Search, ShoppingCart, X } from "lucide-react";
import { Dialog, DialogContent, DialogTitle } from "@/components/ui/dialog";
import { useCart } from "@/contexts/CartContext";
import { useToast } from "@/hooks/use-toast";

type Item = { 
  id: string; 
  url: string; 
  name: string; 
  tags: string[]; 
  category?: string; 
  createdAt: number;
  mimeType?: string;
  code?: string;
};

type Ent = { licensed: boolean; quotaRemaining: number; canDownloadOriginal: boolean };

// Helper functions
async function toTenDigitCode(seed: string): Promise<string> {
  const enc = new TextEncoder().encode(seed);
  const buf = await crypto.subtle.digest("SHA-256", enc);
  const hex = [...new Uint8Array(buf)]
    .map((b) => b.toString(16).padStart(2, "0"))
    .join("");
  const digits = hex.replace(/[a-f]/g, (c) => (parseInt(c, 16) % 10).toString()).slice(0, 10);
  return digits.padEnd(10, "0");
}

const fileTypeFrom = (item: Item) => {
  const t = item.mimeType?.split("/")[1] || (item.name || "").split(".").pop() || "";
  return t.toUpperCase();
};

function useEntitlement(assetId?: string) {
  const [state, setState] = useState<Ent & { loading: boolean }>({
    licensed: false,
    quotaRemaining: 0,
    canDownloadOriginal: false,
    loading: true,
  });

  useEffect(() => {
    if (!assetId) return;
    setState((s) => ({ ...s, loading: true }));
    fetch(`/api/mockups/${assetId}/entitlement`)
      .then((r) => r.json())
      .then((data) => setState({ ...data, loading: false }))
      .catch(() => setState((s) => ({ ...s, loading: false })));
  }, [assetId]);

  return state;
}

// Mockup Card Component with conditional button logic
function MockupCard({ item, onAddToCart, onImageClick }: { item: Item; onAddToCart: (item: Item) => void; onImageClick: (item: Item) => void }) {
  const [code, setCode] = useState<string>("");
  const ent = useEntitlement(item.id);

  useEffect(() => {
    toTenDigitCode(item.id).then(setCode);
  }, [item.id]);

  const displayName = `IBrandBiz | #${code}`;
  const fileType = fileTypeFrom(item);

  const handleDownload = () => {
    window.location.href = `/api/mockups/${item.id}/download`;
  };

  const handlePurchase = () => {
    onAddToCart(item);
  };

  return (
    <Card className="group overflow-hidden hover:shadow-lg transition-all duration-300">
      <div className="aspect-video bg-gray-100 dark:bg-gray-800 overflow-hidden relative cursor-pointer" onClick={() => onImageClick(item)}>
        <img 
          src={item.url} 
          alt={item.name} 
          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300" 
          loading="lazy"
        />
        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
          {ent.loading ? (
            <div className="bg-white/90 px-4 py-2 rounded-lg text-gray-700 text-sm">Checking…</div>
          ) : ent.licensed || ent.quotaRemaining > 0 ? (
            <button
              onClick={handleDownload}
              className="bg-white/90 hover:bg-white text-gray-900 px-4 py-2 rounded-lg font-medium flex items-center gap-2"
              data-testid={`download-${item.id}`}
            >
              <Download className="h-4 w-4" />
              {ent.licensed ? "Download" : "Download (uses 1 credit)"}
            </button>
          ) : (
            <button
              onClick={handlePurchase}
              className="bg-white/90 hover:bg-white text-gray-900 px-4 py-2 rounded-lg font-medium flex items-center gap-2"
              data-testid={`purchase-${item.id}`}
            >
              <ShoppingCart className="h-4 w-4" />
              Purchase $6.99
            </button>
          )}
        </div>
      </div>
      <CardContent className="p-4">
        <div className="mb-2">
          <h3 className="font-semibold text-sm mb-1 line-clamp-1" data-testid={`title-${item.id}`}>
            {displayName}
          </h3>
          <p className="text-xs text-gray-500">File type: {fileType || "—"}</p>
        </div>
        <div className="text-xs text-gray-500 truncate mb-2">{item.name}</div>
        <div className="flex flex-wrap gap-1 mb-2">
          {item.category && (
            <Badge variant="secondary" className="text-xs">
              {item.category}
            </Badge>
          )}
        </div>
        <div className="flex flex-wrap gap-1">
          {item.tags.slice(0, 3).map((tag, index) => (
            <Badge key={index} variant="outline" className="text-xs">
              {tag}
            </Badge>
          ))}
          {item.tags.length > 3 && (
            <Badge variant="outline" className="text-xs">
              +{item.tags.length - 3}
            </Badge>
          )}
        </div>
      </CardContent>
    </Card>
  );
}

export default function Mockups() {
  const [data, setData] = useState<Item[]>([]);
  const [q, setQ] = useState("");
  const [loading, setLoading] = useState(true);
  const [selectedImage, setSelectedImage] = useState<Item | null>(null);
  const { addItem, open } = useCart();
  const { toast } = useToast();

  // zoom/pan state
  const [zoom, setZoom] = useState(1);
  const [pos, setPos] = useState({ x: 0, y: 0 });
  const [drag, setDrag] = useState<{startX:number; startY:number} | null>(null);
  const [selectedImageCode, setSelectedImageCode] = useState("");
  const ent = useEntitlement(selectedImage?.id);

  const MOCKUP_PRICE = 699; // $6.99 in cents

  const handleAddToCart = async (item: Item) => {
    const code = item.code || await toTenDigitCode(item.id);
    const displayName = `IBrandBiz | #${code}`;
    
    addItem({
      kind: "stock",
      assetId: item.id,
      name: displayName,
      priceCents: MOCKUP_PRICE,
      qty: 1,
      previewUrl: `/api/mockups/${item.id}/preview`,
    });
    open();

    toast({
      title: "Added to cart",
      description: `${displayName} added to your cart`,
    });
  };

  // Update selected image code when image changes
  useEffect(() => {
    if (selectedImage) {
      (async () => {
        const code = selectedImage.code || await toTenDigitCode(selectedImage.id);
        setSelectedImageCode(code);
      })();
    }
  }, [selectedImage]);

  // reset when a new image opens
  useEffect(() => {
    if (selectedImage) { setZoom(1); setPos({x:0,y:0}); setDrag(null); }
  }, [selectedImage]);

  // Handle keyboard events for modal
  useEffect(() => {
    const handleKeyDown = (event: KeyboardEvent) => {
      if (event.key === 'Escape' && selectedImage) {
        setSelectedImage(null);
      }
    };

    if (selectedImage) {
      document.addEventListener('keydown', handleKeyDown);
      document.body.style.overflow = 'hidden'; // Prevent background scrolling
    }

    return () => {
      document.removeEventListener('keydown', handleKeyDown);
      document.body.style.overflow = 'unset';
    };
  }, [selectedImage]);

  const onWheelZoom: React.WheelEventHandler<HTMLDivElement> = (e) => {
    e.preventDefault();
    const delta = e.deltaY < 0 ? 0.15 : -0.15;
    setZoom(z => Math.max(1, Math.min(4, +(z + delta).toFixed(2))));
  };

  const onMouseDown: React.MouseEventHandler = (e) => {
    if (zoom === 1) return;
    setDrag({ startX: e.clientX - pos.x, startY: e.clientY - pos.y });
  };

  const onMouseMove: React.MouseEventHandler = (e) => {
    if (!drag) return;
    setPos({ x: e.clientX - drag.startX, y: e.clientY - drag.startY });
  };

  const endDrag = () => setDrag(null);

  const toggleClickZoom: React.MouseEventHandler = (e) => {
    e.stopPropagation();
    if (zoom === 1) setZoom(2);
    else { setZoom(1); setPos({x:0,y:0}); }
  };

  const handleDownload = () => {
    if (selectedImage) {
      window.location.href = `/api/mockups/${selectedImage.id}/download`;
    }
  };

  const handlePurchase = async () => {
    if (selectedImage) {
      await handleAddToCart(selectedImage);
    }
  };

  useEffect(() => { 
    (async () => {
      try {
        const res = await fetch("/api/mockups/list");
        const result = await res.json();
        // Handle both direct array and object with mockups property
        setData(Array.isArray(result) ? result : (result.mockups || []));
      } catch (error) {
        console.error("Failed to fetch mockups:", error);
      } finally {
        setLoading(false);
      }
    })(); 
  }, []);

  const filtered = useMemo(() => {
    const term = q.toLowerCase().trim();
    if (!term) return data;
    return data.filter(x =>
      x.name.toLowerCase().includes(term) ||
      x.tags.join(" ").toLowerCase().includes(term) ||
      (x.category||"").toLowerCase().includes(term)
    );
  }, [data, q]);

  return (
    <DashboardTemplatePage title="Mockups Library" description="Professional device and product mockups to showcase your designs">
      <div className="space-y-6">
        {/* Search Section */}
        <div className="flex items-center gap-4">
          <div className="relative flex-1 max-w-md">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground h-4 w-4" />
            <Input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              placeholder="Search by name, tags, or category..."
              className="pl-10"
              data-testid="input-search"
            />
          </div>
          <div className="text-sm text-muted-foreground">
            {filtered.length} {filtered.length === 1 ? 'mockup' : 'mockups'} found
          </div>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex items-center justify-center py-12">
            <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
          </div>
        )}

        {/* Empty State */}
        {!loading && data.length === 0 && (
          <div className="text-center py-12">
            <div className="text-muted-foreground mb-4">
              <svg className="h-16 w-16 mx-auto mb-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z" />
              </svg>
            </div>
            <h3 className="text-lg font-medium mb-2">No mockups available</h3>
            <p className="text-muted-foreground">Mockups will appear here once uploaded by administrators.</p>
          </div>
        )}

        {/* Mockups Grid */}
        {!loading && filtered.length > 0 && (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
            {filtered.map((item) => <MockupCard key={item.id} item={item} onAddToCart={handleAddToCart} onImageClick={setSelectedImage} />)}
          </div>
        )}

        {/* No Results State */}
        {!loading && data.length > 0 && filtered.length === 0 && (
          <div className="text-center py-12">
            <div className="text-muted-foreground mb-4">
              <Search className="h-16 w-16 mx-auto mb-4" />
            </div>
            <h3 className="text-lg font-medium mb-2">No mockups found</h3>
            <p className="text-muted-foreground">Try adjusting your search terms or browse all mockups.</p>
          </div>
        )}
      </div>

      {/* Image Modal/Lightbox */}
      <Dialog open={!!selectedImage} onOpenChange={() => setSelectedImage(null)}>
        <DialogContent className="max-w-6xl w-[95vw] max-h-[95vh] p-0 bg-transparent border-none shadow-none" data-testid="modal-image-preview">
          <DialogTitle className="sr-only">Mockup Preview</DialogTitle>
          <div className="relative bg-black/90 rounded-lg overflow-hidden h-[90vh] flex flex-col">
            {selectedImage && (
              <>
                {/* Close Button */}
                <button
                  onClick={() => setSelectedImage(null)}
                  className="absolute top-4 right-4 z-10 bg-black/50 hover:bg-black/70 text-white rounded-full p-2 transition-colors"
                  data-testid="modal-close-button"
                >
                  <X className="h-5 w-5" />
                </button>
                <div className="flex items-center justify-center flex-1 p-4 min-h-0">
                  <img 
                    src={selectedImage.url || selectedImage.previewUrl} 
                    alt={selectedImage.name}
                    className="max-h-full max-w-full w-auto h-auto mx-auto object-contain rounded"
                  />
                </div>
                <div className="bg-black/70 text-white p-4 flex-shrink-0">
                  <h3 className="font-semibold text-lg mb-1">IBrandBiz | #{selectedImageCode}</h3>
                  <p className="text-xs text-gray-300 mb-2">File type: {fileTypeFrom(selectedImage)}</p>
                  <div className="flex flex-wrap gap-2 mb-3">
                    {selectedImage.category && (
                      <Badge variant="secondary" className="text-xs bg-white/20 text-white border-white/30">
                        {selectedImage.category}
                      </Badge>
                    )}
                    {selectedImage.tags?.slice(0, 5).map((tag, index) => (
                      <Badge key={index} variant="outline" className="text-xs border-white/30 text-white">
                        {tag}
                      </Badge>
                    ))}
                    {selectedImage.tags && selectedImage.tags.length > 5 && (
                      <Badge variant="outline" className="text-xs border-white/30 text-white">
                        +{selectedImage.tags?.length ? selectedImage.tags.length - 5 : 0}
                      </Badge>
                    )}
                  </div>
                  <div className="text-xs text-gray-300 mb-3">{selectedImage.name}</div>
                  <div className="flex gap-3">
                    {(() => {
                      if (ent.loading) return (
                        <Button disabled className="bg-gray-600 text-white">
                          Checking...
                        </Button>
                      );
                      if (ent.licensed) return (
                        <Button
                          onClick={handleDownload}
                          className="bg-green-600 hover:bg-green-700 text-white"
                          data-testid={`modal-download-${selectedImage.id}`}
                        >
                          <Download className="h-4 w-4 mr-2" />
                          Download
                        </Button>
                      );
                      if (ent.quotaRemaining > 0) return (
                        <Button
                          onClick={handleDownload}
                          className="bg-green-600 hover:bg-green-700 text-white"
                          data-testid={`modal-download-credit-${selectedImage.id}`}
                        >
                          <Download className="h-4 w-4 mr-2" />
                          Download (uses 1 credit)
                        </Button>
                      );
                      return (
                        <Button
                          onClick={handlePurchase}
                          className="bg-blue-600 hover:bg-blue-700 text-white"
                          data-testid={`modal-purchase-${selectedImage.id}`}
                        >
                          <ShoppingCart className="h-4 w-4 mr-2" />
                          Purchase $6.99
                        </Button>
                      );
                    })()}
                  </div>
                </div>
              </>
            )}
          </div>
        </DialogContent>
      </Dialog>
    </DashboardTemplatePage>
  );
}