import React, { useEffect, useMemo, useState } from "react";
import { Heart, HeartOff, Sparkles, Search, Download, Copy, Palette, Layers, Wand2, Image } from "lucide-react";
import { DashboardTemplatePage } from '@/components/DashboardTemplatePage';
import { applySvgColor, sanitizeSvgClient } from "@/utils/svg";
import { useFavorites } from "@/store/favorites";
import { useMutation, useQuery } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import SignInChip from "../../../components/SignInChip";
import { useIconEntitlement } from "@/hooks/useIconEntitlement";
import AttributionNotice from "@/components/AttributionNotice";
import { buildAttribution } from "@/utils/iconAttribution";

/**
 * ICONS — MVP NOTES
 * - Two modes: Browse (DB) and Generate (AI).
 * - Styles: modern, classic, flat, outlined, solid, hand-drawn, isometric (3D), material.
 * - All SVG so colors are easy to change in PPT/Keynote/Slides.
 * - Users can save "Favorites" with cloud sync via Firebase Firestore.
 *
 * TODO (server/back-end integration):
 * - /api/icons/search?query=&style=&page= to query your icon DB (or a provider).
 * - /api/icons/generate { prompt, style } -> returns SVG string.
 */

// --- Style taxonomy ---
const STYLES = [
  { key: "modern", label: "Modern" },
  { key: "classic", label: "Classic" },
  { key: "flat", label: "Flat" },
  { key: "outlined", label: "Outlined" },
  { key: "solid", label: "Solid (Filled)" },
  { key: "handdrawn", label: "Hand-Drawn" },
  { key: "isometric", label: "3D/Isometric" },
  { key: "material", label: "Material Design" },
] as const;

type StyleKey = typeof STYLES[number]["key"];

// --- Frontend Blocklist (Quick Fix) ---
const HIDE_ICON_IDS = new Set(["guest_icon"]);
const HIDE_ICON_NAMES = new Set(["Guest Icon"]);

function stripStoragePrefix(id: string) {
  return id?.startsWith("storage-") ? id.slice("storage-".length) : id;
}

// --- SuperNova's enhanced IconRecord type ---
type IconRecord = {
  id: string;
  name: string;
  style: StyleKey;
  // SuperNova's dual approach: inline SVG for small files, URL for large ones
  svg?: string; // Inline SVG content for small icons
  svgUrl?: string; // URL endpoint for large SVG files
  tags: string[];
  previewUrl?: string; // Fallback PNG preview for object storage icons
};

// 12 demo icons across styles; swap in real source later
const MOCK_ICONS: IconRecord[] = [
  {
    id: "classic-lightbulb",
    name: "Lightbulb",
    style: "classic",
    tags: ["idea", "innovation"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><g fill="none" stroke="currentColor" stroke-width="2"><path d="M9 18h6m-5 3h4"/><path d="M6 10a6 6 0 1 1 12 0c0 2.1-1.2 3.9-3 5v1H9v-1c-1.8-1.1-3-2.9-3-5Z"/></g></svg>`,
  },
  {
    id: "flat-chart",
    name: "Bar Chart",
    style: "flat",
    tags: ["data", "analytics"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><g fill="currentColor"><rect x="3" y="10" width="3" height="10" rx="1"/><rect x="9" y="6" width="3" height="14" rx="1"/><rect x="15" y="3" width="3" height="17" rx="1"/></g></svg>`,
  },
  {
    id: "outlined-user",
    name: "User",
    style: "outlined",
    tags: ["profile", "person"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><g fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="8" r="4"/><path d="M4 20c0-3.314 3.582-6 8-6s8 2.686 8 6"/></g></svg>`,
  },
  {
    id: "solid-shield",
    name: "Shield",
    style: "solid",
    tags: ["security", "protect"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path fill="currentColor" d="M12 3l7 3v6c0 5-3.5 8.5-7 9c-3.5-.5-7-4-7-9V6l7-3Z"/></svg>`,
  },
  {
    id: "handdrawn-heart",
    name: "Heart",
    style: "handdrawn",
    tags: ["love", "like"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M12 20s-6-3.33-8.5-6.5C1 10.5 3 6 6.5 6C9 6 10.5 8 12 9.5C13.5 8 15 6 17.5 6C21 6 23 10.5 20.5 13.5C18 16.67 12 20 12 20Z" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>`,
  },
  {
    id: "iso-cube",
    name: "Isometric Cube",
    style: "isometric",
    tags: ["3D", "box"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><g fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2l8 4v12l-8 4l-8-4V6z"/><path d="M20 6l-8 4l-8-4"/></g></svg>`,
  },
  {
    id: "material-home",
    name: "Home",
    style: "material",
    tags: ["house", "dashboard"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M12 3l9 8h-3v9h-12v-9H3l9-8z" fill="currentColor"/></svg>`,
  },
  {
    id: "outlined-link",
    name: "Link",
    style: "outlined",
    tags: ["url", "chain"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><g fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M10 14a5 5 0 0 1 0-7l1-1a5 5 0 0 1 7 7l-1 1"/><path d="M14 10a5 5 0 0 1 0 7l-1 1a5 5 0 1 1-7-7l1-1"/></g></svg>`,
  },
  {
    id: "solid-download",
    name: "Download",
    style: "solid",
    tags: ["arrow", "save"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path fill="currentColor" d="M12 3a1 1 0 0 1 1 1v8.586l2.293-2.293a1 1 0 1 1 1.414 1.414l-4.007 4.007a1 1 0 0 1-1.4 0l-4.014-4.014a1 1 0 1 1 1.414-1.414L11 12.586V4a1 1 0 0 1 1-1Zm-7 15a1 1 0 0 1 1-1h12a1 1 0 1 1 0 2H6a1 1 0 0 1-1-1Z"/></svg>`,
  },
  {
    id: "flat-star",
    name: "Star",
    style: "flat",
    tags: ["favorite", "rating"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path fill="currentColor" d="m12 2l3.09 6.26L22 9.27l-5 4.87l1.18 6.88L12 17.77l-6.18 3.25L7 14.14l-5-4.87l6.91-1.01z"/></svg>`,
  },
  {
    id: "modern-folder",
    name: "Folder",
    style: "modern",
    tags: ["files", "assets"],
    svg: `<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M3 7a2 2 0 0 1 2-2h4l2 2h8a2 2 0 0 1 2 2v7a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z" fill="none" stroke="currentColor" stroke-width="2"/></svg>`,
  },
];

// --- Helpers: download + clipboard
function downloadTextFile(filename: string, content: string) {
  const blob = new Blob([content], { type: "image/svg+xml;charset=utf-8" });
  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  a.remove();
  URL.revokeObjectURL(url);
}

async function copyToClipboard(text: string) {
  try {
    await navigator.clipboard.writeText(text);
    return true;
  } catch {
    return false;
  }
}

type Mode = "browse" | "generate";

// Circuit breaker to track failed SVG fetches and prevent infinite retries
const failedSvgCache = new Set<string>();

// Clean IconRenderer component with no infinite loops
const IconRenderer: React.FC<{ icon: IconRecord; color: string; size: string }> = ({ icon, color, size }) => {
  const [renderState, setRenderState] = useState<{
    type: 'loading' | 'inline-svg' | 'preview' | 'placeholder' | 'text-fallback';
    content?: string;
    error?: string;
  }>({ type: 'loading' });

  useEffect(() => {
    // Create unique cache key for this icon
    const cacheKey = `${icon.id}-${icon.svgUrl || 'no-url'}`;
    
    // If already failed, go directly to preview
    if (failedSvgCache.has(cacheKey)) {
      setRenderState({ type: 'preview' });
      return;
    }

    // Handle inline SVG first (highest priority)
    if (icon.svg) {
      try {
        setRenderState({ type: 'inline-svg', content: icon.svg });
        return;
      } catch (err) {
        console.error('Failed to process inline SVG:', err);
        setRenderState({ type: 'preview', error: 'Invalid SVG content' });
        return;
      }
    }

    // Handle SVG URL with single attempt (no retries)
    if (icon.svgUrl) {
      setRenderState({ type: 'loading' });
      
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3000); // 3 second timeout
      
      fetch(icon.svgUrl, { signal: controller.signal })
        .then(response => {
          clearTimeout(timeoutId);
          if (!response.ok) {
            throw new Error(`HTTP ${response.status}`);
          }
          return response.text();
        })
        .then(svgText => {
          if (svgText && svgText.includes('<svg')) {
            setRenderState({ type: 'inline-svg', content: svgText });
          } else {
            throw new Error('Invalid SVG response');
          }
        })
        .catch(err => {
          console.error('SVG fetch failed for', icon.id, ':', err.message);
          // Mark as permanently failed to prevent future attempts
          failedSvgCache.add(cacheKey);
          setRenderState({ type: 'preview', error: err.message });
        });
      
      // Cleanup function
      return () => {
        clearTimeout(timeoutId);
        controller.abort();
      };
    }

    // No SVG available - use preview
    setRenderState({ type: 'preview' });
  }, [icon.id, icon.svg, icon.svgUrl]); // Only depend on icon properties

  // Render based on current state
  switch (renderState.type) {
    case 'inline-svg':
      // Render SVG with safe color application
      if (renderState.content) {
        try {
          const isStorage = icon.id.startsWith("storage-");
          const source = isStorage
            ? sanitizeSvgClient(renderState.content)                      // raw for uploaded icons
            : sanitizeSvgClient(applySvgColor(renderState.content, { primary: color, secondary: color, accent: color })); // recolor only mock/demo
          
          return (
            <div
              className={size}
              dangerouslySetInnerHTML={{ __html: source }}
              aria-label={icon.name}
            />
          );
        } catch (colorError) {
          console.error('Failed to apply color to SVG:', colorError);
          // Fallback to original SVG without color
          return (
            <div
              className={size}
              dangerouslySetInnerHTML={{ __html: renderState.content }}
              aria-label={icon.name}
            />
          );
        }
      }
      break;

    case 'loading':
      // Show loading spinner
      return (
        <div className={`${size} flex items-center justify-center text-muted-foreground`}>
          <div className="animate-spin w-4 h-4 border-2 border-primary border-t-transparent rounded-full"></div>
        </div>
      );

    case 'preview':
      // Use preview image as fallback
      if (icon.id.startsWith('storage-')) {
        const previewUrl = `/api/icons/${icon.id.replace('storage-', '')}/preview`;
        return (
          <img
            src={previewUrl}
            alt={icon.name}
            className={`${size} object-contain`}
            style={{ 
              filter: color !== "#000000" ? `hue-rotate(${getHueRotation(color)}deg)` : undefined,
              backgroundColor: 'transparent'
            }}
            onError={(e) => {
              // Replace image with placeholder on error
              const target = e.target as HTMLImageElement;
              const placeholder = document.createElement('div');
              placeholder.className = `${size} flex items-center justify-center text-muted-foreground bg-muted/20 rounded border-2 border-dashed border-muted-foreground/30`;
              placeholder.innerHTML = `<svg class="w-6 h-6" style="color: ${color !== "#000000" ? color : 'currentColor'}" fill="currentColor" viewBox="0 0 24 24"><path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="m21 15l-5-5L5 21"/></svg>`;
              target.parentNode?.replaceChild(placeholder, target);
            }}
          />
        );
      } else if (icon.previewUrl) {
        return (
          <img
            src={icon.previewUrl}
            alt={icon.name}
            className={`${size} object-contain`}
            style={{ 
              filter: color !== "#000000" ? `hue-rotate(${getHueRotation(color)}deg)` : undefined,
              backgroundColor: 'transparent'
            }}
            onError={(e) => {
              // Replace image with placeholder on error
              const target = e.target as HTMLImageElement;
              const placeholder = document.createElement('div');
              placeholder.className = `${size} flex items-center justify-center text-muted-foreground bg-muted/20 rounded border-2 border-dashed border-muted-foreground/30`;
              placeholder.innerHTML = `<svg class="w-6 h-6" style="color: ${color !== "#000000" ? color : 'currentColor'}" fill="currentColor" viewBox="0 0 24 24"><path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="m21 15l-5-5L5 21"/></svg>`;
              target.parentNode?.replaceChild(placeholder, target);
            }}
          />
        );
      } else {
        // No preview available, return placeholder directly
        return (
          <div className={`${size} flex items-center justify-center text-muted-foreground bg-muted/20 rounded border-2 border-dashed border-muted-foreground/30`}>
            <Image 
              className="w-6 h-6" 
              style={{ color: color !== "#000000" ? color : undefined }}
            />
          </div>
        );
      }
      break;

    case 'placeholder':
      // Show a generic placeholder icon
      return (
        <div className={`${size} flex items-center justify-center text-muted-foreground bg-muted/20 rounded border-2 border-dashed border-muted-foreground/30`}>
          <Image 
            className="w-6 h-6" 
            style={{ color: color !== "#000000" ? color : undefined }}
          />
        </div>
      );

    case 'text-fallback':
    default:
      // Final fallback: show icon name as text
      return (
        <div className={`${size} flex items-center justify-center text-xs text-muted-foreground bg-muted/10 rounded border border-muted-foreground/20 p-1`}>
          <span className="text-center leading-tight font-medium">
            {icon.name}
          </span>
        </div>
      );
  }

  // This should never be reached, but provide one more safety net
  return (
    <div className={`${size} flex items-center justify-center text-xs text-muted-foreground bg-muted/10 rounded border border-muted-foreground/20 p-1`}>
      <span className="text-center leading-tight font-medium">
        {icon.name}
      </span>
    </div>
  );
};

// Helper function for PNG color approximation (rough hue rotation)
const getHueRotation = (hexColor: string): number => {
  // Simple hue calculation for basic color matching on PNG fallback
  const hex = hexColor.replace('#', '');
  const r = parseInt(hex.substr(0, 2), 16);
  const g = parseInt(hex.substr(2, 2), 16);
  const b = parseInt(hex.substr(4, 2), 16);
  
  const max = Math.max(r, g, b);
  const min = Math.min(r, g, b);
  const diff = max - min;
  
  if (diff === 0) return 0;
  
  let hue = 0;
  if (max === r) hue = ((g - b) / diff) % 6;
  else if (max === g) hue = (b - r) / diff + 2;
  else hue = (r - g) / diff + 4;
  
  return Math.round(hue * 60);
};

// IconCard component that properly uses useIconEntitlement hook
interface IconCardProps {
  icon: IconRecord;
  color: string;
  onToggleFavorite: (id: string) => void;
  isFavorite: boolean;
  onCopyIcon: (icon: IconRecord, color: string, toast: any) => void;
  onDownloadIcon: (icon: IconRecord, color: string, toast: any) => void;
  toast: any;
}

const IconCard: React.FC<IconCardProps> = ({ 
  icon, 
  color, 
  onToggleFavorite, 
  isFavorite, 
  onCopyIcon, 
  onDownloadIcon, 
  toast 
}) => {
  const ent = useIconEntitlement(icon.id);

  return (
    <div className="group relative rounded-xl border border-border bg-card p-3" data-testid={`card-icon-${icon.id}`}>
      <div className="aspect-square flex items-center justify-center rounded-lg bg-white border border-border">
        <IconRenderer 
          icon={icon} 
          color={color} 
          size="w-16 h-16 md:w-20 md:h-20" 
        />
      </div>

      <div className="mt-3 flex items-center justify-between">
        <div>
          <div className="text-sm font-medium" data-testid={`text-icon-name-${icon.id}`}>{icon.name}</div>
          <div className="text-xs text-muted-foreground">{STYLES.find((s) => s.key === icon.style)?.label}</div>
        </div>

        <button
          className={`p-2 rounded-lg border ${isFavorite ? "border-pink-500/40 text-pink-400 bg-pink-500/10" : "border-border text-muted-foreground hover:bg-muted"}`}
          onClick={() => onToggleFavorite(icon.id)}
          title={isFavorite ? "Remove favorite" : "Save to favorites"}
          data-testid={`button-favorite-${icon.id}`}
        >
          {isFavorite ? <Heart className="w-4 h-4" /> : <HeartOff className="w-4 h-4" />}
        </button>
      </div>

      {/* Enhanced actions with proper licensing */}
      <div className="mt-2 space-y-2">
        {/* Top row: Copy and Download */}
        <div className="grid grid-cols-2 gap-2">
          <button
            className="inline-flex items-center justify-center gap-2 px-3 py-2 text-sm rounded-lg border border-border hover:bg-muted"
            onClick={() => onCopyIcon(icon, color, toast)}
            data-testid={`button-copy-${icon.id}`}
          >
            <Copy className="w-4 h-4" /> Copy SVG
          </button>
          <button
            className="inline-flex items-center justify-center gap-2 px-3 py-2 text-sm rounded-lg border border-border hover:bg-muted"
            onClick={() => onDownloadIcon(icon, color, toast)}
            data-testid={`button-download-${icon.id}`}
          >
            <Download className="w-4 h-4" /> Download
          </button>
        </div>

        {/* Attribution notice for non-Pro users */}
        {ent.requiresAttribution && (
          <div className="text-xs text-muted-foreground text-center">
            <AttributionNotice 
              text={buildAttribution(icon.name).text}
              html={buildAttribution(icon.name).html}
              copyText={buildAttribution(icon.name).copyText}
            />
          </div>
        )}

        {/* Licensed status indicator for Pro users */}
        {ent.licensed && !ent.requiresAttribution && (
          <div className="text-xs text-green-600 text-center font-medium">
            ✅ Licensed - No attribution required
          </div>
        )}
      </div>
    </div>
  );
};

function IconsContent() {
  const [mode, setMode] = useState<Mode>("browse");
  const [query, setQuery] = useState("");
  const [style, setStyle] = useState<StyleKey | "">("");
  const [color, setColor] = useState<string>("#000000"); // default black for preview
  const { favorites, ready, toggleFavorite, isFavorite } = useFavorites();

  const [genPrompt, setGenPrompt] = useState("");
  const [genStyle, setGenStyle] = useState<StyleKey>("outlined");
  const [genSvg, setGenSvg] = useState<string | null>(null);

  const { toast } = useToast();

  // SuperNova's helper functions for icon API management
  const useIconsFromApi = () => {
    return useQuery({
      queryKey: ['/api/icons/list'],
      enabled: mode === "browse",
    });
  };

  // Helper function to lazy-load SVG content from svgUrl
  const fetchSvgOnce = async (svgUrl: string): Promise<string> => {
    try {
      const response = await fetch(svgUrl);
      if (!response.ok) {
        throw new Error(`Failed to fetch SVG: ${response.status}`);
      }
      return await response.text();
    } catch (error) {
      console.error('Failed to fetch SVG:', error);
      throw error;
    }
  };

  // Helper to get SVG content (inline or from URL) with proper error handling
  const getSvgContent = async (icon: IconRecord): Promise<string> => {
    if (icon.svg) {
      return icon.svg;
    }
    if (icon.svgUrl) {
      return await fetchSvgOnce(icon.svgUrl);
    }
    throw new Error('No SVG content available');
  };

  // Enhanced copy handler that supports async SVG fetching
  const handleCopyIcon = async (icon: IconRecord, color: string, toast: any) => {
    try {
      const svgContent = await getSvgContent(icon);
      const safeColored = sanitizeSvgClient(applySvgColor(svgContent, { primary: color, secondary: color, accent: color }));
      await copyToClipboard(safeColored);
      toast({ title: "Copied!", description: "SVG copied to clipboard" });
    } catch (error) {
      console.error('Failed to copy SVG:', error);
      toast({ 
        variant: "destructive", 
        title: "Copy Failed", 
        description: "Could not copy SVG. Please try again." 
      });
    }
  };

  // Enhanced download handler that supports async SVG fetching
  const handleDownloadIcon = async (icon: IconRecord, color: string, toast: any) => {
    try {
      const svgContent = await getSvgContent(icon);
      const safeColored = sanitizeSvgClient(applySvgColor(svgContent, { primary: color, secondary: color, accent: color }));
      downloadTextFile(`${icon.name}.svg`, safeColored);
      toast({ title: "Downloaded!", description: `${icon.name}.svg saved` });
    } catch (error) {
      console.error('Failed to download SVG:', error);
      toast({ 
        variant: "destructive", 
        title: "Download Failed", 
        description: "Could not download SVG. Please try again." 
      });
    }
  };

  // Use the new helper function instead of direct useQuery
  const { data: iconsData, isLoading: isLoadingIcons } = useIconsFromApi();

  // Generate icon mutation using React Query
  const generateIconMutation = useMutation({
    mutationFn: (input: { prompt: string; style: StyleKey }) => 
      apiRequest<{ svg: string }>('POST', '/api/icons/generate', input),
    onSuccess: (data) => {
      setGenSvg(data.svg);
      toast({
        title: "Icon Generated",
        description: "Your custom icon has been generated successfully!",
      });
    },
    onError: (error: any) => {
      console.error('Icon generation error:', error);
      toast({
        variant: "destructive",
        title: "Generation Failed",
        description: error?.message || "Failed to generate icon. Please try again.",
      });
    },
  });

  // Hook automatically handles loading favorites from cloud or localStorage

  // SuperNova's improved icon data processing - no star fallback
  const allIcons = useMemo(() => {
    const storageIcons = iconsData?.icons || [];
    
    // Convert object storage icons to match SuperNova's IconRecord format
    const convertedStorageIcons: IconRecord[] = storageIcons.map((icon: any, index: number) => ({
      id: `storage-${icon.id || `item-${index}`}`, // Ensure unique ID
      name: icon.name || 'Untitled Icon',
      style: (icon.style as StyleKey) || 'outlined',
      // SuperNova's approach: use svg OR svgUrl, never both
      svg: icon.svg, // Inline SVG for small files
      svgUrl: icon.svgUrl, // URL endpoint for large files
      tags: icon.tags || [],
      previewUrl: icon.previewUrl, // PNG fallback if both SVG options fail
    }));

    return [...MOCK_ICONS, ...convertedStorageIcons];
  }, [iconsData]);

  const data = useMemo(() => {
    const q = query.trim().toLowerCase();
    return allIcons.filter((i) => {
      // Check for Guest Icon blocklist - filter out permanently
      const rawId = stripStoragePrefix(i.id);
      if (HIDE_ICON_IDS.has(rawId) || (HIDE_ICON_NAMES.has(i.name) && i.style === "outlined")) {
        return false; // Hide Guest Icon permanently
      }
      
      if (style && i.style !== style) return false;
      if (!q) return true;
      return i.name.toLowerCase().includes(q) || i.tags.some((t) => t.includes(q));
    });
  }, [allIcons, query, style]);

  async function handleToggleFavorite(id: string) {
    await toggleFavorite(id); // hook updates state (cloud or local)
  }

  function renderStyleChips(current: string | "", onPick: (k: StyleKey | "") => void, compact = false) {
    return (
      <div className={`flex flex-wrap gap-2 ${compact ? "" : "mt-3"}`}>
        <button
          className={`px-3 py-1.5 rounded-full text-sm border ${current === "" ? "bg-primary text-white border-primary/20" : "border-border text-muted-foreground hover:bg-muted"}`}
          onClick={() => onPick("")}
          data-testid="filter-all"
        >
          All
        </button>
        {STYLES.map((s) => (
          <button
            key={s.key}
            className={`px-3 py-1.5 rounded-full text-sm border ${current === s.key ? "bg-primary text-white border-primary/20" : "border-border text-muted-foreground hover:bg-muted"}`}
            onClick={() => onPick(s.key)}
            data-testid={`filter-${s.key}`}
          >
            {s.label}
          </button>
        ))}
      </div>
    );
  }

  function generateIcon() {
    if (!genPrompt.trim()) return;
    
    // Clear previous result and trigger mutation
    setGenSvg(null);
    generateIconMutation.mutate({ 
      prompt: genPrompt.trim(), 
      style: genStyle 
    });
  }

  return (
    <div className="space-y-6">
      <div className="flex flex-wrap items-center justify-between gap-3">
        <h1 className="text-2xl font-bold" data-testid="text-page-title">Icons</h1>

        <div className="flex items-center gap-3">
          <SignInChip />
          <div className="inline-flex rounded-xl overflow-hidden border border-border">
            <button
              className={`px-4 py-2 text-sm ${mode === "browse" ? "bg-primary text-white" : "text-muted-foreground hover:bg-muted"}`}
              onClick={() => setMode("browse")}
              data-testid="button-mode-browse"
            >
              <Layers className="inline-block w-4 h-4 mr-2" />
              Browse
            </button>
            <button
              className={`px-4 py-2 text-sm ${mode === "generate" ? "bg-primary text-white" : "text-muted-foreground hover:bg-muted"}`}
              onClick={() => setMode("generate")}
              data-testid="button-mode-generate"
            >
              <Wand2 className="inline-block w-4 h-4 mr-2" />
              Generate
            </button>
          </div>
        </div>
      </div>

      {mode === "browse" ? (
        <>
          {/* Search + style row */}
          <div className="flex flex-wrap items-center gap-3">
            <div className="flex items-center gap-2 flex-1 min-w-[260px]">
              <div className="relative flex-1">
                <Search className="w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground" />
                <input
                  className="w-full pl-9 pr-3 py-2 rounded-lg bg-card border border-border text-foreground placeholder:text-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary/50"
                  placeholder="Search icons (e.g., rocket, chart, heart)…"
                  value={query}
                  onChange={(e) => setQuery(e.target.value)}
                  data-testid="input-search"
                />
              </div>
              <label className="flex items-center gap-2 text-sm text-muted-foreground px-3 py-2 bg-card rounded-lg border border-border">
                <Palette className="w-4 h-4" />
                <input
                  type="color"
                  className="h-6 w-6 bg-transparent cursor-pointer"
                  value={color}
                  onChange={(e) => setColor(e.target.value)}
                  title="Preview color"
                  data-testid="input-color"
                />
              </label>
            </div>
          </div>

          {renderStyleChips(style, (k) => setStyle(k))}

          {/* Grid */}
          <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 xl:grid-cols-6 gap-4 mt-4">
            {data.map((icon) => (
              <IconCard 
                key={icon.id} 
                icon={icon} 
                color={color} 
                onToggleFavorite={handleToggleFavorite}
                isFavorite={isFavorite(icon.id)}
                onCopyIcon={handleCopyIcon}
                onDownloadIcon={handleDownloadIcon}
                toast={toast}
              />
            ))}
          </div>

          {/* Favorites strip */}
          <div className="mt-8">
            <h2 className="text-lg font-semibold mb-3" data-testid="text-favorites-title">Favorites</h2>
            {favorites.length === 0 ? (
              <p className="text-muted-foreground" data-testid="text-no-favorites">No favorites yet. Tap the heart on any icon to save it here.</p>
            ) : (
              <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-5 xl:grid-cols-8 gap-3">
                {favorites.map((id) => {
                  const icon = allIcons.find((i) => i.id === id);
                  if (!icon) return null;
                  return (
                    <div key={id} className="rounded-xl border border-border bg-card p-3" data-testid={`card-favorite-${id}`}>
                      <div className="aspect-square flex items-center justify-center rounded-lg bg-white border border-border">
                        <IconRenderer 
                          icon={icon} 
                          color={color} 
                          size="w-12 h-12" 
                        />
                      </div>
                      <div className="mt-2 text-xs text-muted-foreground truncate">{icon.name}</div>
                      <div className="mt-2 grid grid-cols-2 gap-2">
                        <button
                          className="inline-flex items-center justify-center gap-1.5 px-2 py-1.5 text-xs rounded border border-border hover:bg-muted"
                          onClick={() => handleCopyIcon(icon, color, toast)}
                          data-testid={`button-favorite-copy-${id}`}
                        >
                          <Copy className="w-3.5 h-3.5" /> Copy
                        </button>
                        <button
                          className="inline-flex items-center justify-center gap-1.5 px-2 py-1.5 text-xs rounded border border-border hover:bg-muted"
                          onClick={() => handleDownloadIcon(icon, color, toast)}
                          data-testid={`button-favorite-download-${id}`}
                        >
                          <Download className="w-3.5 h-3.5" /> SVG
                        </button>
                      </div>
                      <button
                        className="mt-2 w-full inline-flex items-center justify-center gap-1.5 px-2 py-1.5 text-xs rounded border border-pink-500/30 text-pink-300 hover:bg-pink-500/10"
                        onClick={() => handleToggleFavorite(id)}
                        data-testid={`button-favorite-remove-${id}`}
                      >
                        <Heart className="w-3.5 h-3.5" /> Remove
                      </button>
                    </div>
                  );
                })}
              </div>
            )}
          </div>
        </>
      ) : (
        // === Generate Mode ===
        <div className="grid md:grid-cols-2 gap-6">
          <div className="rounded-2xl border border-border bg-card p-5">
            <h2 className="text-lg font-semibold" data-testid="text-generate-title">Generate a Unique Icon</h2>
            <p className="text-muted-foreground text-sm mt-1">
              Describe the icon you need. Output will be pure SVG so you can recolor it easily in PPT/Keynote/Slides.
            </p>

            <label className="block mt-4 text-sm text-foreground">Prompt</label>
            <textarea
              className="mt-1 w-full min-h-[120px] rounded-lg bg-muted/50 border border-border p-3 text-foreground placeholder:text-muted-foreground focus:outline-none focus:ring-2 focus:ring-primary/50"
              placeholder="e.g., 'A flat rocket icon pointing upward with a small window, minimal detail'"
              value={genPrompt}
              onChange={(e) => setGenPrompt(e.target.value)}
              data-testid="textarea-prompt"
            />

            <label className="block mt-4 text-sm text-foreground">Style</label>
            {renderStyleChips(genStyle, (k) => setGenStyle((k || "outlined") as StyleKey), true)}

            <div className="mt-4 flex items-center gap-3">
              <label className="flex items-center gap-2 text-sm text-muted-foreground px-3 py-2 bg-card rounded-lg border border-border">
                <Palette className="w-4 h-4" />
                <input
                  type="color"
                  className="h-6 w-6 bg-transparent cursor-pointer"
                  value={color}
                  onChange={(e) => setColor(e.target.value)}
                  title="Preview color"
                  data-testid="input-generate-color"
                />
              </label>

              <button
                className="inline-flex items-center gap-2 px-4 py-2 rounded-lg bg-primary hover:bg-primary/90 text-white disabled:opacity-60"
                onClick={generateIcon}
                disabled={generateIconMutation.isPending || !genPrompt.trim()}
                data-testid="button-generate"
              >
                <Sparkles className="w-4 h-4" />
                {generateIconMutation.isPending ? "Generating…" : "Generate SVG"}
              </button>
            </div>

          </div>

          <div className="rounded-2xl border border-border bg-card p-5">
            <h2 className="text-lg font-semibold" data-testid="text-preview-title">Preview & Export</h2>
            <div className="mt-4 aspect-square rounded-xl border border-border bg-white flex items-center justify-center" data-testid="preview-container">
              {genSvg ? (
                (() => {
                  try {
                    const coloredSvg = applySvgColor(genSvg, { primary: color, secondary: color, accent: color });
                    return (
                      <div
                        className="w-24 h-24"
                        dangerouslySetInnerHTML={{ __html: coloredSvg }}
                      />
                    );
                  } catch (error) {
                    console.error('Failed to apply color to generated SVG:', error);
                    // Fallback to original SVG without color
                    try {
                      return (
                        <div
                          className="w-24 h-24"
                          dangerouslySetInnerHTML={{ __html: genSvg }}
                        />
                      );
                    } catch (fallbackError) {
                      console.error('Failed to render original generated SVG:', fallbackError);
                      // Final fallback: show placeholder
                      return (
                        <div className="w-24 h-24 flex items-center justify-center text-muted-foreground bg-muted/20 rounded border-2 border-dashed border-muted-foreground/30">
                          <Image className="w-8 h-8" />
                        </div>
                      );
                    }
                  }
                })()
              ) : (
                <div className="text-muted-foreground text-sm">No icon generated yet.</div>
              )}
            </div>

            <div className="mt-4 grid grid-cols-2 gap-3">
              <button
                className="inline-flex items-center justify-center gap-2 px-3 py-2 text-sm rounded-lg border border-border hover:bg-muted disabled:opacity-50"
                onClick={() => {
                  if (genSvg) {
                    try {
                      const safeColored = sanitizeSvgClient(applySvgColor(genSvg, { primary: color, secondary: color, accent: color }));
                      copyToClipboard(safeColored);
                    } catch (error) {
                      console.error('SVG sanitization failed:', error);
                    }
                  }
                }}
                disabled={!genSvg}
                data-testid="button-copy-generated"
              >
                <Copy className="w-4 h-4" /> Copy SVG
              </button>
              <button
                className="inline-flex items-center justify-center gap-2 px-3 py-2 text-sm rounded-lg border border-border hover:bg-muted disabled:opacity-50"
                onClick={() => {
                  if (genSvg) {
                    try {
                      const safeColored = sanitizeSvgClient(applySvgColor(genSvg, { primary: color, secondary: color, accent: color }));
                      downloadTextFile(`generated-icon.svg`, safeColored);
                    } catch (error) {
                      console.error('SVG sanitization failed:', error);
                    }
                  }
                }}
                disabled={!genSvg}
                data-testid="button-download-generated"
              >
                <Download className="w-4 h-4" /> Download SVG
              </button>
            </div>

            <p className="mt-3 text-xs text-muted-foreground">
              Tip: Paste SVGs directly into PowerPoint (Insert → Pictures → This Device) or drag in — then change
              fill/stroke colors natively.
            </p>
          </div>
        </div>
      )}
    </div>
  );
}

export default function IconsPage() {
  return (
    <DashboardTemplatePage title="Icons">
      <IconsContent />
    </DashboardTemplatePage>
  );
}