import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { Plus } from 'lucide-react';

export default function WebsiteTemplates() {
  const [selectedCategory, setSelectedCategory] = useState('All Templates');

  const categories = [
    'All Templates', 'Business', 'Portfolio', 'E-commerce', 'Restaurant', 'Agency'
  ];

  const templates = [
    {
      name: 'Business Pro',
      description: 'Perfect for consulting firms and professional services',
      category: 'Business',
      gradient: 'from-slate-100 to-slate-200'
    },
    {
      name: 'Creative Portfolio',
      description: 'Showcase your work with this modern portfolio design',
      category: 'Portfolio',
      gradient: 'from-purple-100 to-blue-100'
    },
    {
      name: 'Shop Starter',
      description: 'Complete e-commerce solution with cart and checkout',
      category: 'E-commerce',
      gradient: 'from-green-100 to-blue-100'
    },
    {
      name: 'Bistro Deluxe',
      description: 'Beautiful restaurant template with menu and reservations',
      category: 'Restaurant',
      gradient: 'from-orange-100 to-red-100'
    },
    {
      name: 'Agency Pro',
      description: 'Modern agency template with team and services showcase',
      category: 'Agency',
      gradient: 'from-indigo-100 to-purple-100'
    },
    {
      name: 'Launch Page',
      description: 'High-converting landing page for product launches',
      category: 'Landing',
      gradient: 'from-cyan-100 to-blue-100'
    }
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-3xl font-bold text-dark">Website Templates</h2>
          <p className="text-muted-foreground mt-2">Professional website templates to launch your business quickly</p>
        </div>
        <Button 
          className="bg-accent hover:bg-accent/90 text-white font-bold" 
          data-testid="button-use-template"
          onClick={() => {
            // Track website creation
            try {
              const currentCount = parseInt(localStorage.getItem('websiteCount') || '0', 10);
              localStorage.setItem('websiteCount', (currentCount + 1).toString());
            } catch (error) {
              console.warn('Failed to track website creation:', error);
            }
          }}
        >
          <Plus className="mr-2 h-4 w-4" />
          Use Template
        </Button>
      </div>

      {/* Template Categories */}
      <div className="flex space-x-4 overflow-x-auto pb-2">
        {categories.map((category) => (
          <button
            key={category}
            onClick={() => setSelectedCategory(category)}
            className={`px-4 py-2 rounded-lg whitespace-nowrap transition-colors ${
              selectedCategory === category
                ? 'bg-primary hover:bg-primary/90 text-white font-bold'
                : 'text-muted-foreground hover:text-foreground'
            }`}
            data-testid={`button-category-${category.toLowerCase().replace(/\s+/g, '-')}`}
          >
            {category}
          </button>
        ))}
      </div>

      {/* Template Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
        {templates.map((template, index) => (
          <div
            key={index}
            className="group bg-card border border-border rounded-lg overflow-hidden hover:shadow-lg transition-shadow"
            data-testid={`template-${index}`}
          >
            {/* Template Preview */}
            <div className={`aspect-[4/3] bg-gradient-to-br ${template.gradient} relative overflow-hidden`}>
              <div className="absolute inset-4 bg-white rounded shadow-sm">
                <div className="h-8 bg-primary/10 border-b border-border/20"></div>
                <div className="p-4 space-y-2">
                  <div className="h-16 bg-primary/20 rounded"></div>
                  <div className="space-y-1">
                    <div className="h-2 bg-gray-200 rounded w-3/4"></div>
                    <div className="h-2 bg-gray-200 rounded w-1/2"></div>
                  </div>
                  <div className="grid grid-cols-3 gap-2">
                    <div className="h-12 bg-gray-100 rounded"></div>
                    <div className="h-12 bg-gray-100 rounded"></div>
                    <div className="h-12 bg-gray-100 rounded"></div>
                  </div>
                </div>
              </div>
              <div className="absolute inset-0 bg-black/0 group-hover:bg-black/5 transition-colors flex items-center justify-center opacity-0 group-hover:opacity-100">
                <Button className="bg-primary hover:bg-primary/90 text-primary-foreground">
                  Preview
                </Button>
              </div>
            </div>
            <div className="p-6">
              <h3 className="text-lg font-semibold text-foreground mb-2">{template.name}</h3>
              <p className="text-sm text-muted-foreground mb-4">{template.description}</p>
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">{template.category}</span>
                <button className="text-primary font-medium text-sm hover:underline">
                  Use Template
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Your Templates */}
      <Card>
        <div className="p-6 border-b border-border">
          <h3 className="text-xl font-semibold text-dark">Your Websites</h3>
          <p className="text-muted-foreground mt-1">Manage your created websites</p>
        </div>
        <CardContent className="p-6">
          <div className="text-center py-8">
            <p className="text-muted-foreground">No websites created yet. Choose a template above to get started!</p>
            <Button className="mt-4" data-testid="button-create-first-website">
              Create Your First Website
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
