import React, { useState, useEffect } from 'react';
import { withPro } from '@/lib/withPro';
import { useParams, useLocation } from 'wouter';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Separator } from '@/components/ui/separator';
import { ResizableHandle, ResizablePanel, ResizablePanelGroup } from '@/components/ui/resizable';
import { Badge } from '@/components/ui/badge';
import { toast } from '@/hooks/use-toast';
import { 
  ArrowLeft, 
  Save, 
  Eye, 
  Download, 
  Upload, 
  RotateCcw, 
  Undo2, 
  Redo2,
  Monitor,
  Tablet,
  Smartphone,
  ZoomIn,
  ZoomOut,
  FolderOpen,
  BookOpen
} from 'lucide-react';

// Import builder components
import ColorCustomizer from '@/components/builder/ColorCustomizer';
import FontCustomizer from '@/components/builder/FontCustomizer';
import ContentEditor from '@/components/builder/ContentEditor';
import LayoutCustomizer from '@/components/builder/LayoutCustomizer';
import SectionManager from '@/components/builder/SectionManager';
import TemplatePreview from '@/components/builder/TemplatePreview';
import SaveTemplateDialog from '@/components/builder/SaveTemplateDialog';
import TemplateLibrary from '@/components/builder/TemplateLibrary';
import ExportDialog from '@/components/builder/ExportDialog';

import { apiRequest } from '@/lib/queryClient';
import type { WebsiteTemplate, UserTemplateCustomization } from '@shared/schema';

type DeviceMode = 'desktop' | 'tablet' | 'mobile';

interface TemplateCustomization {
  colors: {
    primary: string;
    secondary: string;
    accent: string;
    background: string;
    text: string;
  };
  fonts: {
    heading: string;
    body: string;
    sizes: {
      h1: string;
      h2: string;
      h3: string;
      body: string;
    };
  };
  layout: {
    spacing: string;
    maxWidth: string;
    gridColumns: number;
  };
  content: {
    sections: Array<{
      id: string;
      type: string;
      content: any;
      order: number;
    }>;
  };
}

const defaultCustomization: TemplateCustomization = {
  colors: {
    primary: '#00C851',
    secondary: '#427AB9',
    accent: '#FF6F00',
    background: '#FFFFFF',
    text: '#2B2B2D'
  },
  fonts: {
    heading: 'Inter',
    body: 'Inter',
    sizes: {
      h1: '2.5rem',
      h2: '2rem',
      h3: '1.5rem',
      body: '1rem'
    }
  },
  layout: {
    spacing: 'normal',
    maxWidth: '1200px',
    gridColumns: 12
  },
  content: {
    sections: []
  }
};

function TemplateBuilder() {
  const { templateId } = useParams();
  const [, setLocation] = useLocation();
  const queryClient = useQueryClient();
  
  const [deviceMode, setDeviceMode] = useState<DeviceMode>('desktop');
  const [zoomLevel, setZoomLevel] = useState(100);
  const [customization, setCustomization] = useState<TemplateCustomization>(defaultCustomization);
  const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false);
  const [undoStack, setUndoStack] = useState<TemplateCustomization[]>([defaultCustomization]);
  const [redoStack, setRedoStack] = useState<TemplateCustomization[]>([]);
  const [undoIndex, setUndoIndex] = useState(0);
  const [loadedTemplateName, setLoadedTemplateName] = useState<string | null>(null);
  const [isLoadingTemplate, setIsLoadingTemplate] = useState(false);

  // Fetch template data
  const { data: template, isLoading: templateLoading } = useQuery({
    queryKey: ['/api/website-templates', templateId],
    enabled: !!templateId
  });

  // Auto-save mutation
  const autoSaveMutation = useMutation({
    mutationFn: async (customizationData: TemplateCustomization) => {
      return apiRequest(
        'PUT',
        `/api/website-templates/${templateId}/customization`,
        {
          configurations: customizationData,
          contentOverrides: customizationData.content,
          styleOverrides: {
            colors: customizationData.colors,
            fonts: customizationData.fonts,
            layout: customizationData.layout
          }
        }
      );
    },
    onSuccess: () => {
      setHasUnsavedChanges(false);
      queryClient.invalidateQueries({ queryKey: ['/api/website-templates', templateId] });
    }
  });

  // Handle customization changes with history tracking
  const handleCustomizationChange = (newCustomization: Partial<TemplateCustomization>) => {
    const updatedCustomization = { ...customization, ...newCustomization };
    
    // Add to undo stack
    const newUndoStack = undoStack.slice(0, undoIndex + 1);
    newUndoStack.push(updatedCustomization);
    
    setUndoStack(newUndoStack);
    setUndoIndex(newUndoStack.length - 1);
    setRedoStack([]);
    setCustomization(updatedCustomization);
    setHasUnsavedChanges(true);
  };

  // Undo functionality
  const handleUndo = () => {
    if (undoIndex > 0) {
      const newIndex = undoIndex - 1;
      const prevCustomization = undoStack[newIndex];
      
      setRedoStack([customization, ...redoStack]);
      setCustomization(prevCustomization);
      setUndoIndex(newIndex);
      setHasUnsavedChanges(true);
    }
  };

  // Redo functionality
  const handleRedo = () => {
    if (redoStack.length > 0) {
      const nextCustomization = redoStack[0];
      const newRedoStack = redoStack.slice(1);
      
      setUndoStack([...undoStack, customization]);
      setCustomization(nextCustomization);
      setUndoIndex(undoStack.length);
      setRedoStack(newRedoStack);
      setHasUnsavedChanges(true);
    }
  };

  // Auto-save every 3 seconds when there are unsaved changes
  useEffect(() => {
    if (hasUnsavedChanges) {
      const timer = setTimeout(() => {
        autoSaveMutation.mutate(customization);
      }, 3000);

      return () => clearTimeout(timer);
    }
  }, [customization, hasUnsavedChanges]);

  // Save manually
  const handleSave = async () => {
    try {
      await autoSaveMutation.mutateAsync(customization);
      toast({
        title: "Success",
        description: "Template customization saved successfully!"
      });
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to save customization. Please try again.",
        variant: "destructive"
      });
    }
  };

  // Reset to defaults
  const handleReset = () => {
    handleCustomizationChange(defaultCustomization);
    toast({
      title: "Reset",
      description: "Template reset to default settings"
    });
  };

  // Template loading from library
  const handleLoadTemplate = async (savedTemplate: UserTemplateCustomization) => {
    setIsLoadingTemplate(true);
    try {
      // Load the customization from the saved template
      const styleOverrides = savedTemplate.styleOverrides as any || {};
      const configurations = savedTemplate.configurations as any || {};
      const contentOverrides = savedTemplate.contentOverrides as any || {};
      
      const loadedCustomization = {
        colors: styleOverrides.colors || configurations.colors || defaultCustomization.colors,
        fonts: styleOverrides.fonts || configurations.fonts || defaultCustomization.fonts,
        layout: styleOverrides.layout || configurations.layout || defaultCustomization.layout,
        content: contentOverrides || configurations.content || defaultCustomization.content,
      };

      // Reset undo/redo stack with new customization
      setUndoStack([loadedCustomization]);
      setRedoStack([]);
      setUndoIndex(0);
      setCustomization(loadedCustomization);
      setHasUnsavedChanges(false);
      setLoadedTemplateName(savedTemplate.customizationName || null);

      toast({
        title: "Template Loaded",
        description: `Successfully loaded "${savedTemplate.customizationName}".`
      });
    } catch (error) {
      console.error('Failed to load template:', error);
      toast({
        title: "Load Failed",
        description: "Failed to load template. Please try again.",
        variant: "destructive"
      });
    } finally {
      setIsLoadingTemplate(false);
    }
  };

  // Handle template duplication from library
  const handleDuplicateTemplate = (duplicatedTemplate: UserTemplateCustomization) => {
    toast({
      title: "Template Duplicated",
      description: `"${duplicatedTemplate.customizationName}" has been created and can be found in your template library.`
    });
  };

  // Handle successful template save
  const handleSaveComplete = (savedTemplate: UserTemplateCustomization) => {
    setLoadedTemplateName(savedTemplate.customizationName || null);
    setHasUnsavedChanges(false);
  };

  // Export/Preview functions
  const handlePreview = () => {
    const previewUrl = `/template-preview/${templateId}`;
    window.open(previewUrl, '_blank');
  };

  if (templateLoading) {
    return (
      <div className="h-screen flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin w-8 h-8 border-2 border-primary border-t-transparent rounded-full mx-auto mb-4"></div>
          <p>Loading template...</p>
        </div>
      </div>
    );
  }

  if (!template) {
    return (
      <div className="h-screen flex items-center justify-center">
        <div className="text-center">
          <h2 className="text-2xl font-bold mb-2">Template not found</h2>
          <p className="text-muted-foreground mb-4">The template you're looking for doesn't exist.</p>
          <Button onClick={() => setLocation('/website-templates')}>
            Back to Templates
          </Button>
        </div>
      </div>
    );
  }

  return (
    <div className="h-screen bg-background">
      {/* Header */}
      <div className="h-16 border-b bg-card flex items-center justify-between px-6">
        <div className="flex items-center space-x-4">
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setLocation('/website-templates')}
            data-testid="button-back"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back
          </Button>
          <Separator orientation="vertical" className="h-6" />
          <div>
            <h1 className="text-lg font-semibold">{(template as WebsiteTemplate)?.name || 'Template'}</h1>
            <p className="text-sm text-muted-foreground">
              Template Builder
              {loadedTemplateName && (
                <span className="ml-2 text-primary font-medium">
                  • Based on "{loadedTemplateName}"
                </span>
              )}
            </p>
          </div>
          <div className="flex items-center gap-2">
            {hasUnsavedChanges && (
              <Badge variant="secondary">
                Unsaved Changes
              </Badge>
            )}
            {isLoadingTemplate && (
              <Badge variant="outline">
                Loading Template...
              </Badge>
            )}
          </div>
        </div>

        <div className="flex items-center space-x-2">
          {/* Undo/Redo */}
          <Button
            variant="ghost"
            size="sm"
            onClick={handleUndo}
            disabled={undoIndex <= 0}
            data-testid="button-undo"
          >
            <Undo2 className="w-4 h-4" />
          </Button>
          <Button
            variant="ghost"
            size="sm"
            onClick={handleRedo}
            disabled={redoStack.length === 0}
            data-testid="button-redo"
          >
            <Redo2 className="w-4 h-4" />
          </Button>

          <Separator orientation="vertical" className="h-6" />

          {/* Device mode controls moved to TemplatePreview component */}

          {/* Template Management Actions */}
          <TemplateLibrary
            onLoadTemplate={handleLoadTemplate}
            onDuplicateTemplate={handleDuplicateTemplate}
          >
            <Button variant="outline" size="sm" data-testid="button-load-templates">
              <FolderOpen className="w-4 h-4 mr-2" />
              Load Template
            </Button>
          </TemplateLibrary>

          <SaveTemplateDialog
            template={template as WebsiteTemplate}
            customization={customization}
            onSaveComplete={handleSaveComplete}
          >
            <Button variant="outline" size="sm" data-testid="button-save-template">
              <BookOpen className="w-4 h-4 mr-2" />
              Save Template
            </Button>
          </SaveTemplateDialog>

          {/* Actions */}
          <Button variant="outline" size="sm" onClick={handleReset} data-testid="button-reset">
            <RotateCcw className="w-4 h-4 mr-2" />
            Reset
          </Button>
          <Button variant="outline" size="sm" onClick={handlePreview} data-testid="button-preview">
            <Eye className="w-4 h-4 mr-2" />
            Preview
          </Button>
          
          <ExportDialog
            template={template as WebsiteTemplate}
            customization={customization}
          >
            <Button variant="outline" size="sm" data-testid="button-export">
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>
          </ExportDialog>

          <Button 
            onClick={handleSave}
            disabled={autoSaveMutation.isPending}
            data-testid="button-save"
          >
            <Save className="w-4 h-4 mr-2" />
            {autoSaveMutation.isPending ? 'Saving...' : 'Save'}
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="h-[calc(100vh-4rem)]">
        <ResizablePanelGroup direction="horizontal">
          {/* Left Panel - Customization Controls */}
          <ResizablePanel defaultSize={30} minSize={25} maxSize={40}>
            <div className="h-full flex flex-col bg-card border-r">
              <div className="p-4 border-b">
                <h2 className="text-lg font-semibold">Customize</h2>
                <p className="text-sm text-muted-foreground">
                  Modify your template settings
                </p>
              </div>
              
              <div className="flex-1 overflow-auto">
                <Tabs defaultValue="colors" className="h-full">
                  <TabsList className="grid w-full grid-cols-5 m-4">
                    <TabsTrigger value="colors" data-testid="tab-colors">Colors</TabsTrigger>
                    <TabsTrigger value="fonts" data-testid="tab-fonts">Fonts</TabsTrigger>
                    <TabsTrigger value="content" data-testid="tab-content">Content</TabsTrigger>
                    <TabsTrigger value="layout" data-testid="tab-layout">Layout</TabsTrigger>
                    <TabsTrigger value="sections" data-testid="tab-sections">Sections</TabsTrigger>
                  </TabsList>

                  <div className="px-4 pb-4">
                    <TabsContent value="colors" className="space-y-4">
                      <ColorCustomizer
                        colors={customization.colors}
                        onChange={(colors: typeof customization.colors) => handleCustomizationChange({ colors })}
                      />
                    </TabsContent>

                    <TabsContent value="fonts" className="space-y-4">
                      <FontCustomizer
                        fonts={customization.fonts}
                        onChange={(fonts: typeof customization.fonts) => handleCustomizationChange({ fonts })}
                      />
                    </TabsContent>

                    <TabsContent value="content" className="space-y-4">
                      <ContentEditor
                        content={customization.content}
                        onChange={(content: typeof customization.content) => handleCustomizationChange({ content })}
                        template={template as WebsiteTemplate}
                      />
                    </TabsContent>

                    <TabsContent value="layout" className="space-y-4">
                      <LayoutCustomizer
                        layout={customization.layout}
                        onChange={(layout: typeof customization.layout) => handleCustomizationChange({ layout })}
                      />
                    </TabsContent>

                    <TabsContent value="sections" className="space-y-4">
                      <SectionManager
                        sections={customization.content.sections}
                        onChange={(sections: typeof customization.content.sections) => 
                          handleCustomizationChange({ 
                            content: { ...customization.content, sections } 
                          })
                        }
                        template={template as WebsiteTemplate}
                      />
                    </TabsContent>
                  </div>
                </Tabs>
              </div>
            </div>
          </ResizablePanel>

          <ResizableHandle />

          {/* Right Panel - Preview */}
          <ResizablePanel defaultSize={70}>
            <div className="h-full bg-muted/30 p-4">
              <div className="h-full flex flex-col">
                {/* Preview controls are now integrated into TemplatePreview component */}

                {/* Template Preview */}
                <div className="flex-1 overflow-auto">
                  <TemplatePreview
                    template={template as WebsiteTemplate}
                    customization={customization}
                    deviceMode={deviceMode}
                    zoomLevel={zoomLevel}
                    onDeviceModeChange={setDeviceMode}
                    onZoomChange={setZoomLevel}
                    onFullscreen={() => {
                      // Handle fullscreen mode
                      toast({
                        title: "Fullscreen Preview",
                        description: "Opening preview in new tab..."
                      });
                    }}
                    onExport={() => {
                      // Handle export completion
                      toast({
                        title: "Export Complete",
                        description: "Template has been exported successfully!"
                      });
                    }}
                  />
                </div>
              </div>
            </div>
          </ResizablePanel>
        </ResizablePanelGroup>
      </div>
    </div>
  );
}

export default withPro(TemplateBuilder, {
  feature: "website template builder",
  description: "Create and customize professional website templates with drag-and-drop editing"
});