import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { Plus, Heart } from 'lucide-react';

export default function SocialMedia() {
  const [selectedCategory, setSelectedCategory] = useState('All');

  const categories = [
    { id: 'instagram', name: 'Instagram Posts', icon: '📱', description: 'Square and story templates' },
    { id: 'facebook', name: 'Facebook Posts', icon: '📘', description: 'Feed and cover templates' },
    { id: 'twitter', name: 'Twitter Graphics', icon: '🐦', description: 'Header and post templates' },
    { id: 'linkedin', name: 'LinkedIn Posts', icon: '💼', description: 'Professional templates' }
  ];

  const templateCategories = ['All', 'Business', 'Lifestyle', 'Tech'];

  const templates = [
    { name: 'Business Pro', category: 'Instagram Post', gradient: 'from-primary to-accent' },
    { name: 'Warm Vibes', category: 'Instagram Story', gradient: 'from-orange-300 to-pink-400' },
    { name: 'Tech Forward', category: 'LinkedIn Post', gradient: 'from-blue-600 to-purple-700' },
    { name: 'Clean Slate', category: 'Facebook Post', gradient: 'from-gray-200 to-gray-400' }
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-3xl font-bold text-dark">Social Media Kits</h2>
          <p className="text-muted-foreground mt-2">Create cohesive social media templates and graphics for all platforms</p>
        </div>
        <Button 
          className="bg-accent hover:bg-accent/90 text-accent-foreground" 
          data-testid="button-create-new-kit"
          onClick={() => {
            // Track social kit creation
            try {
              const currentCount = parseInt(localStorage.getItem('socialKitCount') || '0', 10);
              localStorage.setItem('socialKitCount', (currentCount + 1).toString());
            } catch (error) {
              console.warn('Failed to track social kit creation:', error);
            }
          }}
        >
          <Plus className="mr-2 h-4 w-4" />
          Create New Kit
        </Button>
      </div>

      {/* Template Categories */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        {categories.map((category) => (
          <button
            key={category.id}
            className="bg-card border border-border rounded-lg p-6 text-center hover:border-primary hover:bg-primary/5 transition-colors"
            data-testid={`button-category-${category.id}`}
          >
            <div className="w-12 h-12 bg-blue-100 rounded-xl flex items-center justify-center mx-auto mb-4">
              <span className="text-blue-600 text-xl">{category.icon}</span>
            </div>
            <h3 className="font-semibold text-foreground mb-2">{category.name}</h3>
            <p className="text-sm text-muted-foreground">{category.description}</p>
          </button>
        ))}
      </div>

      {/* Template Gallery */}
      <Card>
        <div className="p-6 border-b border-border">
          <div className="flex items-center justify-between">
            <h3 className="text-xl font-semibold text-dark">Popular Templates</h3>
            <div className="flex space-x-2">
              {templateCategories.map((category) => (
                <button
                  key={category}
                  onClick={() => setSelectedCategory(category)}
                  className={`px-3 py-1 text-sm rounded-lg transition-colors ${
                    selectedCategory === category
                      ? 'bg-primary hover:bg-primary/90 text-white font-bold'
                      : 'text-muted-foreground hover:text-foreground'
                  }`}
                  data-testid={`button-filter-${category.toLowerCase()}`}
                >
                  {category}
                </button>
              ))}
            </div>
          </div>
        </div>
        <CardContent className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-6">
            {templates.map((template, index) => (
              <div key={index} className="group cursor-pointer" data-testid={`template-${index}`}>
                <div className={`aspect-square bg-gradient-to-br ${template.gradient} rounded-lg mb-3 relative overflow-hidden`}>
                  <div className="absolute inset-0 bg-black/0 group-hover:bg-black/10 transition-colors"></div>
                  <div className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity">
                    <button className="w-8 h-8 bg-white/90 rounded-full flex items-center justify-center">
                      <Heart className="h-4 w-4 text-red-500" />
                    </button>
                  </div>
                  <div className="absolute bottom-4 left-4 text-white">
                    <h4 className="font-bold text-lg">Your Brand</h4>
                    <p className="text-sm opacity-90">{template.name}</p>
                  </div>
                </div>
                <h4 className="font-semibold text-foreground">{template.name}</h4>
                <p className="text-sm text-muted-foreground">{template.category}</p>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Your Kits */}
      <Card>
        <div className="p-6 border-b border-border">
          <h3 className="text-xl font-semibold text-dark">Your Social Media Kits</h3>
        </div>
        <CardContent className="p-6">
          <div className="text-center py-8">
            <p className="text-muted-foreground">No social media kits created yet. Start by creating your first kit!</p>
            <Button className="mt-4" data-testid="button-create-first-kit">
              Create Your First Kit
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
