import React, { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { useAuth } from "@/contexts/AuthContext";
import {
  Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import {
  Select, SelectContent, SelectItem, SelectTrigger, SelectValue,
} from "@/components/ui/select";
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from "@/components/ui/breadcrumb";
import { Wand2, Copy, ThumbsUp, ThumbsDown, Loader2, RotateCcw, RefreshCw, ChevronRight, Download } from "lucide-react";
import { generate, type SloganRequest } from "@/services/ai/slogan";
import { useToast } from "@/hooks/use-toast";
import { useUserPlan } from '@/hooks/useUserPlan';
import { getTier } from '@/utils/tiers';
import { PaywallModal } from '@/components/PaywallModal';

const sloganFormSchema = z.object({
  brandName: z.string().min(1, "Brand name is required"),
  description: z.string().max(200, "Description must be under 200 characters").optional(),
  tone: z.enum(["Professional", "Playful", "Bold", "Minimal", "Luxury", "Friendly"]).optional(),
  industry: z.string().optional(),
  audience: z.string().optional(),
  maxResults: z.coerce.number().min(3).max(20).optional(),
});
type SloganFormValues = z.infer<typeof sloganFormSchema>;


export default function SloganPage() {
  const [slogans, setSlogans] = useState<string[]>([]);
  const [isGenerating, setIsGenerating] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [copiedIndex, setCopiedIndex] = useState<number | null>(null);
  const { toast } = useToast();
  const { currentUser } = useAuth();
  
  // Premium gating
  const { isPremium } = useUserPlan() ?? { isPremium: false };
  const tier = getTier(isPremium);
  const [showPaywall, setShowPaywall] = useState(false);

  const form = useForm<SloganFormValues>({
    resolver: zodResolver(sloganFormSchema),
    defaultValues: {
      brandName: "",
      description: "",
      tone: "Professional",
      industry: "",
      audience: "",
      maxResults: 8,
    },
  });

  async function runGenerate(values: SloganFormValues) {
    setIsGenerating(true);
    setError(null);
    try {
      const request: SloganRequest = {
        brandName: values.brandName.trim(),
        description: values.description?.trim() || undefined,
        tone: isPremium ? values.tone : undefined,
        industry: values.industry?.trim() || undefined,
        audience: values.audience?.trim() || undefined,
        maxResults: tier.maxResults,
      };
      const response = await generate(request);
      setSlogans(response.suggestions);
      console.log("Slogan generated:", {
        brandName: values.brandName,
        tone: values.tone,
        resultsCount: response.suggestions.length,
        isAuthenticated: !!currentUser,
      });
      if (!response.suggestions.length) {
        setError("Nova returned no results. Try again or tweak your description.");
      }
    } catch (err: any) {
      console.error("Failed to generate slogans:", err);
      setError(err?.message?.includes("API")
        ? "Slogan service is unavailable. Please try again."
        : "Couldn't reach Nova right now. Please try again.");
    } finally {
      setIsGenerating(false);
    }
  }
  
  const handleExportAll = () => {
    if (!isPremium) {
      setShowPaywall(true);
      return;
    }
    const text = slogans.join('\n');
    const blob = new Blob([text], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'slogans.txt';
    a.click();
    URL.revokeObjectURL(url);
  };

  const onSubmit = (values: SloganFormValues) => {
    setSlogans([]); // reset before new run
    runGenerate(values);
  };

  const handleClear = () => {
    form.reset();
    setSlogans([]);
    setError(null);
  };

  const handleCopy = async (slogan: string, index: number) => {
    try {
      await navigator.clipboard.writeText(slogan);
      setCopiedIndex(index);
      toast({ title: "Copied to clipboard", description: "Slogan copied successfully!" });
      setTimeout(() => setCopiedIndex(null), 2000);
    } catch {
      toast({ title: "Copy failed", description: "Please try again.", variant: "destructive" });
    }
  };

  const handleFeedback = (slogan: string, index: number, isPositive: boolean) => {
    console.log("Slogan feedback:", {
      slogan, index, feedback: isPositive ? "positive" : "negative", brandName: form.getValues("brandName"),
    });
    toast({ title: "Feedback recorded", description: `Thanks for your ${isPositive ? "positive" : "negative"} feedback!` });
  };

  const toneOptions = ["Professional", "Playful", "Bold", "Minimal", "Luxury", "Friendly"];

  return (
    <div className="p-4 md:p-6">
      {/* Paywall Modal */}
      <PaywallModal
        isOpen={showPaywall}
        onClose={() => setShowPaywall(false)}
        feature="advanced slogan generation"
        description="Get 10 results, tone filters, and export functionality"
      />
      
      {/* Breadcrumbs */}
      <Breadcrumb className="mb-[16px] -mt-1">
        <BreadcrumbList>
          <BreadcrumbItem>
            <BreadcrumbLink href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Business Development</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbPage className="text-[#FF8B00] font-semibold">Slogan Generator</BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>

      <div className="mb-8">
        <h1 className="text-2xl md:text-3xl font-semibold mb-2" data-testid="heading-slogan-generator">
          Slogan Generator
        </h1>
        <p className="text-muted-foreground" data-testid="text-description">
          Craft a memorable tagline that fits your brand's voice.
        </p>
      </div>

      <div className="grid lg:grid-cols-2 gap-8">
        <div className="space-y-6">
          <Form {...form}>
            <form
              onSubmit={form.handleSubmit(onSubmit)}
              className="space-y-6"
              onKeyDown={(e) => {
                if (e.key === "Enter" && !e.shiftKey) {
                  e.preventDefault();
                  form.handleSubmit(onSubmit)();
                }
              }}
            >
              <FormField
                control={form.control}
                name="brandName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Brand Name *</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter your brand name" {...field} data-testid="input-brand-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Brief Description</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="What does your brand do? (optional)"
                        className="resize-none"
                        {...field}
                        data-testid="textarea-description"
                      />
                    </FormControl>
                    <FormDescription>
                      {field.value ? `${field.value.length}/200` : "0/200"} characters
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              {tier.filters && (
                <FormField
                  control={form.control}
                  name="tone"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Tone</FormLabel>
                      <Select onValueChange={field.onChange} value={field.value}>
                        <FormControl>
                          <SelectTrigger data-testid="select-tone">
                            <SelectValue placeholder="Choose a tone (optional)" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {toneOptions.map((tone) => (
                            <SelectItem key={tone} value={tone} data-testid={`option-tone-${tone.toLowerCase()}`}>
                              {tone}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              )}
              <FormField
                control={form.control}
                name="industry"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Industry</FormLabel>
                    <FormControl>
                      <Input placeholder="e.g., Technology, Healthcare (optional)" {...field} data-testid="input-industry" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="audience"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Target Audience</FormLabel>
                    <FormControl>
                      <Input placeholder="e.g., Young professionals (optional)" {...field} data-testid="input-audience" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex gap-3 pt-4">
                <Button
                  type="submit"
                  disabled={isGenerating || !form.getValues("brandName").trim()}
                  className="flex-1 md:flex-initial"
                  data-testid="button-generate"
                >
                  {isGenerating ? (<><Loader2 className="w-4 h-4 mr-2 animate-spin" /> Generating...</>) : (<><Wand2 className="w-4 h-4 mr-2" /> Generate</>)}
                </Button>
                <Button type="button" variant="outline"
                  onClick={() => runGenerate(form.getValues())}
                  disabled={isGenerating || !form.getValues("brandName").trim()}
                  title="Regenerate with same inputs">
                  <RefreshCw className="w-4 h-4 mr-2" />
                  Regenerate
                </Button>
                <Button type="button" variant="secondary" onClick={handleClear} data-testid="button-clear">
                  <RotateCcw className="w-4 h-4 mr-2" />
                  Clear
                </Button>
              </div>
            </form>
          </Form>
        </div>

        {/* Results */}
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <h2 className="text-lg font-semibold" data-testid="heading-results">Results</h2>
            {slogans.length > 0 && (
              <Button
                onClick={handleExportAll}
                variant="outline"
                size="sm"
                disabled={!tier.exportAll}
                data-testid="button-export-all"
              >
                <Download className="h-4 w-4 mr-2" />
                Export All
              </Button>
            )}
          </div>
          {!isPremium && slogans.length > 0 && (
            <div className="text-xs text-gray-500 bg-gray-50 p-2 rounded">
              Showing {tier.maxResults} results. Unlock 10 results, tone filters, and export with Premium.
            </div>
          )}

          {isGenerating && (
            <div className="space-y-3" data-testid="loading-results">
              {[...Array(6)].map((_, i) => (
                <div key={i} className="h-14 bg-muted rounded-lg animate-pulse" />
              ))}
            </div>
          )}

          {error && (
            <div className="p-4 bg-destructive/10 border border-destructive/20 rounded-lg" data-testid="error-message">
              <p className="text-destructive text-sm">{error}</p>
            </div>
          )}

          {!isGenerating && !error && slogans.length === 0 && (
            <div className="p-8 text-center text-muted-foreground" data-testid="empty-state">
              <Wand2 className="w-12 h-12 mx-auto mb-4 opacity-50 transform scale-x-[-1]" />
              <p>Describe your brand and tone to generate a punchy tagline.</p>
            </div>
          )}

          {slogans.length > 0 && (
            <div className="space-y-3" data-testid="results-list">
              {slogans.map((slogan, index) => (
                <div key={index} className="relative group border rounded-lg p-4 hover:border-primary/50 transition-colors" data-testid={`result-item-${index}`}>
                  <p className="text-lg font-medium pr-8 mb-3" data-testid={`text-slogan-${index}`}>"{slogan}"</p>
                  <div className="flex items-center justify-between">
                    <div className="flex gap-2">
                      <Button size="sm" variant="outline" onClick={() => handleCopy(slogan, index)} className="text-xs" data-testid={`button-copy-${index}`}>
                        <Copy className="w-3 h-3 mr-1" /> {copiedIndex === index ? "Copied!" : "Copy"}
                      </Button>
                    </div>
                    <div className="flex gap-1">
                      <Button size="sm" variant="ghost" onClick={() => handleFeedback(slogan, index, true)} className="h-8 w-8 p-0 hover:text-green-600" data-testid={`button-thumbs-up-${index}`}>
                        <ThumbsUp className="w-3 h-3" />
                      </Button>
                      <Button size="sm" variant="ghost" onClick={() => handleFeedback(slogan, index, false)} className="h-8 w-8 p-0 hover:text-red-600" data-testid={`button-thumbs-down-${index}`}>
                        <ThumbsDown className="w-3 h-3" />
                      </Button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
}