import React, { useState, useEffect, useMemo } from 'react';
import { useLocation } from 'wouter';
import { useAuth } from '@/contexts/AuthContext';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { useToast } from '@/hooks/use-toast';
import { Loader2 } from 'lucide-react';
import { apiRequest } from '@/lib/queryClient';
import TemplatePage from '@/components/TemplatePage';

export default function SignupPage() {
  const { currentUser, signup, loading: authLoading } = useAuth();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  
  // Parse URL parameters
  const urlParams = useMemo(() => {
    if (typeof window === 'undefined') return { intent: 'free', billing: 'monthly' };
    const params = new URLSearchParams(window.location.search);
    return {
      intent: params.get('intent') || 'free',
      billing: params.get('billing') || 'monthly'
    };
  }, []);
  
  // Form state
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [displayName, setDisplayName] = useState('');
  const [loading, setLoading] = useState(false);

  // If user is already authenticated, handle based on intent
  useEffect(() => {
    if (currentUser) {
      if (urlParams.intent === 'pro') {
        // For pro intent, go directly to checkout (no modal)
        toast({
          title: "Account Ready!",
          description: "Taking you to checkout...",
        });
        handleProCheckout();
      } else {
        toast({
          title: "Welcome to IBrandBiz!",
          description: "Your account is ready. Start creating your brand!",
        });
        navigate('/dashboard');
      }
    }
  }, [currentUser, navigate, toast, urlParams.intent, urlParams.billing]);

  // Handle Pro subscription checkout
  const handleProCheckout = async () => {
    try {
      const response = await apiRequest('POST', '/api/billing/checkout', {
        priceType: urlParams.billing, // 'monthly' or 'yearly'
        successUrl: `${window.location.origin}/dashboard?welcome=pro`,
        cancelUrl: `${window.location.origin}/pricing?cancelled=true`
      });
      
      const { checkoutUrl } = await response.json();
      if (checkoutUrl) {
        // Redirect to Stripe Checkout
        window.location.href = checkoutUrl;
      } else {
        throw new Error('No checkout URL received');
      }
    } catch (error: any) {
      console.error('Checkout error:', error);
      
      // More detailed error handling
      let errorMessage = "Unable to set up payment. Please try again.";
      
      if (error.message?.includes('401')) {
        errorMessage = "Please sign in to continue with your Pro subscription.";
      } else if (error.message?.includes('404')) {
        errorMessage = "Account not found. Please create your account first.";
      } else if (error.message?.includes('500')) {
        errorMessage = "Payment system temporarily unavailable. Please try again later.";
      } else if (error.message?.includes('configuration')) {
        errorMessage = "Payment setup incomplete. Contact support for assistance.";
      }
      
      toast({
        title: "Payment Setup Failed",
        description: errorMessage,
        variant: "destructive"
      });
      
      // Don't redirect to dashboard immediately for Pro users - let them retry
      if (urlParams.intent !== 'pro') {
        navigate('/dashboard');
      }
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!email || !password || !displayName) {
      toast({
        title: "Error",
        description: "Please fill in all fields",
        variant: "destructive"
      });
      return;
    }

    if (password.length < 6) {
      toast({
        title: "Error", 
        description: "Password must be at least 6 characters",
        variant: "destructive"
      });
      return;
    }

    try {
      setLoading(true);
      await signup(email, password, displayName);
      
      if (urlParams.intent === 'pro') {
        toast({
          title: "Account Created!",
          description: "Setting up your Pro subscription...",
        });
        // useEffect will handle pro checkout redirect
      } else {
        toast({
          title: "Account Created!",
          description: "Welcome to IBrandBiz! Redirecting to your dashboard...",
        });
        // useEffect will handle dashboard redirect
      }
    } catch (error: any) {
      console.error('Signup error:', error);
      toast({
        title: "Signup Failed",
        description: error.message || "Failed to create account. Please try again.",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  if (authLoading) {
    return (
      <TemplatePage>
        <div className="flex items-center justify-center py-16">
          <Loader2 className="h-8 w-8 animate-spin" />
        </div>
      </TemplatePage>
    );
  }

  return (
    <TemplatePage>
      <div className="bg-gray-50 flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
        <div className="max-w-md w-full space-y-8">
          <div className="text-center">
            <h1 className="text-3xl font-bold text-gray-900">
              {urlParams.intent === 'pro' ? 'Upgrade to Pro' : 'Create Your Account'}
            </h1>
            <p className="mt-2 text-gray-600">
              {urlParams.intent === 'pro' 
                ? 'Sign up and we\'ll set up your Pro subscription'
                : 'Join thousands of businesses using IBrandBiz to build their brand'
              }
            </p>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Create Your Account</CardTitle>
              <CardDescription>
                {urlParams.intent === 'pro' 
                  ? `Step 1: Create account, Step 2: Setup ${urlParams.billing} Pro subscription`
                  : 'Join thousands of businesses using IBrandBiz to build their brand'
                }
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <Label htmlFor="displayName">Full Name</Label>
                  <Input
                    id="displayName"
                    type="text"
                    value={displayName}
                    onChange={(e) => setDisplayName(e.target.value)}
                    placeholder="Enter your full name"
                    required
                    data-testid="input-displayName"
                  />
                </div>
                
                <div>
                  <Label htmlFor="email">Email</Label>
                  <Input
                    id="email"
                    type="email"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    placeholder="Enter your email"
                    required
                    data-testid="input-email"
                  />
                </div>
                
                <div>
                  <Label htmlFor="password">Password</Label>
                  <Input
                    id="password"
                    type="password"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    placeholder="Create a password (min 6 characters)"
                    required
                    minLength={6}
                    data-testid="input-password"
                  />
                </div>
                
                <Button 
                  type="submit" 
                  className="w-full" 
                  disabled={loading}
                  data-testid="button-signup"
                >
                  {loading ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Creating Account...
                    </>
                  ) : (
                    urlParams.intent === 'pro' ? 'Create Account & Upgrade to Pro' : 'Create Account'
                  )}
                </Button>
              </form>
              
              <div className="mt-6 text-center">
                <p className="text-sm text-gray-500">
                  Already have an account?{' '}
                  <button 
                    onClick={() => navigate('/pricing')}
                    className="text-green-600 hover:text-green-500 font-medium"
                    data-testid="link-signin"
                  >
                    Sign in instead
                  </button>
                </p>
              </div>
              
              <div className="mt-4 text-center">
                <p className="text-xs text-gray-400">
                  By signing up, you agree to our{' '}
                  <a href="/terms" className="text-green-600 hover:text-green-500">Terms of Service</a>
                  {' '}and{' '}
                  <a href="/privacy" className="text-green-600 hover:text-green-500">Privacy Policy</a>
                </p>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </TemplatePage>
  );
}