import { useState, useEffect, useRef } from "react";
import { Bell, User, Sliders, Lock } from "lucide-react";

type SettingsDTO = {
  theme: "light" | "dark" | "system" | string;
  timezone: string;
  fontScale?: number; // Font scale multiplier (0.85-1.30)
  notifications: { 
    system: number | boolean; 
    billing: number | boolean; 
    project: number | boolean 
  };
};

const tabs = [
  { key: "profile", label: "Profile", icon: <User className="w-4 h-4" /> },
  { key: "notifications", label: "Notifications", icon: <Bell className="w-4 h-4" /> },
  { key: "preferences", label: "Preferences", icon: <Sliders className="w-4 h-4" /> },
  { key: "security", label: "Security", icon: <Lock className="w-4 h-4" /> },
];

export default function SettingsPage() {
  const [active, setActive] = useState("profile");
  const [data, setData] = useState<SettingsDTO | null>(null);
  const [saving, setSaving] = useState(false);
  const saveTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const rafRef = useRef<number | null>(null);
  const latestScaleRef = useRef<number>(1.0);

  useEffect(() => {
    // Load from localStorage immediately for pre-login users
    const localScale = localStorage.getItem('fontScale');
    if (localScale) {
      applyFontScale(parseFloat(localScale), true);
    }
    
    fetch("/api/user/settings", { credentials: "include" })
      .then(r => r.json())
      .then((d: SettingsDTO) => {
        setData(d);
        // Apply font scale on load (server overrides localStorage)
        if (d.fontScale !== undefined) {
          applyFontScale(d.fontScale, true);
        }
      })
      .catch(err => {
        console.error('Failed to load settings:', err);
        // Set default values on error
        setData({
          theme: 'system',
          timezone: '',
          fontScale: 1.0,
          notifications: { system: true, billing: true, project: true }
        });
      });
    
    // Cleanup on unmount
    return () => {
      if (rafRef.current) {
        cancelAnimationFrame(rafRef.current);
      }
      if (saveTimeoutRef.current) {
        clearTimeout(saveTimeoutRef.current);
      }
    };
  }, []);

  const save = async (patch: Partial<SettingsDTO>) => {
    if (!data) return;
    const next = {
      ...data,
      ...patch,
      notifications: { ...data.notifications, ...(patch as any).notifications }
    };
    setData(next);
    setSaving(true);
    
    try {
      const res = await fetch("/api/user/settings", {
        method: "POST", 
        credentials: "include",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(next),
      });
      const out = await res.json();
      setData(out);
    } catch (error) {
      console.error('Failed to save settings:', error);
    } finally {
      setSaving(false);
    }
  };

  // Debounced save for slider (prevent hammering API while dragging)
  const debouncedSave = (patch: Partial<SettingsDTO>) => {
    if (saveTimeoutRef.current) {
      clearTimeout(saveTimeoutRef.current);
    }
    saveTimeoutRef.current = setTimeout(() => {
      save(patch);
    }, 300); // 300ms debounce
  };

  const applyTheme = (theme: string) => {
    const html = document.documentElement;
    if (theme === 'dark') {
      html.classList.add('dark');
    } else if (theme === 'light') {
      html.classList.remove('dark');
    } else { // system
      const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
      if (prefersDark) {
        html.classList.add('dark');
      } else {
        html.classList.remove('dark');
      }
    }
    localStorage.setItem('theme', theme);
  };

  const applyFontScale = (scale: number, immediate = false) => {
    // Always store the latest value in the ref
    latestScaleRef.current = scale;
    
    const updateCSS = () => {
      // Read from ref to ensure we always get the latest value
      const currentScale = latestScaleRef.current;
      document.documentElement.style.setProperty('--font-scale', String(currentScale));
      localStorage.setItem('fontScale', currentScale.toString());
    };
    
    if (immediate) {
      // Immediate update for reset button and initial load
      updateCSS();
    } else {
      // Use RAF for smooth slider interactions
      if (rafRef.current) {
        cancelAnimationFrame(rafRef.current);
      }
      rafRef.current = requestAnimationFrame(updateCSS);
    }
  };

  if (!data) return <div className="p-8">Loading…</div>;

  return (
    <div className="max-w-3xl mx-auto py-8">
      <h1 className="text-2xl font-bold mb-6" data-testid="heading-settings">Settings</h1>

      <div className="flex gap-4 border-b border-gray-200 dark:border-gray-700">
        {tabs.map(t => (
          <button
            key={t.key}
            onClick={() => setActive(t.key)}
            className={`flex items-center gap-2 py-2 px-4 border-b-2 transition-colors ${
              active === t.key ? "border-black dark:border-white font-semibold" : "border-transparent hover:border-gray-300 dark:hover:border-gray-600"
            }`}
            data-testid={`tab-${t.key}`}
          >
            {t.icon} {t.label}
          </button>
        ))}
      </div>

      <div className="mt-6">
        {active === "profile" && (
          <div className="rounded-2xl border border-gray-200 dark:border-gray-700 p-6 bg-white dark:bg-gray-800">
            <h2 className="text-lg font-semibold mb-2">Profile</h2>
            <p className="text-gray-600 dark:text-gray-400 text-sm">
              Edit name, company, and avatar on your{" "}
              <a className="underline text-blue-600 hover:text-blue-800" href="/profile">
                Profile page
              </a>.
            </p>
          </div>
        )}

        {active === "notifications" && (
          <div className="rounded-2xl border border-gray-200 dark:border-gray-700 p-6 bg-white dark:bg-gray-800">
            <h2 className="text-lg font-semibold mb-4">Notification Preferences</h2>
            {(["system", "billing", "project"] as const).map((k) => (
              <label key={k} className="flex items-center gap-3 mb-3">
                <input
                  type="checkbox"
                  checked={!!(data.notifications && data.notifications[k])}
                  onChange={e => save({ notifications: { [k]: e.target.checked } } as any)}
                  className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 dark:focus:ring-blue-600 dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600"
                  data-testid={`checkbox-${k}-notifications`}
                />
                <span className="capitalize text-gray-900 dark:text-white">{k} notifications</span>
              </label>
            ))}
            {saving && <div className="text-xs text-gray-500 mt-2">Saving…</div>}
          </div>
        )}

        {active === "preferences" && (
          <div className="rounded-2xl border border-gray-200 dark:border-gray-700 p-6 bg-white dark:bg-gray-800">
            <h2 className="text-lg font-semibold mb-4">Preferences</h2>
            
            <div className="mb-4">
              <label className="block text-sm font-medium mb-2 text-gray-900 dark:text-white">Theme</label>
              <select
                className="border border-gray-300 dark:border-gray-600 rounded-xl px-3 py-2 bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                value={data.theme}
                onChange={e => {
                  const theme = e.target.value;
                  applyTheme(theme);
                  save({ theme });
                }}
                data-testid="select-theme"
              >
                <option value="system">System</option>
                <option value="light">Light</option>
                <option value="dark">Dark</option>
              </select>
            </div>

            <div className="mb-4">
              <label className="block text-sm font-medium mb-2 text-gray-900 dark:text-white">Time zone</label>
              <input
                className="border border-gray-300 dark:border-gray-600 rounded-xl px-3 py-2 w-full bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                value={data.timezone || ""}
                onChange={e => save({ timezone: e.target.value })}
                placeholder="e.g., America/Chicago"
                data-testid="input-timezone"
              />
            </div>

            <div className="mb-4">
              <div className="flex items-center justify-between mb-2">
                <label className="text-sm font-medium text-gray-900 dark:text-white">
                  Font Scale: {Math.round((data.fontScale || 1.0) * 100)}%
                </label>
                <button
                  type="button"
                  onClick={() => {
                    const resetValue = 1.0;
                    setData({ ...data, fontScale: resetValue });
                    applyFontScale(resetValue, true); // immediate update for button click
                    debouncedSave({ fontScale: resetValue });
                  }}
                  className="px-2 py-1 text-xs bg-gray-200 hover:bg-gray-300 text-gray-700 rounded transition-colors dark:bg-gray-700 dark:hover:bg-gray-600 dark:text-gray-300"
                  data-testid="button-reset-fontscale"
                >
                  Reset to default
                </button>
              </div>
              <input
                type="range"
                min="0.85"
                max="1.30"
                step="0.05"
                value={data.fontScale || 1.0}
                onChange={e => {
                  const fontScale = parseFloat(e.target.value);
                  // Update local state immediately for smooth slider movement
                  setData({ ...data, fontScale });
                  // Apply CSS immediately for instant visual feedback
                  applyFontScale(fontScale);
                  // Debounce the server save to avoid hammering the API
                  debouncedSave({ fontScale });
                }}
                className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer dark:bg-gray-700"
                data-testid="slider-fontscale"
              />
              <div className="flex justify-between text-xs text-gray-500 dark:text-gray-400 mt-1">
                <span>85%</span>
                <span>100%</span>
                <span>130%</span>
              </div>
              {/* Live preview */}
              <div className="mt-3 p-3 bg-gray-50 dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600">
                <p className="text-gray-700 dark:text-gray-300">
                  The quick brown fox jumps over the lazy dog
                </p>
              </div>
            </div>
            
            {saving && <div className="text-xs text-gray-500 mt-2">Saving…</div>}
          </div>
        )}

        {active === "security" && (
          <div className="rounded-2xl border border-gray-200 dark:border-gray-700 p-6 bg-white dark:bg-gray-800">
            <h2 className="text-lg font-semibold mb-2">Security</h2>
            <p className="text-gray-600 dark:text-gray-400 text-sm">
              Change your password on the{" "}
              <a className="underline text-blue-600 hover:text-blue-800" href="/profile">
                Profile
              </a>{" "}
              security tab.
            </p>
          </div>
        )}
      </div>
    </div>
  );
}