import { Link } from "wouter";
import TemplatePage from "@/components/TemplatePage";
import { servicesImage } from "@/assets";

export default function ServicesPage() {
  return (
    <TemplatePage
      bannerTitle="Make Your First Impression Your Best — from Name to Domain"
      bannerSubtitle="From naming to slogans and domains, we give you everything you need to launch and grow your brand with confidence."
      bannerImage={servicesImage}
    >
      <div className="text-slate-800">
        {/* Business Development Section */}
        <section className="mx-auto max-w-6xl px-6 py-16">
          <h2 className="text-2xl font-bold mb-2" style={{color: '#03222e'}}>Business Development</h2>
          <p className="text-slate-600 mb-8">Build your business foundation from the ground up</p>
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
            <ServiceCard
              eyebrow="Brand Name Wizard"
              title="Find a Name People Remember"
              description="AI-powered name ideas based on your keywords, industry, and vibe. Instantly generate unique, brandable names with matching domain suggestions."
              bullets={[
                "Smart prompts tuned for memorability",
                "Instant shortlist with save + copy",
                "Checks domain availability hints"
              ]}
              ctaLabel="Try the Wizard"
              href="/business-development/brand-name-wizard"
              tone="teal"
            />

            <ServiceCard
              eyebrow="Slogan Generator"
              title="Craft the Perfect Tagline"
              description="AI-powered slogans that capture your brand's essence. Generate memorable taglines that connect with your audience and make your brand stick."
              bullets={[
                "Smart prompts based on your industry",
                "Multiple creative variations",
                "Copy & save your favorites instantly"
              ]}
              ctaLabel="Generate Slogans"
              href="/business-development/slogan-generator"
              tone="navy"
            />

            <ServiceCard
              eyebrow="AI Logo Creator"
              title="Generate Custom Logos"
              description="Create unique, professional logos with AI in seconds. Customize colors, fonts, and layouts to match your brand perfectly."
              bullets={[
                "AI-powered logo generation",
                "Full customization in Logo Composer",
                "Export in multiple formats (SVG, PNG, PDF)"
              ]}
              ctaLabel="Create Your Logo"
              href="/business-development/ai-logo-creator"
              tone="purple"
            />
          </div>
        </section>

        {/* Web Services Section */}
        <section className="mx-auto max-w-6xl px-6 py-16">
          <h2 className="text-2xl font-bold mb-2" style={{color: '#03222e'}}>Web Services</h2>
          <p className="text-slate-600 mb-8">Essential solutions for your online presence</p>
          <div className="grid gap-6 md:grid-cols-2">
            <ServiceCard
              eyebrow="Domains"
              title="Secure the Perfect Domain"
              description="Search and secure your domain with our GoDaddy-powered lookup. Lock in your brand's home online before someone else does."
              bullets={[
                "Fast search with sensible suggestions",
                "Filters for extensions (.com, .io, .co)",
                "Straightforward checkout handoff"
              ]}
              ctaLabel="Search Domains"
              href="/web-services/domains"
              tone="orange"
            />

            <ServiceCard
              eyebrow="Web Hosting"
              title="Professional Website Hosting"
              description="Reliable hosting solutions starting at $2.19/month. From personal sites to business applications, we've got you covered."
              bullets={[
                "SSD storage with free SSL certificates",
                "Professional plan includes free domain",
                "Daily backups & priority support available"
              ]}
              ctaLabel="View Hosting Plans"
              href="/pricing/web-services"
              tone="purple"
            />
          </div>
        </section>

        {/* Why It Matters */}
        <section className="bg-slate-50 border-y">
          <div className="mx-auto max-w-6xl px-6 py-16">
            <h2 className="text-2xl font-bold mb-8 text-center" style={{color: '#03222e'}}>Why It Matters</h2>
            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
              <WhyCard
                title="Name First, Trust Follows"
                body="Your name is your first impression. The right one is easy to say, easy to remember, and easy to recommend."
              />
              <WhyCard
                title="Slogans Stick in Memory"
                body="A great slogan makes your brand unforgettable. The perfect tagline captures your essence in just a few words."
              />
              <WhyCard
                title="Domains = Findability"
                body="Your domain is where people expect to find you. Get the clean, on‑brand URL before it's taken."
              />
              <WhyCard
                title="Hosting = Reliability"
                body="Professional hosting ensures your site stays online 24/7. Fast loading speeds and secure infrastructure keep visitors happy."
              />
            </div>
          </div>
        </section>

        {/* Final CTA */}
        <section className="mx-auto max-w-6xl px-6 py-16 text-center">
          <h2 className="text-2xl sm:text-3xl font-bold" style={{color: '#03222e'}}>
            Ready to build something amazing?
          </h2>
          <p className="mt-3 text-slate-600 max-w-2xl mx-auto">
            Start with the Name Wizard, craft the perfect slogan, secure your ideal domain, and get reliable hosting — all in one flow.
          </p>
          <div className="mt-8 flex flex-wrap items-center justify-center gap-3">
            <CTA href="/business-development/brand-name-wizard" label="Try the Name Wizard" />
            <CTA href="/business-development/slogan-generator" variant="outline" label="Generate Slogans" />
            <CTA href="/web-services/domains" variant="ghost" label="Search Domains" />
            <CTA href="/pricing/web-services" variant="ghost" label="View Hosting Plans" />
          </div>
        </section>
      </div>
    </TemplatePage>
  );
}

/* ========================= Helper Components ========================= */

type CTAProps = {
  href: string;
  label: string;
  variant?: "solid" | "outline" | "ghost";
};

function CTA({ href, label, variant = "solid" }: CTAProps) {
  const base =
    "inline-flex items-center justify-center rounded-2xl px-5 py-3 text-sm font-semibold transition focus:outline-none focus-visible:ring-2 focus-visible:ring-offset-2";
  const styles: Record<"solid" | "outline" | "ghost", string> = {
    solid:
      "text-white shadow focus-visible:ring-2 focus-visible:ring-offset-2",
    outline:
      "border border-slate-300 hover:bg-slate-100 focus-visible:ring-2 focus-visible:ring-offset-2",
    ghost:
      "hover:bg-slate-100 focus-visible:ring-2 focus-visible:ring-offset-2",
  };
  const getVariantStyles = () => {
    switch (variant) {
      case "solid":
        return { backgroundColor: '#00C851' };
      case "outline":
        return { backgroundColor: 'white', color: '#03222e', border: '2px solid white' };
      case "ghost":
        return { backgroundColor: 'white', color: '#05445e' };
      default:
        return { backgroundColor: '#00C851' };
    }
  };

  return (
    <Link href={href}>
      <div 
        className={`${base} ${styles[variant]} cursor-pointer`} 
        style={getVariantStyles()}
        data-testid={`button-${label.toLowerCase().replace(/\s+/g, '-')}`}
      >
        {label}
      </div>
    </Link>
  );
}

type ServiceCardProps = {
  eyebrow: string;
  title: string;
  description: string;
  bullets: string[];
  ctaLabel: string;
  href: string;
  tone?: "teal" | "navy" | "orange" | "purple";
};

function ServiceCard({ eyebrow, title, description, bullets, ctaLabel, href, tone = "teal" }: ServiceCardProps) {
  const getToneGradient = (tone: "teal" | "navy" | "orange" | "purple") => {
    switch (tone) {
      case "teal":
        return 'linear-gradient(to right, #d4f1f4, #75e6da)';
      case "navy":
        return 'linear-gradient(to right, #d4f1f4, #5cccdc)';
      case "orange":
        return 'linear-gradient(to right, #fff7ed, #ff8800)';
      case "purple":
        return 'linear-gradient(to right, #f3e8ff, #a855f7)';
      default:
        return 'linear-gradient(to right, #d4f1f4, #75e6da)';
    }
  };

  return (
    <div className="group rounded-3xl border border-slate-200 overflow-hidden shadow-sm hover:shadow-md transition" data-testid={`card-service-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>
      <div className="h-2 w-full" style={{background: getToneGradient(tone)}} />
      <div className="p-6">
        <p className="text-xs uppercase tracking-wider text-slate-500 font-semibold" data-testid={`text-eyebrow-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>{eyebrow}</p>
        <h3 className="mt-2 text-xl font-bold" style={{color: '#03222e'}} data-testid={`text-title-${title.toLowerCase().replace(/\s+/g, '-')}`}>{title}</h3>
        <p className="mt-3 text-sm text-slate-600" data-testid={`text-description-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>{description}</p>
        <ul className="mt-4 space-y-2 text-sm text-slate-700">
          {bullets.map((b, i) => (
            <li key={i} className="flex gap-2" data-testid={`list-item-${eyebrow.toLowerCase().replace(/\s+/g, '-')}-${i}`}>
              <span 
                className="mt-1 inline-block h-2.5 w-2.5 rounded-full transition-colors group-hover:bg-slate-700" 
                style={{backgroundColor: '#189ab4'}}
              />
              <span>{b}</span>
            </li>
          ))}
        </ul>
        <div className="mt-6">
          <CTA href={href} label={ctaLabel} />
        </div>
      </div>
    </div>
  );
}

type WhyCardProps = { title: string; body: string };
function WhyCard({ title, body }: WhyCardProps) {
  return (
    <div className="rounded-3xl border border-slate-200 bg-white p-6 shadow-sm" data-testid={`card-why-${title.toLowerCase().replace(/\s+/g, '-')}`}>
      <h3 className="text-lg font-semibold" style={{color: '#03222e'}} data-testid={`text-why-title-${title.toLowerCase().replace(/\s+/g, '-')}`}>{title}</h3>
      <p className="mt-2 text-slate-600 text-sm" data-testid={`text-why-body-${title.toLowerCase().replace(/\s+/g, '-')}`}>{body}</p>
    </div>
  );
}
