import React, { useEffect, useMemo, useState } from "react";
import { Download, Eye, Search, Tag, ChevronRight } from "lucide-react";
import { Card, CardContent, CardFooter, CardHeader } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { BusinessPlanPreviewDialog } from "@/components/business-plan/BusinessPlanPreviewDialog";
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from "@/components/ui/breadcrumb";
import { Link } from "wouter";

// IBrandBiz palette
const IBB = {
  green: "#00CB51",
  orange: "#FF8B00",
  darkGreen: "#30342d",
  darkGray: "#231f20",
  white: "#ffffff",
};

// Types
export type StoragePaths = {
  docx?: string;
  preview?: string;
  thumb?: string;
};

export type TemplateDoc = {
  title: string;
  slug: string;
  category: string;
  tags?: string[];
  industries?: string[];
  sections?: string[];
  isMaster?: boolean;
  currentVersion?: string;
  storagePaths?: StoragePaths;
  viewerUrl?: string | null;
  previewPages?: string[];
  updatedAt?: number | string | null;
};

const API_BASE = "/api/bp-templates-firebase";

const btnBase = "px-4 py-2 rounded-lg text-sm font-medium transition-colors";
const chip = "px-2 py-1 rounded-full text-xs bg-gray-100 text-gray-700";

function classNames(...cls: (string | false | undefined)[]) {
  return cls.filter(Boolean).join(" ");
}

export default function BusinessPlanTemplatesPage() {
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [items, setItems] = useState<TemplateDoc[]>([]);
  const [q, setQ] = useState("");
  const [tag, setTag] = useState<string>("all");
  const [sort, setSort] = useState<"az" | "new">("az");

  useEffect(() => {
    let alive = true;
    (async () => {
      try {
        setLoading(true);
        const res = await fetch(`${API_BASE}/list`, { cache: "no-store" });
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data: TemplateDoc[] = await res.json();
        if (alive) setItems(data);
      } catch (e: any) {
        setError(e?.message || "Failed to load templates");
      } finally {
        setLoading(false);
      }
    })();
    return () => {
      alive = false;
    };
  }, []);

  // Add canonical tag
  useEffect(() => {
    const canonical = document.querySelector('link[rel="canonical"]');
    const canonicalUrl = `${window.location.origin}/business-assets/templates/business-plan`;
    
    if (canonical) {
      canonical.setAttribute('href', canonicalUrl);
    } else {
      const link = document.createElement('link');
      link.rel = 'canonical';
      link.href = canonicalUrl;
      document.head.appendChild(link);
    }
    
    return () => {
      const link = document.querySelector('link[rel="canonical"]');
      if (link && link.getAttribute('href') === canonicalUrl) {
        link.remove();
      }
    };
  }, []);

  const allTags = useMemo(() => {
    const s = new Set<string>();
    items.forEach((i) => {
      (i.tags || []).forEach((t) => s.add(t));
      (i.industries || []).forEach((t) => s.add(t));
    });
    return ["all", ...Array.from(s).sort()];
  }, [items]);

  const filtered = useMemo(() => {
    const needle = q.trim().toLowerCase();
    let out = items.filter((t) => {
      const inTag =
        tag === "all" ||
        (t.tags || []).includes(tag) ||
        (t.industries || []).includes(tag);
      if (!inTag) return false;
      if (!needle) return true;
      const hay = [
        t.title,
        t.slug,
        t.category,
        ...(t.tags || []),
        ...(t.industries || []),
      ]
        .filter(Boolean)
        .join(" ")
        .toLowerCase();
      return hay.includes(needle);
    });

    if (sort === "az") out = out.sort((a, b) => a.title.localeCompare(b.title));
    if (sort === "new")
      out = out.sort((a, b) => {
        const aTime = typeof a.updatedAt === "number" ? a.updatedAt : 0;
        const bTime = typeof b.updatedAt === "number" ? b.updatedAt : 0;
        return bTime - aTime;
      });

    return out;
  }, [items, q, tag, sort]);

  return (
    <DashboardTemplatePage title="Business Plan Templates">
      {loading ? (
        <div className="text-sm text-gray-500">Loading templates…</div>
      ) : error ? (
        <div className="text-sm text-red-600">Error: {error}</div>
      ) : (
        <>
          <Breadcrumb className="mb-[16px] -mt-1">
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink asChild>
                  <Link href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</Link>
                </BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator>
                <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
              </BreadcrumbSeparator>
              <BreadcrumbItem>
                <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Business Assets</BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator>
                <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
              </BreadcrumbSeparator>
              <BreadcrumbItem>
                <BreadcrumbPage className="text-[#FF8B00] font-semibold">Business Plan Templates</BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>

          <header className="mb-4 flex flex-col gap-3 md:flex-row md:items-center md:justify-between">
            <div className="text-sm text-gray-600">
              <b>{filtered.length}</b> of {items.length} templates
            </div>
            <div className="flex gap-2">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-gray-400" />
                <Input
                  placeholder="Search industries, tags…"
                  value={q}
                  onChange={(e) => setQ(e.target.value)}
                  className="w-56 pl-9"
                  data-testid="input-search-templates"
                />
              </div>
              <select
                value={tag}
                onChange={(e) => setTag(e.target.value)}
                className="rounded-lg border px-3 py-2"
                data-testid="select-tag"
              >
                {allTags.map((t) => (
                  <option key={t} value={t}>
                    {t}
                  </option>
                ))}
              </select>
              <select
                value={sort}
                onChange={(e) => setSort(e.target.value as any)}
                className="rounded-lg border px-3 py-2"
                data-testid="select-sort"
              >
                <option value="az">A–Z</option>
                <option value="new">Newest</option>
              </select>
            </div>
          </header>

          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
            {filtered.map((item) => (
              <TemplateCard key={item.slug} item={item} />
            ))}
          </div>
        </>
      )}
    </DashboardTemplatePage>
  );
}

function TemplateCard({ item }: { item: TemplateDoc }) {
  const [showPreview, setShowPreview] = useState(false);
  const [previewPages, setPreviewPages] = useState<string[]>([]);
  const [loadingPages, setLoadingPages] = useState(false);
  const [freshViewerUrl, setFreshViewerUrl] = useState<string | null>(null);
  
  const origin =
    typeof window !== "undefined"
      ? `${window.location.protocol}//${window.location.host}`
      : "";
  const docxAbs = item.storagePaths?.docx && item.storagePaths.docx.startsWith("http")
    ? item.storagePaths.docx
    : `${origin}${API_BASE}/${encodeURIComponent(item.slug)}/download`;

  const hasMultiPagePreview =
    item.previewPages && item.previewPages.length > 0;

  async function loadMultiPagePreviews() {
    if (loadingPages) return;
    setLoadingPages(true);
    setPreviewPages([]);
    
    try {
      const res = await fetch(`${API_BASE}/${encodeURIComponent(item.slug)}`, { cache: "no-store" });
      if (res.ok) {
        const data: TemplateDoc = await res.json();
        if (data.viewerUrl) {
          // Proxy the PDF through our server for same-origin embedding
          const proxied = `/api/preview/pdf?u=${encodeURIComponent(data.viewerUrl)}&filename=${encodeURIComponent(item.title || 'template.pdf')}`;
          setFreshViewerUrl(proxied);
        }
      }
    } catch (e) {
      console.error("Failed to fetch fresh viewer URL:", e);
    }
    
    setLoadingPages(false);
    setShowPreview(true);
  }

  return (
    <>
      <Card className="overflow-hidden rounded-2xl shadow-sm" data-testid={`card-template-${item.slug}`}>
        <div className="text-center pt-4 pb-2 px-4">
          <h3 className="text-[13px] font-bold">{item.title}</h3>
        </div>
        <a
          href={item.viewerUrl || '#'}
          target="_blank"
          rel="noreferrer"
          className="block aspect-[15/16] bg-white overflow-hidden flex items-center justify-center cursor-pointer hover:opacity-90 transition-opacity relative group"
          data-testid={`card-preview-${item.slug}`}
        >
          {item.storagePaths?.thumb || item.storagePaths?.preview ? (
            <>
              <img
                src={item.storagePaths?.thumb || item.storagePaths?.preview}
                alt={item.title}
                className="object-contain w-full h-full"
                loading="lazy"
                data-testid={`img-thumbnail-${item.slug}`}
              />
              <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                <Eye className="h-12 w-12 text-white" />
              </div>
            </>
          ) : (
            <div
              className="w-full h-full"
              style={{
                background: `linear-gradient(135deg, ${IBB.darkGray} 0%, ${IBB.darkGreen} 100%)`,
              }}
            />
          )}
        </a>

        <div className="text-xs text-center mt-1 font-bold" style={{ color: IBB.orange }}>
          (click the image to view PDF)
        </div>

        <CardContent className="space-y-1 pt-3 pb-2">
          {item.category && (
            <div className="text-xs">
              <span className="font-semibold text-gray-600">Category:</span>{' '}
              <span className="text-gray-700 capitalize">{item.category.replace('-', ' ')}</span>
            </div>
          )}
          {item.industries && item.industries.length > 0 && (
            <div className="text-xs">
              <span className="font-semibold text-gray-600">Industries:</span>{' '}
              <span className="text-gray-700">{item.industries.join(', ')}</span>
            </div>
          )}
        </CardContent>

        <CardFooter className="flex flex-wrap gap-2 pt-2">
          <a
            className={btnBase}
            style={{
              backgroundColor: IBB.orange,
              color: IBB.white,
              fontWeight: "bold",
            }}
            href={`${API_BASE}/${encodeURIComponent(item.slug)}/download`}
            download
            data-testid={`button-download-${item.slug}`}
          >
            Download DOCX
          </a>
          <a
            className={btnBase}
            style={{
              backgroundColor: IBB.green,
              color: IBB.white,
              fontWeight: "bold",
            }}
            href={`https://docs.google.com/gview?embedded=1&url=${encodeURIComponent(
              freshViewerUrl || item.viewerUrl || docxAbs
            )}`}
            target="_blank"
            rel="noreferrer"
            title="View in Google Docs Viewer (then File → Make a copy to edit)"
            data-testid={`button-google-docs-${item.slug}`}
          >
            Open in Google Docs
          </a>
        </CardFooter>
      </Card>

      {showPreview && (
        <BusinessPlanPreviewDialog
          open={showPreview}
          onOpenChange={setShowPreview}
          name={item.title}
          previewPages={previewPages}
          docxUrl={freshViewerUrl || item.viewerUrl || docxAbs}
        />
      )}
    </>
  );
}
