import React, { useEffect, useState } from "react";
import { useParams } from "wouter";
import { motion } from "framer-motion";
import { Download, ArrowLeft, Sparkles, Eye, Calendar, Tag } from "lucide-react";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent } from "@/components/ui/dialog";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";

const IBB = {
  green: "#00CB51",
  blue: "#274b9b",
  darkGray: "#231f20",
  darkGreen: "#30342d",
};

type StoragePaths = {
  docx?: string;
  preview?: string;
  thumb?: string;
};

type TemplateDoc = {
  title: string;
  slug: string;
  category: string;
  industries?: string[];
  isMaster?: boolean;
  currentVersion?: string;
  storagePaths?: StoragePaths;
  updatedAt?: number | string | null;
  sections?: string[];
  description?: string;
};

const API_BASE = "/api/bp-templates-firebase";

export default function BusinessPlanTemplateDetailsPage() {
  const { slug } = useParams<{ slug: string }>();
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [template, setTemplate] = useState<TemplateDoc | null>(null);
  const [previewOpen, setPreviewOpen] = useState(false);

  useEffect(() => {
    if (!slug) return;
    let alive = true;
    (async () => {
      try {
        setLoading(true);
        const res = await fetch(`${API_BASE}/${encodeURIComponent(slug)}`, { cache: "no-store" });
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data: TemplateDoc = await res.json();
        if (alive) setTemplate(data);
      } catch (e: any) {
        setError(e?.message || "Failed to load template");
      } finally {
        setLoading(false);
      }
    })();
    return () => {
      alive = false;
    };
  }, [slug]);

  const onDownload = () => {
    if (!slug) return;
    window.open(`${API_BASE}/${encodeURIComponent(slug)}/download`, "_blank");
  };

  if (loading) {
    return (
      <DashboardTemplatePage title="Template Details">
        <div className="max-w-5xl mx-auto">
          <div className="animate-pulse space-y-6">
            <div className="h-8 w-2/3 bg-gray-200 rounded" />
            <div className="h-64 bg-gray-200 rounded-xl" />
            <div className="h-32 bg-gray-200 rounded-xl" />
          </div>
        </div>
      </DashboardTemplatePage>
    );
  }

  if (error || !template) {
    return (
      <DashboardTemplatePage title="Template Not Found">
        <div className="max-w-5xl mx-auto">
          <div className="rounded-xl border border-red-200 bg-red-50 p-4 text-red-800">
            {error || "Template not found"}
          </div>
          <Button
            className="mt-4"
            variant="outline"
            onClick={() => window.location.href = "/business-assets/templates/business-plan"}
            data-testid="button-back-gallery"
          >
            <ArrowLeft className="mr-2 h-4 w-4" /> Back to Gallery
          </Button>
        </div>
      </DashboardTemplatePage>
    );
  }

  return (
    <DashboardTemplatePage title={template.title}>
      <div className="max-w-5xl mx-auto">
        {/* Back button */}
        <Button
          variant="ghost"
          className="mb-4"
          onClick={() => window.location.href = "/business-assets/templates/business-plan"}
          data-testid="button-back"
        >
          <ArrowLeft className="mr-2 h-4 w-4" /> Back to Templates
        </Button>

        {/* Hero section */}
        <motion.div
          initial={{ opacity: 0, y: 10 }}
          animate={{ opacity: 1, y: 0 }}
          className="mb-8"
        >
          <div className="flex items-start justify-between gap-4 mb-4">
            <div>
              <h1 className="text-3xl md:text-4xl font-bold text-gray-900 mb-2">
                {template.title}
              </h1>
              <div className="flex flex-wrap items-center gap-2">
                <Badge className="bg-[rgba(0,203,81,0.15)] text-[#00CB51] hover:bg-[rgba(0,203,81,0.25)]">
                  <Tag className="mr-1 h-3 w-3" />
                  {template.category || "General"}
                </Badge>
                {template.isMaster && (
                  <Badge className="bg-yellow-100 text-yellow-800">
                    Master Template
                  </Badge>
                )}
                {template.currentVersion && (
                  <Badge variant="secondary">
                    Version {template.currentVersion}
                  </Badge>
                )}
              </div>
            </div>
          </div>

          {template.description && (
            <p className="text-gray-600 text-lg mb-6">{template.description}</p>
          )}

          {/* Action buttons */}
          <div className="flex flex-wrap gap-3">
            <Button
              className="rounded-xl hover:bg-emerald-600"
              style={{ backgroundColor: IBB.green }}
              onClick={onDownload}
              size="lg"
              data-testid="button-download-main"
            >
              <Download className="mr-2 h-5 w-5" /> Download Template
            </Button>
            <Button
              variant="outline"
              className="rounded-xl"
              onClick={() => setPreviewOpen(true)}
              size="lg"
              data-testid="button-preview-main"
            >
              <Eye className="mr-2 h-5 w-5" /> Preview
            </Button>
          </div>
        </motion.div>

        {/* Preview image */}
        {template.storagePaths?.preview && (
          <Card className="mb-8 overflow-hidden">
            <CardContent className="p-0">
              <img
                src={template.storagePaths.preview}
                alt={`${template.title} preview`}
                className="w-full h-auto cursor-pointer hover:opacity-90 transition-opacity"
                onClick={() => setPreviewOpen(true)}
                data-testid="img-preview-main"
              />
            </CardContent>
          </Card>
        )}

        {/* Metadata */}
        <div className="grid md:grid-cols-2 gap-6 mb-8">
          {template.industries && template.industries.length > 0 && (
            <Card>
              <CardContent className="p-6">
                <h3 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                  <Tag className="h-4 w-4" />
                  Industries
                </h3>
                <div className="flex flex-wrap gap-2">
                  {template.industries.map((industry) => (
                    <Badge key={industry} variant="secondary" className="bg-gray-100 text-gray-800">
                      {industry}
                    </Badge>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}

          {template.updatedAt && (
            <Card>
              <CardContent className="p-6">
                <h3 className="font-semibold text-gray-900 mb-3 flex items-center gap-2">
                  <Calendar className="h-4 w-4" />
                  Last Updated
                </h3>
                <p className="text-gray-600">
                  {typeof template.updatedAt === "number"
                    ? new Date(template.updatedAt).toLocaleDateString()
                    : template.updatedAt || "N/A"}
                </p>
              </CardContent>
            </Card>
          )}
        </div>

        {/* Sections */}
        {template.sections && template.sections.length > 0 && (
          <Card className="mb-8">
            <CardContent className="p-6">
              <h3 className="font-semibold text-gray-900 mb-4 text-lg">
                Template Sections
              </h3>
              <ul className="space-y-2">
                {template.sections.map((section, idx) => (
                  <li key={idx} className="flex items-start gap-2 text-gray-700">
                    <span className="text-[#00CB51] mt-1">•</span>
                    <span>{section}</span>
                  </li>
                ))}
              </ul>
            </CardContent>
          </Card>
        )}

        {/* CTA to AI Builder */}
        <Card
          className="mb-8 overflow-hidden"
          style={{
            background: `linear-gradient(135deg, ${IBB.blue} 0%, ${IBB.green} 100%)`,
          }}
        >
          <CardContent className="p-8 text-white">
            <div className="flex items-start gap-4">
              <Sparkles className="h-8 w-8 flex-shrink-0 mt-1" />
              <div className="flex-1">
                <h3 className="text-xl font-bold mb-2">
                  Want AI to write your business plan for you?
                </h3>
                <p className="text-white/90 mb-4">
                  Skip the blank template and let our AI Business Plan Builder generate a complete, 
                  customized plan based on your business information. Smart, fast, professional.
                </p>
                <Button
                  variant="secondary"
                  className="bg-white text-gray-900 hover:bg-gray-100"
                  onClick={() => window.location.href = "/business-plan-builder"}
                  data-testid="button-ai-builder-cta"
                >
                  Try AI Business Plan Builder →
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Preview Modal */}
      <Dialog open={previewOpen} onOpenChange={setPreviewOpen}>
        <DialogContent className="max-w-5xl max-h-[90vh] overflow-hidden flex flex-col">
          <div className="flex-1 overflow-auto p-4">
            <h2 className="text-2xl font-bold mb-4">{template.title}</h2>
            {template.storagePaths?.preview ? (
              <div className="bg-white rounded-lg shadow-lg overflow-hidden">
                <img
                  src={template.storagePaths.preview}
                  alt={`${template.title} preview`}
                  className="w-full h-auto"
                  data-testid="img-preview-modal"
                />
              </div>
            ) : (
              <div className="flex items-center justify-center h-64 bg-gray-100 rounded-lg">
                <p className="text-gray-500">No preview available</p>
              </div>
            )}
            <div className="mt-6 flex gap-3">
              <Button
                className="rounded-xl"
                style={{ backgroundColor: IBB.green }}
                onClick={() => {
                  onDownload();
                  setPreviewOpen(false);
                }}
                data-testid="button-download-modal"
              >
                <Download className="mr-2 h-4 w-4" /> Download Template
              </Button>
              <Button
                variant="outline"
                className="rounded-xl"
                onClick={() => setPreviewOpen(false)}
                data-testid="button-close-modal"
              >
                Close
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </DashboardTemplatePage>
  );
}
