import { useState, useEffect } from "react";
import { useAuth } from "@/contexts/AuthContext";
import { Eye, EyeOff, Check, X, AlertCircle, Shield, Copy, Download, QrCode, Smartphone, Key } from "lucide-react";
import { updatePassword, reauthenticateWithCredential, EmailAuthProvider, linkWithCredential } from "firebase/auth";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";

// Password strength validation
const validatePasswordStrength = (password: string) => {
  const requirements = {
    length: password.length >= 8,
    number: /\d/.test(password),
    symbol: /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)
  };
  
  const score = Object.values(requirements).filter(Boolean).length;
  return { requirements, score, isStrong: score === 3 };
};

export function SecurityTab() {
  const { currentUser } = useAuth();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  
  // Password state
  const [current, setCurrent] = useState("");
  const [next, setNext] = useState("");
  const [confirm, setConfirm] = useState("");
  const [saving, setSaving] = useState(false);

  // TOTP state
  const [totpSetupData, setTotpSetupData] = useState<{qrCode: string; secret: string} | null>(null);
  const [totpVerifyToken, setTotpVerifyToken] = useState("");
  const [totpDisableToken, setTotpDisableToken] = useState("");
  const [showTotpSecret, setShowTotpSecret] = useState(false);
  const [backupCodes, setBackupCodes] = useState<string[]>([]);
  const [showBackupCodes, setShowBackupCodes] = useState(false);

  // TOTP Status Query
  const { data: totpStatus, isLoading: totpStatusLoading } = useQuery({
    queryKey: ['/api/security/totp/status'],
    enabled: !!currentUser,
  });

  // TOTP Setup Mutation
  const setupTotpMutation = useMutation({
    mutationFn: () => apiRequest('POST', '/api/security/totp/setup'),
    onSuccess: (data) => {
      setTotpSetupData(data);
      toast({
        title: "TOTP Setup Ready",
        description: "Scan the QR code with your authenticator app, then verify to enable 2FA.",
      });
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        title: "Setup Failed",
        description: error?.message || "Failed to setup two-factor authentication. Please try again.",
      });
    },
  });

  // TOTP Verify Mutation
  const verifyTotpMutation = useMutation({
    mutationFn: (token: string) => apiRequest('POST', '/api/security/totp/verify', { token }),
    onSuccess: (data) => {
      setBackupCodes(data.backupCodes);
      setShowBackupCodes(true);
      setTotpSetupData(null);
      setTotpVerifyToken("");
      queryClient.invalidateQueries({ queryKey: ['/api/security/totp/status'] });
      toast({
        title: "Two-Factor Authentication Enabled",
        description: "Save your backup codes in a secure location. You can use them if you lose access to your authenticator app.",
      });
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        title: "Verification Failed",
        description: error?.message || "Invalid TOTP code. Please try again.",
      });
    },
  });

  // TOTP Disable Mutation
  const disableTotpMutation = useMutation({
    mutationFn: (data: { totpToken?: string; currentPassword?: string }) => 
      apiRequest('POST', '/api/security/totp/disable', data),
    onSuccess: () => {
      setTotpDisableToken("");
      queryClient.invalidateQueries({ queryKey: ['/api/security/totp/status'] });
      toast({
        title: "Two-Factor Authentication Disabled",
        description: "2FA has been disabled for your account.",
      });
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        title: "Disable Failed",
        description: error?.message || "Failed to disable two-factor authentication. Please try again.",
      });
    },
  });

  // Function to revoke all sessions for enhanced security after password change
  const revokeOtherSessions = async () => {
    try {
      await apiRequest('POST', '/api/auth/revoke-sessions');
      console.log('✅ Successfully revoked refresh tokens for other sessions');
    } catch (error) {
      console.warn('⚠️ Failed to revoke other sessions:', error);
      // Note: We don't show this error to user as it's a secondary security measure
      // The password change itself was successful
    }
  };
  const [showCurrentPassword, setShowCurrentPassword] = useState(false);
  const [showNewPassword, setShowNewPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [passwordStrength, setPasswordStrength] = useState({ requirements: { length: false, number: false, symbol: false }, score: 0, isStrong: false });
  const [showPasswordMismatch, setShowPasswordMismatch] = useState(false);

  // Check if user is OAuth-only (no password provider)
  const isOAuthOnly = currentUser ? 
    !currentUser.providerData.some(provider => provider.providerId === 'password') : false;

  // Get OAuth provider names for display
  const oauthProviders = currentUser ? 
    currentUser.providerData
      .filter(provider => provider.providerId !== 'password')
      .map(provider => {
        switch (provider.providerId) {
          case 'google.com':
            return 'Google';
          case 'apple.com':
            return 'Apple';
          case 'facebook.com':
            return 'Facebook';
          case 'twitter.com':
            return 'Twitter';
          default:
            return provider.providerId;
        }
      }) : [];

  // Real-time password validation
  useEffect(() => {
    if (next) {
      setPasswordStrength(validatePasswordStrength(next));
    }
  }, [next]);

  // Check password mismatch in real-time
  useEffect(() => {
    if (confirm && next && confirm !== next) {
      setShowPasswordMismatch(true);
    } else {
      setShowPasswordMismatch(false);
    }
  }, [next, confirm]);

  // Form validation - different for OAuth vs email/password accounts
  const isFormValid = () => {
    if (isOAuthOnly) {
      // For OAuth-only accounts adding a password
      return (
        next.trim() !== "" &&
        confirm.trim() !== "" &&
        passwordStrength.isStrong &&
        next === confirm
      );
    } else {
      // For existing email/password accounts changing password
      return (
        current.trim() !== "" &&
        next.trim() !== "" &&
        confirm.trim() !== "" &&
        passwordStrength.isStrong &&
        next === confirm
      );
    }
  };

  const change = async () => {
    if (!currentUser || !isFormValid()) return;

    setSaving(true);
    try {
      if (isOAuthOnly) {
        // For OAuth-only accounts, link an email/password credential
        const credential = EmailAuthProvider.credential(currentUser.email!, next);
        await linkWithCredential(currentUser, credential);
        
        // Refresh the user's auth data to immediately reflect the new password provider
        await currentUser.reload();
        
        // Revoke other sessions for enhanced security
        await revokeOtherSessions();
        
        toast({
          title: "Password added successfully",
          description: "You can now sign in with your email and password. You may be asked to sign in again on other devices."
        });
        
        setNext("");
        setConfirm("");
        setPasswordStrength({ requirements: { length: false, number: false, symbol: false }, score: 0, isStrong: false });
        setShowPasswordMismatch(false);
      } else {
        // For existing email/password accounts, change the password
        // First, reauthenticate the user with their current password
        const credential = EmailAuthProvider.credential(currentUser.email!, current);
        await reauthenticateWithCredential(currentUser, credential);
        
        // Then update the password
        await updatePassword(currentUser, next);
        
        // Revoke other sessions for enhanced security
        await revokeOtherSessions();
        
        toast({
          title: "Password updated",
          description: "Password updated successfully. You may be asked to sign in again on other devices."
        });
        
        setCurrent("");
        setNext("");
        setConfirm("");
        setPasswordStrength({ requirements: { length: false, number: false, symbol: false }, score: 0, isStrong: false });
        setShowPasswordMismatch(false);
      }
    } catch (error: any) {
      console.error('Password operation error:', error);
      
      let errorMessage = isOAuthOnly ? "Failed to add password" : "Failed to update password";
      
      if (error.code === 'auth/wrong-password') {
        errorMessage = "That doesn't match your current password.";
      } else if (error.code === 'auth/weak-password') {
        errorMessage = "Try a stronger password (8+ chars incl. number & symbol).";
      } else if (error.code === 'auth/requires-recent-login') {
        errorMessage = "For your security, please sign in again and try again.";
      } else if (error.code === 'auth/email-already-in-use') {
        errorMessage = "This email is already associated with another account.";
      } else if (error.code === 'auth/credential-already-in-use') {
        errorMessage = "This email/password combination is already linked to your account.";
      }
      
      toast({
        variant: "destructive",
        title: isOAuthOnly ? "Add Password Failed" : "Password Change Failed",
        description: errorMessage
      });
    } finally {
      setSaving(false);
    }
  };

  const PasswordInput = ({ 
    label, 
    value, 
    onChange, 
    showPassword, 
    toggleShow,
    placeholder,
    showStrength = false,
    showMismatch = false,
    testId
  }: {
    label: string;
    value: string;
    onChange: (e: React.ChangeEvent<HTMLInputElement>) => void;
    showPassword: boolean;
    toggleShow: () => void;
    placeholder: string;
    showStrength?: boolean;
    showMismatch?: boolean;
    testId?: string;
  }) => (
    <div className="mb-4">
      <label className="block text-sm mb-1 font-medium">{label}</label>
      <div className="relative">
        <input 
          data-testid={testId}
          className={`w-full border rounded-xl px-3 py-2 pr-10 focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent ${
            showMismatch ? 'border-red-300 focus:ring-red-500' : ''
          }`}
          type={showPassword ? "text" : "password"} 
          value={value} 
          onChange={onChange}
          placeholder={placeholder}
        />
        <button
          type="button"
          onClick={toggleShow}
          className="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-400 hover:text-gray-600"
          data-testid={`${testId}-toggle`}
        >
          {showPassword ? (
            <EyeOff className="h-4 w-4" />
          ) : (
            <Eye className="h-4 w-4" />
          )}
        </button>
      </div>
      
      {/* Password strength indicator */}
      {showStrength && value && (
        <div className="mt-2 space-y-2">
          <div className="text-xs text-gray-600">Password requirements:</div>
          <div className="space-y-1">
            <div className={`flex items-center text-xs ${
              passwordStrength.requirements.length ? 'text-green-600' : 'text-gray-400'
            }`}>
              {passwordStrength.requirements.length ? (
                <Check className="h-3 w-3 mr-1" />
              ) : (
                <X className="h-3 w-3 mr-1" />
              )}
              At least 8 characters
            </div>
            <div className={`flex items-center text-xs ${
              passwordStrength.requirements.number ? 'text-green-600' : 'text-gray-400'
            }`}>
              {passwordStrength.requirements.number ? (
                <Check className="h-3 w-3 mr-1" />
              ) : (
                <X className="h-3 w-3 mr-1" />
              )}
              At least one number
            </div>
            <div className={`flex items-center text-xs ${
              passwordStrength.requirements.symbol ? 'text-green-600' : 'text-gray-400'
            }`}>
              {passwordStrength.requirements.symbol ? (
                <Check className="h-3 w-3 mr-1" />
              ) : (
                <X className="h-3 w-3 mr-1" />
              )}
              At least one special character
            </div>
          </div>
        </div>
      )}
      
      {/* Password mismatch warning */}
      {showMismatch && (
        <div className="mt-1 flex items-center text-xs text-red-600">
          <AlertCircle className="h-3 w-3 mr-1" />
          Passwords don't match
        </div>
      )}
    </div>
  );

  return (
    <div className="rounded-2xl border p-6">
      <h2 className="text-xl font-semibold mb-4">Security</h2>
      
      <div className="mb-6">
        <h3 className="font-medium mb-2">
          {isOAuthOnly ? 'Add Password' : 'Change Password'}
        </h3>
        <p className="text-sm text-gray-600 mb-4">
          {isOAuthOnly 
            ? `Your account uses ${oauthProviders.join(' and ')} sign-in. Add a password to enable email/password login as an additional option.`
            : 'Update your password to keep your account secure. Make sure to use a strong password.'
          }
        </p>
      </div>

      {/* Current account info for OAuth users */}
      {isOAuthOnly && (
        <div className="mb-4">
          <label className="block text-sm mb-1 font-medium">Email</label>
          <input 
            data-testid="input-email-display"
            className="w-full border rounded-xl px-3 py-2 bg-gray-50 text-gray-500"
            type="email" 
            value={currentUser?.email || ''}
            readOnly
            disabled
          />
        </div>
      )}

      {/* Current password field - only for existing email/password accounts */}
      {!isOAuthOnly && (
        <PasswordInput
          label="Current password"
          value={current}
          onChange={e => setCurrent(e.target.value)}
          showPassword={showCurrentPassword}
          toggleShow={() => setShowCurrentPassword(!showCurrentPassword)}
          placeholder="Enter your current password"
          testId="input-current-password"
        />
      )}

      <PasswordInput
        label={isOAuthOnly ? "Password" : "New password"}
        value={next}
        onChange={e => setNext(e.target.value)}
        showPassword={showNewPassword}
        toggleShow={() => setShowNewPassword(!showNewPassword)}
        placeholder={isOAuthOnly 
          ? "Create a password (8+ chars, number & symbol)"
          : "Enter your new password (8+ chars, number & symbol)"
        }
        showStrength={true}
        testId="input-new-password"
      />

      <PasswordInput
        label={isOAuthOnly ? "Confirm password" : "Confirm new password"}
        value={confirm}
        onChange={e => setConfirm(e.target.value)}
        showPassword={showConfirmPassword}
        toggleShow={() => setShowConfirmPassword(!showConfirmPassword)}
        placeholder={isOAuthOnly ? "Confirm your password" : "Confirm your new password"}
        showMismatch={showPasswordMismatch}
        testId="input-confirm-password"
      />

      <button 
        className={`rounded-xl px-4 py-2 transition-colors disabled:opacity-50 disabled:cursor-not-allowed ${
          isFormValid() && !saving
            ? 'bg-black text-white hover:bg-gray-800'
            : 'bg-gray-200 text-gray-500'
        }`}
        onClick={change}
        disabled={!isFormValid() || saving}
        data-testid={isOAuthOnly ? "button-add-password" : "button-update-password"}
      >
        {saving ? (
          <span className="flex items-center">
            <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
              <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
              <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
            </svg>
            {isOAuthOnly ? 'Adding...' : 'Updating...'}
          </span>
        ) : (
          isOAuthOnly ? "Add password" : "Update password"
        )}
      </button>

      <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
        <div className="flex">
          <div className="flex-shrink-0">
            <AlertCircle className="h-5 w-5 text-blue-400" />
          </div>
          <div className="ml-3">
            <h3 className="text-sm font-medium text-blue-800">
              {isOAuthOnly ? 'Adding a Password' : 'Password Security Tips'}
            </h3>
            <div className="mt-1 text-sm text-blue-700">
              {isOAuthOnly ? (
                <ul className="list-disc list-inside space-y-1">
                  <li>Adding a password doesn't replace your {oauthProviders.join(' or ')} sign-in</li>
                  <li>You'll be able to sign in with either method</li>
                  <li>Use a unique password you don't use elsewhere</li>
                  <li>This provides a backup sign-in method if needed</li>
                </ul>
              ) : (
                <ul className="list-disc list-inside space-y-1">
                  <li>Use a unique password you don't use elsewhere</li>
                  <li>Avoid using personal information like birthdays or names</li>
                  <li>Consider using a password manager for stronger security</li>
                  <li>Update your password regularly for best security</li>
                </ul>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* TOTP Two-Factor Authentication Section */}
      <div className="mt-8 rounded-2xl border p-6">
        <div className="flex items-center gap-2 mb-4">
          <Shield className="h-5 w-5 text-green-600" />
          <h3 className="text-lg font-semibold">Two-Factor Authentication</h3>
          {totpStatus?.enabled && (
            <span className="px-2 py-1 text-xs font-medium bg-green-100 text-green-800 rounded-full">
              Enabled
            </span>
          )}
        </div>
        
        <p className="text-sm text-gray-600 mb-6">
          Add an extra layer of security to your account by enabling two-factor authentication (2FA). 
          You'll need your phone or authenticator app to sign in.
        </p>

        {totpStatusLoading ? (
          <div className="flex items-center justify-center py-8">
            <div className="animate-spin h-6 w-6 border-2 border-primary border-t-transparent rounded-full"></div>
            <span className="ml-2 text-sm text-gray-600">Loading 2FA status...</span>
          </div>
        ) : !totpStatus?.enabled ? (
          // TOTP Setup Flow
          <div className="space-y-6">
            {!totpSetupData ? (
              // Initial setup button
              <div className="text-center py-4">
                <button
                  onClick={() => setupTotpMutation.mutate()}
                  disabled={setupTotpMutation.isPending}
                  className="inline-flex items-center gap-2 px-4 py-2 bg-black text-white rounded-xl hover:bg-gray-800 disabled:opacity-50 disabled:cursor-not-allowed"
                  data-testid="button-setup-totp"
                >
                  {setupTotpMutation.isPending ? (
                    <>
                      <div className="animate-spin h-4 w-4 border-2 border-white border-t-transparent rounded-full"></div>
                      Setting up...
                    </>
                  ) : (
                    <>
                      <Smartphone className="h-4 w-4" />
                      Enable Two-Factor Authentication
                    </>
                  )}
                </button>
              </div>
            ) : (
              // QR Code and verification step
              <div className="space-y-6">
                <div className="border rounded-lg p-6 bg-gray-50">
                  <h4 className="font-medium mb-4 flex items-center gap-2">
                    <QrCode className="h-4 w-4" />
                    Step 1: Scan QR Code
                  </h4>
                  
                  <div className="flex flex-col md:flex-row gap-6 items-start">
                    <div className="flex-shrink-0">
                      <img 
                        src={totpSetupData.qrCode} 
                        alt="TOTP QR Code" 
                        className="w-32 h-32 border rounded-lg"
                        data-testid="img-totp-qr"
                      />
                    </div>
                    
                    <div className="flex-1 min-w-0">
                      <p className="text-sm text-gray-600 mb-3">
                        Scan this QR code with your authenticator app (Google Authenticator, Authy, 1Password, etc.).
                      </p>
                      
                      <div className="mb-4">
                        <p className="text-xs text-gray-500 mb-1">Manual entry key:</p>
                        <div className="flex items-center gap-2">
                          <code className="px-2 py-1 bg-gray-100 text-xs font-mono rounded break-all">
                            {showTotpSecret ? totpSetupData.secret : '••••••••••••••••'}
                          </code>
                          <button
                            onClick={() => setShowTotpSecret(!showTotpSecret)}
                            className="text-gray-400 hover:text-gray-600"
                            data-testid="button-toggle-totp-secret"
                          >
                            {showTotpSecret ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                          </button>
                          <button
                            onClick={() => {
                              navigator.clipboard.writeText(totpSetupData.secret);
                              toast({ title: "Copied", description: "Secret key copied to clipboard" });
                            }}
                            className="text-gray-400 hover:text-gray-600"
                            data-testid="button-copy-totp-secret"
                          >
                            <Copy className="h-4 w-4" />
                          </button>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="border rounded-lg p-6">
                  <h4 className="font-medium mb-4 flex items-center gap-2">
                    <Key className="h-4 w-4" />
                    Step 2: Verify Setup
                  </h4>
                  
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm mb-1 font-medium">
                        Enter the 6-digit code from your authenticator app
                      </label>
                      <input
                        type="text"
                        maxLength={6}
                        placeholder="000000"
                        value={totpVerifyToken}
                        onChange={(e) => setTotpVerifyToken(e.target.value.replace(/\D/g, ''))}
                        className="w-32 text-center tracking-widest font-mono text-lg border rounded-xl px-3 py-2 focus:outline-none focus:ring-2 focus:ring-primary"
                        data-testid="input-totp-verify"
                      />
                    </div>
                    
                    <div className="flex gap-3">
                      <button
                        onClick={() => verifyTotpMutation.mutate(totpVerifyToken)}
                        disabled={totpVerifyToken.length !== 6 || verifyTotpMutation.isPending}
                        className="px-4 py-2 bg-green-600 text-white rounded-xl hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed"
                        data-testid="button-verify-totp"
                      >
                        {verifyTotpMutation.isPending ? 'Verifying...' : 'Enable 2FA'}
                      </button>
                      
                      <button
                        onClick={() => {
                          setTotpSetupData(null);
                          setTotpVerifyToken("");
                        }}
                        className="px-4 py-2 border rounded-xl hover:bg-gray-50"
                        data-testid="button-cancel-totp-setup"
                      >
                        Cancel
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        ) : (
          // TOTP is enabled - show status and disable option
          <div className="space-y-6">
            <div className="flex items-center justify-between p-4 bg-green-50 border border-green-200 rounded-lg">
              <div className="flex items-center gap-3">
                <Check className="h-5 w-5 text-green-600" />
                <div>
                  <p className="text-sm font-medium text-green-800">Two-factor authentication is enabled</p>
                  <p className="text-xs text-green-600">Your account is protected with 2FA</p>
                </div>
              </div>
              {totpStatus?.hasBackupCodes && (
                <span className="text-xs text-green-600">
                  {totpStatus.backupCodesCount} backup codes remaining
                </span>
              )}
            </div>

            <div className="border rounded-lg p-6 bg-red-50 border-red-200">
              <h4 className="font-medium text-red-800 mb-4">Disable Two-Factor Authentication</h4>
              <p className="text-sm text-red-600 mb-4">
                This will make your account less secure. You'll only need your password to sign in.
              </p>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm mb-1 font-medium text-red-700">
                    Enter current 2FA code to disable
                  </label>
                  <input
                    type="text"
                    maxLength={6}
                    placeholder="000000"
                    value={totpDisableToken}
                    onChange={(e) => setTotpDisableToken(e.target.value.replace(/\D/g, ''))}
                    className="w-32 text-center tracking-widest font-mono text-lg border border-red-300 rounded-xl px-3 py-2 focus:outline-none focus:ring-2 focus:ring-red-500"
                    data-testid="input-totp-disable"
                  />
                </div>
                
                <button
                  onClick={() => disableTotpMutation.mutate({ totpToken: totpDisableToken })}
                  disabled={totpDisableToken.length !== 6 || disableTotpMutation.isPending}
                  className="px-4 py-2 bg-red-600 text-white rounded-xl hover:bg-red-700 disabled:opacity-50 disabled:cursor-not-allowed"
                  data-testid="button-disable-totp"
                >
                  {disableTotpMutation.isPending ? 'Disabling...' : 'Disable 2FA'}
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Backup Codes Modal-like Section */}
        {showBackupCodes && backupCodes.length > 0 && (
          <div className="mt-6 border rounded-lg p-6 bg-yellow-50 border-yellow-200">
            <div className="flex items-center gap-2 mb-4">
              <AlertCircle className="h-5 w-5 text-yellow-600" />
              <h4 className="font-medium text-yellow-800">Save Your Backup Codes</h4>
            </div>
            
            <p className="text-sm text-yellow-700 mb-4">
              These backup codes can be used to access your account if you lose your authenticator device. 
              Save them in a secure location - they won't be shown again.
            </p>
            
            <div className="grid grid-cols-2 gap-2 mb-4 font-mono text-sm">
              {backupCodes.map((code, index) => (
                <div key={index} className="p-2 bg-white border rounded text-center" data-testid={`backup-code-${index}`}>
                  {code}
                </div>
              ))}
            </div>
            
            <div className="flex gap-3">
              <button
                onClick={() => {
                  const codesText = backupCodes.join('\n');
                  navigator.clipboard.writeText(codesText);
                  toast({ title: "Copied", description: "Backup codes copied to clipboard" });
                }}
                className="inline-flex items-center gap-2 px-3 py-1 border rounded-xl hover:bg-gray-50"
                data-testid="button-copy-backup-codes"
              >
                <Copy className="h-4 w-4" />
                Copy Codes
              </button>
              
              <button
                onClick={() => {
                  const codesText = backupCodes.join('\n');
                  const blob = new Blob([codesText], { type: 'text/plain' });
                  const url = URL.createObjectURL(blob);
                  const a = document.createElement('a');
                  a.href = url;
                  a.download = 'ibrandbiz-backup-codes.txt';
                  a.click();
                  URL.revokeObjectURL(url);
                }}
                className="inline-flex items-center gap-2 px-3 py-1 border rounded-xl hover:bg-gray-50"
                data-testid="button-download-backup-codes"
              >
                <Download className="h-4 w-4" />
                Download
              </button>
              
              <button
                onClick={() => {
                  setShowBackupCodes(false);
                  setBackupCodes([]);
                }}
                className="px-3 py-1 bg-yellow-600 text-white rounded-xl hover:bg-yellow-700"
                data-testid="button-close-backup-codes"
              >
                I've Saved These Codes
              </button>
            </div>
          </div>
        )}

        <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
          <div className="flex">
            <div className="flex-shrink-0">
              <AlertCircle className="h-5 w-5 text-blue-400" />
            </div>
            <div className="ml-3">
              <h3 className="text-sm font-medium text-blue-800">
                About Two-Factor Authentication
              </h3>
              <div className="mt-1 text-sm text-blue-700">
                <ul className="list-disc list-inside space-y-1">
                  <li>Adds an extra layer of security beyond your password</li>
                  <li>Works with popular authenticator apps like Google Authenticator, Authy, and 1Password</li>
                  <li>Backup codes let you access your account if you lose your phone</li>
                  <li>Required for accessing sensitive account features and admin functions</li>
                </ul>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}