import { useState } from "react";
import { useAuth } from "@/contexts/AuthContext";

export function ProfileInfoTab({ me, onUpdated }: any) {
  const { currentUser } = useAuth();
  const [name, setName] = useState(me.name || "");
  const [company, setCompany] = useState(me.company || "");
  const [saving, setSaving] = useState(false);

  const save = async () => {
    if (!currentUser) return;
    
    setSaving(true);
    try {
      const token = await currentUser.getIdToken();
      const res = await fetch("/api/profile/update", {
        method: "POST", 
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${token}`
        },
        body: JSON.stringify({ name, company }),
      });
      
      const out = await res.json();
      if (res.ok) {
        onUpdated((prev: any) => ({ ...prev, name: out.name, company: out.company }));
        alert("Profile updated successfully!");
      } else {
        alert(out.error || "Failed to save");
      }
    } catch (error) {
      console.error('Save error:', error);
      alert("Failed to save profile");
    } finally {
      setSaving(false);
    }
  };

  const onAvatar = async (e: any) => {
    if (!currentUser) return;
    
    const f = e.target.files?.[0]; 
    if (!f) return;
    
    const form = new FormData(); 
    form.append("file", f);
    
    try {
      const token = await currentUser.getIdToken();
      const res = await fetch("/api/profile/avatar", { 
        method: "POST", 
        headers: {
          "Authorization": `Bearer ${token}`
        },
        body: form 
      });
      
      const out = await res.json();
      if (res.ok) {
        onUpdated((prev: any) => ({ ...prev, avatarUrl: out.url }));
        alert("Avatar updated successfully!");
      } else {
        alert(out.error || "Upload failed");
      }
    } catch (error) {
      console.error('Avatar upload error:', error);
      alert("Upload failed");
    }
  };

  return (
    <div className="rounded-2xl border p-6">
      <h2 className="text-xl font-semibold mb-4">Profile Info</h2>
      
      <label className="block text-sm mb-1 font-medium">Avatar</label>
      <div className="flex items-center gap-3 mb-4">
        <div className="w-14 h-14 bg-accent rounded-full flex items-center justify-center">
          {me.avatarUrl ? (
            <img 
              src={me.avatarUrl} 
              className="w-14 h-14 rounded-full object-cover" 
              alt="Avatar"
            />
          ) : (
            <span className="text-accent-foreground font-medium">
              {name ? name.charAt(0).toUpperCase() : me.email.charAt(0).toUpperCase()}
            </span>
          )}
        </div>
        <input 
          type="file" 
          accept="image/*" 
          onChange={onAvatar} 
          className="text-sm text-gray-600 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-medium file:bg-gray-50 file:text-gray-700 hover:file:bg-gray-100"
        />
      </div>

      <label className="block text-sm mb-1 font-medium">Full name</label>
      <input 
        className="w-full border rounded-xl px-3 py-2 mb-3 focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent" 
        value={name} 
        onChange={e => setName(e.target.value)}
        placeholder="Enter your full name"
      />

      <label className="block text-sm mb-1 font-medium">Company / Brand</label>
      <input 
        className="w-full border rounded-xl px-3 py-2 mb-6 focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent" 
        value={company} 
        onChange={e => setCompany(e.target.value)}
        placeholder="Enter your company or brand name"
      />

      <button 
        className="rounded-xl bg-black text-white px-4 py-2 hover:bg-gray-800 transition-colors disabled:opacity-50 disabled:cursor-not-allowed" 
        onClick={save} 
        disabled={saving}
      >
        {saving ? "Saving..." : "Save changes"}
      </button>
    </div>
  );
}