import { useState, useEffect } from "react";
import { useAuth } from "@/contexts/AuthContext";

export function PreferencesTab({ me, onUpdated }: any) {
  const { currentUser } = useAuth();
  const [emailNews, setEmailNews] = useState(true);
  const [marketingEmails, setMarketingEmails] = useState(false);
  const [productUpdates, setProductUpdates] = useState(true);
  
  // Notification preferences
  const [systemNotifications, setSystemNotifications] = useState(true);
  const [billingNotifications, setBillingNotifications] = useState(true);
  const [projectNotifications, setProjectNotifications] = useState(true);
  
  const [saving, setSaving] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const loadPreferences = async () => {
      if (!currentUser) return;
      
      try {
        const token = await currentUser.getIdToken();
        const res = await fetch("/api/profile/preferences", {
          method: "GET",
          headers: {
            "Authorization": `Bearer ${token}`
          }
        });
        
        if (res.ok) {
          const prefs = await res.json();
          setEmailNews(prefs.emailNews ?? true);
          setMarketingEmails(prefs.marketingEmails ?? false);
          setProductUpdates(prefs.productUpdates ?? true);
          
          // Notification preferences
          setSystemNotifications(prefs.systemNotifications ?? true);
          setBillingNotifications(prefs.billingNotifications ?? true);
          setProjectNotifications(prefs.projectNotifications ?? true);
        }
      } catch (error) {
        console.error('Load preferences error:', error);
      } finally {
        setLoading(false);
      }
    };

    loadPreferences();
  }, [currentUser]);

  const save = async () => {
    if (!currentUser) return;
    
    setSaving(true);
    try {
      const token = await currentUser.getIdToken();
      const res = await fetch("/api/profile/preferences", {
        method: "POST", 
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${token}`
        },
        body: JSON.stringify({ 
          emailNews, 
          marketingEmails, 
          productUpdates,
          systemNotifications,
          billingNotifications,
          projectNotifications
        }),
      });
      
      if (res.ok) {
        alert("Preferences saved successfully!");
      } else {
        const out = await res.json();
        alert(out.error || "Failed to save preferences");
      }
    } catch (error) {
      console.error('Save preferences error:', error);
      alert("Failed to save preferences");
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <div className="rounded-2xl border p-6">
        <h2 className="text-xl font-semibold mb-4">Preferences</h2>
        <div className="text-center py-4">
          <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-primary mx-auto"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="rounded-2xl border p-6">
      <h2 className="text-xl font-semibold mb-4">Preferences</h2>
      
      <div className="space-y-6 mb-6">
        {/* Email Preferences Section */}
        <div>
          <h3 className="font-medium mb-3 text-gray-900">Email Preferences</h3>
          <div className="space-y-3">
            <div className="flex items-start gap-3 p-3 border rounded-lg">
              <input 
                type="checkbox" 
                id="productUpdates"
                checked={productUpdates} 
                onChange={e => setProductUpdates(e.target.checked)}
                className="mt-0.5 h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded"
              />
              <div className="flex-1">
                <label htmlFor="productUpdates" className="font-medium text-sm cursor-pointer">
                  Product Updates
                </label>
                <p className="text-xs text-gray-600">
                  Get notified about new features, improvements, and important updates to IBrandBiz
                </p>
              </div>
            </div>

            <div className="flex items-start gap-3 p-3 border rounded-lg">
              <input 
                type="checkbox" 
                id="emailNews"
                checked={emailNews} 
                onChange={e => setEmailNews(e.target.checked)}
                className="mt-0.5 h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded"
              />
              <div className="flex-1">
                <label htmlFor="emailNews" className="font-medium text-sm cursor-pointer">
                  Educational Content
                </label>
                <p className="text-xs text-gray-600">
                  Receive tips, tutorials, and best practices for building your brand
                </p>
              </div>
            </div>

            <div className="flex items-start gap-3 p-3 border rounded-lg">
              <input 
                type="checkbox" 
                id="marketingEmails"
                checked={marketingEmails} 
                onChange={e => setMarketingEmails(e.target.checked)}
                className="mt-0.5 h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded"
              />
              <div className="flex-1">
                <label htmlFor="marketingEmails" className="font-medium text-sm cursor-pointer">
                  Marketing Communications
                </label>
                <p className="text-xs text-gray-600">
                  Get special offers, promotions, and announcements about IBrandBiz services
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Notification Preferences Section */}
        <div>
          <h3 className="font-medium mb-3 text-gray-900">In-App Notifications</h3>
          <div className="space-y-3">
            <div className="flex items-start gap-3 p-3 border rounded-lg">
              <input 
                type="checkbox" 
                id="systemNotifications"
                checked={systemNotifications} 
                onChange={e => setSystemNotifications(e.target.checked)}
                className="mt-0.5 h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded"
              />
              <div className="flex-1">
                <label htmlFor="systemNotifications" className="font-medium text-sm cursor-pointer">
                  System Updates
                </label>
                <p className="text-xs text-gray-600">
                  Important system announcements, maintenance alerts, and platform updates
                </p>
              </div>
            </div>

            <div className="flex items-start gap-3 p-3 border rounded-lg">
              <input 
                type="checkbox" 
                id="billingNotifications"
                checked={billingNotifications} 
                onChange={e => setBillingNotifications(e.target.checked)}
                className="mt-0.5 h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded"
              />
              <div className="flex-1">
                <label htmlFor="billingNotifications" className="font-medium text-sm cursor-pointer">
                  Billing Alerts
                </label>
                <p className="text-xs text-gray-600">
                  Payment confirmations, subscription changes, and billing-related notifications
                </p>
              </div>
            </div>

            <div className="flex items-start gap-3 p-3 border rounded-lg">
              <input 
                type="checkbox" 
                id="projectNotifications"
                checked={projectNotifications} 
                onChange={e => setProjectNotifications(e.target.checked)}
                className="mt-0.5 h-4 w-4 text-primary focus:ring-primary border-gray-300 rounded"
              />
              <div className="flex-1">
                <label htmlFor="projectNotifications" className="font-medium text-sm cursor-pointer">
                  Project Activity
                </label>
                <p className="text-xs text-gray-600">
                  Updates on brand kits, business names, and other project-related activities
                </p>
              </div>
            </div>
          </div>
        </div>
      </div>

      <button 
        className="rounded-xl bg-black text-white px-4 py-2 hover:bg-gray-800 transition-colors disabled:opacity-50 disabled:cursor-not-allowed" 
        onClick={save}
        disabled={saving}
      >
        {saving ? "Saving..." : "Save preferences"}
      </button>
      
      <div className="mt-4 text-xs text-gray-500">
        <p>
          You can unsubscribe from any of these emails at any time. Essential account notifications will still be sent.
        </p>
      </div>
    </div>
  );
}