import { useEffect, useState } from "react";
import { useAuth } from "@/contexts/AuthContext";
import { ProfileInfoTab } from "./tabs/ProfileInfoTab";
import { SubscriptionTab } from "./tabs/SubscriptionTab";
import { PreferencesTab } from "./tabs/PreferencesTab";
import { SecurityTab } from "./tabs/SecurityTab";

type Me = {
  authenticated: boolean;
  email: string;
  name?: string;
  company?: string;
  avatarUrl?: string;
  isPaid?: boolean;
  proActivatedAt?: string | null;
  proWelcomeDismissed?: boolean;
  isAdmin?: boolean;
};

export default function ProfilePage() {
  const { currentUser } = useAuth();
  const [me, setMe] = useState<Me | null>(null);
  const [tab, setTab] = useState<"profile"|"subscription"|"preferences"|"security">("profile");
  const [adminStatus, setAdminStatus] = useState<boolean>(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchMe = async () => {
      if (!currentUser) {
        setLoading(false);
        return;
      }

      try {
        const token = await currentUser.getIdToken();
        
        // Check admin status - in development, check if user has admin claim
        let hasAdminAccess = false;
        try {
          // Try Firebase token claim check first
          const tokenResult = await currentUser.getIdTokenResult();
          hasAdminAccess = !!tokenResult.claims.admin;
          
          // If no admin claim, fallback to API health check
          if (!hasAdminAccess) {
            const adminCheck = await fetch('/api/admin/health', {
              headers: {
                'Authorization': `Bearer ${token}`,
              },
            });
            hasAdminAccess = adminCheck.ok;
          }
        } catch (error) {
          console.log('Admin status check failed:', error);
          hasAdminAccess = false;
        }
        setAdminStatus(hasAdminAccess);
        
        const response = await fetch('/api/auth/me', {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        });

        if (response.ok) {
          const userData = await response.json();
          setMe({
            authenticated: true,
            email: currentUser.email || '',
            name: currentUser.displayName || userData.name || '',
            company: userData.company || '',
            avatarUrl: userData.avatarUrl || '',
            isPaid: userData.isPaid || false,
            proActivatedAt: userData.proActivatedAt,
            proWelcomeDismissed: userData.proWelcomeDismissed || false,
            isAdmin: hasAdminAccess
          });
        } else {
          setMe({
            authenticated: true,
            email: currentUser.email || '',
            name: currentUser.displayName || '',
            company: '',
            avatarUrl: '',
            isPaid: false,
            proActivatedAt: null,
            proWelcomeDismissed: false,
            isAdmin: hasAdminAccess
          });
        }
      } catch (error) {
        console.error('Failed to fetch user profile:', error);
        setMe({
          authenticated: true,
          email: currentUser.email || '',
          name: currentUser.displayName || '',
          company: '',
          avatarUrl: '',
          isPaid: false,
          proActivatedAt: null,
          proWelcomeDismissed: false,
          isAdmin: adminStatus
        });
      } finally {
        setLoading(false);
      }
    };

    fetchMe();
  }, [currentUser]);

  if (loading) return (
    <div className="p-8 text-center">
      <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto"></div>
      <p className="mt-4 text-muted-foreground">Loading...</p>
    </div>
  );
  
  if (!me?.authenticated) return (
    <div className="p-8 text-center">
      <p className="text-muted-foreground">Please sign in.</p>
    </div>
  );

  return (
    <div className="max-w-5xl mx-auto py-8 grid grid-cols-12 gap-6">
      {/* Left column: card */}
      <aside className="col-span-12 md:col-span-4">
        <div className="rounded-2xl border p-6">
          <div className="flex items-center gap-3 mb-4">
            <div className="w-16 h-16 bg-accent rounded-full flex items-center justify-center">
              {me.avatarUrl ? (
                <img 
                  className="w-16 h-16 rounded-full object-cover" 
                  src={me.avatarUrl} 
                  alt="Profile"
                />
              ) : (
                <span className="text-accent-foreground text-xl font-medium">
                  {me.name ? me.name.charAt(0).toUpperCase() : me.email.charAt(0).toUpperCase()}
                </span>
              )}
            </div>
            <div>
              <div className="font-semibold">{me.name || "Your Name"}</div>
              <div className="text-sm text-gray-500">{me.email}</div>
            </div>
          </div>
          <div className="grid grid-cols-2 gap-2 text-sm">
            <div className="rounded-lg border p-3">
              <div className="text-gray-500">Plan</div>
              <div className="font-semibold">{me.isPaid ? "Pro" : "Free"}</div>
            </div>
            <div className="rounded-lg border p-3">
              <div className="text-gray-500">Member since</div>
              <div className="font-semibold">{me.proActivatedAt ? new Date(me.proActivatedAt).toLocaleDateString() : "—"}</div>
            </div>
          </div>
          
          {/* Admin Status */}
          {me.isAdmin && (
            <div className="mt-4 rounded-lg border-2 border-orange-200 bg-orange-50 p-3">
              <div className="flex items-center justify-between">
                <div>
                  <div className="text-orange-800 font-semibold text-sm">🛡️ Administrator</div>
                  <div className="text-orange-600 text-xs">You have admin privileges</div>
                </div>
                <button
                  onClick={async () => {
                    const { setAdminTokenFromFirebase } = await import('@/admin/adminGuard');
                    const token = await setAdminTokenFromFirebase();
                    if (token) {
                      window.location.href = '/admin/dashboard';
                    } else {
                      console.error('Failed to set admin token');
                    }
                  }}
                  className="text-orange-800 hover:text-orange-900 text-xs underline bg-transparent border-none cursor-pointer"
                  data-testid="button-admin-panel"
                >
                  Admin Panel
                </button>
              </div>
            </div>
          )}
        </div>
        
        {/* Tabs */}
        <div className="mt-4 rounded-2xl border p-2">
          {[
            ["profile","Profile Info"],
            ["subscription","Subscription"],
            ["preferences","Preferences"],
            ["security","Security"],
          ].map(([k, label]) => (
            <button
              key={k}
              onClick={() => setTab(k as any)}
              className={`w-full text-left px-3 py-2 rounded-xl text-sm transition-colors ${
                tab === k ? "bg-black text-white" : "hover:bg-gray-100"
              }`}
              data-testid={`tab-${k}`}
            >
              {label}
            </button>
          ))}
        </div>
      </aside>

      {/* Right column: tab content */}
      <main className="col-span-12 md:col-span-8">
        {tab === "profile" && <ProfileInfoTab me={me} onUpdated={setMe} />}
        {tab === "subscription" && <SubscriptionTab me={me} />}
        {tab === "preferences" && <PreferencesTab me={me} onUpdated={setMe} />}
        {tab === "security" && <SecurityTab />}
      </main>
    </div>
  );
}