import { Link } from "wouter";
import TemplatePage from "@/components/TemplatePage";
import productImage from "@/assets/product.png";

/**
 * IBrandBiz — Products Page (Pre‑Login)
 * Active Sections:
 *  - AI Business Plan Developer
 *  - Professional Business Templates
 * 
 * Brand palette:
 *  #5cccdc, #f99f1b, #05445e, #189ab4, #75e6da, #d4f1f4, #03222e
 */

export default function Products() {
  return (
    <TemplatePage
      bannerTitle="Everything You Need to Build Your Brand"
      bannerSubtitle="Pick a product category to start fast and grow with confidence."
      bannerImage={productImage}
    >
      <main className="mx-auto max-w-6xl px-6 py-12 text-[#03222e]">
        {/* Business Development Section */}
        <section className="mb-12">
          <h2 className="text-2xl font-bold mb-6">Business Development</h2>
          <div className="grid gap-6 md:grid-cols-2">
            <ProductCard
              eyebrow="Brand Kit"
              title="Complete Brand Identity"
              desc="Build a cohesive brand identity with our 4-step wizard. Get color palettes, typography, logo variations, and brand guidelines."
              bullets={[
                "Professional color palettes",
                "Typography and logo systems",
                "Downloadable brand guidelines",
              ]}
              href="/business-development/brand-kit"
              tone="teal"
            />

            <ProductCard
              eyebrow="Business Plan Developer"
              title="AI-Powered Business Plans"
              desc="Create professional business plans with AI assistance. 12-section structure with templates, AI writing, and export options."
              bullets={[
                "AI content generation for each section",
                "Structured templates (SWOT, Personas)",
                "Export to DOCX, PDF, or PowerPoint",
              ]}
              href="/business-development/ai-business-plan-developer"
              tone="navy"
            />
          </div>
        </section>

        {/* Business Assets Section */}
        <section>
          <h2 className="text-2xl font-bold mb-2">Business Assets</h2>
          <p className="text-[#05445e] mb-6">Professional templates and resources for your brand</p>
          <div className="grid gap-6 md:grid-cols-2">
            <ProductCard
              eyebrow="Business Plan Templates"
              title="Free Professional Templates"
              desc="Download professionally designed business plan templates in DOCX format. Start with a solid foundation and customize to your needs."
              bullets={[
                "12-section business plan structure",
                "Free DOCX downloads",
                "Professional formatting ready to use",
              ]}
              href="/business-assets/templates/business-plan"
              tone="navy"
            />

            <ProductCard
              eyebrow="Logo Templates"
              title="Pre-Designed Logo Collection"
              desc="Browse hundreds of professional logo templates. Download as SVG or customize in our Logo Composer for a perfect fit."
              bullets={[
                "Hundreds of professional designs",
                "Free SVG downloads",
                "Customize colors, text, and layout",
              ]}
              href="/business-assets/logo-templates"
              tone="teal"
            />
          </div>
        </section>

          {/* Hidden for future use */}
          {/*
          <ProductCard
            eyebrow="Website Templates"
            title="Professional Sites, Ready to Launch"
            desc="Modern, responsive templates that look great on every device. Choose, customize, and go live."
            bullets={[
              "Clean, conversion‑focused layouts",
              "Responsive by default",
              "Easy to customize",
            ]}
            href="/website-templates"
            tone="navy"
          />

          <ProductCard
            eyebrow="Social Media Kits"
            title="Cohesive Posts & Profiles"
            desc="On‑brand templates for posts, stories, banners, and ads — sized for every major platform."
            bullets={[
              "Formats for IG, FB, X, TikTok",
              "Editable captions & overlays",
              "Batch export options",
            ]}
            href="/social-media-kits"
            tone="orange"
          />

          <ProductCard
            eyebrow="Stock Photos"
            title="High‑Quality Visuals"
            desc="Curated photography to power your brand visuals across web, print, and social."
            bullets={[
              "Commercial‑use friendly",
              "Collections by industry",
              "Hi‑res, web‑ready versions",
            ]}
            href="/business-assets/stock/photos"
            tone="teal"
          />

          <ProductCard
            eyebrow="Brand Assets"
            title="Business‑Ready Templates"
            desc="Reusable files for decks, proposals, letterheads, packaging, signage, and more."
            bullets={[
              "Presentation & document kits",
              "Print & packaging templates",
              "Easy brand swap‑ins",
            ]}
            href="/brand-assets"
            tone="navy"
          />
          */}

        {/* Helpful note / micro‑trust */}
        <div className="mt-12 rounded-3xl border border-slate-200 bg-[#d4f1f4] p-6 text-[#05445e]" data-testid="text-guidance-note">
          <p className="mb-3">
            Looking for guidance on what to choose first?
          </p>
          <p>
            Start with a <span className="font-semibold text-[#05445e]">Brand Kit</span> to establish your identity, create a comprehensive <span className="font-semibold text-[#05445e]">Business Plan</span> to guide your growth strategy, then browse our free templates for polished materials and logos.
          </p>
        </div>
      </main>

      {/* Final CTA */}
      <section className="mx-auto max-w-6xl px-6 pb-16 pt-10 text-center">
        <h2 className="text-2xl sm:text-3xl font-bold">Ready to build something amazing?</h2>
        <p className="mt-3 text-[#05445e] max-w-2xl mx-auto">
          Make your first impression your best — choose a starting point and get moving.
        </p>
        <div className="mt-8 flex flex-wrap items-center justify-center gap-3">
          <CTA href="/business-development/brand-kit" label="Build Your Brand" />
          <CTA href="/pricing/business-development" variant="outline" label="Get Business Plan Builder" />
          <CTA href="/business-assets/templates/business-plan" variant="ghost" label="Browse Free Templates" />
        </div>
      </section>
    </TemplatePage>
  );
}

/* ========================= Components ========================= */

type CTAProps = {
  href: string;
  label: string;
  variant?: "solid" | "outline" | "ghost";
};

function CTA({ href, label, variant = "solid" }: CTAProps) {
  const base =
    "inline-flex items-center justify-center rounded-2xl px-5 py-3 text-sm font-semibold transition focus:outline-none focus-visible:ring-2 focus-visible:ring-offset-2";
  const styles: Record<NonNullable<CTAProps["variant"]>, string> = {
    solid:
      "bg-[#00C851] text-white shadow hover:bg-[#00B347] focus-visible:ring-[#5cccdc]",
    outline:
      "border border-[#00C851] text-[#00C851] hover:bg-[#d4f1f4] focus-visible:ring-[#5cccdc]",
    ghost:
      "text-[#00C851] hover:bg-[#d4f1f4] focus-visible:ring-[#5cccdc]",
  };
  return (
    <Link href={href} className={`${base} ${styles[variant]}`} data-testid={`button-cta-${label.toLowerCase().replace(/\s+/g, '-')}`}>
      {label}
    </Link>
  );
}

type ProductCardProps = {
  eyebrow: string;
  title: string;
  desc: string;
  bullets: string[];
  href: string;
  tone?: "teal" | "navy" | "orange";
};

function ProductCard({ eyebrow, title, desc, bullets, href, tone = "teal" }: ProductCardProps) {
  const getToneGradient = (tone: "teal" | "navy" | "orange") => {
    switch (tone) {
      case "teal":
        return 'linear-gradient(to right, #d4f1f4, #75e6da)';
      case "navy":
        return 'linear-gradient(to right, #d4f1f4, #5cccdc)';
      case "orange":
        return 'linear-gradient(to right, #fff7ed, #ff8800)';
      default:
        return 'linear-gradient(to right, #d4f1f4, #75e6da)';
    }
  };

  return (
    <div className="group rounded-3xl border border-slate-200 overflow-hidden bg-white shadow-sm hover:shadow-md transition" data-testid={`card-product-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>
      <div className="h-2 w-full" style={{background: getToneGradient(tone)}} />
      <div className="p-6">
        <p className="text-xs uppercase tracking-wider text-[#05445e] font-semibold" data-testid={`text-eyebrow-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>{eyebrow}</p>
        <h3 className="mt-2 text-xl font-bold" data-testid={`text-title-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>{title}</h3>
        <p className="mt-3 text-sm text-[#05445e]" data-testid={`text-desc-${eyebrow.toLowerCase().replace(/\s+/g, '-')}`}>{desc}</p>
        <ul className="mt-4 space-y-2 text-sm text-[#05445e]">
          {bullets.map((b, i) => (
            <li key={i} className="flex gap-2" data-testid={`text-bullet-${eyebrow.toLowerCase().replace(/\s+/g, '-')}-${i}`}>
              <span className="mt-1 inline-block h-2.5 w-2.5 rounded-full bg-[#189ab4] group-hover:bg-[#05445e]" />
              <span>{b}</span>
            </li>
          ))}
        </ul>
        <div className="mt-6">
          <CTA href={href} label={`Explore ${eyebrow}`} />
        </div>
      </div>
    </div>
  );
}