import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/contexts/AuthContext";
import { useCart } from "@/contexts/CartContext";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Slider } from "@/components/ui/slider";
import { Separator } from "@/components/ui/separator";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { toast } from "sonner";
import { Link, useLocation } from "wouter";
import {
  Search,
  TrendingUp,
  Grid3X3,
  List,
  ShoppingCart,
  Eye,
  User,
  Calendar,
  DollarSign,
  FileImage,
  FileText,
  File,
  X,
  SlidersHorizontal,
  Star
} from "lucide-react";

interface Creator {
  id: string;
  name: string;
  email: string;
  profileData?: { bio?: string; portfolio?: string; };
}

interface Asset {
  fileName: string;
  fileUrl: string;
  previewUrl?: string;
  fileSize: number;
  mimeType: string;
  assetType: string;
  category?: string;
  tags: string[];
  downloadCount: number;
}

interface MarketplaceAsset {
  id: string;
  assetId: string;
  title: string;
  description?: string;
  price: number;
  isExclusive: boolean;
  salesCount: number;
  createdAt: string;
  asset: Asset;
  creator: Creator;
}

interface MarketplaceStats { totalAssets: number; totalSales: number; }

interface MarketplaceResponse {
  assets: MarketplaceAsset[];
  stats: MarketplaceStats;
  totalCount: number;
  page: number;
  limit: number;
  hasMore: boolean;
}

interface Filters {
  search: string;
  category: string;
  assetType: string;
  minPrice: number;
  maxPrice: number;
  sortBy: string;
  creatorId: string;
}

const DEFAULT_FILTERS: Filters = {
  search: "",
  category: "all",
  assetType: "all",
  minPrice: 0,
  maxPrice: 5000,
  sortBy: "newest",
  creatorId: ""
};

const SORT_OPTIONS = [
  { value: "newest", label: "Newest First" },
  { value: "oldest", label: "Oldest First" },
  { value: "price_low", label: "Price: Low to High" },
  { value: "price_high", label: "Price: High to Low" },
  { value: "popular", label: "Most Popular" }
];

const ASSET_TYPES = [
  { value: "all", label: "All Types" },
  { value: "image", label: "Images" },
  { value: "video", label: "Videos" },
  { value: "audio", label: "Audio" },
  { value: "document", label: "Documents" },
  { value: "template", label: "Templates" }
];

const CATEGORIES = [
  { value: "all", label: "All Categories" },
  { value: "graphics", label: "Graphics" },
  { value: "icons", label: "Icons" },
  { value: "illustrations", label: "Illustrations" },
  { value: "photos", label: "Photos" },
  { value: "templates", label: "Templates" },
  { value: "ui-kits", label: "UI Kits" },
  { value: "fonts", label: "Fonts" }
];

export default function MarketplacePage() {
  const { currentUser } = useAuth();
  const { addItem, hasItem } = useCart();
  const [location, setLocation] = useLocation();

  // URL ←→ filters (single source of truth: URL holds actual filters only)
  const getFiltersFromURL = () => {
    const urlParams = new URLSearchParams(window.location.search);
    return {
      search: urlParams.get('search') || DEFAULT_FILTERS.search,
      category: urlParams.get('category') || DEFAULT_FILTERS.category,
      assetType: urlParams.get('assetType') || DEFAULT_FILTERS.assetType,
      minPrice: parseInt(urlParams.get('minPrice') || DEFAULT_FILTERS.minPrice.toString()),
      maxPrice: parseInt(urlParams.get('maxPrice') || DEFAULT_FILTERS.maxPrice.toString()),
      sortBy: urlParams.get('sortBy') || DEFAULT_FILTERS.sortBy,
      creatorId: urlParams.get('creatorId') || DEFAULT_FILTERS.creatorId,
    };
  };

  const [filters, setFilters] = useState<Filters>(getFiltersFromURL());
  const [viewMode, setViewMode] = useState<'grid' | 'list'>('grid');
  const [page, setPage] = useState(1);
  const [showFilters, setShowFilters] = useState(false);
  const [searchDebounce, setSearchDebounce] = useState(() => getFiltersFromURL().search);

  // Debounce search input
  useEffect(() => {
    const timer = setTimeout(() => setSearchDebounce(filters.search), 300);
    return () => clearTimeout(timer);
  }, [filters.search]);

  // Reset page when filters change
  useEffect(() => {
    setPage(1);
  }, [searchDebounce, filters.category, filters.assetType, filters.minPrice, filters.maxPrice, filters.sortBy, filters.creatorId]);

  // Write filters to URL (only non-defaults)
  useEffect(() => {
    const urlParams = new URLSearchParams();
    if (searchDebounce) urlParams.set('search', searchDebounce);
    if (filters.category !== 'all') urlParams.set('category', filters.category);
    if (filters.assetType !== 'all') urlParams.set('assetType', filters.assetType);
    if (filters.minPrice > 0) urlParams.set('minPrice', filters.minPrice.toString());
    if (filters.maxPrice < 5000) urlParams.set('maxPrice', filters.maxPrice.toString());
    if (filters.sortBy !== 'newest') urlParams.set('sortBy', filters.sortBy);
    if (filters.creatorId) urlParams.set('creatorId', filters.creatorId);

    const newSearch = urlParams.toString();
    const newUrl = `${window.location.pathname}${newSearch ? `?${newSearch}` : ''}`;
    if (newUrl !== window.location.pathname + window.location.search) {
      window.history.replaceState({}, '', newUrl);
    }
  }, [searchDebounce, filters.category, filters.assetType, filters.minPrice, filters.maxPrice, filters.sortBy, filters.creatorId]);

  // Fetch assets
  const { data: marketplaceData, isLoading, error, refetch } = useQuery<MarketplaceResponse>({
    queryKey: ['/api/marketplace/assets', page, searchDebounce, filters.category, filters.assetType, filters.minPrice, filters.maxPrice, filters.sortBy, filters.creatorId],
    queryFn: async () => {
      const params = new URLSearchParams({
        page: page.toString(),
        limit: '20',
        ...(searchDebounce && { search: searchDebounce }),
        ...(filters.category !== 'all' && { category: filters.category }),
        ...(filters.assetType !== 'all' && { assetType: filters.assetType }),
        ...(filters.minPrice > 0 && { minPrice: filters.minPrice.toString() }),
        ...(filters.maxPrice < 5000 && { maxPrice: filters.maxPrice.toString() }),
        ...(filters.sortBy && { sortBy: filters.sortBy }),
        ...(filters.creatorId && { creatorId: filters.creatorId })
      });

      const response = await fetch(`/api/marketplace/assets?${params}`, { credentials: 'include' });
      if (!response.ok) throw new Error('Failed to fetch marketplace assets');
      return response.json();
    },
  });

  const handleFilterChange = (key: keyof Filters, value: any) => setFilters(prev => ({ ...prev, [key]: value }));
  const clearFilters = () => { setFilters(DEFAULT_FILTERS); setPage(1); };
  const hasActiveFilters = () =>
    filters.search !== "" ||
    (filters.category !== "all") ||
    (filters.assetType !== "all") ||
    filters.minPrice > 0 ||
    filters.maxPrice < 5000 ||
    filters.creatorId !== "";

  const formatPrice = (cents: number) => `$${(cents / 100).toFixed(2)}`;
  const getAssetIcon = (assetType: string) => {
    switch (assetType) {
      case 'image': return <FileImage className="h-4 w-4" />;
      case 'video': return <FileText className="h-4 w-4" />;
      case 'audio': return <FileText className="h-4 w-4" />;
      case 'document': return <FileText className="h-4 w-4" />;
      case 'template': return <FileText className="h-4 w-4" />;
      default: return <File className="h-4 w-4" />;
    }
  };

  const handleAddToCart = (asset: MarketplaceAsset) => {
    try {
      const success = addItem({
        itemType: 'creator_asset',
        itemId: asset.id,
        itemName: asset.title,
        itemPrice: asset.price,
        quantity: 1,
        metadata: {
          creatorId: asset.creator.id,
          assetId: asset.assetId,
          previewUrl: asset.asset?.previewUrl,
          assetType: asset.asset?.assetType,
          category: asset.asset?.category,
          creatorName: asset.creator.name
        }
      });
      if (success) toast.success(`"${asset.title}" added to cart!`);
    } catch (e) {
      toast.error("Failed to add item to cart");
      console.error("Add to cart error:", e);
    }
  };

  if (error) {
    return (
      <div className="container mx-auto px-4 py-8">
        <Alert className="mb-6"><AlertDescription>Failed to load marketplace assets. Please try again.</AlertDescription></Alert>
        <Button onClick={() => refetch()} data-testid="button-retry">Try Again</Button>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-white dark:bg-gray-900 border-b">
        <div className="container mx-auto px-4 py-8">
          <div className="text-center mb-8">
            <h1 className="text-4xl font-bold text-foreground mb-4" data-testid="page-title">Creator Marketplace</h1>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Discover unique digital assets created by talented creators worldwide. Find the perfect graphics, templates, and resources for your projects.
            </p>
          </div>

          {/* Stats */}
          {marketplaceData?.stats && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 max-w-md mx-auto">
              <Card><CardContent className="flex items-center p-4"><TrendingUp className="h-8 w-8 text-primary mr-3" /><div><div className="text-2xl font-bold">{marketplaceData.stats.totalAssets.toLocaleString()}</div><div className="text-sm text-muted-foreground">Assets Available</div></div></CardContent></Card>
              <Card><CardContent className="flex items-center p-4"><Star className="h-8 w-8 text-yellow-500 mr-3" /><div><div className="text-2xl font-bold">{marketplaceData.stats.totalSales.toLocaleString()}</div><div className="text-sm text-muted-foreground">Total Sales</div></div></CardContent></Card>
            </div>
          )}
        </div>
      </div>

      <div className="container mx-auto px-4 py-8">
        {/* Controls */}
        <div className="flex flex-col lg:flex-row gap-4 mb-8">
          <div className="flex-1 relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search assets, creators, or tags..."
              value={filters.search}
              onChange={(e) => handleFilterChange('search', e.target.value)}
              className="pl-10"
              data-testid="input-search"
            />
          </div>

          <div className="flex items-center gap-2">
            <Button
              variant={showFilters ? "default" : "outline"}
              size="sm"
              onClick={() => setShowFilters(!showFilters)}
              data-testid="button-toggle-filters"
            >
              <SlidersHorizontal className="h-4 w-4 mr-2" />
              Filters
              {hasActiveFilters() && <Badge variant="secondary" className="ml-2">{Object.values(filters).filter(v => v !== "" && v !== 0 && v !== 5000).length}</Badge>}
            </Button>

            <Separator orientation="vertical" className="h-6" />

            <div className="flex rounded-md border">
              <Button variant={viewMode === 'grid' ? "default" : "ghost"} size="sm" onClick={() => setViewMode('grid')} className="rounded-r-none" data-testid="button-grid-view">
                <Grid3X3 className="h-4 w-4" />
              </Button>
              <Button variant={viewMode === 'list' ? "default" : "ghost"} size="sm" onClick={() => setViewMode('list')} className="rounded-l-none" data-testid="button-list-view">
                <List className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </div>

        {/* Filters Panel */}
        {showFilters && (
          <Card className="mb-8">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>Filters</CardTitle>
                {hasActiveFilters() && (
                  <Button variant="outline" size="sm" onClick={clearFilters} data-testid="button-clear-filters">
                    <X className="h-4 w-4 mr-2" /> Clear All
                  </Button>
                )}
              </div>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div>
                  <label className="text-sm font-medium mb-2 block">Category</label>
                  <Select value={filters.category} onValueChange={(v) => handleFilterChange('category', v)}>
                    <SelectTrigger data-testid="select-category"><SelectValue placeholder="All Categories" /></SelectTrigger>
                    <SelectContent>{CATEGORIES.map(c => (<SelectItem key={c.value} value={c.value}>{c.label}</SelectItem>))}</SelectContent>
                  </Select>
                </div>
                <div>
                  <label className="text-sm font-medium mb-2 block">Asset Type</label>
                  <Select value={filters.assetType} onValueChange={(v) => handleFilterChange('assetType', v)}>
                    <SelectTrigger data-testid="select-asset-type"><SelectValue placeholder="All Types" /></SelectTrigger>
                    <SelectContent>{ASSET_TYPES.map(t => (<SelectItem key={t.value} value={t.value}>{t.label}</SelectItem>))}</SelectContent>
                  </Select>
                </div>
                <div>
                  <label className="text-sm font-medium mb-2 block">Sort By</label>
                  <Select value={filters.sortBy} onValueChange={(v) => handleFilterChange('sortBy', v)}>
                    <SelectTrigger data-testid="select-sort"><SelectValue /></SelectTrigger>
                    <SelectContent>{SORT_OPTIONS.map(o => (<SelectItem key={o.value} value={o.value}>{o.label}</SelectItem>))}</SelectContent>
                  </Select>
                </div>
                <div>
                  <label className="text-sm font-medium mb-2 block">
                    Price Range: {formatPrice(filters.minPrice)} - {formatPrice(filters.maxPrice)}
                  </label>
                  <div className="px-3">
                    <Slider
                      value={[filters.minPrice, filters.maxPrice]}
                      onValueChange={([min, max]) => { handleFilterChange('minPrice', min); handleFilterChange('maxPrice', max); }}
                      max={5000}
                      min={0}
                      step={100}
                      className="w-full"
                      data-testid="slider-price-range"
                    />
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Results header */}
        <div className="mb-6">
          {marketplaceData && (
            <div className="flex items-center justify-between">
              <p className="text-muted-foreground">Showing {marketplaceData.assets.length} of {marketplaceData.totalCount} assets</p>
            </div>
          )}
        </div>

        {/* Grid/List */}
        {isLoading ? (
          <div className={viewMode === 'grid' ? "grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6" : "space-y-4"}>
            {[...Array(8)].map((_, i) => (
              <Card key={i}>
                <Skeleton className="aspect-video w-full" />
                <CardContent className="p-4">
                  <Skeleton className="h-4 w-3/4 mb-2" />
                  <Skeleton className="h-3 w-1/2 mb-4" />
                  <div className="flex justify-between items-center"><Skeleton className="h-6 w-16" /><Skeleton className="h-8 w-20" /></div>
                </CardContent>
              </Card>
            ))}
          </div>
        ) : marketplaceData?.assets.length === 0 ? (
          <div className="text-center py-12">
            <div className="text-muted-foreground mb-4"><Search className="h-16 w-16 mx-auto mb-4" /></div>
            <h3 className="text-lg font-medium mb-2">No assets found</h3>
            <p className="text-muted-foreground mb-4">Try adjusting your search criteria or browse all available assets.</p>
            {hasActiveFilters() && <Button onClick={clearFilters} data-testid="button-clear-filters-empty">Clear All Filters</Button>}
          </div>
        ) : (
          <>
            <div className={viewMode === 'grid' ? "grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6" : "space-y-4"}>
              {marketplaceData!.assets.map((asset) => (
                <AssetCard key={asset.id} asset={asset} viewMode={viewMode} onAddToCart={handleAddToCart} isInCart={hasItem(asset.id, 'creator_asset')} />
              ))}
            </div>
            {marketplaceData && marketplaceData.hasMore && (
              <div className="flex justify-center mt-8">
                <Button onClick={() => setPage(prev => prev + 1)} disabled={isLoading} data-testid="button-load-more">Load More Assets</Button>
              </div>
            )}
          </>
        )}
      </div>
    </div>
  );
}

interface AssetCardProps {
  asset: MarketplaceAsset;
  viewMode: 'grid' | 'list';
  onAddToCart: (asset: MarketplaceAsset) => void;
  isInCart: boolean;
}

function AssetCard({ asset, viewMode, onAddToCart, isInCart }: AssetCardProps) {
  const formatPrice = (cents: number) => `$${(cents / 100).toFixed(2)}`;

  const getAssetIcon = (assetType: string) => {
    switch (assetType) {
      case 'image': return <FileImage className="h-4 w-4" />;
      case 'video': return <FileText className="h-4 w-4" />;
      case 'audio': return <FileText className="h-4 w-4" />;
      case 'document': return <FileText className="h-4 w-4" />;
      case 'template': return <FileText className="h-4 w-4" />;
      default: return <File className="h-4 w-4" />;
    }
  };

  if (viewMode === 'list') {
    return (
      <Card className="hover:shadow-md transition-shadow" data-testid={`asset-card-${asset.id}`}>
        <CardContent className="p-6">
          <div className="flex items-start gap-4">
            <div className="w-24 h-24 bg-gray-100 dark:bg-gray-800 rounded-lg overflow-hidden flex-shrink-0">
              {asset.asset?.previewUrl ? (
                <img src={asset.asset.previewUrl} alt={asset.title} className="w-full h-full object-cover" loading="lazy" />
              ) : (
                <div className="w-full h-full flex items-center justify-center">{getAssetIcon(asset.asset?.assetType || 'unknown')}</div>
              )}
            </div>

            <div className="flex-1 min-w-0">
              <div className="flex items-start justify-between mb-2">
                <div className="flex-1">
                  <Link href={`/marketplace/asset/${asset.id}`}>
                    <h3 className="text-lg font-semibold hover:text-primary cursor-pointer" data-testid={`asset-title-${asset.id}`}>{asset.title}</h3>
                  </Link>
                  <p className="text-sm text-muted-foreground line-clamp-2 mt-1">{asset.description || 'No description available'}</p>
                </div>
                <div className="text-right ml-4">
                  <div className="text-lg font-bold text-green-600" data-testid={`asset-price-${asset.id}`}>{formatPrice(asset.price)}</div>
                  {asset.isExclusive && (<Badge variant="secondary" className="mt-1">Exclusive</Badge>)}
                </div>
              </div>

              <div className="flex items-center gap-4 text-sm text-muted-foreground mb-3">
                <div className="flex items-center">
                  <User className="h-4 w-4 mr-1" />
                  <Link href={`/marketplace/creator/${asset.creator.id}`} className="hover:text-primary">{asset.creator.name || asset.creator.email}</Link>
                </div>
                <div className="flex items-center"><Calendar className="h-4 w-4 mr-1" />{new Date(asset.createdAt).toLocaleDateString()}</div>
                <div className="flex items-center"><DollarSign className="h-4 w-4 mr-1" />{asset.salesCount} sales</div>
              </div>

              <div className="flex items-center justify-between">
                <div className="flex flex-wrap gap-1">
                  {asset.asset?.category && (<Badge variant="outline">{asset.asset.category}</Badge>)}
                  {(asset.asset?.tags || []).slice(0, 3).map((tag, i) => (<Badge key={i} variant="outline" className="text-xs">{tag}</Badge>))}
                </div>
                <div className="flex gap-2">
                  <Link href={`/marketplace/asset/${asset.id}`}>
                    <Button variant="outline" size="sm" data-testid={`button-view-${asset.id}`}> <Eye className="h-4 w-4 mr-2" /> View </Button>
                  </Link>
                  <Button onClick={() => onAddToCart(asset)} disabled={isInCart} size="sm" data-testid={`button-add-cart-${asset.id}`}>
                    <ShoppingCart className="h-4 w-4 mr-2" /> {isInCart ? 'In Cart' : 'Add to Cart'}
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    );
  }

  // Grid
  return (
    <Card className="group overflow-hidden hover:shadow-lg transition-all duration-300" data-testid={`asset-card-${asset.id}`}>
      <div className="aspect-video bg-gray-100 dark:bg-gray-800 overflow-hidden relative">
        {asset.asset?.previewUrl ? (
          <img src={asset.asset.previewUrl} alt={asset.title} className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300" loading="lazy" />
        ) : (
          <div className="w-full h-full flex items-center justify-center">{getAssetIcon(asset.asset?.assetType || 'unknown')}</div>
        )}
        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
          <Link href={`/marketplace/asset/${asset.id}`}><Button variant="secondary" size="sm" data-testid={`button-view-details-${asset.id}`}><Eye className="h-4 w-4 mr-2" /> View Details</Button></Link>
        </div>
        {asset.isExclusive && (<Badge className="absolute top-2 right-2 bg-yellow-500">Exclusive</Badge>)}
      </div>

      <CardContent className="p-4">
        <div className="mb-2">
          <Link href={`/marketplace/asset/${asset.id}`}>
            <h3 className="font-semibold hover:text-primary transition-colors cursor-pointer" data-testid={`asset-title-${asset.id}`}>{asset.title}</h3>
          </Link>
          <p className="text-sm text-muted-foreground line-clamp-2 mt-1">{asset.description || 'No description available'}</p>
        </div>

        <div className="flex items-center gap-2 text-xs text-muted-foreground mb-3">
          <div className="flex items-center">
            <User className="h-3 w-3 mr-1" />
            <Link href={`/marketplace/creator/${asset.creator.id}`} className="hover:text-primary truncate">{asset.creator.name || asset.creator.email}</Link>
          </div>
          <span>•</span>
          <div className="flex items-center">
            <DollarSign className="h-3 w-3 mr-1" />
            {asset.salesCount} sales
          </div>
        </div>

        <div className="flex items-center justify-between mb-3">
          <div className="text-lg font-bold text-green-600" data-testid={`asset-price-${asset.id}`}>{formatPrice(asset.price)}</div>
          <div className="flex gap-1">
            {asset.asset?.category && (<Badge variant="outline" className="text-xs">{asset.asset.category}</Badge>)}
            {asset.isExclusive && (<Badge variant="secondary" className="text-xs">Exclusive</Badge>)}
          </div>
        </div>

        <div className="flex gap-2">
          <Link href={`/marketplace/asset/${asset.id}`} className="flex-1">
            <Button variant="outline" size="sm" className="w-full" data-testid={`button-view-${asset.id}`}>
              <Eye className="h-4 w-4 mr-2" /> View
            </Button>
          </Link>
          <Button onClick={() => onAddToCart(asset)} disabled={isInCart} size="sm" className="flex-1" data-testid={`button-add-cart-${asset.id}`}>
            <ShoppingCart className="h-4 w-4 mr-2" /> {isInCart ? 'In Cart' : 'Add to Cart'}
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}