import React, { useEffect, useMemo, useState, useRef, useCallback } from "react";
import { useParams, useLocation } from "wouter";
import { LogoComposer } from "../types/logo-types";
import { ArrowLeft, Download, Brush, PenTool, FileText, RotateCcw } from "lucide-react";
import { Link } from "wouter";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { applySvgColor, resolveSvgColorsForExport, normalizeSvgWithLogoRoot } from "@/utils/svg";
import { Toolbar } from "@/components/Toolbar";
import { ExportModal } from "@/components/export/ExportModal";
import { FontPicker } from "@/components/logo/FontPicker";
import { useToast } from "@/hooks/use-toast";
import { prepareSvgForExport } from "@/lib/prepareSvgForExport";
import { Button } from "@/components/ui/button";

export default function LogoCustomizerPage() {
  const { composerId } = useParams<{ composerId: string }>();
  // Extract the actual template ID from composerId
  // Handle: "logo-composer-coffee-cup-v1" -> "coffee-cup-v1"
  // Handle: "logo-composer-logo-wordmark-coffee-cup" -> "coffee-cup" (legacy)
  let id = composerId?.startsWith('logo-composer-') 
    ? composerId.replace('logo-composer-', '') 
    : composerId;
  // Strip legacy "logo-wordmark-" prefix if present
  id = id?.startsWith('logo-wordmark-') 
    ? id.replace('logo-wordmark-', '') 
    : id;
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [template, setTemplate] = useState<LogoComposer | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  // Export modal state
  const [exportModalOpen, setExportModalOpen] = useState(false);

  // Form state
  const [displayName, setDisplayName] = useState("");
  const [colors, setColors] = useState<{ primary: string; secondary: string; accent: string }>({
    primary: "#000000",
    secondary: "#000000", 
    accent: "#000000",
  });
  
  // Text colors state
  const [textColors, setTextColors] = useState<{ brandName: string; tagline: string; estYear: string }>({
    brandName: "#000000",
    tagline: "#000000",
    estYear: "#000000",
  });
  const [brandName, setBrandName] = useState("");
  const [tagline, setTagline] = useState("");
  const [estYear, setEstYear] = useState("");
  const [tags, setTags] = useState("modern, emblem");
  const [logoSize, setLogoSize] = useState(100); // Logo/image size percentage (100% = 25% of preview pane)
  const [maxLogoScale, setMaxLogoScale] = useState(2.0); // Maximum allowed scale based on bbox
  const [isDragging, setIsDragging] = useState(false); // Track dragging state for visual guide
  const baseScaleRef = useRef(1.0); // Baseline scale for logo fitting
  const originalColorsRef = useRef<{ primary: string; secondary: string; accent: string }>({
    primary: "#000000",
    secondary: "#000000",
    accent: "#000000"
  }); // Store original template colors
  const [colorsModified, setColorsModified] = useState(false); // Track if user has changed colors
  const [brandNameSize, setBrandNameSize] = useState(100);
  const [taglineSize, setTaglineSize] = useState(100);
  const [estYearSize, setEstYearSize] = useState(100);
  const [brandNameFont, setBrandNameFont] = useState("Inter");
  const [taglineFont, setTaglineFont] = useState("Inter");
  const [estYearFont, setEstYearFont] = useState("Inter");
  
  // Shape state
  const [shapes, setShapes] = useState<{
    circle: { enabled: boolean; quantity: number; positions: Array<{ x: number; y: number; id: string }> };
    square: { enabled: boolean; quantity: number; positions: Array<{ x: number; y: number; id: string }> };
    triangle: { enabled: boolean; quantity: number; positions: Array<{ x: number; y: number; id: string }> };
    diamond: { enabled: boolean; quantity: number; positions: Array<{ x: number; y: number; id: string }> };
  }>({
    circle: { enabled: false, quantity: 1, positions: [] },
    square: { enabled: false, quantity: 1, positions: [] },
    triangle: { enabled: false, quantity: 1, positions: [] },
    diamond: { enabled: false, quantity: 1, positions: [] }
  });
  const [shapeSize, setShapeSize] = useState(100); // Shape size percentage
  const [shapeColor, setShapeColor] = useState("#000000"); // Shape color
  const [shapeRotation, setShapeRotation] = useState(0); // Shape rotation in degrees
  const [shapeFill, setShapeFill] = useState({
    circle: false,
    square: false,
    triangle: false,
    diamond: false
  }); // Shape fill settings
  
  // Position state for drag and drop (percentage-based)
  const [positions, setPositions] = useState<{
    [key: string]: { xPct: number; yPct: number }
  }>({});
  const [previewRef, setPreviewRef] = useState<HTMLDivElement | null>(null);
  const [logoRotation, setLogoRotation] = useState(0); // Logo rotation in degrees
  const [logoTx, setLogoTx] = useState(0); // Logo translation X in SVG units (centered by default)
  const [logoTy, setLogoTy] = useState(0); // Logo translation Y in SVG units (centered by default)

  // Ref to #logo-root for direct DOM manipulation (SN's smooth drag spec)
  const logoRootRef = useRef<SVGGElement | null>(null);
  
  // Ref to store original state for "Reset to Original" functionality
  const originalStateRef = useRef<{
    template: LogoComposer | null;
    colors: { primary: string; secondary: string; accent: string };
    textColors: { brandName: string; tagline: string; estYear: string };
    brandName: string;
    tagline: string;
    estYear: string;
    logoSize: number;
    logoRotation: number;
    logoTx: number;
    logoTy: number;
    brandNameSize: number;
    taglineSize: number;
    estYearSize: number;
    brandNameFont: string;
    taglineFont: string;
    estYearFont: string;
    shapes: typeof shapes;
    shapeSize: number;
    shapeColor: string;
    shapeRotation: number;
    shapeFill: typeof shapeFill;
    positions: typeof positions;
  } | null>(null);
  
  // Drag state using useRef as per SN's specification
  const dragRef = useRef<{
    id: string | null;
    startXPct: number;
    startYPct: number;
    startTx: number; // For logo translation
    startTy: number; // For logo translation
    originX: number;
    originY: number;
    rect: DOMRect | null;
    latestX: number; // Latest mouse X for rAF loop
    latestY: number; // Latest mouse Y for rAF loop
    dragging: boolean; // Flag for rAF loop
    scale: number; // Cached scale for transform
    rotation: number; // Cached rotation for transform
    rafId: number | null; // rAF ID for cleanup
  }>({ 
    id: null, 
    startXPct: 0, 
    startYPct: 0, 
    startTx: 0, 
    startTy: 0, 
    originX: 0, 
    originY: 0, 
    rect: null,
    latestX: 0,
    latestY: 0,
    dragging: false,
    scale: 1,
    rotation: 0,
    rafId: null
  });

  // Legacy state mapping for backward compatibility - convert xPct/yPct to x/y format
  const textPositions = useMemo(() => {
    const converted: { [key: string]: { x: number; y: number } } = {};
    Object.keys(positions).forEach(key => {
      if (positions[key]) {
        converted[key] = {
          x: positions[key].xPct,
          y: positions[key].yPct
        };
      }
    });
    return converted;
  }, [positions]);
  
  const logoPosition = { x: positions['logo']?.xPct || 50, y: positions['logo']?.yPct || 50 };

  // Helper function to clamp values between 0 and 100
  const clamp = (value: number, min: number = 0, max: number = 100) => Math.max(min, Math.min(max, value));
  
  // Percent to pixel converters for canonical preview space (SN's spec)
  const getPreviewRect = () => {
    const wrapper = document.querySelector('[data-logo-preview]') as HTMLElement;
    return wrapper?.getBoundingClientRect() || null;
  };
  
  const px = (pct: number, rect: DOMRect) => (pct / 100) * rect.width;
  const py = (pct: number, rect: DOMRect) => (pct / 100) * rect.height;
  
  // Convert px back to percent
  const toPctX = (pixels: number, rect: DOMRect) => (pixels / rect.width) * 100;
  const toPctY = (pixels: number, rect: DOMRect) => (pixels / rect.height) * 100;

  // Reset functions for text positions
  const resetTextPosition = (textId: 'brand-name' | 'tagline' | 'est-year') => {
    const defaultPositions = {
      'brand-name': { xPct: 25, yPct: 50 },
      'tagline': { xPct: 50, yPct: 50 },
      'est-year': { xPct: 75, yPct: 50 }
    };
    
    setPositions(prev => ({
      ...prev,
      [textId]: defaultPositions[textId]
    }));
  };

  // Get current position for any element ID
  const getCurrentPosition = (elementId: string) => {
    // Check positions state first
    if (positions[elementId]) {
      return { xPct: positions[elementId].xPct, yPct: positions[elementId].yPct };
    }
    
    // Check shapes for shape IDs
    let foundPosition = null;
    Object.values(shapes).forEach(shape => {
      const pos = shape.positions.find(p => p.id === elementId);
      if (pos) foundPosition = { xPct: pos.x, yPct: pos.y };
    });
    if (foundPosition) return foundPosition;
    
    // Default positions - lined up horizontally at center
    if (elementId === 'brand-name') return { xPct: 25, yPct: 50 };
    if (elementId === 'tagline') return { xPct: 50, yPct: 50 };
    if (elementId === 'est-year') return { xPct: 75, yPct: 50 };
    if (elementId === 'logo') return { xPct: 50, yPct: 50 };
    
    return { xPct: 50, yPct: 50 };
  };

  // Silky-smooth drag system (SN's spec: ref-based, single rAF loop, DOM mutation only)
  const handleDragStart = (e: React.PointerEvent, dragId: string) => {
    // Check if target is a text/shape overlay - bail if so (for logo drag)
    if (dragId === 'logo') {
      const target = e.target as HTMLElement;
      if (target.closest('[data-text-id], [data-shape-id]')) {
        return; // Don't start logo drag if clicking on text/shape
      }
    }
    
    // Use the wrapper with data-logo-preview for correct bounding rect
    const wrapper = document.querySelector('[data-logo-preview]') as HTMLElement;
    if (!wrapper) return;
    
    const rect = wrapper.getBoundingClientRect();
    
    // Get current position for this element
    const currentPos = getCurrentPosition(dragId);
    
    // For logo: cache scale and rotation, start rAF loop
    if (dragId === 'logo') {
      dragRef.current = {
        id: dragId,
        startXPct: 0,
        startYPct: 0,
        startTx: logoTx,
        startTy: logoTy,
        originX: e.clientX,
        originY: e.clientY,
        rect,
        latestX: e.clientX,
        latestY: e.clientY,
        dragging: true,
        scale: logoSize / 100, // Convert percentage to scale factor
        rotation: logoRotation,
        rafId: null
      };
      
      // Show drag guide
      setIsDragging(true);
      
      // Start the rAF loop
      const rafLoop = () => {
        if (!dragRef.current.dragging) return;
        
        // Calculate screen-space delta
        const dx = dragRef.current.latestX - dragRef.current.originX;
        const dy = dragRef.current.latestY - dragRef.current.originY;
        
        // Convert screen pixels to SVG coordinate units
        // SVG is scaled by preserveAspectRatio, so we need to account for that
        const svgElement = logoRootRef.current?.ownerSVGElement;
        let scaleFactorX = 1, scaleFactorY = 1;
        if (svgElement && dragRef.current.rect) {
          const viewBox = svgElement.viewBox.baseVal;
          const rect = svgElement.getBoundingClientRect();
          if (rect.width > 0 && viewBox.width > 0) {
            scaleFactorX = viewBox.width / rect.width;
            scaleFactorY = viewBox.height / rect.height;
          }
        }
        
        // Apply conversion: screen pixels → SVG units
        let tx = dragRef.current.startTx + (dx * scaleFactorX);
        let ty = dragRef.current.startTy + (dy * scaleFactorY);
        
        // Clamp using screen-space bbox (once per frame)
        if (logoRootRef.current && dragRef.current.rect) {
          // Get viewBox center for transform calculation
          const svgElement = logoRootRef.current.ownerSVGElement;
          let centerX = 200, centerY = 200;
          if (svgElement) {
            const viewBox = svgElement.viewBox.baseVal;
            centerX = viewBox.width / 2;
            centerY = viewBox.height / 2;
          }
          
          // Build transform matching buildCompositeSVG format
          const actualScale = dragRef.current.scale * baseScaleRef.current * 0.25;
          const transformStr = `translate(${centerX + tx} ${centerY + ty}) rotate(${dragRef.current.rotation}) scale(${actualScale}) translate(${-centerX} ${-centerY})`;
          logoRootRef.current.setAttribute('transform', transformStr);
          
          const r = logoRootRef.current.getBoundingClientRect();
          const previewRect = dragRef.current.rect;
          
          // Calculate clamping corrections in screen pixels
          let dxCorrect = 0, dyCorrect = 0;
          if (r.left < previewRect.left) dxCorrect = previewRect.left - r.left;
          if (r.right > previewRect.right) dxCorrect = Math.min(dxCorrect, previewRect.right - r.right);
          if (r.top < previewRect.top) dyCorrect = previewRect.top - r.top;
          if (r.bottom > previewRect.bottom) dyCorrect = Math.min(dyCorrect, previewRect.bottom - r.bottom);
          
          // Convert corrections to SVG units before applying (critical for multi-drag consistency)
          tx += dxCorrect * scaleFactorX;
          ty += dyCorrect * scaleFactorY;
        }
        
        // Apply final transform with centering
        if (logoRootRef.current) {
          // Get viewBox center for transform calculation
          const svgElement = logoRootRef.current.ownerSVGElement;
          let centerX = 200, centerY = 200;
          if (svgElement) {
            const viewBox = svgElement.viewBox.baseVal;
            centerX = viewBox.width / 2;
            centerY = viewBox.height / 2;
          }
          
          // Build transform matching buildCompositeSVG format
          const actualScale = dragRef.current.scale * baseScaleRef.current * 0.25;
          const transformStr = `translate(${centerX + tx} ${centerY + ty}) rotate(${dragRef.current.rotation}) scale(${actualScale}) translate(${-centerX} ${-centerY})`;
          logoRootRef.current.setAttribute('transform', transformStr);
        }
        
        dragRef.current.rafId = requestAnimationFrame(rafLoop);
      };
      
      dragRef.current.rafId = requestAnimationFrame(rafLoop);
    } else {
      // Text/shape drag (keep existing logic)
      dragRef.current = {
        id: dragId,
        startXPct: currentPos.xPct,
        startYPct: currentPos.yPct,
        startTx: 0,
        startTy: 0,
        originX: e.clientX,
        originY: e.clientY,
        rect,
        latestX: e.clientX,
        latestY: e.clientY,
        dragging: false,
        scale: 1,
        rotation: 0,
        rafId: null
      };
    }
    
    // Set pointer capture and prevent default
    (e.currentTarget as HTMLElement).setPointerCapture?.(e.pointerId);
    e.preventDefault();
  };

  const handleDragMove = (e: React.PointerEvent) => {
    if (!dragRef.current.id || !dragRef.current.rect) return;
    
    e.preventDefault();
    
    const dragId = dragRef.current.id;
    
    // Logo drag: just update latest mouse position (rAF loop handles the rest)
    if (dragId === 'logo') {
      dragRef.current.latestX = e.clientX;
      dragRef.current.latestY = e.clientY;
      return;
    }
    
    // Text/shape drag uses percentage-based positioning
    const rect = dragRef.current.rect;
    const dxPct = ((e.clientX - dragRef.current.originX) / rect.width) * 100;
    const dyPct = ((e.clientY - dragRef.current.originY) / rect.height) * 100;
    
    let newXPct = dragRef.current.startXPct + dxPct;
    let newYPct = dragRef.current.startYPct + dyPct;
    
    // Size-aware clamping for text/shapes
    const element = document.querySelector(`[data-text-id="${dragId}"], [data-shape-id="${dragId}"]`) as HTMLElement;
    if (element) {
      const elRect = element.getBoundingClientRect();
      const halfWPct = toPctX(elRect.width / 2, rect);
      const halfHPct = toPctY(elRect.height / 2, rect);
      
      newXPct = clamp(newXPct, 0 + halfWPct, 100 - halfWPct);
      newYPct = clamp(newYPct, 0 + halfHPct, 100 - halfHPct);
    } else {
      newXPct = clamp(newXPct);
      newYPct = clamp(newYPct);
    }
    
    // Update positions state (text/shapes still use setState)
    requestAnimationFrame(() => {
      setPositions(prev => ({
        ...prev,
        [dragId]: { xPct: newXPct, yPct: newYPct }
      }));
      
      // Update shapes state if it's a shape
      if (dragId.length > 10) {
        setShapes(prev => {
          const newShapes = { ...prev };
          Object.keys(newShapes).forEach(shapeType => {
            const shapeData = { ...newShapes[shapeType as keyof typeof shapes] };
            const positionIndex = shapeData.positions.findIndex(p => p.id === dragId);
            if (positionIndex !== -1) {
              shapeData.positions = [...shapeData.positions];
              shapeData.positions[positionIndex] = {
                ...shapeData.positions[positionIndex],
                x: newXPct,
                y: newYPct
              };
              newShapes[shapeType as keyof typeof shapes] = shapeData;
            }
          });
          return newShapes;
        });
      }
    });
  };

  const handleDragEnd = (e: React.PointerEvent) => {
    if (dragRef.current.id) {
      const dragId = dragRef.current.id;
      
      // Logo drag: stop rAF loop and commit final tx/ty to state
      if (dragId === 'logo') {
        dragRef.current.dragging = false;
        if (dragRef.current.rafId !== null) {
          cancelAnimationFrame(dragRef.current.rafId);
        }
        
        // Commit final position to state (MUST convert screen pixels to SVG units)
        const dx = dragRef.current.latestX - dragRef.current.originX;
        const dy = dragRef.current.latestY - dragRef.current.originY;
        
        // Convert screen pixels to SVG coordinate units (same as rAF loop)
        const svgElement = logoRootRef.current?.ownerSVGElement;
        let scaleFactorX = 1, scaleFactorY = 1;
        if (svgElement && dragRef.current.rect) {
          const viewBox = svgElement.viewBox.baseVal;
          const rect = svgElement.getBoundingClientRect();
          if (rect.width > 0 && viewBox.width > 0) {
            scaleFactorX = viewBox.width / rect.width;
            scaleFactorY = viewBox.height / rect.height;
          }
        }
        
        const finalTx = dragRef.current.startTx + (dx * scaleFactorX);
        const finalTy = dragRef.current.startTy + (dy * scaleFactorY);
        
        // Apply clamping one final time
        if (logoRootRef.current && dragRef.current.rect) {
          const transformStr = `translate(${finalTx}, ${finalTy}) rotate(${dragRef.current.rotation}) scale(${dragRef.current.scale})`;
          logoRootRef.current.setAttribute('transform', transformStr);
          
          const r = logoRootRef.current.getBoundingClientRect();
          const previewRect = dragRef.current.rect;
          
          // Calculate clamping corrections in screen pixels
          let dxCorrect = 0, dyCorrect = 0;
          if (r.left < previewRect.left) dxCorrect = previewRect.left - r.left;
          if (r.right > previewRect.right) dxCorrect = Math.min(dxCorrect, previewRect.right - r.right);
          if (r.top < previewRect.top) dyCorrect = previewRect.top - r.top;
          if (r.bottom > previewRect.bottom) dyCorrect = Math.min(dyCorrect, previewRect.bottom - r.bottom);
          
          // Convert corrections to SVG units before storing in state
          setLogoTx(finalTx + (dxCorrect * scaleFactorX));
          setLogoTy(finalTy + (dyCorrect * scaleFactorY));
        } else {
          setLogoTx(finalTx);
          setLogoTy(finalTy);
        }
      }
      
      // Clear drag state
      dragRef.current = { 
        id: null, 
        startXPct: 0, 
        startYPct: 0, 
        startTx: 0,
        startTy: 0,
        originX: 0, 
        originY: 0, 
        rect: null,
        latestX: 0,
        latestY: 0,
        dragging: false,
        scale: 1,
        rotation: 0,
        rafId: null
      };
      
      // Hide drag guide
      setIsDragging(false);
      
      // Release capture
      (e.currentTarget as HTMLElement).releasePointerCapture?.(e.pointerId);
      e.preventDefault();
    }
  };


  // Helper function to generate random positions for shapes (percentage-based)
  const generateShapePositions = (quantity: number, shapeType: string) => {
    const positions = [];
    for (let i = 0; i < quantity; i++) {
      positions.push({
        x: 10 + Math.random() * 80, // Random x between 10-90% of frame width
        y: 10 + Math.random() * 80, // Random y between 10-90% of frame height
        id: `${shapeType}-${Date.now()}-${i}`
      });
    }
    return positions;
  };

  // Handle shape enable/disable
  const toggleShape = (shapeType: keyof typeof shapes) => {
    setShapes(prev => ({
      ...prev,
      [shapeType]: {
        ...prev[shapeType],
        enabled: !prev[shapeType].enabled,
        positions: !prev[shapeType].enabled 
          ? generateShapePositions(prev[shapeType].quantity, shapeType)
          : []
      }
    }));
  };


  // Handle shape quantity change
  const updateShapeQuantity = (shapeType: keyof typeof shapes, quantity: number) => {
    setShapes(prev => ({
      ...prev,
      [shapeType]: {
        ...prev[shapeType],
        quantity: Math.max(1, Math.min(10, quantity)), // Limit 1-10
        positions: prev[shapeType].enabled 
          ? generateShapePositions(Math.max(1, Math.min(10, quantity)), shapeType)
          : prev[shapeType].positions
      }
    }));
  };

  // Add a new shape of the specified type
  const addShape = (shapeType: keyof typeof shapes) => {
    setShapes(prev => {
      const currentShape = prev[shapeType];
      const isCurrentlyEnabled = currentShape.enabled;
      const currentQuantity = currentShape.quantity;
      
      // Don't add if already at maximum
      if (isCurrentlyEnabled && currentQuantity >= 10) {
        return prev; // No change if at max limit
      }
      
      // Generate a new position for the additional shape
      const newPosition = {
        x: 10 + Math.random() * 80, // Random x between 10-90% of frame width
        y: 10 + Math.random() * 80, // Random y between 10-90% of frame height
        id: `${shapeType}-${Date.now()}-${currentQuantity + 1}`
      };

      return {
        ...prev,
        [shapeType]: {
          ...currentShape,
          enabled: true, // Enable the shape if not already enabled
          quantity: isCurrentlyEnabled ? Math.min(10, currentQuantity + 1) : 1,
          positions: isCurrentlyEnabled 
            ? [...currentShape.positions, newPosition] // Add to existing positions
            : [newPosition] // Just add one shape when first enabling
        }
      };
    });
  };

  // Clear all shapes
  const clearAllShapes = () => {
    setShapes({
      circle: { enabled: false, quantity: 1, positions: [] },
      square: { enabled: false, quantity: 1, positions: [] },
      triangle: { enabled: false, quantity: 1, positions: [] },
      diamond: { enabled: false, quantity: 1, positions: [] }
    });
  };

  // Text shape state for rotation and skew (renamed from curve)
  type TextShape = {
    rotation: number;
    curve: number; // 0 = straight, >0 = arc up, <0 = arc down (now skew)
  };
  
  const [textShapes, setTextShapes] = useState<{
    [key: string]: TextShape
  }>({
    "brand-name": { rotation: 0, curve: 0 },
    "tagline": { rotation: 0, curve: 0 },
    "est-year": { rotation: 0, curve: 0 }
  });

  // New curve state for textPath curved text functionality
  type TextCurve = {
    enabled: boolean;
    type: "arc" | "bezier"; // Start with "arc" as per SN's spec
    radius: number; // px or % of preview width
    angleDeg: number; // sweep angle (-120° to +120°)
    direction: "up" | "down"; // flip curvature
    startOffsetPct: number; // 0-100, where text begins along path
  };

  const [textCurves, setTextCurves] = useState<{
    [key: string]: TextCurve
  }>({
    "brand-name": { 
      enabled: false, 
      type: "arc", 
      radius: 200, 
      angleDeg: 30, 
      direction: "up", 
      startOffsetPct: 0 
    },
    "tagline": { 
      enabled: false, 
      type: "arc", 
      radius: 200, 
      angleDeg: 30, 
      direction: "up", 
      startOffsetPct: 0 
    },
    "est-year": { 
      enabled: false, 
      type: "arc", 
      radius: 200, 
      angleDeg: 30, 
      direction: "up", 
      startOffsetPct: 0 
    }
  });

  // SVG Arc Generator for curved text (skew functionality)
  const buildArcPath = (x: number, y: number, width: number, curve: number) => {
    // curve is sagitta: how "tall" the arc bows (positive = up, negative = down)
    const x0 = x - width / 2;
    const x1 = x + width / 2;
    const y0 = y;
    const y1 = y;

    const arcFlag = curve > 0 ? 0 : 1;
    const rx = Math.abs(curve) + width / 2;

    return `M ${x0},${y0} A ${rx},${rx} 0 0,${arcFlag} ${x1},${y1}`;
  };

  // SVG TextPath Arc Generator for proper curved text
  const buildTextPathArc = (
    centerX: number, 
    centerY: number, 
    textWidth: number, 
    curve: TextCurve
  ) => {
    // Calculate arc radius based on curve parameters
    const baseRadius = Math.max(curve.radius || 200, textWidth / 2);
    
    // Convert angle to radians 
    const angleRad = (curve.angleDeg * Math.PI) / 180;
    
    // Calculate chord length (straight line distance between arc endpoints)
    const chordLength = Math.min(textWidth * 1.2, baseRadius * Math.sin(angleRad));
    
    // Calculate arc radius based on chord and angle
    const arcRadius = chordLength / (2 * Math.sin(angleRad / 2));
    
    // Start and end points for the arc
    const halfChord = chordLength / 2;
    const x1 = centerX - halfChord;
    const x2 = centerX + halfChord;
    
    // Calculate y position based on direction
    const arcHeight = arcRadius * (1 - Math.cos(angleRad / 2));
    const y1 = curve.direction === "up" ? centerY + arcHeight : centerY - arcHeight;
    const y2 = y1; // Arc endpoints are at same y level
    
    // Arc flags: large-arc-flag, sweep-flag
    const largeArcFlag = angleRad > Math.PI ? 1 : 0;
    const sweepFlag = curve.direction === "up" ? 0 : 1;
    
    // Generate SVG arc path using A command
    const pathData = `M ${x1},${y1} A ${arcRadius},${arcRadius} 0 ${largeArcFlag},${sweepFlag} ${x2},${y2}`;
    
    return {
      pathData,
      pathId: `textpath-${Math.random().toString(36).substr(2, 9)}`,
      arcRadius,
      chordLength
    };
  };

  // Capture #logo-root ref after SVG renders (SN's smooth drag spec)
  useEffect(() => {
    const logoRoot = document.querySelector('#logo-root') as SVGGElement | null;
    if (logoRoot) {
      logoRootRef.current = logoRoot;
      // Ensure no CSS transitions on #logo-root (causes lag)
      logoRoot.style.transition = 'none';
    }
  }, [template, logoTx, logoTy, logoSize, logoRotation]); // Re-capture when SVG changes

  // Load template data
  useEffect(() => {
    if (!id) return;
    
    const loadTemplate = async () => {
      try {
        setLoading(true);
        setError(null);
        
        // Draft flow: load from sessionStorage and short-circuit
        if (id.startsWith("draft-")) {
          const raw = sessionStorage.getItem(`logoDraft:${id}`);
          if (!raw) { 
            setError("This draft isn't available anymore. Generate a new logo and try again."); 
            setLoading(false);
            return; 
          }
          const { svg } = JSON.parse(raw) as { svg: string };
          setTemplate({
            id, 
            name: "AI Draft", 
            svg,
            defaultFields: {}, 
            defaultColors: { primary: "#231f20", secondary: "#6dc282", accent: "#00CB51" }
          });
          setLoading(false);
          return;
        }
        
        // First try to load from Firebase via backend API
        try {
          // Try original ID first, then try without "logo-wordmark-" prefix for Firebase templates
          let firebaseTemplate: any = null;
          try {
            const response = await fetch(`/api/logo-templates-firebase/${id}`);
            if (response.ok) {
              firebaseTemplate = await response.json();
            }
          } catch (fetchError) {
            console.warn(`Backend API fetch failed for ${id}:`, fetchError);
          }
          
          if (!firebaseTemplate && id.startsWith('logo-wordmark-')) {
            const shortId = id.replace('logo-wordmark-', '');
            try {
              const response = await fetch(`/api/logo-templates-firebase/${shortId}`);
              if (response.ok) {
                firebaseTemplate = await response.json();
              }
            } catch (fetchError) {
              console.warn(`Backend API fetch failed for ${shortId}:`, fetchError);
            }
          }
          
          if (firebaseTemplate) {
            // Map Firebase template structure to LogoTemplate
            const previewUrl =
              firebaseTemplate.assets?.previewDownloadUrl
              || firebaseTemplate.previewUrl                // if your doc stores it here
              || firebaseTemplate.thumbnailUrl              // if you saved a thumb
              || firebaseTemplate.assets?.svgDownloadUrl;   // last-resort: show the SVG in <img>

            const loadedTemplate: LogoTemplate = {
              id: firebaseTemplate.id,
              name: firebaseTemplate.name,
              previewUrl,            // ← guaranteed (unless doc truly has nothing)
              svg: "",               // will try to fetch real SVG below
              defaultFields: {
                Brand_Name: firebaseTemplate.defaults?.brandName || "Your Brand",
                Tagline: firebaseTemplate.defaults?.tagline || "Your Tagline",
                Est_Year: firebaseTemplate.defaults?.estYear || "2025"
              },
              fieldOrder: ["Brand_Name", "Tagline", "Est_Year"],
              defaultColors: { primary: "#000000", secondary: "#978752", accent: "#6dc282" }
            };
            
            // Try to fetch SVG content
            console.log(`📱 Starting SVG fetch for template:`, firebaseTemplate.id);
            console.log(`📱 Firebase template structure:`, firebaseTemplate);
            
            try {
              // If SVG URL is available, try fetching via backend proxy
              if (firebaseTemplate.assets?.svgDownloadUrl) {
                console.log(`📱 Trying backend proxy for:`, firebaseTemplate.assets.svgDownloadUrl);
                const proxyUrl = `/api/firebase-proxy?url=${encodeURIComponent(firebaseTemplate.assets.svgDownloadUrl)}`;
                console.log(`📱 Proxy URL:`, proxyUrl);
                
                const proxyResponse = await fetch(proxyUrl);
                console.log(`📱 Proxy response status:`, proxyResponse.status, proxyResponse.statusText);
                
                if (proxyResponse.ok) {
                  const svgContent = await proxyResponse.text();
                  console.log(`✅ Successfully fetched SVG via proxy (${svgContent.length} chars)`);
                  loadedTemplate.svg = svgContent;
                } else {
                  console.log(`❌ Proxy fetch failed:`, await proxyResponse.text());
                }
              } else {
                console.log(`📱 No SVG URL available for proxy fetch`);
              }
            } catch (svgError) {
              console.log(`❌ SVG fetch error:`, svgError);
              // This is expected behavior when Firebase Storage has access restrictions
              // The template will work with the preview image instead of interactive SVG
            }
            
            // Belt-and-suspenders: ensure previewUrl is never lost
            if (!loadedTemplate.previewUrl) {
              loadedTemplate.previewUrl =
                firebaseTemplate.assets?.previewDownloadUrl
                || firebaseTemplate.previewUrl
                || firebaseTemplate.thumbnailUrl
                || firebaseTemplate.assets?.svgDownloadUrl;
            }
            setTemplate(loadedTemplate);
            setDisplayName(firebaseTemplate.name);
            
            // Initialize form with template defaults
            setColors(loadedTemplate.defaultColors);
            originalColorsRef.current = loadedTemplate.defaultColors; // Store for reset functionality
            // Note: We don't set default values for text fields - they stay empty to show placeholders
            
            return; // Successfully loaded from Firebase
          }
        } catch (firebaseError) {
          console.log("Firebase template not found, falling back to manifest:", firebaseError);
        }
        
        // Fallback to manifest for static templates
        const manifestRes = await fetch("/site/data/manifest.logo.json", { cache: "no-cache" });
        if (!manifestRes.ok) throw new Error("Failed to load template manifest");
        const manifest = await manifestRes.json();
        
        const templateItem = manifest.items?.find((item: any) => item.id === id);
        if (!templateItem) throw new Error(`Template ${id} not found in Firebase or manifest`);
        
        // Load the SVG content
        const svgRes = await fetch(templateItem.svgUrl, { cache: "no-cache" });
        if (!svgRes.ok) throw new Error(`Failed to load SVG: ${templateItem.svgUrl}`);
        const svgText = await svgRes.text();
        
        const loadedTemplate: LogoTemplate = {
          id: templateItem.id,
          name: templateItem.name,
          previewUrl: templateItem.previewUrl,
          svg: svgText,
          defaultFields: templateItem.defaultFields,
          fieldOrder: templateItem.fieldOrder,
          defaultColors: templateItem.defaultColors
        };
        
        setTemplate(loadedTemplate);
        setDisplayName(templateItem.name);
        
        // Initialize form with template defaults
        if (templateItem.defaultColors) {
          setColors(templateItem.defaultColors);
          originalColorsRef.current = templateItem.defaultColors; // Store for reset functionality
        }
        // Note: We don't set default values for text fields - they stay empty to show placeholders
        
      } catch (e) {
        console.error(e);
        setError((e as any)?.message || "Failed to load template");
      } finally {
        setLoading(false);
      }
    };
    
    loadTemplate();
  }, [id]);

  // Capture original state once (first time template is set)
  useEffect(() => {
    if (template && !originalStateRef.current) {
      originalStateRef.current = JSON.parse(JSON.stringify({
        template,
        colors,
        textColors,
        brandName,
        tagline,
        estYear,
        logoSize,
        logoRotation,
        logoTx,
        logoTy,
        brandNameSize,
        taglineSize,
        estYearSize,
        brandNameFont,
        taglineFont,
        estYearFont,
        shapes,
        shapeSize,
        shapeColor,
        shapeRotation,
        shapeFill,
        positions
      }));
    }
  }, [template, colors, textColors, brandName, tagline, estYear, logoSize, logoRotation, logoTx, logoTy, brandNameSize, taglineSize, estYearSize, brandNameFont, taglineFont, estYearFont, shapes, shapeSize, shapeColor, shapeRotation, shapeFill, positions]);

  // Reset to original state
  const resetToOriginal = useCallback(() => {
    if (!originalStateRef.current) return;
    
    const original = JSON.parse(JSON.stringify(originalStateRef.current));
    setTemplate(original.template);
    setColors(original.colors);
    setTextColors(original.textColors);
    setBrandName(original.brandName);
    setTagline(original.tagline);
    setEstYear(original.estYear);
    setLogoSize(original.logoSize);
    setLogoRotation(original.logoRotation);
    setLogoTx(original.logoTx);
    setLogoTy(original.logoTy);
    setBrandNameSize(original.brandNameSize);
    setTaglineSize(original.taglineSize);
    setEstYearSize(original.estYearSize);
    setBrandNameFont(original.brandNameFont);
    setTaglineFont(original.taglineFont);
    setEstYearFont(original.estYearFont);
    setShapes(original.shapes);
    setShapeSize(original.shapeSize);
    setShapeColor(original.shapeColor);
    setShapeRotation(original.shapeRotation);
    setShapeFill(original.shapeFill);
    setPositions(original.positions);
    setColorsModified(false);
    
    toast({
      title: "Reset Complete",
      description: "Logo restored to original state",
      variant: "default"
    });
  }, [toast]);

  // Check if state has changed from original
  const hasChangedFromOriginal = useMemo(() => {
    if (!originalStateRef.current) return false;
    
    const current = {
      template,
      colors,
      textColors,
      brandName,
      tagline,
      estYear,
      logoSize,
      logoRotation,
      logoTx,
      logoTy,
      brandNameSize,
      taglineSize,
      estYearSize,
      brandNameFont,
      taglineFont,
      estYearFont,
      shapes,
      shapeSize,
      shapeColor,
      shapeRotation,
      shapeFill,
      positions
    };
    
    return JSON.stringify(current) !== JSON.stringify(originalStateRef.current);
  }, [template, colors, textColors, brandName, tagline, estYear, logoSize, logoRotation, logoTx, logoTy, brandNameSize, taglineSize, estYearSize, brandNameFont, taglineFont, estYearFont, shapes, shapeSize, shapeColor, shapeRotation, shapeFill, positions]);

  // Build base SVG with only color application (new overlay approach)
  const baseSvg = useMemo(() => {
    if (!template) return "";
    let svg = template.svg;

    // Remove any existing text placeholders to prevent conflicts
    svg = svg
      .replace(/\{Brand_Name\}/g, "")
      .replace(/\{Tagline\}/g, "")
      .replace(/\{Est_Year\}/g, "");

    // Remove any existing text elements to prevent conflicts
    svg = svg.replace(/<text[^>]*>.*?<\/text>/g, "");

    // Normalize SVG with #logo-root wrapper (SN's architecture)
    svg = normalizeSvgWithLogoRoot(svg);

    // Apply the color swatches to the SVG only if user has modified colors
    // This preserves original logo colors on first load
    if (colorsModified) {
      svg = applySvgColor(svg, colors);
    }

    // Simple SVG sizing - fit within container maintaining aspect ratio
    const sizeStyles = `display: block; width: 100%; height: 100%; object-fit: contain;`;
    
    if (svg.includes('style="')) {
      // SVG has existing style, append to it
      svg = svg.replace(
        /<svg([^>]*?)style="([^"]*)"/,
        `<svg$1style="$2 ${sizeStyles}"`
      );
    } else {
      // SVG has no style attribute, add one
      svg = svg.replace(
        /<svg/,
        `<svg style="${sizeStyles}"`
      );
    }

    return svg;
  }, [template, colors, logoSize, colorsModified]);


  // Build SVG string with tokens replaced and CSS vars injected (old approach - to be removed)
  const resolvedSvg = useMemo(() => {
    if (!template) return "";
    let svg = template.svg;
    let pathDefinitions = ""; // Store arc path definitions

    // Remove text tokens - the overlay text system will handle all text rendering
    svg = svg
      .replace(/\{Brand_Name\}/g, "")
      .replace(/\{Tagline\}/g, "")
      .replace(/\{Est_Year\}/g, "");
    
    // Remove all <text> elements from template SVG - overlay system handles all text
    svg = svg.replace(/<text[^>]*>[\s\S]*?<\/text>/g, "");

    // Ensure CSS vars are set on root <svg> for text colors and proper sizing
    svg = svg.replace(
      /<svg([^>]*?)>/,
      `<svg$1 style="--brand-name-color:${textColors.brandName};--tagline-color:${textColors.tagline};--est-year-color:${textColors.estYear}; width: 100%; height: 100%;" preserveAspectRatio="xMidYMid meet">`
    );

    // Collect text elements to add at the end for proper z-order
    const textElements: string[] = [];
    
    // Process existing text elements and collect them instead of replacing inline
    svg = svg.replace(/<text([^>]*?)>([^<]*?)<\/text>/g, (match, attrs, content) => {
      let sizeMultiplier = 1;
      let textId = "";
      
      // Decode HTML entities to properly match text content
      const decodeHtmlEntities = (str: string) => {
        const txt = document.createElement("textarea");
        txt.innerHTML = str;
        return txt.value;
      };
      
      const decodedContent = decodeHtmlEntities(content).trim();
      
      // Check which text field this belongs to and assign ID
      if (brandName.trim() && decodedContent === brandName.trim()) {
        sizeMultiplier = brandNameSize / 100;
        textId = "brand-name";
      } else if (tagline.trim() && decodedContent === tagline.trim()) {
        sizeMultiplier = taglineSize / 100;
        textId = "tagline";
      } else if (estYear.trim() && decodedContent === estYear.trim()) {
        sizeMultiplier = estYearSize / 100;
        textId = "est-year";
      } else {
        // This is existing text content that doesn't match user input
        // Apply default styling and let it render
        textId = "default-text";
        sizeMultiplier = 1;
      }
      
      // Extract existing font-size or use a reasonable default
      let baseSize = 14;
      const fontSizeMatch = attrs.match(/font-size="([^"]*?)"/);
      if (fontSizeMatch) {
        baseSize = parseFloat(fontSizeMatch[1]) || 14;
      }
      
      // Calculate new font size
      const scaledSize = baseSize * sizeMultiplier;
      
      // Extract existing position or set defaults within viewBox
      let x = 120; // Center of viewBox (238.34/2)
      let y = 20; // Start much higher to avoid overlapping with main content
      const xMatch = attrs.match(/x="([^"]*?)"/);
      const yMatch = attrs.match(/y="([^"]*?)"/);
      if (xMatch) x = parseFloat(xMatch[1]) || x;
      if (yMatch) y = parseFloat(yMatch[1]) || y;
      
      // Position text elements at top of SVG where there's usually empty space
      if (textId === "brand-name") y = 20;
      if (textId === "tagline") y = 200; // Bottom of SVG
      if (textId === "est-year") y = 40;
      
      // Apply custom position if set
      if (textId && textPositions[textId]) {
        x = textPositions[textId].x;
        y = textPositions[textId].y;
      }
      
      // Get text shape settings
      const shape = textShapes[textId] || { rotation: 0, curve: 0 };
      
      // Remove existing attributes and rebuild
      let newAttrs = attrs.replace(/\s*font-size="[^"]*"/g, '')
                         .replace(/\s*fill="[^"]*"/g, ''); // Remove existing fill to apply new color
      newAttrs = newAttrs.replace(/\s*style="[^"]*"/g, '');
      newAttrs = newAttrs.replace(/\s*x="[^"]*"/g, '');
      newAttrs = newAttrs.replace(/\s*y="[^"]*"/g, '');
      newAttrs = newAttrs.replace(/\s*id="[^"]*"/g, '');
      newAttrs = newAttrs.replace(/\s*transform="[^"]*"/g, '');
      
      // If curve !== 0, render text along a textPath
      if (shape.curve !== 0) {
        const textWidth = content.length * scaledSize * 0.6; // Rough estimate
        const arcPath = buildArcPath(x, y, textWidth, shape.curve);
        const arcId = `arc-${textId}`;
        
        // Add path definition - path updates automatically when x,y changes via useMemo dependencies
        pathDefinitions += `<path id="${arcId}" d="${arcPath}" />`;
        
        // Apply rotation to curved text as well
        let transform = "";
        if (shape.rotation !== 0) {
          transform = ` transform="rotate(${shape.rotation} ${x} ${y})"`;
        }
        
        // Get text color based on textId
        let textColor = "#000000";
        if (textId === "brand-name") textColor = textColors.brandName;
        else if (textId === "tagline") textColor = textColors.tagline;
        else if (textId === "est-year") textColor = textColors.estYear;
        else textColor = "#000000"; // default color for existing text
        
        // Add new attributes for curved text - PRESERVE position data for drag handlers
        newAttrs += ` id="${textId}" x="${x}" y="${y}" font-size="${scaledSize}" fill="${textColor}" font-weight="bold" data-curve="${shape.curve}" data-text-width="${textWidth}"${transform} style="cursor: move; user-select: none;"`;
        
        const textElement = `<text${newAttrs}><textPath href="#${arcId}" startOffset="50%" text-anchor="middle">${content}</textPath></text>`;
        textElements.push(textElement);
        return ""; // Remove from original position
      } else {
        // Regular text with optional rotation
        let transform = "";
        if (shape.rotation !== 0) {
          transform = `transform="rotate(${shape.rotation} ${x} ${y})"`;
        }
        
        // Get text color based on textId
        let textColor = "#000000";
        if (textId === "brand-name") textColor = textColors.brandName;
        else if (textId === "tagline") textColor = textColors.tagline;
        else if (textId === "est-year") textColor = textColors.estYear;
        else textColor = "#000000"; // default color for existing text
        
        // Add new attributes for straight text using actual scaled size
        newAttrs += ` id="${textId}" x="${x}" y="${y}" font-size="${scaledSize}" fill="${textColor}" text-anchor="middle" font-weight="bold" ${transform} style="cursor: move; user-select: none;"`;
        
        const textElement = `<text${newAttrs}>${content}</text>`;
        textElements.push(textElement);
        return ""; // Remove from original position
      }
    });

    // Insert path definitions after the opening <svg> tag
    if (pathDefinitions) {
      svg = svg.replace(/(<svg[^>]*>)/, `$1<defs>${pathDefinitions}</defs>`);
    }

    // Add all text elements at the very end for proper z-order (they render on top)
    if (textElements.length > 0) {
      svg = svg.replace(/<\/svg>$/, `${textElements.join("")}</svg>`);
    }

    // Text elements successfully processed and positioned at end of SVG

    // Apply the color swatches to the SVG using currentColor and CSS variables
    svg = applySvgColor(svg, colors);
    
    // Re-apply individual text colors after applySvgColor (which converts to currentColor)
    // Set specific colors for text elements that need individual control
    svg = svg.replace(/(<text[^>]*id="brand-name"[^>]*)fill="[^"]*"/g, `$1fill="${textColors.brandName}"`);
    svg = svg.replace(/(<text[^>]*id="tagline"[^>]*)fill="[^"]*"/g, `$1fill="${textColors.tagline}"`);
    svg = svg.replace(/(<text[^>]*id="est-year"[^>]*)fill="[^"]*"/g, `$1fill="${textColors.estYear}"`);

    // Generate shape elements and add them to the SVG
    const shapeElements: string[] = [];
    
    // Calculate actual shape size from percentage (base size 30px at 100%)
    const actualShapeSize = (30 * shapeSize) / 100;
    
    // Add circles
    if (shapes.circle.enabled) {
      shapes.circle.positions.forEach(pos => {
        shapeElements.push(
          `<circle cx="${pos.x}" cy="${pos.y}" r="${actualShapeSize / 2}" fill="${shapeFill.circle ? shapeColor : 'none'}" stroke="${shapeColor}" stroke-width="2" data-shape-id="${pos.id}" style="cursor: move;" />`
        );
      });
    }
    
    // Add squares  
    if (shapes.square.enabled) {
      shapes.square.positions.forEach(pos => {
        shapeElements.push(
          `<rect x="${pos.x - actualShapeSize / 2}" y="${pos.y - actualShapeSize / 2}" width="${actualShapeSize}" height="${actualShapeSize}" fill="${shapeFill.square ? shapeColor : 'none'}" stroke="${shapeColor}" stroke-width="2" data-shape-id="${pos.id}" style="cursor: move;" />`
        );
      });
    }
    
    // Add triangles
    if (shapes.triangle.enabled) {
      shapes.triangle.positions.forEach(pos => {
        const size = actualShapeSize;
        const points = `${pos.x},${pos.y - size/2} ${pos.x + size/2},${pos.y + size/2} ${pos.x - size/2},${pos.y + size/2}`;
        shapeElements.push(
          `<polygon points="${points}" fill="${shapeFill.triangle ? shapeColor : 'none'}" stroke="${shapeColor}" stroke-width="2" data-shape-id="${pos.id}" style="cursor: move;" />`
        );
      });
    }
    
    // Add diamonds
    if (shapes.diamond.enabled) {
      shapes.diamond.positions.forEach(pos => {
        const size = actualShapeSize;
        const points = `${pos.x},${pos.y - size/2} ${pos.x + size/2},${pos.y} ${pos.x},${pos.y + size/2} ${pos.x - size/2},${pos.y}`;
        shapeElements.push(
          `<polygon points="${points}" fill="${shapeFill.diamond ? shapeColor : 'none'}" stroke="${shapeColor}" stroke-width="2" data-shape-id="${pos.id}" style="cursor: move;" />`
        );
      });
    }

    // Insert shapes before the closing </svg> tag
    if (shapeElements.length > 0) {
      svg = svg.replace('</svg>', shapeElements.join('') + '</svg>');
    }

    return svg;
  }, [template, brandName, tagline, estYear, colors, textColors, logoSize, brandNameSize, taglineSize, estYearSize, textPositions, textShapes, shapes, shapeSize, shapeColor, shapeFill, brandNameFont, taglineFont, estYearFont]);



  // Helper function to resolve CSS variables and currentColor to actual hex values
  const resolveColorValue = (colorValue: string): string => {
    if (colorValue === 'var(--primary)') return colors.primary;
    if (colorValue === 'var(--secondary)') return colors.secondary;
    if (colorValue === 'var(--accent)') return colors.accent;
    if (colorValue === 'currentColor') return colors.primary; // Default currentColor to primary
    return colorValue; // Return as-is if it's already a hex value
  };

  // Build composite SVG with overlay elements
  // includeText: false for preview (overlay handles text), true for exports
  // includeShapes: false for preview (overlay handles shapes), true for exports
  const buildCompositeSVG = (includeText = true, includeShapes = true) => {
    if (!baseSvg) return "";
    
    let svg = baseSvg;
    
    // Apply logo transforms to #logo-root: translate → rotate → scale (SN's spec)
    const viewBoxMatch = svg.match(/viewBox="([^"]*?)"/);
    let centerX = 200, centerY = 200; // Default center
    let viewBoxWidth = 400, viewBoxHeight = 400;
    if (viewBoxMatch) {
      const parts = viewBoxMatch[1].split(/\s+/);
      viewBoxWidth = parseFloat(parts[2]);
      viewBoxHeight = parseFloat(parts[3]);
      centerX = viewBoxWidth / 2;
      centerY = viewBoxHeight / 2;
    }
    
    // Build transform string in exact order: translate → rotate → scale
    // The logo needs to be centered when logoTx=0 and logoTy=0
    // We'll apply transforms around the viewBox center point
    const transforms = [];
    
    // User translation (relative to center)
    transforms.push(`translate(${centerX + logoTx} ${centerY + logoTy})`);
    
    if (logoRotation !== 0) {
      transforms.push(`rotate(${logoRotation})`);
    }
    
    // Scale: At 100%, logo should be 25% of preview pane height (96px for 384px pane)
    // Preview pane is 384px, viewBox is typically 400 units
    // Target: 100% slider = 25% of viewBox = 100 units
    const targetSize = viewBoxHeight * 0.25; // 25% of viewBox height
    const scale = (logoSize / 100) * baseScaleRef.current * (targetSize / viewBoxHeight);
    transforms.push(`scale(${scale})`);
    
    // Translate back to origin for proper centering
    transforms.push(`translate(${-centerX} ${-centerY})`);
    
    // Apply transform to #logo-root group
    if (transforms.length > 0) {
      const transformStr = transforms.join(' ');
      svg = svg.replace(/<g id="logo-root"([^>]*)>/, `<g id="logo-root"$1 transform="${transformStr}">`);
    }
    
    // SVG dimensions for positioning calculations (reuse viewBoxMatch from above)
    const svgWidth = 400; // Default width for calculations
    const svgHeight = 400; // Default height for calculations
    
    const textElements: string[] = [];
    const shapeElements: string[] = [];
    const pathDefinitions: string[] = []; // Store arc path definitions for textPath
    
    // Add text elements (only for exports, not for preview where overlays handle text)
    if (includeText && brandName) {
      const pos = textPositions['brand-name'] || { x: 50, y: 20 };
      const x = (pos.x / 100) * svgWidth;
      const y = (pos.y / 100) * svgHeight;
      const fontSize = (brandNameSize / 100) * 16;
      const rotation = textShapes['brand-name']?.rotation || 0;
      const curve = textCurves['brand-name'];
      
      if (curve?.enabled) {
        // Use textPath for curved text
        const textWidth = brandName.length * fontSize * 0.6; // Estimate text width
        const arcData = buildTextPathArc(x, y, textWidth, curve);
        
        // Add path definition
        pathDefinitions.push(`<path id="${arcData.pathId}" d="${arcData.pathData}" fill="none" stroke="none" />`);
        
        // Create textPath element
        const startOffsetAttr = curve.startOffsetPct > 0 ? ` startOffset="${curve.startOffsetPct}%"` : '';
        textElements.push(
          `<text font-family="${brandNameFont}, sans-serif" font-size="${fontSize}" fill="${resolveColorValue(textColors.brandName)}" font-weight="bold" text-anchor="middle" transform="rotate(${rotation} ${x} ${y})">
            <textPath href="#${arcData.pathId}"${startOffsetAttr}>${brandName}</textPath>
          </text>`
        );
      } else {
        // Use regular text
        textElements.push(
          `<text x="${x}" y="${y}" font-family="${brandNameFont}, sans-serif" font-size="${fontSize}" fill="${resolveColorValue(textColors.brandName)}" font-weight="bold" text-anchor="middle" transform="rotate(${rotation} ${x} ${y})">${brandName}</text>`
        );
      }
    }
    
    if (includeText && tagline) {
      const pos = textPositions['tagline'] || { x: 50, y: 70 };
      const x = (pos.x / 100) * svgWidth;
      const y = (pos.y / 100) * svgHeight;
      const fontSize = (taglineSize / 100) * 14;
      const rotation = textShapes['tagline']?.rotation || 0;
      const curve = textCurves['tagline'];
      
      if (curve?.enabled) {
        // Use textPath for curved text
        const textWidth = tagline.length * fontSize * 0.6; // Estimate text width
        const arcData = buildTextPathArc(x, y, textWidth, curve);
        
        // Add path definition
        pathDefinitions.push(`<path id="${arcData.pathId}" d="${arcData.pathData}" fill="none" stroke="none" />`);
        
        // Create textPath element
        const startOffsetAttr = curve.startOffsetPct > 0 ? ` startOffset="${curve.startOffsetPct}%"` : '';
        textElements.push(
          `<text font-family="${taglineFont}, sans-serif" font-size="${fontSize}" fill="${resolveColorValue(textColors.tagline)}" text-anchor="middle" transform="rotate(${rotation} ${x} ${y})">
            <textPath href="#${arcData.pathId}"${startOffsetAttr}>${tagline}</textPath>
          </text>`
        );
      } else {
        // Use regular text
        textElements.push(
          `<text x="${x}" y="${y}" font-family="${taglineFont}, sans-serif" font-size="${fontSize}" fill="${resolveColorValue(textColors.tagline)}" text-anchor="middle" transform="rotate(${rotation} ${x} ${y})">${tagline}</text>`
        );
      }
    }
    
    if (includeText && estYear) {
      const pos = textPositions['est-year'] || { x: 50, y: 45 };
      const x = (pos.x / 100) * svgWidth;
      const y = (pos.y / 100) * svgHeight;
      const fontSize = (estYearSize / 100) * 12;
      const rotation = textShapes['est-year']?.rotation || 0;
      const curve = textCurves['est-year'];
      
      if (curve?.enabled) {
        // Use textPath for curved text
        const textWidth = estYear.length * fontSize * 0.6; // Estimate text width
        const arcData = buildTextPathArc(x, y, textWidth, curve);
        
        // Add path definition
        pathDefinitions.push(`<path id="${arcData.pathId}" d="${arcData.pathData}" fill="none" stroke="none" />`);
        
        // Create textPath element
        const startOffsetAttr = curve.startOffsetPct > 0 ? ` startOffset="${curve.startOffsetPct}%"` : '';
        textElements.push(
          `<text font-family="${estYearFont}, sans-serif" font-size="${fontSize}" fill="${resolveColorValue(textColors.estYear)}" text-anchor="middle" transform="rotate(${rotation} ${x} ${y})">
            <textPath href="#${arcData.pathId}"${startOffsetAttr}>${estYear}</textPath>
          </text>`
        );
      } else {
        // Use regular text
        textElements.push(
          `<text x="${x}" y="${y}" font-family="${estYearFont}, sans-serif" font-size="${fontSize}" fill="${resolveColorValue(textColors.estYear)}" text-anchor="middle" transform="rotate(${rotation} ${x} ${y})">${estYear}</text>`
        );
      }
    }
    
    // Add shape elements (only for exports, not for preview where overlays handle shapes)
    if (includeShapes) {
      const logoDisplayWidth = (logoSize / 2) / 100; // Logo width as fraction of container
      const shapeMultiplier = shapeSize / 100; // Shape size as multiplier (200% = 2x logo)
      const actualShapeSize = logoDisplayWidth * shapeMultiplier * svgWidth; // Convert to SVG units
      
      if (shapes.circle.enabled) {
      shapes.circle.positions.forEach(pos => {
        const x = (pos.x / 100) * svgWidth;
        const y = (pos.y / 100) * svgHeight;
        shapeElements.push(
          `<circle cx="${x}" cy="${y}" r="${actualShapeSize / 2}" fill="${shapeFill.circle ? resolveColorValue(shapeColor) : 'none'}" stroke="${resolveColorValue(shapeColor)}" stroke-width="2" transform="rotate(${shapeRotation} ${x} ${y})" />`
        );
      });
    }
    
    if (shapes.square.enabled) {
      shapes.square.positions.forEach(pos => {
        const x = (pos.x / 100) * svgWidth - actualShapeSize / 2;
        const y = (pos.y / 100) * svgHeight - actualShapeSize / 2;
        shapeElements.push(
          `<rect x="${x}" y="${y}" width="${actualShapeSize}" height="${actualShapeSize}" fill="${shapeFill.square ? resolveColorValue(shapeColor) : 'none'}" stroke="${resolveColorValue(shapeColor)}" stroke-width="2" transform="rotate(${shapeRotation} ${x + actualShapeSize/2} ${y + actualShapeSize/2})" />`
        );
      });
    }
    
    if (shapes.triangle.enabled) {
      shapes.triangle.positions.forEach(pos => {
        const x = (pos.x / 100) * svgWidth;
        const y = (pos.y / 100) * svgHeight;
        const size = actualShapeSize;
        const points = `${x},${y - size/2} ${x + size/2},${y + size/2} ${x - size/2},${y + size/2}`;
        shapeElements.push(
          `<polygon points="${points}" fill="${shapeFill.triangle ? resolveColorValue(shapeColor) : 'none'}" stroke="${resolveColorValue(shapeColor)}" stroke-width="2" transform="rotate(${shapeRotation} ${x} ${y})" />`
        );
      });
    }
    
    if (shapes.diamond.enabled) {
      shapes.diamond.positions.forEach(pos => {
        const x = (pos.x / 100) * svgWidth;
        const y = (pos.y / 100) * svgHeight;
        const size = actualShapeSize;
        const points = `${x},${y - size/2} ${x + size/2},${y} ${x},${y + size/2} ${x - size/2},${y}`;
        shapeElements.push(
          `<polygon points="${points}" fill="${shapeFill.diamond ? resolveColorValue(shapeColor) : 'none'}" stroke="${resolveColorValue(shapeColor)}" stroke-width="2" transform="rotate(${45 + shapeRotation} ${x} ${y})" />`
        );
      });
    }
  } // Close includeShapes block
    
    // Insert path definitions and elements before closing </svg> tag
    const defsContent = pathDefinitions.length > 0 ? `<defs>${pathDefinitions.join('')}</defs>` : '';
    const allElements = [...textElements, ...shapeElements];
    
    if (defsContent || allElements.length > 0) {
      const elementsToInsert = defsContent + allElements.join('');
      svg = svg.replace('</svg>', elementsToInsert + '</svg>');
    }
    
    return svg;
  };

  // Download SVG (updated for overlay system with color resolution)
  const downloadSVG = () => {
    const compositeSvg = buildCompositeSVG();
    if (!compositeSvg) return;
    
    // Resolve CSS variables and currentColor to actual hex values for export
    const resolvedSvg = resolveSvgColorsForExport(compositeSvg, colors);
    
    const blob = new Blob([resolvedSvg], { type: "image/svg+xml;charset=utf-8" });
    const url = URL.createObjectURL(blob);
    triggerDownload(url, `${template?.name || template?.id || "logo"}.svg`);
    URL.revokeObjectURL(url);
  };

  // Download PNG using an offscreen canvas (updated for overlay system with color resolution)
  const downloadPNG = async (px = 2000) => {
    const compositeSvg = buildCompositeSVG();
    if (!compositeSvg) return;
    
    // Resolve CSS variables and currentColor to actual hex values for export
    const resolvedSvg = resolveSvgColorsForExport(compositeSvg, colors);
    
    const svgBlob = new Blob([resolvedSvg], { type: "image/svg+xml;charset=utf-8" });
    const url = URL.createObjectURL(svgBlob);
    const img = new Image();
    img.decoding = "async";
    img.onload = () => {
      const canvas = document.createElement("canvas");
      const ratio = img.width ? px / img.width : 1;
      canvas.width = px;
      canvas.height = Math.round(img.height * ratio);
      const ctx = canvas.getContext("2d");
      if (!ctx) return;
      ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
      canvas.toBlob((blob) => {
        if (!blob) return;
        const out = URL.createObjectURL(blob);
        triggerDownload(out, `${template?.name || template?.id || "logo"}@${px}px.png`);
        URL.revokeObjectURL(out);
        URL.revokeObjectURL(url);
      }, "image/png");
    };
    img.onerror = () => URL.revokeObjectURL(url);
    img.src = url;
  };

  // Handle send to brand kit request
  const handleSendToBrandKit = async () => {
    try {
      const compositeSvg = buildCompositeSVG();
      if (!compositeSvg) {
        toast({
          title: "Error",
          description: "Unable to send to brand kit. Please try again.",
          variant: "destructive",
        });
        return;
      }

      const resolvedSvg = resolveSvgColorsForExport(compositeSvg, colors);
      const colorList = [colors.primary, colors.secondary, colors.accent].filter(c => c !== "#000000");
      
      const response = await fetch('/api/brand-kit/queue', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          svgMarkup: resolvedSvg,
          brandName: brandName || 'Untitled Brand',
          colors: colorList,
          fonts: [brandNameFont, taglineFont, estYearFont].filter((f, i, arr) => arr.indexOf(f) === i), // Unique fonts chosen in Logo Creator
        }),
      });

      if (!response.ok) {
        throw new Error('Failed to send to brand kit');
      }

      toast({
        title: "Sent to Brand Kit",
        description: "We'll notify you when it's ready.",
      });
    } catch (error) {
      console.error('Brand kit error:', error);
      toast({
        title: "Error",
        description: "Failed to send to brand kit. Please try again.",
        variant: "destructive",
      });
    }
  };

  if (loading) {
    return (
      <DashboardTemplatePage title="Loading...">
        <div className="flex items-center justify-center py-12">
          <div className="text-gray-500">Loading template...</div>
        </div>
      </DashboardTemplatePage>
    );
  }

  if (error || !template) {
    return (
      <DashboardTemplatePage title="Error">
        <div className="flex flex-col items-center justify-center py-12">
          <div className="text-red-600 mb-4">{error || "Template not found"}</div>
          <Link href="/business-development/ai-logo-creator">
            <button className="text-blue-600 hover:underline">← Back to AI Logo Creator</button>
          </Link>
        </div>
      </DashboardTemplatePage>
    );
  }

  return (
    <DashboardTemplatePage title={template?.name || "Logo Customizer"}>
      <div className="space-y-6">
        {/* Back button and actions */}
        <div className="flex items-center justify-between">
          <Link href="/business-development/ai-logo-creator">
            <button className="flex items-center space-x-2 text-gray-600 hover:text-gray-900">
              <ArrowLeft size={20} />
              <span>Back to AI Logo Creator</span>
            </button>
          </Link>
          <div className="flex items-center gap-3">
            <button
              onClick={handleSendToBrandKit}
              className="flex items-center gap-2 px-4 py-2.5 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
              data-testid="button-send-to-brand-kit"
            >
              <FileText size={18} />
              <span>Send to Brand Kit</span>
            </button>
            <button
              onClick={() => setExportModalOpen(true)}
              className="flex items-center gap-2 px-4 py-2.5 bg-[#10B981] text-white rounded-lg hover:bg-[#059669] transition-colors font-medium shadow-sm"
              data-testid="button-download"
            >
              <Download size={18} />
              <span>Download</span>
            </button>
          </div>
        </div>

        {/* Step 2 Header */}
        {template && (
          <div className="mb-3">
            <h2 className="text-xl font-semibold text-gray-900 dark:text-gray-100">Step 2: Customize Your Logo</h2>
            <p className="text-sm text-gray-600 dark:text-gray-400">Edit text, shapes, and colors. When finished, click "Send to Brand Kit".</p>
          </div>
        )}

        {/* Main Content */}
        <div className="flex items-start" style={{ gap: '1px' }}>
          {/* Left side - Tools area */}
          <div style={{ width: '180px' }} className="flex-shrink-0">
            {/* Toolbar Component */}
            <Toolbar
            logoEditorContent={
              <div className="space-y-4">
                {/* Logo Size Controls */}
                <div className="space-y-2">
                  <div className="flex items-center justify-between text-sm text-gray-600">
                    <span>Logo Size</span>
                    <div className="flex items-center gap-2">
                      <span>{logoSize}%</span>
                      <button
                        type="button"
                        onClick={() => setLogoSize(100)}
                        className="text-xs px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded transition-colors"
                        disabled={logoSize === 100}
                        title="Reset size to 100%"
                      >
                        Reset
                      </button>
                    </div>
                  </div>
                  <input
                    type="range"
                    min="50"
                    max="200"
                    value={logoSize}
                    onChange={(e) => setLogoSize(parseInt(e.target.value))}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    data-testid="slider-logo-size"
                  />
                  <div className="flex justify-between text-xs text-gray-400">
                    <span>50%</span>
                    <span>200%</span>
                  </div>
                </div>

                {/* Logo Rotation Controls */}
                <div className="space-y-2">
                  <div className="flex items-center justify-between text-sm text-gray-600">
                    <span>Logo Rotation</span>
                    <div className="flex items-center gap-2">
                      <span>{logoRotation}°</span>
                      <button
                        type="button"
                        onClick={() => setLogoRotation(0)}
                        className="text-xs px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded transition-colors"
                        disabled={logoRotation === 0}
                        title="Reset rotation to 0°"
                      >
                        Reset
                      </button>
                    </div>
                  </div>
                  <input
                    type="range"
                    min={-180}
                    max={180}
                    step={5}
                    value={logoRotation}
                    onChange={(e) => setLogoRotation(Number(e.target.value))}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    data-testid="slider-logo-rotation"
                  />
                  <div className="flex justify-between text-xs text-gray-400">
                    <span>-180°</span>
                    <span>180°</span>
                  </div>
                </div>

              </div>
            }
            textEditorContent={
              <div className="space-y-4">
                {/* Brand Name Section */}
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <label className="block text-sm font-medium text-gray-700">
                      Brand Name
                    </label>
                    <button
                      type="button"
                      onClick={() => resetTextPosition('brand-name')}
                      className="text-xs px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded transition-colors"
                      title="Reset to default position"
                    >
                      Reset
                    </button>
                  </div>
                  <input
                    type="text"
                    value={brandName}
                    onChange={(e) => setBrandName(e.target.value)}
                    placeholder="Enter Brand Name"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    data-testid="input-brand-name"
                  />
                  
                  {/* Font Picker */}
                  <div className="mt-3">
                    <div className="text-xs text-gray-600 mb-2">Font</div>
                    <FontPicker
                      value={brandNameFont}
                      onChange={setBrandNameFont}
                      sampleText="Sample Text"
                    />
                  </div>
                  
                  {/* Text Size */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Size</span>
                      <span>{brandNameSize}%</span>
                    </div>
                    <input
                      type="range"
                      min={25}
                      max={200}
                      step={5}
                      value={brandNameSize}
                      onChange={(e) => setBrandNameSize(parseInt(e.target.value))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Rotation */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Rotation</span>
                      <div className="flex items-center gap-2">
                        <span>{textShapes['brand-name']?.rotation || 0}°</span>
                        <button
                          type="button"
                          onClick={() => setTextShapes(prev => ({
                            ...prev,
                            'brand-name': { ...prev['brand-name'], rotation: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={(textShapes['brand-name']?.rotation || 0) === 0}
                        >
                          Reset
                        </button>
                      </div>
                    </div>
                    <input
                      type="range"
                      min={-180}
                      max={180}
                      step={5}
                      value={textShapes['brand-name']?.rotation || 0}
                      onChange={(e) => setTextShapes(prev => ({
                        ...prev,
                        'brand-name': { ...prev['brand-name'], rotation: Number(e.target.value) }
                      }))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Skew */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Skew</span>
                      <div className="flex items-center gap-2">
                        <span>{textShapes['brand-name']?.curve || 0}</span>
                        <button
                          type="button"
                          onClick={() => setTextShapes(prev => ({
                            ...prev,
                            'brand-name': { ...prev['brand-name'], curve: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={(textShapes['brand-name']?.curve || 0) === 0}
                        >
                          Reset
                        </button>
                      </div>
                    </div>
                    <input
                      type="range"
                      min={-50}
                      max={50}
                      step={1}
                      value={textShapes['brand-name']?.curve || 0}
                      onChange={(e) => setTextShapes(prev => ({
                        ...prev,
                        'brand-name': { ...prev['brand-name'], curve: Number(e.target.value) }
                      }))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Curve */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Curve</span>
                      <div className="flex items-center gap-2">
                        <label className="flex items-center gap-1">
                          <input
                            type="checkbox"
                            checked={textCurves['brand-name']?.enabled || false}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'brand-name': { ...prev['brand-name'], enabled: e.target.checked }
                            }))}
                            className="w-3 h-3"
                          />
                          <span className="text-xs">Enable</span>
                        </label>
                        <button
                          type="button"
                          onClick={() => setTextCurves(prev => ({
                            ...prev,
                            'brand-name': { ...prev['brand-name'], enabled: false, angleDeg: 30, direction: 'up', startOffsetPct: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={!textCurves['brand-name']?.enabled}
                        >
                          Reset
                        </button>
                      </div>
                    </div>

                    {textCurves['brand-name']?.enabled && (
                      <div className="space-y-2">
                        {/* Amount Slider */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Amount</span>
                            <span>{textCurves['brand-name']?.angleDeg || 30}°</span>
                          </div>
                          <input
                            type="range"
                            min={5}
                            max={120}
                            step={5}
                            value={textCurves['brand-name']?.angleDeg || 30}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'brand-name': { ...prev['brand-name'], angleDeg: Number(e.target.value) }
                            }))}
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                          />
                        </div>

                        {/* Direction Toggle */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Direction</span>
                            <div className="flex gap-1">
                              <button
                                type="button"
                                onClick={() => setTextCurves(prev => ({
                                  ...prev,
                                  'brand-name': { ...prev['brand-name'], direction: 'up' }
                                }))}
                                className={`px-2 py-0.5 text-xs rounded transition-colors ${
                                  textCurves['brand-name']?.direction === 'up' 
                                    ? 'bg-blue-500 text-white' 
                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                }`}
                              >
                                ↑ Up
                              </button>
                              <button
                                type="button"
                                onClick={() => setTextCurves(prev => ({
                                  ...prev,
                                  'brand-name': { ...prev['brand-name'], direction: 'down' }
                                }))}
                                className={`px-2 py-0.5 text-xs rounded transition-colors ${
                                  textCurves['brand-name']?.direction === 'down' 
                                    ? 'bg-blue-500 text-white' 
                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                }`}
                              >
                                ↓ Down
                              </button>
                            </div>
                          </div>
                        </div>

                        {/* Offset Slider */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Offset</span>
                            <span>{textCurves['brand-name']?.startOffsetPct || 0}%</span>
                          </div>
                          <input
                            type="range"
                            min={0}
                            max={100}
                            step={5}
                            value={textCurves['brand-name']?.startOffsetPct || 0}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'brand-name': { ...prev['brand-name'], startOffsetPct: Number(e.target.value) }
                            }))}
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                          />
                        </div>
                      </div>
                    )}
                  </div>

                </div>

                {/* Tagline Section */}
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <label className="block text-sm font-medium text-gray-700">
                      Tagline
                    </label>
                    <button
                      type="button"
                      onClick={() => resetTextPosition('tagline')}
                      className="text-xs px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded transition-colors"
                      title="Reset to default position"
                    >
                      Reset
                    </button>
                  </div>
                  <input
                    type="text"
                    value={tagline}
                    onChange={(e) => setTagline(e.target.value)}
                    placeholder="Enter Tagline"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    data-testid="input-tagline"
                  />
                  
                  {/* Font Picker */}
                  <div className="mt-3">
                    <div className="text-xs text-gray-600 mb-2">Font</div>
                    <FontPicker
                      value={taglineFont}
                      onChange={setTaglineFont}
                      sampleText="Sample Text"
                    />
                  </div>
                  
                  {/* Text Size */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Size</span>
                      <span>{taglineSize}%</span>
                    </div>
                    <input
                      type="range"
                      min={25}
                      max={200}
                      step={5}
                      value={taglineSize}
                      onChange={(e) => setTaglineSize(parseInt(e.target.value))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Rotation */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Rotation</span>
                      <div className="flex items-center gap-2">
                        <span>{textShapes['tagline']?.rotation || 0}°</span>
                        <button
                          type="button"
                          onClick={() => setTextShapes(prev => ({
                            ...prev,
                            'tagline': { ...prev['tagline'], rotation: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={(textShapes['tagline']?.rotation || 0) === 0}
                        >
                          Reset
                        </button>
                      </div>
                    </div>
                    <input
                      type="range"
                      min={-180}
                      max={180}
                      step={5}
                      value={textShapes['tagline']?.rotation || 0}
                      onChange={(e) => setTextShapes(prev => ({
                        ...prev,
                        'tagline': { ...prev['tagline'], rotation: Number(e.target.value) }
                      }))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Skew */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Skew</span>
                      <div className="flex items-center gap-2">
                        <span>{textShapes['tagline']?.curve || 0}</span>
                        <button
                          type="button"
                          onClick={() => setTextShapes(prev => ({
                            ...prev,
                            'tagline': { ...prev['tagline'], curve: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={(textShapes['tagline']?.curve || 0) === 0}
                        >
                          Reset
                        </button>
                      </div>
                    </div>
                    <input
                      type="range"
                      min={-50}
                      max={50}
                      step={1}
                      value={textShapes['tagline']?.curve || 0}
                      onChange={(e) => setTextShapes(prev => ({
                        ...prev,
                        'tagline': { ...prev['tagline'], curve: Number(e.target.value) }
                      }))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Curve */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Curve</span>
                      <div className="flex items-center gap-2">
                        <label className="flex items-center gap-1">
                          <input
                            type="checkbox"
                            checked={textCurves['tagline']?.enabled || false}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'tagline': { ...prev['tagline'], enabled: e.target.checked }
                            }))}
                            className="w-3 h-3"
                          />
                          <span className="text-xs">Enable</span>
                        </label>
                        <button
                          type="button"
                          onClick={() => setTextCurves(prev => ({
                            ...prev,
                            'tagline': { ...prev['tagline'], enabled: false, angleDeg: 30, direction: 'up', startOffsetPct: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={!textCurves['tagline']?.enabled}
                        >
                          Reset
                        </button>
                      </div>
                    </div>

                    {textCurves['tagline']?.enabled && (
                      <div className="space-y-2">
                        {/* Amount Slider */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Amount</span>
                            <span>{textCurves['tagline']?.angleDeg || 30}°</span>
                          </div>
                          <input
                            type="range"
                            min={5}
                            max={120}
                            step={5}
                            value={textCurves['tagline']?.angleDeg || 30}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'tagline': { ...prev['tagline'], angleDeg: Number(e.target.value) }
                            }))}
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                          />
                        </div>

                        {/* Direction Toggle */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Direction</span>
                            <div className="flex gap-1">
                              <button
                                type="button"
                                onClick={() => setTextCurves(prev => ({
                                  ...prev,
                                  'tagline': { ...prev['tagline'], direction: 'up' }
                                }))}
                                className={`px-2 py-0.5 text-xs rounded transition-colors ${
                                  textCurves['tagline']?.direction === 'up' 
                                    ? 'bg-blue-500 text-white' 
                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                }`}
                              >
                                ↑ Up
                              </button>
                              <button
                                type="button"
                                onClick={() => setTextCurves(prev => ({
                                  ...prev,
                                  'tagline': { ...prev['tagline'], direction: 'down' }
                                }))}
                                className={`px-2 py-0.5 text-xs rounded transition-colors ${
                                  textCurves['tagline']?.direction === 'down' 
                                    ? 'bg-blue-500 text-white' 
                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                }`}
                              >
                                ↓ Down
                              </button>
                            </div>
                          </div>
                        </div>

                        {/* Offset Slider */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Offset</span>
                            <span>{textCurves['tagline']?.startOffsetPct || 0}%</span>
                          </div>
                          <input
                            type="range"
                            min={0}
                            max={100}
                            step={5}
                            value={textCurves['tagline']?.startOffsetPct || 0}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'tagline': { ...prev['tagline'], startOffsetPct: Number(e.target.value) }
                            }))}
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                          />
                        </div>
                      </div>
                    )}
                  </div>

                </div>

                {/* Est Year Section */}
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <label className="block text-sm font-medium text-gray-700">
                      Est Year
                    </label>
                    <button
                      type="button"
                      onClick={() => resetTextPosition('est-year')}
                      className="text-xs px-2 py-1 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded transition-colors"
                      title="Reset to default position"
                    >
                      Reset
                    </button>
                  </div>
                  <input
                    type="text"
                    value={estYear}
                    onChange={(e) => setEstYear(e.target.value)}
                    placeholder="Enter Est Year"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    data-testid="input-est-year"
                  />
                  
                  {/* Font Picker */}
                  <div className="mt-3">
                    <div className="text-xs text-gray-600 mb-2">Font</div>
                    <FontPicker
                      value={estYearFont}
                      onChange={setEstYearFont}
                      sampleText="Sample Text"
                    />
                  </div>
                  
                  {/* Text Size */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Size</span>
                      <span>{estYearSize}%</span>
                    </div>
                    <input
                      type="range"
                      min={25}
                      max={200}
                      step={5}
                      value={estYearSize}
                      onChange={(e) => setEstYearSize(parseInt(e.target.value))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Rotation */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Rotation</span>
                      <div className="flex items-center gap-2">
                        <span>{textShapes['est-year']?.rotation || 0}°</span>
                        <button
                          type="button"
                          onClick={() => setTextShapes(prev => ({
                            ...prev,
                            'est-year': { ...prev['est-year'], rotation: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={(textShapes['est-year']?.rotation || 0) === 0}
                        >
                          Reset
                        </button>
                      </div>
                    </div>
                    <input
                      type="range"
                      min={-180}
                      max={180}
                      step={5}
                      value={textShapes['est-year']?.rotation || 0}
                      onChange={(e) => setTextShapes(prev => ({
                        ...prev,
                        'est-year': { ...prev['est-year'], rotation: Number(e.target.value) }
                      }))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Skew */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Skew</span>
                      <div className="flex items-center gap-2">
                        <span>{textShapes['est-year']?.curve || 0}</span>
                        <button
                          type="button"
                          onClick={() => setTextShapes(prev => ({
                            ...prev,
                            'est-year': { ...prev['est-year'], curve: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={(textShapes['est-year']?.curve || 0) === 0}
                        >
                          Reset
                        </button>
                      </div>
                    </div>
                    <input
                      type="range"
                      min={-50}
                      max={50}
                      step={1}
                      value={textShapes['est-year']?.curve || 0}
                      onChange={(e) => setTextShapes(prev => ({
                        ...prev,
                        'est-year': { ...prev['est-year'], curve: Number(e.target.value) }
                      }))}
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                  </div>

                  {/* Text Curve */}
                  <div className="mt-3">
                    <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                      <span>Curve</span>
                      <div className="flex items-center gap-2">
                        <label className="flex items-center gap-1">
                          <input
                            type="checkbox"
                            checked={textCurves['est-year']?.enabled || false}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'est-year': { ...prev['est-year'], enabled: e.target.checked }
                            }))}
                            className="w-3 h-3"
                          />
                          <span className="text-xs">Enable</span>
                        </label>
                        <button
                          type="button"
                          onClick={() => setTextCurves(prev => ({
                            ...prev,
                            'est-year': { ...prev['est-year'], enabled: false, angleDeg: 30, direction: 'up', startOffsetPct: 0 }
                          }))}
                          className="text-xs px-1 py-0.5 bg-gray-100 hover:bg-gray-200 text-gray-600 rounded"
                          disabled={!textCurves['est-year']?.enabled}
                        >
                          Reset
                        </button>
                      </div>
                    </div>

                    {textCurves['est-year']?.enabled && (
                      <div className="space-y-2">
                        {/* Amount Slider */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Amount</span>
                            <span>{textCurves['est-year']?.angleDeg || 30}°</span>
                          </div>
                          <input
                            type="range"
                            min={5}
                            max={120}
                            step={5}
                            value={textCurves['est-year']?.angleDeg || 30}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'est-year': { ...prev['est-year'], angleDeg: Number(e.target.value) }
                            }))}
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                          />
                        </div>

                        {/* Direction Toggle */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Direction</span>
                            <div className="flex gap-1">
                              <button
                                type="button"
                                onClick={() => setTextCurves(prev => ({
                                  ...prev,
                                  'est-year': { ...prev['est-year'], direction: 'up' }
                                }))}
                                className={`px-2 py-0.5 text-xs rounded transition-colors ${
                                  textCurves['est-year']?.direction === 'up' 
                                    ? 'bg-blue-500 text-white' 
                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                }`}
                              >
                                ↑ Up
                              </button>
                              <button
                                type="button"
                                onClick={() => setTextCurves(prev => ({
                                  ...prev,
                                  'est-year': { ...prev['est-year'], direction: 'down' }
                                }))}
                                className={`px-2 py-0.5 text-xs rounded transition-colors ${
                                  textCurves['est-year']?.direction === 'down' 
                                    ? 'bg-blue-500 text-white' 
                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                }`}
                              >
                                ↓ Down
                              </button>
                            </div>
                          </div>
                        </div>

                        {/* Offset Slider */}
                        <div>
                          <div className="flex items-center justify-between text-xs text-gray-500 mb-1">
                            <span>Offset</span>
                            <span>{textCurves['est-year']?.startOffsetPct || 0}%</span>
                          </div>
                          <input
                            type="range"
                            min={0}
                            max={100}
                            step={5}
                            value={textCurves['est-year']?.startOffsetPct || 0}
                            onChange={(e) => setTextCurves(prev => ({
                              ...prev,
                              'est-year': { ...prev['est-year'], startOffsetPct: Number(e.target.value) }
                            }))}
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                          />
                        </div>
                      </div>
                    )}
                  </div>

                </div>
              </div>
            }
            shapesEditorContent={
              <div className="space-y-4">
                <div>
                  <div className="flex items-center justify-between mb-3">
                    <label className="text-sm font-medium text-gray-700">Add Basic Shapes</label>
                    <button
                      onClick={clearAllShapes}
                      className="px-3 py-1 text-xs text-white rounded transition-all duration-200 hover:shadow-md focus:outline-none focus:ring-2 focus:ring-orange-300"
                      style={{ backgroundColor: '#FF6600' }}
                      onMouseEnter={(e) => e.currentTarget.style.backgroundColor = '#CC4E00'}
                      onMouseLeave={(e) => e.currentTarget.style.backgroundColor = '#FF6600'}
                      onFocus={(e) => e.currentTarget.style.backgroundColor = '#CC4E00'}
                      onBlur={(e) => e.currentTarget.style.backgroundColor = '#FF6600'}
                      title="Clear all shapes"
                      data-testid="button-clear-shapes"
                    >
                      Clear
                    </button>
                  </div>
                  <div className="grid grid-cols-2 gap-3 mb-4">
                    {/* Circle */}
                    <div className="flex items-center space-x-2 p-2 border rounded-lg">
                      <div className="flex items-center space-x-2">
                        <svg width="20" height="20" viewBox="0 0 20 20" className="flex-shrink-0">
                          <circle cx="10" cy="10" r="8" fill="none" stroke="currentColor" strokeWidth="2" />
                        </svg>
                        <span className="text-xs">Circle</span>
                      </div>
                      <button
                        onClick={() => addShape('circle')}
                        className="px-2 py-1 text-xs bg-blue-500 text-white rounded hover:bg-blue-600"
                      >
                        Add
                      </button>
                    </div>

                    {/* Square */}
                    <div className="flex items-center space-x-2 p-2 border rounded-lg">
                      <div className="flex items-center space-x-2">
                        <svg width="20" height="20" viewBox="0 0 20 20" className="flex-shrink-0">
                          <rect x="2" y="2" width="16" height="16" fill="none" stroke="currentColor" strokeWidth="2" />
                        </svg>
                        <span className="text-xs">Square</span>
                      </div>
                      <button
                        onClick={() => addShape('square')}
                        className="px-2 py-1 text-xs bg-blue-500 text-white rounded hover:bg-blue-600"
                      >
                        Add
                      </button>
                    </div>

                    {/* Triangle */}
                    <div className="flex items-center space-x-2 p-2 border rounded-lg">
                      <div className="flex items-center space-x-2">
                        <svg width="20" height="20" viewBox="0 0 20 20" className="flex-shrink-0">
                          <polygon points="10,2 18,18 2,18" fill="none" stroke="currentColor" strokeWidth="2" />
                        </svg>
                        <span className="text-xs">Triangle</span>
                      </div>
                      <button
                        onClick={() => addShape('triangle')}
                        className="px-2 py-1 text-xs bg-blue-500 text-white rounded hover:bg-blue-600"
                      >
                        Add
                      </button>
                    </div>

                    {/* Diamond */}
                    <div className="flex items-center space-x-2 p-2 border rounded-lg">
                      <div className="flex items-center space-x-2">
                        <svg width="20" height="20" viewBox="0 0 20 20" className="flex-shrink-0">
                          <polygon points="10,2 18,10 10,18 2,10" fill="none" stroke="currentColor" strokeWidth="2" />
                        </svg>
                        <span className="text-xs">Diamond</span>
                      </div>
                      <button
                        onClick={() => addShape('diamond')}
                        className="px-2 py-1 text-xs bg-blue-500 text-white rounded hover:bg-blue-600"
                      >
                        Add
                      </button>
                    </div>
                  </div>
                </div>

                {/* Shape Size */}
                <div>
                  <div className="flex items-center justify-between text-xs text-gray-600 mb-1">
                    <span>Shape Size</span>
                    <span>{shapeSize}%</span>
                  </div>
                  <input
                    type="range"
                    min={25}
                    max={300}
                    step={5}
                    value={shapeSize}
                    onChange={(e) => setShapeSize(parseInt(e.target.value))}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    data-testid="slider-shape-size"
                  />
                </div>
              </div>
            }
            colorEditorContent={
              <div className="space-y-4">
                {/* Color Swatch System */}
                <div>
                  <h3 className="text-sm font-medium text-gray-700 mb-3">Logo Color Swatch</h3>
                  <div className="space-y-3">
                    <ColorPicker
                      label="Primary"
                      value={colors.primary}
                      onChange={(v) => { setColors((s) => ({ ...s, primary: v })); setColorsModified(true); }}
                      onReset={() => { setColors((s) => ({ ...s, primary: originalColorsRef.current.primary })); setColorsModified(true); }}
                    />
                    <ColorPicker
                      label="Secondary"
                      value={colors.secondary}
                      onChange={(v) => { setColors((s) => ({ ...s, secondary: v })); setColorsModified(true); }}
                      onReset={() => { setColors((s) => ({ ...s, secondary: originalColorsRef.current.secondary })); setColorsModified(true); }}
                    />
                    <ColorPicker
                      label="Accent"
                      value={colors.accent}
                      onChange={(v) => { setColors((s) => ({ ...s, accent: v })); setColorsModified(true); }}
                      onReset={() => { setColors((s) => ({ ...s, accent: originalColorsRef.current.accent })); setColorsModified(true); }}
                    />
                  </div>
                </div>

                {/* Element Color Assignment */}
                <div>
                  <h3 className="text-sm font-medium text-gray-700 mb-3">Element Colors</h3>
                  <div className="space-y-2">
                    {/* Shape Color */}
                    <ColorPicker
                      label="Shape Color"
                      value={shapeColor}
                      onChange={setShapeColor}
                      onReset={() => setShapeColor("#000000")}
                    />
                    
                    <div className="flex items-center justify-between text-xs">
                      <span className="text-gray-600">Brand Name:</span>
                      <select 
                        value={textColors.brandName.startsWith('var(') ? textColors.brandName : 'custom'}
                        onChange={(e) => {
                          if (e.target.value === 'custom') return;
                          setTextColors(prev => ({ ...prev, brandName: e.target.value }));
                        }}
                        className="px-2 py-1 text-xs border border-gray-300 rounded"
                      >
                        <option value="var(--primary)">Primary</option>
                        <option value="var(--secondary)">Secondary</option>
                        <option value="var(--accent)">Accent</option>
                        <option value="currentColor">Current Color</option>
                        <option value="custom">Custom</option>
                      </select>
                    </div>
                    {textColors.brandName !== 'var(--primary)' && textColors.brandName !== 'var(--secondary)' && textColors.brandName !== 'var(--accent)' && textColors.brandName !== 'currentColor' && (
                      <ColorPicker
                        label="Brand Name Custom"
                        value={textColors.brandName}
                        onChange={(v) => setTextColors(prev => ({ ...prev, brandName: v }))}
                        onReset={() => setTextColors(prev => ({ ...prev, brandName: "#000000" }))}
                      />
                    )}
                    
                    <div className="flex items-center justify-between text-xs">
                      <span className="text-gray-600">Tagline:</span>
                      <select 
                        value={textColors.tagline.startsWith('var(') ? textColors.tagline : 'custom'}
                        onChange={(e) => {
                          if (e.target.value === 'custom') return;
                          setTextColors(prev => ({ ...prev, tagline: e.target.value }));
                        }}
                        className="px-2 py-1 text-xs border border-gray-300 rounded"
                      >
                        <option value="var(--primary)">Primary</option>
                        <option value="var(--secondary)">Secondary</option>
                        <option value="var(--accent)">Accent</option>
                        <option value="currentColor">Current Color</option>
                        <option value="custom">Custom</option>
                      </select>
                    </div>
                    {textColors.tagline !== 'var(--primary)' && textColors.tagline !== 'var(--secondary)' && textColors.tagline !== 'var(--accent)' && textColors.tagline !== 'currentColor' && (
                      <ColorPicker
                        label="Tagline Custom"
                        value={textColors.tagline}
                        onChange={(v) => setTextColors(prev => ({ ...prev, tagline: v }))}
                        onReset={() => setTextColors(prev => ({ ...prev, tagline: "#000000" }))}
                      />
                    )}
                    
                    <div className="flex items-center justify-between text-xs">
                      <span className="text-gray-600">Est Year:</span>
                      <select 
                        value={textColors.estYear.startsWith('var(') ? textColors.estYear : 'custom'}
                        onChange={(e) => {
                          if (e.target.value === 'custom') return;
                          setTextColors(prev => ({ ...prev, estYear: e.target.value }));
                        }}
                        className="px-2 py-1 text-xs border border-gray-300 rounded"
                      >
                        <option value="var(--primary)">Primary</option>
                        <option value="var(--secondary)">Secondary</option>
                        <option value="var(--accent)">Accent</option>
                        <option value="currentColor">Current Color</option>
                        <option value="custom">Custom</option>
                      </select>
                    </div>
                    {textColors.estYear !== 'var(--primary)' && textColors.estYear !== 'var(--secondary)' && textColors.estYear !== 'var(--accent)' && textColors.estYear !== 'currentColor' && (
                      <ColorPicker
                        label="Est Year Custom"
                        value={textColors.estYear}
                        onChange={(v) => setTextColors(prev => ({ ...prev, estYear: v }))}
                        onReset={() => setTextColors(prev => ({ ...prev, estYear: "#000000" }))}
                      />
                    )}
                  </div>
                </div>
              </div>
            }
            />
          </div>

          {/* Right side - Preview area */}
          <div style={{ width: '510px' }}>
            <div className="bg-white rounded-xl border border-gray-200">
            <div className="p-6 pb-0">
              <div className="flex items-start justify-between mb-2">
                <div>
                  <div className="mb-2 text-lg font-semibold text-gray-900">{template?.name}</div>
                  <div className="text-sm font-medium text-gray-700">Preview</div>
                </div>
                <Button
                  onClick={resetToOriginal}
                  variant="outline"
                  disabled={!hasChangedFromOriginal}
                  size="sm"
                  className="flex items-center gap-1.5"
                  aria-label="Reset to Original"
                  title="Reset to Original"
                  data-testid="button-reset-to-original"
                >
                  <RotateCcw size={16} />
                  <span>Reset to Original</span>
                </Button>
              </div>
              <div className="mb-4"></div>
            </div>
            <div
              ref={setPreviewRef}
              className="w-full bg-gray-50 border-t"
              style={{ 
                height: "384px",
                position: "relative",
                overflow: "hidden",
                touchAction: "none",
                userSelect: "none",
                zIndex: 0,
                '--primary': colors.primary,
                '--secondary': colors.secondary,
                '--accent': colors.accent
              } as React.CSSProperties}
              data-logo-preview
              onPointerMove={handleDragMove}
              onPointerUp={handleDragEnd}
            >
              {baseSvg ? (
                <div 
                  className="relative w-full h-full"
                  style={{ touchAction: 'none', userSelect: 'none' }}
                >
                  {/* Full preview frame guide during dragging */}
                  {isDragging && (
                    <div
                      className="absolute pointer-events-none"
                      style={{
                        left: 0,
                        top: 0,
                        right: 0,
                        bottom: 0,
                        border: '1px dashed rgba(0, 0, 0, 0.15)',
                        borderRadius: '4px',
                        zIndex: 9999
                      }}
                    />
                  )}
                  {/* Logo stage - NO padding, SVG fills 100%, optimized for smooth drag */}
                  <div
                    data-logo-stage
                    className="absolute inset-0"
                    style={{ 
                      padding: 0,
                      cursor: 'grab',
                      pointerEvents: 'auto',
                      boxSizing: 'border-box',
                      willChange: 'transform', // SN's spec: helps Chrome optimize transforms
                      touchAction: 'none', // SN's spec: disable browser gestures
                      userSelect: 'none', // SN's spec: prevent text selection during drag
                      display: 'flex',
                      alignItems: 'center',
                      justifyContent: 'center'
                    }}
                    onPointerDown={(e) => handleDragStart(e, 'logo')}
                    dangerouslySetInnerHTML={{ 
                      __html: buildCompositeSVG(false, false) // Don't include text or shapes - overlays handle them
                    }}
                  />
                  
                  {/* Text Overlay Layer */}
                  <div className="absolute inset-0 pointer-events-none">
                    {brandName && (
                      textCurves['brand-name']?.enabled ? (
                        // Curved text using SVG textPath
                        <svg
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${(positions['brand-name']?.xPct || 25)}%`,
                            top: `${(positions['brand-name']?.yPct || 50)}%`,
                            transform: `translate(-50%, -50%)`,
                            overflow: 'visible',
                            cursor: 'grab',
                          }}
                          data-text-id="brand-name"
                          onPointerDown={(e) => handleDragStart(e, 'brand-name')}
                        >
                          <defs>
                            <path
                              id="brand-name-curve-path"
                              d={(() => {
                                const fontSize = (brandNameSize / 100) * 16;
                                const textWidth = brandName.length * fontSize * 0.6;
                                const curve = textCurves['brand-name'];
                                const arcData = buildTextPathArc(0, 0, textWidth, curve);
                                return arcData.pathData;
                              })()}
                              fill="none"
                            />
                          </defs>
                          <text
                            fontSize={`${(brandNameSize / 100) * 16}px`}
                            fill={textColors.brandName}
                            fontFamily={`${brandNameFont}, sans-serif`}
                            fontWeight="bold"
                            textAnchor="middle"
                            transform={`rotate(${textShapes['brand-name']?.rotation || 0})`}
                          >
                            <textPath
                              href="#brand-name-curve-path"
                              startOffset={`${textCurves['brand-name']?.startOffsetPct || 50}%`}
                            >
                              {brandName}
                            </textPath>
                          </text>
                        </svg>
                      ) : (
                        // Regular text
                        <div
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${(positions['brand-name']?.xPct || 25)}%`,
                            top: `${(positions['brand-name']?.yPct || 50)}%`,
                            fontSize: `${(brandNameSize / 100) * 16}px`,
                            color: textColors.brandName,
                            fontFamily: `${brandNameFont}, sans-serif`,
                            transform: `translate(-50%, -50%) rotate(${textShapes['brand-name']?.rotation || 0}deg) ${textShapes['brand-name']?.curve ? `skewY(${textShapes['brand-name'].curve * 0.5}deg)` : ''}`,
                            transformOrigin: 'center center',
                            cursor: 'grab',
                            userSelect: 'none',
                            fontWeight: 'bold',
                            position: 'absolute',
                            pointerEvents: 'auto'
                          }}
                          data-text-id="brand-name"
                          onPointerDown={(e) => handleDragStart(e, 'brand-name')}
                        >
                          {brandName}
                        </div>
                      )
                    )}
                    
                    {tagline && (
                      textCurves['tagline']?.enabled ? (
                        // Curved text using SVG textPath
                        <svg
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${(positions['tagline']?.xPct || 50)}%`,
                            top: `${(positions['tagline']?.yPct || 50)}%`,
                            transform: `translate(-50%, -50%)`,
                            overflow: 'visible',
                            cursor: 'grab',
                          }}
                          data-text-id="tagline"
                          onPointerDown={(e) => handleDragStart(e, 'tagline')}
                        >
                          <defs>
                            <path
                              id="tagline-curve-path"
                              d={(() => {
                                const fontSize = (taglineSize / 100) * 14;
                                const textWidth = tagline.length * fontSize * 0.6;
                                const curve = textCurves['tagline'];
                                const arcData = buildTextPathArc(0, 0, textWidth, curve);
                                return arcData.pathData;
                              })()}
                              fill="none"
                            />
                          </defs>
                          <text
                            fontSize={`${(taglineSize / 100) * 14}px`}
                            fill={textColors.tagline}
                            fontFamily={`${taglineFont}, sans-serif`}
                            fontWeight="normal"
                            textAnchor="middle"
                            transform={`rotate(${textShapes['tagline']?.rotation || 0})`}
                          >
                            <textPath
                              href="#tagline-curve-path"
                              startOffset={`${textCurves['tagline']?.startOffsetPct || 50}%`}
                            >
                              {tagline}
                            </textPath>
                          </text>
                        </svg>
                      ) : (
                        // Regular text
                        <div
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${(positions['tagline']?.xPct || 50)}%`,
                            top: `${(positions['tagline']?.yPct || 50)}%`,
                            fontSize: `${(taglineSize / 100) * 14}px`,
                            color: textColors.tagline,
                            fontFamily: `${taglineFont}, sans-serif`,
                            transform: `translate(-50%, -50%) rotate(${textShapes['tagline']?.rotation || 0}deg) ${textShapes['tagline']?.curve ? `skewY(${textShapes['tagline'].curve * 0.5}deg)` : ''}`,
                            transformOrigin: 'center center',
                            cursor: 'grab',
                            userSelect: 'none',
                            fontWeight: 'normal',
                            position: 'absolute',
                            pointerEvents: 'auto'
                          }}
                          data-text-id="tagline"
                          onPointerDown={(e) => handleDragStart(e, 'tagline')}
                        >
                          {tagline}
                        </div>
                      )
                    )}
                    
                    {estYear && (
                      textCurves['est-year']?.enabled ? (
                        // Curved text using SVG textPath
                        <svg
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${(positions['est-year']?.xPct || 75)}%`,
                            top: `${(positions['est-year']?.yPct || 50)}%`,
                            transform: `translate(-50%, -50%)`,
                            overflow: 'visible',
                            cursor: 'grab',
                          }}
                          data-text-id="est-year"
                          onPointerDown={(e) => handleDragStart(e, 'est-year')}
                        >
                          <defs>
                            <path
                              id="est-year-curve-path"
                              d={(() => {
                                const fontSize = (estYearSize / 100) * 12;
                                const textWidth = estYear.length * fontSize * 0.6;
                                const curve = textCurves['est-year'];
                                const arcData = buildTextPathArc(0, 0, textWidth, curve);
                                return arcData.pathData;
                              })()}
                              fill="none"
                            />
                          </defs>
                          <text
                            fontSize={`${(estYearSize / 100) * 12}px`}
                            fill={textColors.estYear}
                            fontFamily={`${estYearFont}, sans-serif`}
                            fontWeight="normal"
                            textAnchor="middle"
                            transform={`rotate(${textShapes['est-year']?.rotation || 0})`}
                          >
                            <textPath
                              href="#est-year-curve-path"
                              startOffset={`${textCurves['est-year']?.startOffsetPct || 50}%`}
                            >
                              {estYear}
                            </textPath>
                          </text>
                        </svg>
                      ) : (
                        // Regular text
                        <div
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${(positions['est-year']?.xPct || 75)}%`,
                            top: `${(positions['est-year']?.yPct || 50)}%`,
                            fontSize: `${(estYearSize / 100) * 12}px`,
                            color: textColors.estYear,
                            fontFamily: `${estYearFont}, sans-serif`,
                            transform: `translate(-50%, -50%) rotate(${textShapes['est-year']?.rotation || 0}deg) ${textShapes['est-year']?.curve ? `skewY(${textShapes['est-year'].curve * 0.5}deg)` : ''}`,
                            transformOrigin: 'center center',
                            cursor: 'grab',
                            userSelect: 'none',
                            fontWeight: 'normal',
                            position: 'absolute',
                            pointerEvents: 'auto'
                          }}
                          data-text-id="est-year"
                          onPointerDown={(e) => handleDragStart(e, 'est-year')}
                        >
                          {estYear}
                        </div>
                      )
                    )}
                  </div>
                  
                  {/* Shape Overlay Layer */}
                  <div className="absolute inset-0 pointer-events-none">
                    {/* Circle shapes */}
                    {shapes.circle.enabled && shapes.circle.positions.map(pos => {
                      // Calculate shape size relative to logo size
                      const logoDisplayWidth = (logoSize / 2) / 100; // Logo width as fraction of container (logoSize/2 gives %, /100 gives fraction)
                      const shapeMultiplier = shapeSize / 100; // Shape size as multiplier (200% = 2x logo)
                      const shapeSizeInPx = logoDisplayWidth * shapeMultiplier * 400; // 400px container
                      
                      return (
                        <div
                          key={pos.id}
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${pos.x}%`,
                            top: `${pos.y}%`,
                            width: `${shapeSizeInPx}px`,
                            height: `${shapeSizeInPx}px`,
                            backgroundColor: shapeFill.circle ? shapeColor : 'transparent',
                            border: `2px solid ${shapeColor}`,
                            borderRadius: '50%',
                            cursor: 'grab',
                            transform: `translate(-50%, -50%) rotate(${shapeRotation}deg)`
                          }}
                          data-shape-id={pos.id}
                          onPointerDown={(e) => handleDragStart(e, pos.id)}
                        />
                      );
                    })}
                    
                    {/* Square shapes */}
                    {shapes.square.enabled && shapes.square.positions.map(pos => {
                      const logoDisplayWidth = (logoSize / 2) / 100;
                      const shapeMultiplier = shapeSize / 100;
                      const shapeSizeInPx = logoDisplayWidth * shapeMultiplier * 400;
                      
                      return (
                        <div
                          key={pos.id}
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${pos.x}%`,
                            top: `${pos.y}%`,
                            width: `${shapeSizeInPx}px`,
                            height: `${shapeSizeInPx}px`,
                            backgroundColor: shapeFill.square ? shapeColor : 'transparent',
                            border: `2px solid ${shapeColor}`,
                            cursor: 'grab',
                            transform: `translate(-50%, -50%) rotate(${shapeRotation}deg)`
                          }}
                          data-shape-id={pos.id}
                          onPointerDown={(e) => handleDragStart(e, pos.id)}
                        />
                      );
                    })}
                    
                    {/* Triangle shapes */}
                    {shapes.triangle.enabled && shapes.triangle.positions.map(pos => {
                      const logoDisplayWidth = (logoSize / 2) / 100;
                      const shapeMultiplier = shapeSize / 100;
                      const shapeSizeInPx = logoDisplayWidth * shapeMultiplier * 400;
                      
                      // Create triangle points for outline
                      const size = shapeSizeInPx;
                      const points = `0,${size/2} ${size/2},-${size/2} -${size/2},-${size/2}`;
                      
                      return (
                        <div
                          key={pos.id}
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${pos.x}%`,
                            top: `${pos.y}%`,
                            width: `${size}px`,
                            height: `${size}px`,
                            cursor: 'grab',
                            transform: `translate(-50%, -50%) rotate(${shapeRotation}deg)`
                          }}
                          data-shape-id={pos.id}
                          onPointerDown={(e) => handleDragStart(e, pos.id)}
                        >
                          <svg
                            width={size}
                            height={size}
                            viewBox={`-${size/2} -${size/2} ${size} ${size}`}
                            style={{ overflow: 'visible' }}
                          >
                            <polygon
                              points={points}
                              fill={shapeFill.triangle ? shapeColor : "none"}
                              stroke={shapeColor}
                              strokeWidth="2"
                            />
                          </svg>
                        </div>
                      );
                    })}
                    
                    {/* Diamond shapes */}
                    {shapes.diamond.enabled && shapes.diamond.positions.map(pos => {
                      const logoDisplayWidth = (logoSize / 2) / 100;
                      const shapeMultiplier = shapeSize / 100;
                      const shapeSizeInPx = logoDisplayWidth * shapeMultiplier * 400;
                      
                      return (
                        <div
                          key={pos.id}
                          className="absolute pointer-events-auto"
                          style={{
                            left: `${pos.x}%`,
                            top: `${pos.y}%`,
                            width: `${shapeSizeInPx}px`,
                            height: `${shapeSizeInPx}px`,
                            backgroundColor: shapeFill.diamond ? shapeColor : 'transparent',
                            border: `2px solid ${shapeColor}`,
                            cursor: 'grab',
                            transform: `translate(-50%, -50%) rotate(${45 + shapeRotation}deg)`
                          }}
                          data-shape-id={pos.id}
                          onPointerDown={(e) => handleDragStart(e, pos.id)}
                        />
                      );
                    })}
                  </div>
                </div>
              ) : template?.previewUrl ? (
                <img 
                  src={template.previewUrl} 
                  alt={template.name || "Logo Preview"}
                  className="object-contain"
                  style={{ 
                    cursor: isDragging ? 'grabbing' : 'default',
                    width: `${logoSize / 2.2}%`, // At 220%, width = 100% (touches edges)
                    height: 'auto',
                    maxWidth: '100%',
                    maxHeight: '100%'
                  }}
                />
              ) : (
                <div className="text-gray-500 text-center">
                  <div className="text-lg mb-2">Logo Preview</div>
                  <div className="text-sm">Loading...</div>
                </div>
              )}
            </div>
            </div>
          </div>
        </div>
      </div>

      {/* Export Modal */}
      <ExportModal
        open={exportModalOpen}
        onOpenChange={setExportModalOpen}
        svgMarkup={buildCompositeSVG() || ''}
        brandName={brandName || template?.name || 'logo'}
      />
    </DashboardTemplatePage>
  );
}

function ColorPicker({
  label,
  value,
  onChange,
  onReset,
}: {
  label: string;
  value: string;
  onChange: (v: string) => void;
  onReset?: () => void;
}) {
  const [isOpen, setIsOpen] = useState(false);
  const [tempColor, setTempColor] = useState(value);
  const pickerRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    setTempColor(value);
  }, [value]);

  // Close picker when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (pickerRef.current && !pickerRef.current.contains(event.target as Node)) {
        setIsOpen(false);
        setTempColor(value); // Reset to original value if not applied
      }
    };

    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
    }
    return () => {
      document.removeEventListener('mousedown', handleClickOutside);
    };
  }, [isOpen, value]);

  const handleApply = () => {
    onChange(tempColor);
    setIsOpen(false);
  };

  return (
    <div className="relative" ref={pickerRef}>
      <div className="flex items-center justify-between mb-2">
        <label className="text-sm font-medium text-gray-700">{label}</label>
        {/* Reset button */}
        {onReset && (
          <button
            type="button"
            onClick={onReset}
            className="px-2 py-1 text-xs bg-gray-200 hover:bg-gray-300 text-gray-700 rounded transition-colors"
            data-testid={`reset-${label.toLowerCase()}`}
            title="Reset to original color"
          >
            Reset
          </button>
        )}
      </div>
      <div className="flex items-center gap-2">
        {/* Color swatch button */}
        <button
          type="button"
          onClick={() => setIsOpen(!isOpen)}
          className="w-8 h-8 rounded border-2 border-gray-300 cursor-pointer hover:border-gray-400 transition-colors"
          style={{ backgroundColor: value }}
          data-testid={`color-${label.toLowerCase()}`}
        />
        {/* Hex display */}
        <input
          type="text"
          value={value}
          readOnly
          className="flex-1 px-2 py-1 text-sm border border-gray-300 rounded bg-gray-50 cursor-pointer"
          onClick={() => setIsOpen(!isOpen)}
          data-testid={`hex-${label.toLowerCase()}`}
        />
      </div>

      {/* Color picker dropdown */}
      {isOpen && (
        <div className="absolute z-50 mt-2 bg-white border-2 border-gray-300 rounded-lg shadow-xl p-3" style={{ minWidth: '240px' }}>
          {/* Native color picker */}
          <input
            type="color"
            value={tempColor}
            onChange={(e) => setTempColor(e.target.value)}
            className="w-full h-8 rounded border border-gray-300 cursor-pointer mb-3"
          />

          {/* Apply button */}
          <button
            type="button"
            onClick={handleApply}
            className="w-full bg-orange-500 hover:bg-orange-600 text-white font-medium py-2 px-4 rounded transition-colors"
            data-testid={`apply-${label.toLowerCase()}`}
          >
            Apply
          </button>
        </div>
      )}
    </div>
  );
}

function escapeRegExp(s: string) {
  return s.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
}

function escapeXml(unsafe: string) {
  return unsafe
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#039;");
}

function triggerDownload(url: string, filename: string) {
  const a = document.createElement("a");
  a.href = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  a.remove();
}
