import { useEffect, useMemo, useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from "@/components/ui/breadcrumb";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Search, ExternalLink, CheckCircle, XCircle, AlertTriangle, Crown, Gift, User, ChevronRight } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

type DomainPriceResponse = {
  domain: string;
  tld: string;
  available: boolean;
  premium: boolean;
  pricing: {
    wholesale: {
      registration: number;    // in dollars
      currency: string;
    };
    retail: {
      registration: number;    // in dollars
      currency: string;
    };
  };
  credit: {
    credit_id: string;
    eligible: boolean;
    cap_cents: number;
    covered_cents: number;
    expires_at: string;
    ineligible_reason?: string;
  } | null;
  out_of_pocket_cents: number;
};

type DomainResult = {
  domain: string;
  available: boolean;
  premium: boolean;
  priceCents: number;        // retail price in cents
  currency: string;
  credit: {
    credit_id: string;
    eligible: boolean;
    covered_cents: number;
    expires_at: string;
    ineligible_reason?: string;
  } | null;
  outOfPocketCents: number;
};

const ENABLE_INTERNAL_CHECKOUT =
  (import.meta as any).env?.VITE_ENABLE_INTERNAL_DOMAIN_CHECKOUT === "true";

// Contact form schema for domain registration
const contactFormSchema = z.object({
  firstName: z.string().min(1, "First name is required"),
  lastName: z.string().min(1, "Last name is required"),
  email: z.string().email("Valid email is required"),
  phone: z.string().min(1, "Phone number is required"),
  address: z.string().min(1, "Address is required"),
  city: z.string().min(1, "City is required"),
  state: z.string().min(1, "State is required"),
  zip: z.string().min(1, "ZIP code is required"),
  country: z.string().length(2, "Country code must be 2 characters"),
  organization: z.string().optional()
});

export default function FindDomainPage() {
  const [query, setQuery] = useState("");
  const [results, setResults] = useState<DomainResult[]>([]);
  const [isLoadingPricing, setIsLoadingPricing] = useState<{ [domain: string]: boolean }>({});
  const [showContactDialog, setShowContactDialog] = useState(false);
  const [selectedDomain, setSelectedDomain] = useState<DomainResult | null>(null);
  const [isRegistering, setIsRegistering] = useState(false);
  const [isConfigured, setIsConfigured] = useState<boolean>(true); // Domain service configuration status
  const { toast } = useToast();

  // Check if OpenSRS credentials are configured on mount
  useEffect(() => {
    const checkHealth = async () => {
      try {
        const response = await apiRequest<{ configured: boolean }>("GET", "/api/domain/health");
        setIsConfigured(response.configured);
      } catch (error) {
        console.warn("Failed to check domain service health:", error);
        // Assume configured on error to avoid blocking users unnecessarily
        setIsConfigured(true);
      }
    };
    checkHealth();
  }, []);

  // Contact form for domain registration
  const contactForm = useForm<z.infer<typeof contactFormSchema>>({
    resolver: zodResolver(contactFormSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      phone: "",
      address: "",
      city: "",
      state: "",
      zip: "",
      country: "US",
      organization: ""
    }
  });

  // --- Domain pricing check function ---
  const checkDomainPrice = async (domain: string): Promise<DomainResult | null> => {
    const url = `/api/domain/price?q=${encodeURIComponent(domain)}`;
    
    try {
      const response = await apiRequest<DomainPriceResponse>("GET", url);
      
      // Convert response to internal format
      return {
        domain: response.domain,
        available: response.available,
        premium: response.premium,
        priceCents: Math.round(response.pricing.retail.registration * 100),
        currency: response.pricing.retail.currency,
        credit: response.credit ? {
          credit_id: response.credit.credit_id,
          eligible: response.credit.eligible,
          covered_cents: response.credit.covered_cents,
          expires_at: response.credit.expires_at,
          ineligible_reason: response.credit.ineligible_reason
        } : null,
        outOfPocketCents: response.out_of_pocket_cents
      };
    } catch (error: any) {
      // Silent fail - errors are handled by mutation toast
      return null;
    }
  };
  
  // --- Multi-domain search function ---
  const searchDomains = async (baseName: string) => {
    setResults([]);
    
    // Generate domain variations to check
    const tlds = ['.com', '.net', '.org', '.co', '.io', '.ai'];
    const domainsToCheck = tlds.map(tld => `${baseName}${tld}`);
    
    // Check each domain individually with pricing
    const domainPromises = domainsToCheck.map(async (domain) => {
      setIsLoadingPricing(prev => ({ ...prev, [domain]: true }));
      const result = await checkDomainPrice(domain);
      setIsLoadingPricing(prev => ({ ...prev, [domain]: false }));
      return result;
    });
    
    const domainResults = await Promise.all(domainPromises);
    const validResults = domainResults.filter((result): result is DomainResult => result !== null);
    
    setResults(validResults);
    
    if (validResults.length === 0) {
      toast({
        variant: "destructive",
        title: "Search failed",
        description: "Unable to check domain availability. Please try again.",
      });
    }
  };
  
  const [isSearching, setIsSearching] = useState(false);
  
  // --- Search mutation for UI consistency ---
  const searchMutation = {
    isPending: isSearching,
    mutate: async (q: string) => {
      setIsSearching(true);
      try {
        await searchDomains(q);
      } catch (error: any) {
        toast({
          variant: "destructive",
          title: "Search failed",
          description: error?.message || "Unable to search domains right now.",
        });
      } finally {
        setIsSearching(false);
      }
    }
  };

  // ----- Prefill from URL: ?query= -----
  useEffect(() => {
    const params = new URLSearchParams(window.location.search);
    const q = params.get("query");
    if (q) {
      setQuery(q);
      // Clean URL
      window.history.replaceState({}, "", window.location.pathname);
      doSearch(q);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Normalizes user input (strip protocol/www, ensure TLD if desired)
  const normalize = (raw: string) => {
    let s = raw.trim().toLowerCase();
    s = s.replace(/^https?:\/\//, "").replace(/^www\./, "");
    s = s.split("/")[0];
    if (!s.includes(".")) s += ".com"; // default
    return s;
  };

  const doSearch = (raw?: string) => {
    const q = (raw ?? query).trim().toLowerCase();
    if (!q) {
      toast({ variant: "destructive", title: "Enter a domain", description: "Try something like cafemojo or cafemojo.com" });
      return;
    }
    
    const hasDot = q.includes(".");
    if (!hasDot) {
      toast({ 
        variant: "destructive", 
        title: "Add a TLD", 
        description: "Add a TLD (e.g., cafemojo.com) or try .com, .net, .org, .co." 
      });
      return;
    }
    
    // Extract just the base name without TLD for searching
    const baseName = normalize(q).split('.')[0];
    searchMutation.mutate(baseName);
  };

  const formatPrice = (cents?: number, currency: string = "USD") => {
    if (typeof cents !== "number") return "";
    return new Intl.NumberFormat("en-US", { style: "currency", currency }).format(cents / 100);
  };

  const statusFor = (result: DomainResult) => {
    return result.available ? { text: "Available", variant: "default" as const } : { text: "Taken", variant: "destructive" as const };
  };
  
  // Check if domain is eligible for "Free with Pro" badge
  const isFreeWithPro = (result: DomainResult) => {
    return result.available && result.credit?.eligible && result.outOfPocketCents === 0;
  };
  
  // Get display pricing with credit breakdown
  const getPricingDisplay = (result: DomainResult) => {
    const originalPrice = formatPrice(result.priceCents, result.currency);
    
    if (result.credit?.eligible && result.credit.covered_cents > 0) {
      const creditDiscount = formatPrice(result.credit.covered_cents, result.currency);
      const finalPrice = formatPrice(result.outOfPocketCents, result.currency);
      
      return {
        original: originalPrice,
        creditApplied: creditDiscount,
        final: finalPrice,
        isFree: result.outOfPocketCents === 0
      };
    }
    
    return {
      original: originalPrice,
      creditApplied: null,
      final: originalPrice,
      isFree: false
    };
  };

  // --- Buy / Checkout with Credit Integration ---
  const buyDomain = async (domain: string) => {
    if (!ENABLE_INTERNAL_CHECKOUT) {
      // fallback: external GoDaddy search page
      const url = `https://www.godaddy.com/domainsearch/find?domainToCheck=${encodeURIComponent(domain)}`;
      window.open(url, "_blank", "noopener,noreferrer");
      return;
    }

    // Find domain result
    const domainResult = results.find(r => r.domain === domain);
    if (!domainResult) {
      toast({
        variant: "destructive",
        title: "Domain not found",
        description: "Please search for the domain again.",
      });
      return;
    }

    // Store selected domain and show contact form
    setSelectedDomain(domainResult);
    setShowContactDialog(true);
  };

  // Handle domain registration after contact form submission
  const handleDomainRegistration = async (contactData: z.infer<typeof contactFormSchema>) => {
    if (!selectedDomain) return;

    setIsRegistering(true);
    
    try {
      const isFree = isFreeWithPro(selectedDomain);
      
      if (isFree) {
        toast({
          title: "🎉 Starting FREE domain registration!",
          description: `Using your Pro credit for ${selectedDomain.domain}. You'll save ${formatPrice(selectedDomain.priceCents, selectedDomain.currency)}!`,
        });
      }

      // Call the domain register API
      const response = await apiRequest<{
        success: boolean;
        message: string;
        order_id?: string;
        payment_intent?: {
          id: string;
          client_secret: string;
          amount: number;
        };
      }>(
        "POST",
        "/api/domain/register",
        {
          domain: selectedDomain.domain,
          contact: contactData,
          nameservers: [], // Use default nameservers
          // Only include payment_method_id if we need to handle payment processing later
        }
      );

      if (response.success) {
        // Registration successful
        setShowContactDialog(false);
        setSelectedDomain(null);
        contactForm.reset();
        
        const savedAmount = selectedDomain.credit?.eligible && selectedDomain.credit.covered_cents > 0
          ? formatPrice(selectedDomain.credit.covered_cents, selectedDomain.currency)
          : null;

        toast({
          title: "🎉 Domain registration started!",
          description: savedAmount 
            ? `Registration for ${selectedDomain.domain} has been initiated. You saved ${savedAmount} with your Pro credit!`
            : `Registration for ${selectedDomain.domain} has been initiated.`,
        });
      } else if (response.payment_intent) {
        // Payment required - this would need Stripe integration for paid domains
        // For now, show message that payment is needed
        toast({
          variant: "destructive",
          title: "Payment required",
          description: `Payment of ${formatPrice(response.payment_intent.amount, selectedDomain.currency)} is required to complete registration.`,
        });
      } else {
        // Registration failed
        toast({
          variant: "destructive",
          title: "Registration failed",
          description: response.message || "Could not register domain.",
        });
      }
    } catch (e: any) {
      console.error('Domain registration error:', e);
      toast({
        variant: "destructive",
        title: "Registration failed",
        description: e?.message || "Could not register domain.",
      });
    } finally {
      setIsRegistering(false);
    }
  };

  // derived: any available hits and free domains?
  const anyAvail = useMemo(() => results.some(r => r.available), [results]);
  const anyFree = useMemo(() => results.some(r => isFreeWithPro(r)), [results]);

  return (
    <div className="p-4 md:p-6">
      {/* Breadcrumbs */}
      <Breadcrumb className="mb-[16px] -mt-1">
        <BreadcrumbList>
          <BreadcrumbItem>
            <BreadcrumbLink href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Web Services</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbPage className="text-[#FF8B00] font-semibold">Domains</BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>

      <div className="max-w-4xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">Find a Domain</h1>
          <p className="text-muted-foreground">
            Search availability and secure your perfect web address. <span className="text-green-600 font-medium">Pro users get FREE domains</span> on eligible TLDs (.com, .net, .org, .co) with their subscription!
          </p>
        </div>

      {/* Configuration Warning Banner */}
      {!isConfigured && (
        <Alert className="mb-6 border-yellow-500 bg-yellow-50 dark:bg-yellow-950 text-yellow-900 dark:text-yellow-100" data-testid="alert-domain-not-configured">
          <AlertTriangle className="h-4 w-4 text-yellow-600 dark:text-yellow-400" />
          <AlertDescription className="text-yellow-800 dark:text-yellow-200">
            Domain search is not configured for this environment. Please contact your administrator to set up OpenSRS credentials.
          </AlertDescription>
        </Alert>
      )}

      {/* Search */}
      <Card className="mb-8">
        <CardHeader>
          <CardTitle>Search</CardTitle>
          <CardDescription>Enter a domain to check availability and pricing</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex gap-4">
            <Input
              className="flex-1"
              placeholder="e.g., cafemojo.com or cafemojo"
              value={query}
              onChange={(e) => setQuery(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && doSearch()}
              data-testid="input-domain-search"
            />
            <Tooltip>
              <TooltipTrigger asChild>
                <div>
                  <Button
                    onClick={() => doSearch()}
                    disabled={searchMutation.isPending || !isConfigured}
                    className="bg-green-600 hover:bg-green-700 text-white font-bold disabled:opacity-50 disabled:cursor-not-allowed"
                    data-testid="button-search-domains"
                  >
                    {searchMutation.isPending ? (
                      <>
                        <div className="h-4 w-4 animate-spin rounded-full border-2 border-current border-t-transparent mr-2" />
                        Searching...
                      </>
                    ) : (
                      <>
                        <Search className="h-4 w-4 mr-2" />
                        Search
                      </>
                    )}
                  </Button>
                </div>
              </TooltipTrigger>
              {!isConfigured && (
                <TooltipContent>
                  <p>Domain search is not available in this environment</p>
                </TooltipContent>
              )}
            </Tooltip>
          </div>
        </CardContent>
      </Card>

      {/* Loading */}
      {searchMutation.isPending && (
        <div className="space-y-4">
          <h2 className="text-xl font-semibold">Searching domains and checking credits...</h2>
          <div className="grid gap-4">
            {[1, 2, 3, 4, 5, 6].map((i) => (
              <Card key={i}>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="space-y-2 flex-1">
                      <Skeleton className="h-6 w-48" />
                      <div className="flex gap-2">
                        <Skeleton className="h-4 w-20" />
                        <Skeleton className="h-4 w-24" />
                      </div>
                    </div>
                    <Skeleton className="h-10 w-28" />
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      )}

      {/* Results */}
      {results.length > 0 && !searchMutation.isPending && (
        <div className="space-y-6">
          <div className="flex items-center justify-between">
            <h2 className="text-2xl font-semibold">Results</h2>
            {anyFree ? (
              <span className="text-sm text-green-600 font-medium">🎉 You have free domains available with your Pro subscription!</span>
            ) : anyAvail ? (
              <span className="text-sm text-muted-foreground">Tip: lock an available domain before someone else does.</span>
            ) : (
              <span className="text-sm text-muted-foreground">No standard matches available—try a shorter name or another TLD.</span>
            )}
          </div>

          <div className="grid gap-4">
            {results.map((result, idx) => {
              const status = statusFor(result);
              const pricing = getPricingDisplay(result);
              const freeWithPro = isFreeWithPro(result);
              const isLoading = isLoadingPricing[result.domain];
              
              const icon = result.available
                ? <CheckCircle className="h-5 w-5 text-green-500" />
                : <XCircle className="h-5 w-5 text-red-500" />;

              return (
                <Card key={`${result.domain}-${idx}`} data-testid={`domain-result-${idx}`} className={freeWithPro ? "ring-2 ring-green-500" : ""}>
                  <CardContent className="p-6">
                    {isLoading ? (
                      <div className="flex items-center justify-between">
                        <div className="space-y-2 flex-1">
                          <Skeleton className="h-6 w-48" />
                          <Skeleton className="h-4 w-32" />
                        </div>
                        <Skeleton className="h-10 w-28" />
                      </div>
                    ) : (
                      <>
                        <div className="flex items-center justify-between">
                          <div className="flex items-center space-x-4">
                            {icon}
                            <div>
                              <div className="flex items-center gap-2">
                                <h3 className="text-base font-semibold" data-testid={`domain-name-${idx}`}>{result.domain}</h3>
                                {freeWithPro && (
                                  <Badge variant="default" className="bg-green-600 hover:bg-green-700 text-white gap-1">
                                    <Gift className="h-3 w-3" /> Free with Pro
                                  </Badge>
                                )}
                              </div>
                              
                              <div className="flex items-center gap-2 mt-1">
                                <Badge variant={status.variant} data-testid={`domain-status-${idx}`}>
                                  {status.text}
                                </Badge>

                                {result.premium && (
                                  <Badge variant="outline" className="gap-1">
                                    <Crown className="h-3 w-3" /> Premium
                                  </Badge>
                                )}
                                
                                {result.available && (
                                  <div className="flex flex-col gap-1" data-testid={`domain-price-${idx}`}>
                                    {pricing.creditApplied ? (
                                      <div className="flex flex-col text-sm">
                                        <span className="text-gray-500 line-through">{pricing.original}/yr</span>
                                        <div className="flex items-center gap-2">
                                          <span className="text-green-600 font-semibold">Credit: -{pricing.creditApplied}</span>
                                          <span className="text-base font-bold text-green-600">
                                            {pricing.isFree ? "FREE" : pricing.final}/yr
                                          </span>
                                        </div>
                                      </div>
                                    ) : (
                                      <span className="text-base font-bold text-green-600">
                                        {pricing.original}/yr
                                      </span>
                                    )}
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>

                          {result.available && (
                            <Button
                              onClick={() => buyDomain(result.domain)}
                              className={`text-white ml-4 ${freeWithPro ? 'bg-green-600 hover:bg-green-700' : ''}`}
                              style={!freeWithPro ? { backgroundColor: '#FF8800' } : {}}
                              onMouseEnter={(e) => !freeWithPro && (e.currentTarget.style.backgroundColor = '#E67700')}
                              onMouseLeave={(e) => !freeWithPro && (e.currentTarget.style.backgroundColor = '#FF8800')}
                              data-testid={`button-buy-${idx}`}
                            >
                              {freeWithPro ? (
                                <><Gift className="h-4 w-4 mr-2" />Claim FREE</>
                              ) : (
                                <><ExternalLink className="h-4 w-4 mr-2" />{ENABLE_INTERNAL_CHECKOUT ? "Checkout" : "Buy Now"}</>
                              )}
                            </Button>
                          )}
                        </div>

                        {result.credit?.eligible === false && result.credit.ineligible_reason && (
                          <Alert className="mt-4">
                            <AlertTriangle className="h-4 w-4" />
                            <AlertDescription>
                              {result.credit.ineligible_reason}
                            </AlertDescription>
                          </Alert>
                        )}
                        
                        {result.credit?.eligible && result.credit.expires_at && (
                          <div className="mt-3 text-sm text-muted-foreground">
                            Credit expires: {new Date(result.credit.expires_at).toLocaleDateString()}
                          </div>
                        )}
                      </>
                    )}
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>
      )}

      {/* Empty */}
      {results.length === 0 && !searchMutation.isPending && (
        <Card>
          <CardContent className="p-12 text-center">
            <Search className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-xl font-semibold mb-2">Start your domain search</h3>
            <p className="text-muted-foreground">
              Enter a domain above to check availability and see pricing.
            </p>
          </CardContent>
        </Card>
      )}

      {/* Contact Information Dialog */}
      <Dialog open={showContactDialog} onOpenChange={setShowContactDialog}>
        <DialogContent className="sm:max-w-[500px] max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <User className="h-5 w-5" />
              Domain Registration Details
            </DialogTitle>
            <DialogDescription>
              {selectedDomain && (
                <div className="mt-2">
                  <div className="flex items-center gap-2 mb-2">
                    <strong className="text-lg">{selectedDomain.domain}</strong>
                    {isFreeWithPro(selectedDomain) && (
                      <Badge variant="default" className="bg-green-600 text-white gap-1">
                        <Gift className="h-3 w-3" /> Free with Pro
                      </Badge>
                    )}
                  </div>
                  <div className="text-sm text-muted-foreground">
                    {isFreeWithPro(selectedDomain) 
                      ? `You'll save ${formatPrice(selectedDomain.priceCents, selectedDomain.currency)} with your Pro credit!`
                      : `Registration price: ${formatPrice(selectedDomain.outOfPocketCents, selectedDomain.currency)}/yr`
                    }
                  </div>
                </div>
              )}
              <div className="mt-3">
                Please provide your contact information for domain registration. This information will be used for the domain's WHOIS record.
              </div>
            </DialogDescription>
          </DialogHeader>

          <Form {...contactForm}>
            <form onSubmit={contactForm.handleSubmit(handleDomainRegistration)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={contactForm.control}
                  name="firstName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>First Name</FormLabel>
                      <FormControl>
                        <Input placeholder="John" {...field} data-testid="input-first-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={contactForm.control}
                  name="lastName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Last Name</FormLabel>
                      <FormControl>
                        <Input placeholder="Doe" {...field} data-testid="input-last-name" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={contactForm.control}
                name="email"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Email</FormLabel>
                    <FormControl>
                      <Input type="email" placeholder="john@example.com" {...field} data-testid="input-email" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={contactForm.control}
                name="phone"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Phone</FormLabel>
                    <FormControl>
                      <Input placeholder="+1-555-123-4567" {...field} data-testid="input-phone" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={contactForm.control}
                name="address"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Address</FormLabel>
                    <FormControl>
                      <Input placeholder="123 Main Street" {...field} data-testid="input-address" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={contactForm.control}
                  name="city"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>City</FormLabel>
                      <FormControl>
                        <Input placeholder="Houston" {...field} data-testid="input-city" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={contactForm.control}
                  name="state"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>State/Province</FormLabel>
                      <FormControl>
                        <Input placeholder="TX" {...field} data-testid="input-state" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={contactForm.control}
                  name="zip"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>ZIP/Postal Code</FormLabel>
                      <FormControl>
                        <Input placeholder="77001" {...field} data-testid="input-zip" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={contactForm.control}
                  name="country"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Country Code</FormLabel>
                      <FormControl>
                        <Input placeholder="US" maxLength={2} {...field} data-testid="input-country" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={contactForm.control}
                name="organization"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Organization (Optional)</FormLabel>
                    <FormControl>
                      <Input placeholder="Company name" {...field} data-testid="input-organization" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex gap-3 pt-4">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => {
                    setShowContactDialog(false);
                    setSelectedDomain(null);
                    contactForm.reset();
                  }}
                  disabled={isRegistering}
                  data-testid="button-cancel-registration"
                >
                  Cancel
                </Button>
                <Button
                  type="submit"
                  disabled={isRegistering}
                  className={`flex-1 ${selectedDomain && isFreeWithPro(selectedDomain) ? 'bg-green-600 hover:bg-green-700' : ''}`}
                  data-testid="button-confirm-registration"
                >
                  {isRegistering ? (
                    <>
                      <div className="h-4 w-4 animate-spin rounded-full border-2 border-current border-t-transparent mr-2" />
                      Registering...
                    </>
                  ) : selectedDomain && isFreeWithPro(selectedDomain) ? (
                    <>
                      <Gift className="h-4 w-4 mr-2" />
                      Register FREE Domain
                    </>
                  ) : (
                    <>
                      <ExternalLink className="h-4 w-4 mr-2" />
                      Register Domain
                    </>
                  )}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
      </div>
    </div>
  );
}