import { useEffect, useState } from "react";
import { useLocation, useParams } from "wouter";
import { useQuery } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Progress } from "@/components/ui/progress";
import { Separator } from "@/components/ui/separator";
import { 
  CheckCircle, 
  Clock, 
  AlertCircle, 
  XCircle, 
  CreditCard, 
  Globe, 
  User, 
  Calendar, 
  DollarSign,
  RefreshCw,
  ArrowLeft,
  CheckCircle2,
  Loader2
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { Link } from "wouter";
import { format } from "date-fns";

interface DomainOrder {
  id: string;
  userId: string;
  domain: string;
  years: number;
  privacy: boolean;
  priceCents: number;
  status: "pending" | "checkout_open" | "paid" | "registering" | "active" | "failed";
  stripeSessionId?: string;
  providerRegId?: string;
  nameservers?: string[];
  expiresAt?: string;
  registrantContact: {
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
    address: string;
    city: string;
    state: string;
    zip: string;
    country: string;
    organization?: string;
  };
  errorMessage?: string;
  createdAt: string;
  updatedAt: string;
}

const StatusConfig = {
  pending: {
    label: "Payment Pending",
    description: "Waiting for payment confirmation",
    icon: Clock,
    color: "bg-yellow-500",
    progress: 20,
    variant: "secondary" as const
  },
  checkout_open: {
    label: "Payment Processing",
    description: "Processing payment through Stripe",
    icon: CreditCard,
    color: "bg-blue-500",
    progress: 40,
    variant: "secondary" as const
  },
  paid: {
    label: "Payment Confirmed",
    description: "Payment received, preparing for registration",
    icon: CheckCircle,
    color: "bg-green-500",
    progress: 60,
    variant: "secondary" as const
  },
  registering: {
    label: "Registering Domain",
    description: "Domain registration in progress with registry",
    icon: Globe,
    color: "bg-blue-500",
    progress: 80,
    variant: "secondary" as const
  },
  active: {
    label: "Registration Complete",
    description: "Domain successfully registered and active",
    icon: CheckCircle2,
    color: "bg-green-500",
    progress: 100,
    variant: "default" as const
  },
  failed: {
    label: "Registration Failed",
    description: "Domain registration encountered an error",
    icon: XCircle,
    color: "bg-red-500",
    progress: 0,
    variant: "destructive" as const
  }
};

export default function DomainOrderStatusPage() {
  const { orderId } = useParams<{ orderId: string }>();
  const [location, setLocation] = useLocation();
  const [pollingEnabled, setPollingEnabled] = useState(true);
  const { toast } = useToast();

  // Parse URL parameters
  const searchParams = new URLSearchParams(location.split('?')[1] || '');
  const isSuccess = searchParams.get('success') === 'true';
  const isCanceled = searchParams.get('canceled') === 'true';

  // Query for domain order data with polling for non-final states
  const { 
    data: domainOrder, 
    error,
    isLoading,
    refetch,
    isRefetching
  } = useQuery<DomainOrder>({
    queryKey: ['/api/domains/orders', orderId],
    enabled: !!orderId,
    refetchInterval: (data) => {
      // Stop polling if disabled or if order is in final state
      if (!pollingEnabled) return false;
      if (!data) return 5000; // Poll every 5 seconds while loading
      
      const finalStates = ['active', 'failed'];
      return finalStates.includes(data.status) ? false : 5000;
    },
    refetchIntervalInBackground: true,
  });

  // Show success toast on initial success redirect
  useEffect(() => {
    if (isSuccess && domainOrder) {
      toast({
        title: "Payment Successful!",
        description: `Your domain registration for ${domainOrder.domain} is now processing.`,
        duration: 5000,
      });
      
      // Clean up URL parameters
      const newUrl = location.split('?')[0];
      window.history.replaceState({}, '', newUrl);
    }
  }, [isSuccess, domainOrder, location, toast]);

  // Show canceled message
  useEffect(() => {
    if (isCanceled) {
      toast({
        title: "Payment Canceled",
        description: "Your domain order was canceled. You can try again when ready.",
        variant: "destructive",
        duration: 5000,
      });
    }
  }, [isCanceled, toast]);

  const handleRefresh = () => {
    refetch();
  };

  const togglePolling = () => {
    setPollingEnabled(!pollingEnabled);
    if (!pollingEnabled) {
      refetch(); // Trigger a refresh when re-enabling polling
    }
  };

  if (isLoading && !domainOrder) {
    return (
      <div className="max-w-4xl mx-auto p-6 space-y-6">
        <div className="flex items-center gap-4 mb-6">
          <Button variant="ghost" size="sm" asChild>
            <Link href="/web-services/domains" data-testid="button-back">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Domains
            </Link>
          </Button>
          <div className="h-6 border-l border-gray-300" />
          <Skeleton className="h-8 w-48" />
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <div className="lg:col-span-2 space-y-6">
            <Card>
              <CardHeader>
                <Skeleton className="h-6 w-32" />
                <Skeleton className="h-4 w-48" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-4 w-full mb-4" />
                <Skeleton className="h-8 w-full" />
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <Skeleton className="h-6 w-24" />
              </CardHeader>
              <CardContent className="space-y-4">
                <Skeleton className="h-16 w-full" />
                <Skeleton className="h-16 w-full" />
              </CardContent>
            </Card>
          </div>

          <div className="space-y-6">
            <Card>
              <CardHeader>
                <Skeleton className="h-6 w-24" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-32 w-full" />
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="max-w-4xl mx-auto p-6">
        <div className="flex items-center gap-4 mb-6">
          <Button variant="ghost" size="sm" asChild>
            <Link href="/web-services/domains" data-testid="button-back">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Domains
            </Link>
          </Button>
        </div>

        <Alert className="border-red-200 bg-red-50">
          <AlertCircle className="h-4 w-4 text-red-600" />
          <AlertDescription>
            Failed to load domain order details. This could be because the order doesn't exist or you don't have permission to view it.
          </AlertDescription>
        </Alert>

        <div className="mt-6 flex gap-4">
          <Button onClick={handleRefresh} disabled={isLoading} data-testid="button-retry">
            <RefreshCw className="h-4 w-4 mr-2" />
            Try Again
          </Button>
          <Button variant="outline" asChild>
            <Link href="/web-services/domains" data-testid="button-domains">
              Go to Domains
            </Link>
          </Button>
        </div>
      </div>
    );
  }

  if (!domainOrder) {
    return (
      <div className="max-w-4xl mx-auto p-6">
        <div className="flex items-center gap-4 mb-6">
          <Button variant="ghost" size="sm" asChild>
            <Link href="/web-services/domains" data-testid="button-back">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Domains
            </Link>
          </Button>
        </div>
        
        <Alert>
          <AlertCircle className="h-4 w-4" />
          <AlertDescription>
            Domain order not found or you don't have permission to view it.
          </AlertDescription>
        </Alert>
      </div>
    );
  }

  const statusConfig = StatusConfig[domainOrder.status];
  const StatusIcon = statusConfig.icon;
  const price = (domainOrder.priceCents / 100).toFixed(2);

  return (
    <div className="max-w-4xl mx-auto p-6 space-y-6" data-testid="page-domain-order-status">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="sm" asChild>
            <Link href="/web-services/domains" data-testid="button-back">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Domains
            </Link>
          </Button>
          <div className="h-6 border-l border-gray-300" />
          <div>
            <h1 className="text-2xl font-bold text-gray-900 dark:text-white" data-testid="text-domain-name">
              {domainOrder.domain}
            </h1>
            <p className="text-sm text-gray-500" data-testid="text-order-id">
              Order #{domainOrder.id.slice(-8).toUpperCase()}
            </p>
          </div>
        </div>

        <div className="flex items-center gap-2">
          <Button 
            variant="outline" 
            size="sm" 
            onClick={togglePolling}
            data-testid={pollingEnabled ? "button-disable-polling" : "button-enable-polling"}
          >
            {pollingEnabled ? (
              <>
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                Auto-refresh ON
              </>
            ) : (
              <>
                <RefreshCw className="h-4 w-4 mr-2" />
                Auto-refresh OFF
              </>
            )}
          </Button>
          <Button 
            variant="outline" 
            size="sm" 
            onClick={handleRefresh} 
            disabled={isRefetching}
            data-testid="button-refresh"
          >
            <RefreshCw className={`h-4 w-4 mr-2 ${isRefetching ? 'animate-spin' : ''}`} />
            Refresh
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Main Content */}
        <div className="lg:col-span-2 space-y-6">
          {/* Status Card */}
          <Card data-testid="card-status">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-3">
                  <div className={`p-2 rounded-full ${statusConfig.color}`}>
                    <StatusIcon className="h-5 w-5 text-white" />
                  </div>
                  <span data-testid={`status-${domainOrder.status}`}>{statusConfig.label}</span>
                </CardTitle>
                <Badge variant={statusConfig.variant} data-testid="badge-status">
                  {domainOrder.status.toUpperCase()}
                </Badge>
              </div>
              <CardDescription data-testid="text-status-description">
                {statusConfig.description}
              </CardDescription>
            </CardHeader>
            <CardContent>
              {domainOrder.status === 'failed' && domainOrder.errorMessage && (
                <Alert className="border-red-200 bg-red-50 mb-4">
                  <XCircle className="h-4 w-4 text-red-600" />
                  <AlertDescription data-testid="text-error-message">
                    {domainOrder.errorMessage}
                  </AlertDescription>
                </Alert>
              )}
              
              <div className="space-y-3">
                <div className="flex justify-between text-sm">
                  <span>Registration Progress</span>
                  <span data-testid="text-progress">{statusConfig.progress}%</span>
                </div>
                <Progress value={statusConfig.progress} className="w-full" data-testid="progress-bar" />
              </div>
            </CardContent>
          </Card>

          {/* Registration Timeline */}
          <Card data-testid="card-timeline">
            <CardHeader>
              <CardTitle>Registration Timeline</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-start gap-4">
                  <div className="flex-shrink-0">
                    <div className="w-8 h-8 rounded-full bg-green-500 flex items-center justify-center">
                      <CheckCircle className="h-4 w-4 text-white" />
                    </div>
                  </div>
                  <div>
                    <h4 className="font-medium text-gray-900 dark:text-white">Order Created</h4>
                    <p className="text-sm text-gray-500" data-testid="text-created-date">
                      {format(new Date(domainOrder.createdAt), 'MMM dd, yyyy at h:mm aa')}
                    </p>
                  </div>
                </div>

                {domainOrder.status !== 'pending' && (
                  <div className="flex items-start gap-4">
                    <div className="flex-shrink-0">
                      <div className={`w-8 h-8 rounded-full ${
                        ['checkout_open', 'paid', 'registering', 'active'].includes(domainOrder.status) 
                          ? 'bg-green-500' 
                          : 'bg-gray-300'
                      } flex items-center justify-center`}>
                        <CreditCard className="h-4 w-4 text-white" />
                      </div>
                    </div>
                    <div>
                      <h4 className="font-medium text-gray-900 dark:text-white">Payment Processed</h4>
                      <p className="text-sm text-gray-500">
                        Payment confirmed via Stripe
                      </p>
                    </div>
                  </div>
                )}

                {['registering', 'active', 'failed'].includes(domainOrder.status) && (
                  <div className="flex items-start gap-4">
                    <div className="flex-shrink-0">
                      <div className={`w-8 h-8 rounded-full ${
                        domainOrder.status === 'active' 
                          ? 'bg-green-500' 
                          : domainOrder.status === 'failed' 
                            ? 'bg-red-500' 
                            : 'bg-blue-500'
                      } flex items-center justify-center`}>
                        {domainOrder.status === 'active' ? (
                          <CheckCircle className="h-4 w-4 text-white" />
                        ) : domainOrder.status === 'failed' ? (
                          <XCircle className="h-4 w-4 text-white" />
                        ) : (
                          <Globe className="h-4 w-4 text-white" />
                        )}
                      </div>
                    </div>
                    <div>
                      <h4 className="font-medium text-gray-900 dark:text-white">
                        {domainOrder.status === 'active' 
                          ? 'Registration Complete' 
                          : domainOrder.status === 'failed' 
                            ? 'Registration Failed' 
                            : 'Registration In Progress'}
                      </h4>
                      <p className="text-sm text-gray-500" data-testid="text-updated-date">
                        {format(new Date(domainOrder.updatedAt), 'MMM dd, yyyy at h:mm aa')}
                      </p>
                    </div>
                  </div>
                )}

                {domainOrder.status === 'registering' && (
                  <div className="flex items-start gap-4">
                    <div className="flex-shrink-0">
                      <div className="w-8 h-8 rounded-full border-2 border-gray-300 flex items-center justify-center">
                        <Clock className="h-4 w-4 text-gray-400" />
                      </div>
                    </div>
                    <div>
                      <h4 className="font-medium text-gray-500">Awaiting Completion</h4>
                      <p className="text-sm text-gray-400">
                        Domain will be active shortly
                      </p>
                    </div>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Sidebar */}
        <div className="space-y-6">
          {/* Order Summary */}
          <Card data-testid="card-order-summary">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Globe className="h-5 w-5" />
                Order Summary
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex justify-between">
                <span className="text-sm text-gray-600">Domain:</span>
                <span className="font-medium" data-testid="text-summary-domain">
                  {domainOrder.domain}
                </span>
              </div>
              
              <div className="flex justify-between">
                <span className="text-sm text-gray-600">Registration Period:</span>
                <span className="font-medium" data-testid="text-summary-years">
                  {domainOrder.years} year{domainOrder.years > 1 ? 's' : ''}
                </span>
              </div>
              
              <div className="flex justify-between">
                <span className="text-sm text-gray-600">Privacy Protection:</span>
                <Badge variant={domainOrder.privacy ? "default" : "secondary"} data-testid="badge-privacy">
                  {domainOrder.privacy ? "Enabled" : "Disabled"}
                </Badge>
              </div>
              
              {domainOrder.expiresAt && (
                <div className="flex justify-between">
                  <span className="text-sm text-gray-600">Expires:</span>
                  <span className="font-medium text-sm" data-testid="text-expires-date">
                    {format(new Date(domainOrder.expiresAt), 'MMM dd, yyyy')}
                  </span>
                </div>
              )}

              <Separator />
              
              <div className="flex justify-between items-center">
                <span className="text-sm text-gray-600">Total Paid:</span>
                <span className="font-bold text-lg" data-testid="text-total-price">
                  ${price}
                </span>
              </div>
            </CardContent>
          </Card>

          {/* Contact Information */}
          <Card data-testid="card-contact-info">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="h-5 w-5" />
                Registrant Contact
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div data-testid="text-contact-name">
                <p className="font-medium">
                  {domainOrder.registrantContact.firstName} {domainOrder.registrantContact.lastName}
                </p>
                {domainOrder.registrantContact.organization && (
                  <p className="text-sm text-gray-600" data-testid="text-contact-org">
                    {domainOrder.registrantContact.organization}
                  </p>
                )}
              </div>
              
              <div className="text-sm space-y-1">
                <p data-testid="text-contact-email">{domainOrder.registrantContact.email}</p>
                <p data-testid="text-contact-phone">{domainOrder.registrantContact.phone}</p>
                <div data-testid="text-contact-address">
                  <p>{domainOrder.registrantContact.address}</p>
                  <p>{domainOrder.registrantContact.city}, {domainOrder.registrantContact.state} {domainOrder.registrantContact.zip}</p>
                  <p>{domainOrder.registrantContact.country}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Technical Details */}
          {domainOrder.nameservers && domainOrder.nameservers.length > 0 && (
            <Card data-testid="card-technical-details">
              <CardHeader>
                <CardTitle className="text-sm">Technical Details</CardTitle>
              </CardHeader>
              <CardContent>
                <div>
                  <p className="text-sm text-gray-600 mb-2">Nameservers:</p>
                  <div className="space-y-1">
                    {domainOrder.nameservers.map((ns, index) => (
                      <p key={index} className="text-sm font-mono" data-testid={`text-nameserver-${index}`}>
                        {ns}
                      </p>
                    ))}
                  </div>
                </div>
                
                {domainOrder.providerRegId && (
                  <div className="mt-3 pt-3 border-t">
                    <p className="text-sm text-gray-600">Registry ID:</p>
                    <p className="text-sm font-mono" data-testid="text-provider-reg-id">
                      {domainOrder.providerRegId}
                    </p>
                  </div>
                )}
              </CardContent>
            </Card>
          )}

          {/* Action Buttons */}
          {domainOrder.status === 'failed' && (
            <Card>
              <CardContent className="pt-6">
                <Button className="w-full" onClick={() => setLocation('/domains')} data-testid="button-try-again">
                  <RefreshCw className="h-4 w-4 mr-2" />
                  Try Another Domain
                </Button>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
}