import React, { useState, useEffect, useMemo } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { useAuth } from '@/contexts/AuthContext';
import { ProUpgradeModal } from '@/components/ProUpgradeModal';
import { useLocation } from 'wouter';
import { 
  Palette, 
  Share2, 
  Globe, 
  Download, 
  Lightbulb,
  MessageSquare,
  FileText,
  X,
  Presentation,
  Image,
  Layers,
  BarChart3,
  Camera,
  Grid3x3,
  CaseSensitive,
  GalleryVerticalEnd,
  Server,
  Wand2
} from 'lucide-react';
import { bulletCubeWhite, bulletCubeGreen, logoComposerIcon, aiLogosIcon } from '@/assets';

// ProWelcomeCard Component
function ProWelcomeCard({ onDismiss }: { onDismiss: () => void }) {
  const [, setLocation] = useLocation();
  
  return (
    <div className="rounded-2xl border p-5 bg-gradient-to-br from-teal-50 to-white mb-6 relative" data-testid="card-pro-welcome">
      <button 
        onClick={onDismiss}
        className="absolute top-3 right-3 text-gray-400 hover:text-gray-600 transition-colors"
        data-testid="button-dismiss-welcome"
      >
        <X className="h-4 w-4" />
      </button>
      
      <h3 className="text-lg font-semibold mb-1">Welcome to Pro 🎉</h3>
      <p className="text-sm text-gray-600 mb-4">
        You're unlocked. Start here to build your Business Development Kit:
      </p>
      <div className="flex flex-wrap gap-2">
        <button 
          className="rounded-xl px-3 py-2 bg-black text-white hover:bg-gray-800 transition-colors" 
          onClick={() => setLocation('/brand-kit')}
          data-testid="button-welcome-brand-kit"
        >
          Open Brand Kit
        </button>
        <button 
          className="rounded-xl px-3 py-2 border hover:bg-gray-50 transition-colors" 
          onClick={() => setLocation('/slogans')}
          data-testid="button-welcome-slogan"
        >
          Slogan Generator
        </button>
        <button 
          className="rounded-xl px-3 py-2 border hover:bg-gray-50 transition-colors" 
          onClick={() => setLocation('/plan')}
          data-testid="button-welcome-plan"
        >
          Business Plan
        </button>
      </div>
      <button 
        className="mt-4 text-sm underline text-gray-500 hover:text-gray-700" 
        onClick={onDismiss}
        data-testid="button-dismiss-text"
      >
        Dismiss
      </button>
    </div>
  );
}

export default function Dashboard() {
  const { currentUser } = useAuth();
  const [, setLocation] = useLocation();
  const [me, setMe] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  // Parse URL parameters for upgrade modal
  const urlParams = useMemo(() => {
    if (typeof window === 'undefined') return { upgrade: null, billing: 'monthly' };
    const params = new URLSearchParams(window.location.search);
    return {
      upgrade: params.get('upgrade'),
      billing: (params.get('billing') as 'monthly' | 'yearly') || 'monthly'
    };
  }, []);

  // Modal state
  const [showUpgradeModal, setShowUpgradeModal] = useState(false);

  // Show upgrade modal if URL params indicate
  useEffect(() => {
    if (urlParams.upgrade === 'pro' && currentUser && !me?.isPaid) {
      setShowUpgradeModal(true);
    }
  }, [urlParams.upgrade, currentUser, me?.isPaid]);

  // Close modal and clean URL
  const handleCloseUpgradeModal = () => {
    setShowUpgradeModal(false);
    // Clean URL params without causing a page reload
    const url = new URL(window.location.href);
    url.searchParams.delete('upgrade');
    url.searchParams.delete('billing');
    window.history.replaceState({}, '', url.toString());
  };

  const quickActions = [
    // Business Development
    {
      title: 'Brand Name Wizard',
      description: 'AI-powered name generation for your brand with matching domain suggestions.',
      icon: () => <Wand2 className="w-6 h-6 transform scale-x-[-1]" />,
      path: '/business-development/brand-name-wizard',
      color: 'bg-white border-2 border-blue-500 group-hover:bg-blue-50 text-blue-600',
      badge: 'Free'
    },
    {
      title: 'Slogan Generator',
      description: 'Create memorable slogans and taglines with AI-powered suggestions.',
      icon: () => <MessageSquare className="w-6 h-6" />,
      path: '/business-development/slogan-generator',
      color: 'bg-purple-100 group-hover:bg-purple-200 text-purple-600',
      badge: 'Free'
    },
    {
      title: 'AI Logo Creator',
      description: 'AI-powered logo generation with customizable templates.',
      icon: () => <img src={aiLogosIcon} alt="AI Logo Creator" className="w-6 h-6" style={{ filter: 'invert(36%) sepia(93%) saturate(4628%) hue-rotate(330deg) brightness(95%) contrast(95%)' }} />,
      path: '/business-development/ai-logo-creator',
      color: 'bg-white border-2 border-rose-500 group-hover:bg-rose-50 text-rose-600',
      badge: 'Membership'
    },
    {
      title: 'Brand Kit',
      description: 'Design your complete brand identity with colors, fonts, and logos.',
      icon: () => <img src={bulletCubeGreen} alt="Brand Kit" className="w-6 h-6" />,
      path: '/business-development/brand-kit',
      color: 'bg-white border-2 border-green-500 group-hover:bg-green-50 text-green-600',
      badge: 'Membership'
    },
    {
      title: 'AI Business Plan Developer',
      description: 'Generate comprehensive business plans tailored to your industry.',
      icon: () => <GalleryVerticalEnd className="w-6 h-6" />,
      path: '/business-development/ai-business-plan-developer',
      color: 'bg-white border-2 border-orange-500 group-hover:bg-orange-50 text-orange-600',
      badge: 'Membership'
    },
    // Business Assets
    {
      title: 'Logo Templates',
      description: 'Browse and customize professional logo templates for your brand.',
      icon: () => <img src={logoComposerIcon} alt="Logo Templates" className="w-6 h-6 opacity-50 grayscale" />,
      path: '/business-assets/logo-templates',
      color: 'bg-white border-2 border-gray-500 group-hover:bg-gray-50 text-gray-500',
      badge: 'Free'
    },
    // Web Services
    {
      title: 'Find a Domain',
      description: 'Search and secure the perfect domain for your business.',
      icon: () => <Globe className="w-6 h-6" />,
      path: '/web-services/domains',
      color: 'bg-white border-2 border-cyan-500 group-hover:bg-cyan-50 text-cyan-600',
      badge: 'Membership'
    },
    {
      title: 'Web Hosting',
      description: 'Reliable hosting solutions for your websites and applications.',
      icon: () => <Server className="w-6 h-6" />,
      path: '/web-services/web-hosting',
      color: 'bg-white border-2 border-amber-500 group-hover:bg-amber-50 text-amber-600',
      badge: 'Membership'
    }
  ];

  // Get counts from localStorage
  const downloadCount = localStorage.getItem('downloadCount') || '0';
  const brandKitCount = localStorage.getItem('brandKitCount') || '0';
  const websiteCount = localStorage.getItem('websiteCount') || '0';
  const socialKitCount = localStorage.getItem('socialKitCount') || '0';
  
  const stats = [
    { label: 'Brand Kits', value: brandKitCount, icon: () => <img src={bulletCubeGreen} alt="Brand Kits" className="w-6 h-6" />, color: 'text-green-600' },
    { label: 'Slogans', value: localStorage.getItem('sloganCount') || '0', icon: MessageSquare, color: 'text-purple-600' },
    { label: 'Business Plans', value: localStorage.getItem('businessPlanCount') || '0', icon: FileText, color: 'text-rose-600' },
    { label: 'Downloads', value: downloadCount, icon: Download, color: 'text-purple-600' }
  ];

  const firstName = currentUser?.displayName?.split(' ')[0] || 'User';

  // Fetch user profile data
  useEffect(() => {
    const fetchUserProfile = async () => {
      if (!currentUser) {
        setLoading(false);
        return;
      }

      try {
        const token = await currentUser.getIdToken();
        const response = await fetch('/api/auth/me', {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        });

        if (response.ok) {
          const userData = await response.json();
          setMe(userData);
        }
      } catch (error) {
        console.error('Failed to fetch user profile:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchUserProfile();
  }, [currentUser]);

  // Show Pro welcome card logic
  const showProWelcome = me && !!me.isPaid && !me.proWelcomeDismissed && !!me.proActivatedAt;

  // Dismiss welcome card
  const dismissWelcome = async () => {
    if (!currentUser) return;

    try {
      const token = await currentUser.getIdToken();
      await fetch('/api/profile/dismiss-pro-welcome', { 
        method: 'POST', 
        headers: {
          'Authorization': `Bearer ${token}`,
        },
      });
      setMe({ ...me, proWelcomeDismissed: true });
    } catch (error) {
      console.error('Failed to dismiss welcome card:', error);
    }
  };

  if (loading) {
    return (
      <div className="space-y-6">
        <div className="gradient-bg rounded-xl p-6 text-primary-foreground animate-pulse">
          <div className="h-6 bg-white/20 rounded w-1/2 mb-2"></div>
          <div className="h-4 bg-white/10 rounded w-3/4"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Pro Upgrade Modal */}
      <ProUpgradeModal 
        isOpen={showUpgradeModal}
        onClose={handleCloseUpgradeModal}
        billing={urlParams.billing as 'monthly' | 'yearly'}
      />
      
      {/* Pro Welcome Card */}
      {showProWelcome && <ProWelcomeCard onDismiss={dismissWelcome} />}
      
      {/* Welcome Section */}
      <div className="gradient-bg rounded-xl p-6 text-primary-foreground">
        <h2 className="text-2xl font-bold mb-2">
          Welcome back, <span data-testid="text-welcome-name">{firstName}</span>!
        </h2>
        <p className="text-primary-foreground/80">
          Ready to build your brand? Let's create something amazing today.
        </p>
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        {stats.map((stat, index) => {
          const Icon = stat.icon;
          return (
            <Card key={index} data-testid={`stat-${stat.label.toLowerCase().replace(' ', '-')}`}>
              <CardContent className="pt-6">
                <div className="flex items-center">
                  <div className="w-10 h-10 bg-primary/10 rounded-lg flex items-center justify-center mr-4">
                    <Icon className={`h-5 w-5 ${stat.color}`} />
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">{stat.label}</p>
                    <p className="text-2xl font-bold text-dark">{stat.value}</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {/* Quick Actions */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {quickActions.map((action, index) => {
          const Icon = action.icon;
          return (
            <button
              key={index}
              onClick={() => setLocation(action.path)}
              className="bg-card border border-border rounded-lg p-6 text-left hover:shadow-md transition-shadow group"
              data-testid={`button-${action.title.toLowerCase().replace(/\s+/g, '-')}`}
            >
              <div className={`w-12 h-12 rounded-xl flex items-center justify-center mb-2 transition-colors ${action.color}`}>
                <Icon />
              </div>
              <h3 className="font-semibold text-dark mb-2 text-left flex items-center gap-2">
                {action.title}
                <span className={`text-xs font-normal px-2 py-0.5 rounded ${
                  action.badge === 'Free' 
                    ? 'bg-green-100 text-green-700' 
                    : 'bg-blue-100 text-blue-700'
                }`}>
                  ({action.badge})
                </span>
              </h3>
              <p className="text-sm text-muted-foreground">{action.description}</p>
            </button>
          );
        })}
      </div>

      {/* Recent Projects */}
      <Card>
        <div className="p-6 border-b border-border">
          <h3 className="text-lg font-semibold text-dark">Recent Projects</h3>
        </div>
        <CardContent className="p-6">
          {(() => {
            const recentProjects = JSON.parse(localStorage.getItem('brandKits') || '[]')
              .sort((a: any, b: any) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime())
              .slice(0, 3);
            
            if (recentProjects.length === 0) {
              return (
                <div className="text-center py-8">
                  <p className="text-muted-foreground">No projects yet. Start creating your first brand kit!</p>
                  <Button 
                    onClick={() => setLocation('/brand-kit')}
                    className="mt-4"
                    data-testid="button-create-first-project"
                  >
                    Create Your First Project
                  </Button>
                </div>
              );
            }
            
            return (
              <div className="space-y-3">
                {recentProjects.map((project: any, index: number) => (
                  <div key={project.id || index} className="flex items-center p-3 rounded-lg hover:bg-gray-50 cursor-pointer" data-testid={`project-${index}`}>
                    <div className="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                      <img src={bulletCubeGreen} alt="Brand Kit" className="w-6 h-6" />
                    </div>
                    <div className="flex-1">
                      <h4 className="font-medium text-dark">{project.businessName || 'Untitled Project'}</h4>
                      <p className="text-sm text-muted-foreground">
                        {project.industry && `${project.industry} · `}
                        {new Date(project.createdAt).toLocaleDateString()}
                      </p>
                    </div>
                    <span className="text-xs px-2 py-1 bg-primary/10 text-primary rounded">
                      {project.personality || 'Brand Kit'}
                    </span>
                  </div>
                ))}
                <div className="pt-2">
                  <Button 
                    variant="outline" 
                    size="sm" 
                    onClick={() => setLocation('/brand-kit')}
                    className="w-full"
                    data-testid="button-create-new-project"
                  >
                    Create New Project
                  </Button>
                </div>
              </div>
            );
          })()}
        </CardContent>
      </Card>
    </div>
  );
}
