import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/contexts/AuthContext";
import { useCart } from "@/contexts/CartContext";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { toast } from "sonner";
import { Link, useParams, useLocation } from "wouter";
import {
  ArrowLeft,
  ExternalLink,
  MapPin,
  Calendar,
  Star,
  TrendingUp,
  User,
  ShoppingCart,
  Eye,
  DollarSign,
  FileImage,
  FileText,
  File,
  Award,
  Users
} from "lucide-react";

interface Creator {
  id: string;
  name: string;
  email: string;
  profileData?: {
    bio?: string;
    portfolio?: string;
    avatar?: string;
    location?: string;
    website?: string;
    socialLinks?: {
      twitter?: string;
      instagram?: string;
      behance?: string;
      dribbble?: string;
    };
  };
  totalEarnings: number;
  isActive: boolean;
  createdAt: string;
}

interface CreatorStats {
  totalAssets: number;
  totalSales: number;
  totalRevenue: number;
  averageRating: number;
  totalReviews: number;
  joinedDate: string;
}

interface Asset {
  fileName: string;
  fileUrl: string;
  previewUrl?: string;
  fileSize: number;
  mimeType: string;
  assetType: string;
  category?: string;
  tags: string[];
  downloadCount: number;
}

interface CreatorAsset {
  id: string;
  assetId: string;
  title: string;
  description?: string;
  price: number;
  isExclusive: boolean;
  salesCount: number;
  createdAt: string;
  asset: Asset;
}

interface CreatorProfile {
  creator: Creator;
  stats: CreatorStats;
  assets: CreatorAsset[];
  totalCount: number;
  hasMore: boolean;
}

export default function CreatorProfilePage() {
  const { id } = useParams();
  const { currentUser } = useAuth();
  const { addItem, hasItem } = useCart();
  const [, setLocation] = useLocation();
  const [page, setPage] = useState(1);

  // Fetch creator profile data
  const { data: creatorData, isLoading: isCreatorLoading, error: creatorError } = useQuery({
    queryKey: ['/api/marketplace/creators', id, 'profile'],
    queryFn: async () => {
      const response = await fetch(`/api/marketplace/creators/${id}/assets?limit=1`, {
        credentials: 'include'
      });

      if (!response.ok) {
        if (response.status === 404) {
          throw new Error('Creator not found');
        }
        throw new Error('Failed to fetch creator profile');
      }

      const data = await response.json();
      
      // Extract creator and stats data only
      return {
        creator: {
          id: data.creator.id,
          name: data.creator.name,
          email: data.creator.name || '', // Using name as email fallback since API doesn't expose email
          profileData: {
            bio: data.creator.bio,
            portfolio: data.creator.portfolio,
          },
          totalEarnings: 0, // Not exposed by API for security
          isActive: true, // Assumed if data is returned
          createdAt: new Date().toISOString(), // Default since not provided
        },
        stats: {
          totalAssets: data.stats.totalAssets,
          totalSales: data.stats.totalSales,
          totalRevenue: 0, // Not exposed by API
          averageRating: 0, // Not implemented yet
          totalReviews: 0, // Not implemented yet
          joinedDate: new Date().toISOString(), // Default since not provided
        }
      };
    },
    enabled: !!id,
  });

  // Fetch creator assets separately
  const { data: assetsData, isLoading: isAssetsLoading, error: assetsError } = useQuery({
    queryKey: ['/api/marketplace/creators', id, 'assets', page],
    queryFn: async () => {
      const response = await fetch(`/api/marketplace/creators/${id}/assets?page=${page}&limit=12`, {
        credentials: 'include'
      });

      if (!response.ok) {
        if (response.status === 404) {
          throw new Error('Creator assets not found');
        }
        throw new Error('Failed to fetch creator assets');
      }

      const data = await response.json();
      
      // Transform assets to match expected interface
      const transformedAssets = data.assets.map((asset: any) => ({
        id: asset.id,
        assetId: asset.assetId,
        title: asset.title,
        description: asset.description,
        price: asset.price,
        isExclusive: asset.isExclusive,
        salesCount: asset.salesCount,
        createdAt: asset.createdAt,
        asset: {
          fileName: asset.originalFileName,
          fileUrl: '',
          previewUrl: `/api/marketplace/assets/${asset.id}/preview`, // Construct preview URL
          fileSize: asset.fileSize,
          mimeType: asset.mimeType || 'application/octet-stream',
          assetType: asset.assetType,
          category: asset.category,
          tags: asset.tags || [],
          downloadCount: asset.downloadCount || 0,
        }
      }));

      return {
        assets: transformedAssets,
        totalCount: data.stats.totalAssets,
        hasMore: transformedAssets.length === 12, // Has more if we got full page
      };
    },
    enabled: !!id,
  });

  // Combine loading and error states
  const isLoading = isCreatorLoading || isAssetsLoading;
  const error = creatorError || assetsError;

  const formatPrice = (cents: number) => {
    return `$${(cents / 100).toFixed(2)}`;
  };

  const getAssetIcon = (assetType: string) => {
    switch (assetType) {
      case 'image': return <FileImage className="h-4 w-4" />;
      case 'video': return <FileText className="h-4 w-4" />;
      case 'audio': return <FileText className="h-4 w-4" />;
      case 'document': return <FileText className="h-4 w-4" />;
      case 'template': return <FileText className="h-4 w-4" />;
      default: return <File className="h-4 w-4" />;
    }
  };

  const handleAddToCart = (asset: CreatorAsset) => {
    try {
      const success = addItem({
        itemType: 'creator_asset',
        itemId: asset.id,
        itemName: asset.title,
        itemPrice: asset.price,
        quantity: 1,
        metadata: {
          creatorId: creatorData?.creator.id,
          assetId: asset.assetId,
          previewUrl: asset.asset.previewUrl,
          assetType: asset.asset.assetType,
          category: asset.asset.category,
          creatorName: creatorData?.creator.name
        }
      });
      
      // Only show success toast if item was actually added (cart constraints allow it)
      if (success) {
        toast.success(`"${asset.title}" added to cart!`);
      }
      // Error toasts are handled by the CartProvider
    } catch (error) {
      toast.error("Failed to add item to cart");
      console.error("Add to cart error:", error);
    }
  };

  if (error) {
    return (
      <div className="container mx-auto px-4 py-8">
        <Button
          variant="ghost"
          onClick={() => setLocation('/marketplace')}
          className="mb-6"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back to Marketplace
        </Button>
        
        <Alert>
          <AlertDescription>
            {error instanceof Error ? error.message : 'Failed to load creator profile'}
          </AlertDescription>
        </Alert>
      </div>
    );
  }

  if (isLoading || !creatorData) {
    return (
      <div className="container mx-auto px-4 py-8">
        <Button
          variant="ghost"
          onClick={() => setLocation('/marketplace')}
          className="mb-6"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back to Marketplace
        </Button>

        <div className="space-y-8">
          {/* Profile Header Skeleton */}
          <Card>
            <CardContent className="pt-6">
              <div className="flex items-start gap-6">
                <Skeleton className="h-24 w-24 rounded-full" />
                <div className="flex-1 space-y-3">
                  <Skeleton className="h-8 w-48" />
                  <Skeleton className="h-4 w-96" />
                  <Skeleton className="h-4 w-64" />
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Stats Skeleton */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            {[...Array(4)].map((_, i) => (
              <Card key={i}>
                <CardContent className="pt-6">
                  <Skeleton className="h-8 w-16 mb-2" />
                  <Skeleton className="h-4 w-20" />
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Assets Grid Skeleton */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {[...Array(8)].map((_, i) => (
              <Card key={i}>
                <Skeleton className="aspect-video w-full" />
                <CardContent className="p-4">
                  <Skeleton className="h-4 w-3/4 mb-2" />
                  <Skeleton className="h-3 w-1/2" />
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </div>
    );
  }

  const creator = creatorData?.creator;
  const stats = creatorData?.stats;
  const assets = assetsData?.assets || [];

  return (
    <div className="min-h-screen bg-background">
      <div className="container mx-auto px-4 py-8">
        {/* Navigation */}
        <Button
          variant="ghost"
          onClick={() => setLocation('/marketplace')}
          className="mb-8"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back to Marketplace
        </Button>

        {/* Creator Profile Header */}
        <Card className="mb-8">
          <CardContent className="pt-6">
            <div className="flex flex-col lg:flex-row items-start gap-6">
              {/* Avatar */}
              <Avatar className="h-24 w-24 lg:h-32 lg:w-32">
                <AvatarImage src={creator.profileData?.avatar} />
                <AvatarFallback className="text-2xl">
                  {(creator.name || creator.email).charAt(0).toUpperCase()}
                </AvatarFallback>
              </Avatar>

              {/* Profile Info */}
              <div className="flex-1">
                <div className="flex flex-col lg:flex-row lg:items-start lg:justify-between mb-4">
                  <div>
                    <h1 className="text-3xl font-bold mb-2" data-testid="creator-name">
                      {creator.name || creator.email}
                    </h1>
                    {creator.profileData?.bio && (
                      <p className="text-muted-foreground text-lg mb-4 leading-relaxed">
                        {creator.profileData.bio}
                      </p>
                    )}
                  </div>
                  <Badge variant={creator.isActive ? "default" : "secondary"} className="lg:ml-4">
                    {creator.isActive ? "Active" : "Inactive"}
                  </Badge>
                </div>

                {/* Creator Metadata */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 text-sm text-muted-foreground mb-6">
                  <div className="flex items-center">
                    <Calendar className="h-4 w-4 mr-2" />
                    Joined {new Date(creator.createdAt).toLocaleDateString()}
                  </div>
                  {creator.profileData?.location && (
                    <div className="flex items-center">
                      <MapPin className="h-4 w-4 mr-2" />
                      {creator.profileData.location}
                    </div>
                  )}
                  {stats.averageRating > 0 && (
                    <div className="flex items-center">
                      <Star className="h-4 w-4 mr-2 text-yellow-500" />
                      {stats.averageRating.toFixed(1)} ({stats.totalReviews} reviews)
                    </div>
                  )}
                </div>

                {/* External Links */}
                <div className="flex flex-wrap gap-2">
                  {creator.profileData?.portfolio && (
                    <Button variant="outline" size="sm" asChild>
                      <a href={creator.profileData.portfolio} target="_blank" rel="noopener noreferrer">
                        <ExternalLink className="h-4 w-4 mr-2" />
                        Portfolio
                      </a>
                    </Button>
                  )}
                  {creator.profileData?.website && (
                    <Button variant="outline" size="sm" asChild>
                      <a href={creator.profileData.website} target="_blank" rel="noopener noreferrer">
                        <ExternalLink className="h-4 w-4 mr-2" />
                        Website
                      </a>
                    </Button>
                  )}
                  {creator.profileData?.socialLinks?.behance && (
                    <Button variant="outline" size="sm" asChild>
                      <a href={creator.profileData.socialLinks.behance} target="_blank" rel="noopener noreferrer">
                        <ExternalLink className="h-4 w-4 mr-2" />
                        Behance
                      </a>
                    </Button>
                  )}
                  {creator.profileData?.socialLinks?.dribbble && (
                    <Button variant="outline" size="sm" asChild>
                      <a href={creator.profileData.socialLinks.dribbble} target="_blank" rel="noopener noreferrer">
                        <ExternalLink className="h-4 w-4 mr-2" />
                        Dribbble
                      </a>
                    </Button>
                  )}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Stats Cards */}
        <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
          <Card>
            <CardContent className="pt-6">
              <div className="flex items-center">
                <TrendingUp className="h-8 w-8 text-primary mr-3" />
                <div>
                  <div className="text-2xl font-bold" data-testid="total-assets">
                    {stats.totalAssets}
                  </div>
                  <div className="text-sm text-muted-foreground">Assets</div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="pt-6">
              <div className="flex items-center">
                <Users className="h-8 w-8 text-green-600 mr-3" />
                <div>
                  <div className="text-2xl font-bold" data-testid="total-sales">
                    {stats.totalSales}
                  </div>
                  <div className="text-sm text-muted-foreground">Sales</div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="pt-6">
              <div className="flex items-center">
                <DollarSign className="h-8 w-8 text-blue-600 mr-3" />
                <div>
                  <div className="text-2xl font-bold" data-testid="total-revenue">
                    {formatPrice(stats.totalRevenue)}
                  </div>
                  <div className="text-sm text-muted-foreground">Revenue</div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="pt-6">
              <div className="flex items-center">
                <Star className="h-8 w-8 text-yellow-500 mr-3" />
                <div>
                  <div className="text-2xl font-bold" data-testid="average-rating">
                    {stats.averageRating > 0 ? stats.averageRating.toFixed(1) : 'N/A'}
                  </div>
                  <div className="text-sm text-muted-foreground">Rating</div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Assets Section */}
        <div className="mb-8">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-2xl font-bold">Assets ({stats.totalAssets})</h2>
            {assets.length > 0 && (
              <div className="text-sm text-muted-foreground">
                Showing {assets.length} of {profileData.totalCount} assets
              </div>
            )}
          </div>

          {assets.length === 0 ? (
            <Card>
              <CardContent className="py-12 text-center">
                <FileImage className="h-16 w-16 mx-auto mb-4 text-muted-foreground" />
                <h3 className="text-lg font-medium mb-2">No assets yet</h3>
                <p className="text-muted-foreground">
                  This creator hasn't published any assets yet. Check back later!
                </p>
              </CardContent>
            </Card>
          ) : (
            <>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                {assets.map((asset) => (
                  <Card key={asset.id} className="group overflow-hidden hover:shadow-lg transition-all duration-300" data-testid={`asset-card-${asset.id}`}>
                    <div className="aspect-video bg-gray-100 dark:bg-gray-800 overflow-hidden relative">
                      {asset.asset.previewUrl ? (
                        <img
                          src={asset.asset.previewUrl}
                          alt={asset.title}
                          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                          loading="lazy"
                        />
                      ) : (
                        <div className="w-full h-full flex items-center justify-center">
                          {getAssetIcon(asset.asset.assetType)}
                        </div>
                      )}
                      
                      <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
                        <Link href={`/marketplace/asset/${asset.id}`}>
                          <Button variant="secondary" size="sm">
                            <Eye className="h-4 w-4 mr-2" />
                            View Details
                          </Button>
                        </Link>
                      </div>

                      {asset.isExclusive && (
                        <Badge className="absolute top-2 right-2 bg-yellow-500">
                          Exclusive
                        </Badge>
                      )}
                    </div>

                    <CardContent className="p-4">
                      <div className="mb-2">
                        <Link href={`/marketplace/asset/${asset.id}`}>
                          <h3 className="font-semibold text-sm mb-1 line-clamp-1 hover:text-primary cursor-pointer" data-testid={`asset-title-${asset.id}`}>
                            {asset.title}
                          </h3>
                        </Link>
                        <p className="text-xs text-muted-foreground line-clamp-2">
                          {asset.description || 'No description available'}
                        </p>
                      </div>

                      <div className="flex items-center justify-between mb-2">
                        <div className="text-sm font-bold text-green-600" data-testid={`asset-price-${asset.id}`}>
                          {formatPrice(asset.price)}
                        </div>
                        <div className="text-xs text-muted-foreground">
                          {asset.salesCount} sales
                        </div>
                      </div>

                      <div className="flex flex-wrap gap-1 mb-3">
                        {asset.asset.category && (
                          <Badge variant="outline" className="text-xs">
                            {asset.asset.category}
                          </Badge>
                        )}
                        {asset.asset.tags.slice(0, 2).map((tag, index) => (
                          <Badge key={index} variant="outline" className="text-xs">
                            {tag}
                          </Badge>
                        ))}
                        {asset.asset.tags.length > 2 && (
                          <Badge variant="outline" className="text-xs">
                            +{asset.asset.tags.length - 2}
                          </Badge>
                        )}
                      </div>

                      <div className="text-xs text-muted-foreground mb-3">
                        {new Date(asset.createdAt).toLocaleDateString()}
                      </div>

                      <Button
                        onClick={() => handleAddToCart(asset)}
                        disabled={hasItem(asset.id, 'creator_asset')}
                        className="w-full"
                        size="sm"
                        data-testid={`button-add-cart-${asset.id}`}
                      >
                        <ShoppingCart className="h-4 w-4 mr-2" />
                        {hasItem(asset.id, 'creator_asset') ? 'In Cart' : 'Add to Cart'}
                      </Button>
                    </CardContent>
                  </Card>
                ))}
              </div>

              {/* Load More */}
              {profileData.hasMore && (
                <div className="flex justify-center mt-8">
                  <Button
                    onClick={() => setPage(prev => prev + 1)}
                    disabled={isLoading}
                    data-testid="button-load-more"
                  >
                    Load More Assets
                  </Button>
                </div>
              )}
            </>
          )}
        </div>
      </div>
    </div>
  );
}