import { useState } from "react";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { toast } from "sonner";
import { Upload, Eye, Plus, X } from "lucide-react";

// Style categories for infographics
const STYLE_CATEGORIES = [
  { value: 'business', label: 'Business' },
  { value: 'professional', label: 'Professional' },
  { value: 'creative', label: 'Creative' },
  { value: 'modern', label: 'Modern' }
];

export default function UploadInfographicTemplate() {
  const [title, setTitle] = useState("");
  const [category, setCategory] = useState("business");
  const [price, setPrice] = useState("14.99");
  const [descriptors, setDescriptors] = useState<string[]>([]);
  const [newDescriptor, setNewDescriptor] = useState("");
  const [previewImage, setPreviewImage] = useState("");
  const [pptxUrl, setPptxUrl] = useState("");
  const [keynoteUrl, setKeynoteUrl] = useState("");
  const [gslidesUrl, setGslidesUrl] = useState("");
  const [bundleUrl, setBundleUrl] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);

  const addDescriptor = () => {
    if (newDescriptor.trim() && !descriptors.includes(newDescriptor.trim().toLowerCase()) && descriptors.length < 10) {
      setDescriptors([...descriptors, newDescriptor.trim().toLowerCase()]);
      setNewDescriptor("");
    }
  };

  const removeDescriptor = (descriptor: string) => {
    setDescriptors(descriptors.filter(d => d !== descriptor));
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      addDescriptor();
    }
  };

  const submit = async () => {
    // Basic validation
    if (!title.trim()) {
      toast.error("Title is required");
      return;
    }
    if (!previewImage.trim()) {
      toast.error("Preview image URL is required");
      return;
    }
    if (descriptors.length === 0) {
      toast.error("At least one descriptor is required");
      return;
    }

    // At least one format URL should be provided
    if (!pptxUrl.trim() && !keynoteUrl.trim() && !gslidesUrl.trim() && !bundleUrl.trim()) {
      toast.error("At least one format URL (PPTX, Keynote, Google Slides, or Bundle) is required");
      return;
    }

    setIsSubmitting(true);
    try {
      const res = await fetch("/api/creator/infographics", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          title: title.trim(),
          category,
          priceCents: Math.max(0, parseFloat(price) * 100),
          descriptors,
          previewImageUrl: previewImage.trim(),
          pptxUrl: pptxUrl.trim() || null,
          keynoteUrl: keynoteUrl.trim() || null,
          gslidesUrl: gslidesUrl.trim() || null,
          downloadBundleUrl: bundleUrl.trim() || null
        })
      });

      if (!res.ok) {
        const errorData = await res.json().catch(() => ({ error: "Upload failed" }));
        throw new Error(errorData.error || "Upload failed");
      }

      toast.success("Infographic template submitted for review. We'll activate it once approved!");
      
      // Clear the form
      setTitle("");
      setCategory("business");
      setPrice("14.99");
      setDescriptors([]);
      setPreviewImage("");
      setPptxUrl("");
      setKeynoteUrl("");
      setGslidesUrl("");
      setBundleUrl("");
      
    } catch (error: any) {
      console.error("Upload error:", error);
      toast.error(error.message || "Failed to upload template");
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <DashboardTemplatePage title="Upload Infographic Template">
      <div className="max-w-4xl mx-auto space-y-6">
        <div className="grid md:grid-cols-2 gap-6">
          {/* Form Section */}
          <div className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Template Details</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="title">Template Title *</Label>
                  <Input 
                    id="title"
                    placeholder="e.g., Financial Performance Dashboard" 
                    value={title} 
                    onChange={e => setTitle(e.target.value)}
                    data-testid="input-title"
                  />
                </div>

                <div>
                  <Label htmlFor="category">Style Category *</Label>
                  <Select value={category} onValueChange={setCategory}>
                    <SelectTrigger data-testid="select-category">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {STYLE_CATEGORIES.map(cat => (
                        <SelectItem key={cat.value} value={cat.value}>
                          {cat.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="price">Price (USD) *</Label>
                  <Input 
                    id="price"
                    placeholder="14.99" 
                    value={price} 
                    onChange={e => setPrice(e.target.value)}
                    data-testid="input-price"
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    Note: Infographics are sold in bundles at $14.99 regardless of individual price
                  </p>
                </div>

                <div>
                  <Label htmlFor="descriptors">Descriptors *</Label>
                  <div className="flex items-center gap-2">
                    <Input 
                      id="descriptors"
                      placeholder="e.g., dashboard, financial, charts"
                      value={newDescriptor}
                      onChange={e => setNewDescriptor(e.target.value)}
                      onKeyPress={handleKeyPress}
                      data-testid="input-new-descriptor"
                    />
                    <Button 
                      type="button" 
                      size="sm" 
                      onClick={addDescriptor}
                      disabled={!newDescriptor.trim() || descriptors.length >= 10}
                      data-testid="button-add-descriptor"
                    >
                      <Plus className="h-4 w-4" />
                    </Button>
                  </div>
                  <div className="flex flex-wrap gap-1 mt-2">
                    {descriptors.map(descriptor => (
                      <Badge key={descriptor} variant="secondary" className="capitalize">
                        {descriptor}
                        <button
                          onClick={() => removeDescriptor(descriptor)}
                          className="ml-1 hover:text-destructive"
                          data-testid={`remove-descriptor-${descriptor}`}
                        >
                          <X className="h-3 w-3" />
                        </button>
                      </Badge>
                    ))}
                  </div>
                  <p className="text-xs text-muted-foreground mt-1">
                    Add descriptive tags (max 10). Press Enter to add each tag.
                  </p>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>File URLs</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="preview">Preview Image URL *</Label>
                  <Input 
                    id="preview"
                    placeholder="https://example.com/preview.jpg" 
                    value={previewImage} 
                    onChange={e => setPreviewImage(e.target.value)}
                    data-testid="input-preview-image"
                  />
                </div>

                <div>
                  <Label htmlFor="pptx">PowerPoint (.pptx) URL</Label>
                  <Input 
                    id="pptx"
                    placeholder="https://example.com/template.pptx" 
                    value={pptxUrl} 
                    onChange={e => setPptxUrl(e.target.value)}
                    data-testid="input-pptx-url"
                  />
                </div>

                <div>
                  <Label htmlFor="keynote">Keynote (.key) URL</Label>
                  <Input 
                    id="keynote"
                    placeholder="https://example.com/template.key" 
                    value={keynoteUrl} 
                    onChange={e => setKeynoteUrl(e.target.value)}
                    data-testid="input-keynote-url"
                  />
                </div>

                <div>
                  <Label htmlFor="gslides">Google Slides Link</Label>
                  <Input 
                    id="gslides"
                    placeholder="https://docs.google.com/presentation/..." 
                    value={gslidesUrl} 
                    onChange={e => setGslidesUrl(e.target.value)}
                    data-testid="input-gslides-url"
                  />
                </div>

                <div>
                  <Label htmlFor="bundle">Bundle ZIP URL</Label>
                  <Input 
                    id="bundle"
                    placeholder="https://example.com/bundle.zip" 
                    value={bundleUrl} 
                    onChange={e => setBundleUrl(e.target.value)}
                    data-testid="input-bundle-url"
                  />
                </div>

                <p className="text-xs text-muted-foreground">
                  * At least one format URL is required
                </p>
              </CardContent>
            </Card>
          </div>

          {/* Preview Section */}
          <div>
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Eye className="h-4 w-4" />
                  Preview
                </CardTitle>
              </CardHeader>
              <CardContent>
                {previewImage ? (
                  <div className="space-y-3">
                    <div className="relative">
                      <img
                        src={previewImage}
                        alt={title || "Template Preview"}
                        className="w-full h-48 object-cover rounded border"
                        onError={(e) => {
                          e.currentTarget.style.display = 'none';
                          e.currentTarget.nextElementSibling!.classList.remove('hidden');
                        }}
                        data-testid="preview-image"
                      />
                      <div className="hidden bg-muted rounded border w-full h-48 flex items-center justify-center text-muted-foreground">
                        Failed to load image
                      </div>
                    </div>
                    
                    <div>
                      <h3 className="font-medium" data-testid="preview-title">
                        {title || "Untitled Template"}
                      </h3>
                      <div className="flex items-center gap-2 mt-1">
                        <Badge variant="outline" className="capitalize" data-testid="preview-category">
                          {category}
                        </Badge>
                        {descriptors.slice(0, 3).map(descriptor => (
                          <Badge key={descriptor} variant="secondary" className="text-xs capitalize">
                            {descriptor}
                          </Badge>
                        ))}
                        {descriptors.length > 3 && (
                          <Badge variant="outline" className="text-xs">
                            +{descriptors.length - 3}
                          </Badge>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground mt-1">
                        ${price} • Bundle pricing: $14.99
                      </p>
                      
                      {/* Available formats */}
                      <div className="flex items-center gap-1 mt-2">
                        {pptxUrl && <Badge variant="outline" className="text-xs">PPTX</Badge>}
                        {keynoteUrl && <Badge variant="outline" className="text-xs">Keynote</Badge>}
                        {gslidesUrl && <Badge variant="outline" className="text-xs">G.Slides</Badge>}
                        {bundleUrl && <Badge variant="outline" className="text-xs">Bundle</Badge>}
                      </div>
                    </div>
                  </div>
                ) : (
                  <div className="bg-muted rounded border h-48 flex items-center justify-center text-muted-foreground">
                    <div className="text-center">
                      <Upload className="h-8 w-8 mx-auto mb-2" />
                      <p>Enter preview image URL to see preview</p>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </div>

        <div className="flex justify-end">
          <Button 
            onClick={submit} 
            disabled={isSubmitting}
            size="lg"
            data-testid="button-submit"
          >
            {isSubmitting ? (
              <>Submitting...</>
            ) : (
              <>
                <Upload className="h-4 w-4 mr-2" />
                Submit for Review
              </>
            )}
          </Button>
        </div>
      </div>
    </DashboardTemplatePage>
  );
}