import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Link } from "wouter";
import { toast } from "sonner";
import { BarChart3, DollarSign, ShoppingBag, Upload, Eye, X } from "lucide-react";

type CreatorAsset = {
  id: string;
  title: string;
  description?: string;
  price: number;
  salesCount: number;
  createdAt: string;
  previewUrl?: string;
  approvalStatus: "approved" | "pending" | "rejected" | "cancelled";
};

type PayoutSummary = {
  lifetimeEarningsCents: number;
  pendingPayoutCents: number;
  lastPayoutAt?: string;
};

const fmt = (cents: number = 0) => `$${(cents / 100).toFixed(2)}`;

export default function MyMarketplacePage() {
  const qc = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["creator-me-marketplace"],
    queryFn: async () => {
      const res = await fetch("/api/creator/me/marketplace", { credentials: "include" });
      if (!res.ok) throw new Error("Failed to load");
      return res.json() as Promise<{
        assets: { approved: CreatorAsset[]; pending: CreatorAsset[]; rejected: CreatorAsset[]; cancelled: CreatorAsset[]; };
        stats: { totalSales: number; totalRevenueCents: number; };
      }>;
    }
  });

  const { data: payouts } = useQuery({
    queryKey: ["creator-payouts-summary"],
    queryFn: async () => {
      const res = await fetch("/api/creator/me/payouts/summary", { credentials: "include" });
      if (!res.ok) throw new Error("Failed to load payouts");
      return res.json() as Promise<PayoutSummary>;
    }
  });

  const cancelMutation = useMutation({
    mutationFn: async (id: string) => {
      const res = await fetch(`/api/creator/assets/${id}/cancel`, { method: "PATCH", credentials: "include" });
      if (!res.ok) throw new Error("Failed to cancel");
      return res.json();
    },
    onSuccess: () => {
      toast.success("Listing cancelled.");
      qc.invalidateQueries({ queryKey: ["creator-me-marketplace"] });
    },
    onError: () => toast.error("Could not cancel listing.")
  });

  const loadingCards = (
    <div className="grid grid-cols-1 sm:grid-cols-2 xl:grid-cols-3 gap-6">
      {[...Array(6)].map((_, i) => (
        <Card key={i}><Skeleton className="aspect-video w-full" /><CardContent className="p-4"><Skeleton className="h-4 w-3/4 mb-2" /><Skeleton className="h-3 w-1/2" /></CardContent></Card>
      ))}
    </div>
  );

  const AssetGrid = ({ items, cancellable = false }: { items: CreatorAsset[]; cancellable?: boolean }) => (
    <div className="grid grid-cols-1 sm:grid-cols-2 xl:grid-cols-3 gap-6">
      {items.map(a => (
        <Card key={a.id} className="overflow-hidden">
          <div className="aspect-video bg-gray-100 dark:bg-gray-800">
            {a.previewUrl && <img src={a.previewUrl} alt={a.title} className="w-full h-full object-cover" />}
          </div>
          <CardContent className="p-4 space-y-2">
            <div className="flex items-start justify-between">
              <div>
                <div className="font-semibold">{a.title}</div>
                <div className="text-xs text-muted-foreground">Created {new Date(a.createdAt).toLocaleDateString()}</div>
              </div>
              <Badge variant="secondary">{fmt(a.price)}</Badge>
            </div>
            <div className="text-xs text-muted-foreground">Sales: {a.salesCount}</div>
            <div className="flex gap-2">
              <Link href={`/marketplace/asset/${a.id}`}><Button variant="outline" size="sm"><Eye className="h-4 w-4 mr-2" />View</Button></Link>
              {cancellable && <Button variant="destructive" size="sm" onClick={() => cancelMutation.mutate(a.id)}><X className="h-4 w-4 mr-2" />Cancel</Button>}
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );

  const approved = data?.assets.approved ?? [];
  const pending = data?.assets.pending ?? [];
  const rejected = data?.assets.rejected ?? [];
  const cancelled = data?.assets.cancelled ?? [];

  return (
    <div className="space-y-8">
      {/* Overview stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card><CardContent className="p-4 flex items-center"><DollarSign className="h-8 w-8 text-primary mr-3" /><div><div className="text-2xl font-bold">{fmt(payouts?.lifetimeEarningsCents)}</div><div className="text-sm text-muted-foreground">Lifetime Earnings</div></div></CardContent></Card>
        <Card><CardContent className="p-4 flex items-center"><ShoppingBag className="h-8 w-8 text-primary mr-3" /><div><div className="text-2xl font-bold">{approved.reduce((n, a) => n + (a.salesCount || 0), 0).toLocaleString()}</div><div className="text-sm text-muted-foreground">Sales (Active)</div></div></CardContent></Card>
        <Card><CardContent className="p-4 flex items-center"><BarChart3 className="h-8 w-8 text-primary mr-3" /><div><div className="text-2xl font-bold">{fmt(payouts?.pendingPayoutCents)}</div><div className="text-sm text-muted-foreground">Pending Payout</div></div></CardContent></Card>
      </div>

      {/* Tabs */}
      <Tabs defaultValue="listings">
        <TabsList className="bg-orange-600 p-1 rounded-md">
          <TabsTrigger 
            value="listings" 
            className="text-white data-[state=active]:bg-white data-[state=active]:text-black data-[state=active]:shadow-sm"
          >
            Listings
          </TabsTrigger>
          <TabsTrigger 
            value="pending"
            className="text-white data-[state=active]:bg-white data-[state=active]:text-black data-[state=active]:shadow-sm"
          >
            Pending
          </TabsTrigger>
          <TabsTrigger 
            value="rejected"
            className="text-white data-[state=active]:bg-white data-[state=active]:text-black data-[state=active]:shadow-sm"
          >
            Rejected
          </TabsTrigger>
          <TabsTrigger 
            value="cancelled"
            className="text-white data-[state=active]:bg-white data-[state=active]:text-black data-[state=active]:shadow-sm"
          >
            Cancelled
          </TabsTrigger>
          <TabsTrigger 
            value="upload"
            className="text-white data-[state=active]:bg-white data-[state=active]:text-black data-[state=active]:shadow-sm"
          >
            Upload
          </TabsTrigger>
          <TabsTrigger 
            value="payouts"
            className="text-white data-[state=active]:bg-white data-[state=active]:text-black data-[state=active]:shadow-sm"
          >
            Payouts
          </TabsTrigger>
        </TabsList>

        <TabsContent value="listings" className="mt-6">
          {isLoading ? loadingCards : (approved.length ? <AssetGrid items={approved} cancellable /> : <Card><CardContent className="p-6 text-muted-foreground">No active listings. Click Upload.</CardContent></Card>)}
        </TabsContent>

        <TabsContent value="pending" className="mt-6">
          {isLoading ? loadingCards : (pending.length ? <AssetGrid items={pending} /> : <Card><CardContent className="p-6 text-muted-foreground">Nothing pending review.</CardContent></Card>)}
        </TabsContent>

        <TabsContent value="rejected" className="mt-6">
          {isLoading ? loadingCards : (rejected.length ? <AssetGrid items={rejected} /> : <Card><CardContent className="p-6 text-muted-foreground">No rejected listings.</CardContent></Card>)}
        </TabsContent>

        <TabsContent value="cancelled" className="mt-6">
          {isLoading ? loadingCards : (cancelled.length ? <AssetGrid items={cancelled} /> : <Card><CardContent className="p-6 text-muted-foreground">No cancelled assets.</CardContent></Card>)}
        </TabsContent>

        <TabsContent value="upload" className="mt-6">
          <Card><CardContent className="p-6 space-y-4">
            <Link href="/creator/upload"><Button><Upload className="h-4 w-4 mr-2" />Upload Asset</Button></Link>
            <Link href="/creator/upload-cover-template"><Button variant="outline"><Upload className="h-4 w-4 mr-2" />Upload Cover Template</Button></Link>
          </CardContent></Card>
        </TabsContent>

        <TabsContent value="payouts" className="mt-6">
          <Card><CardContent className="p-6 space-y-2">
            <div>Lifetime: <strong>{fmt(payouts?.lifetimeEarningsCents)}</strong></div>
            <div>Pending: <strong>{fmt(payouts?.pendingPayoutCents)}</strong></div>
            <div className="text-sm text-muted-foreground">Last payout: {payouts?.lastPayoutAt ? new Date(payouts.lastPayoutAt).toLocaleDateString() : '—'}</div>
            <Link href="/creator/earnings"><Button variant="outline" className="mt-2">Open Earnings</Button></Link>
          </CardContent></Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}