import { useState, useRef } from "react";
import { useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Progress } from "@/components/ui/progress";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { toast } from "sonner";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { auth } from "@/lib/firebase";
import { Link } from "wouter";
import { 
  Upload,
  X,
  FileImage,
  FileText,
  File,
  CheckCircle,
  AlertCircle,
  DollarSign,
  Tag,
  Info,
  ArrowLeft
} from "lucide-react";
import { z } from "zod";
import { CREATOR_MIN_PRICE_CENTS, CREATOR_MAX_PRICE_CENTS } from "@shared/schema";

const MAX_FILE_SIZE = 50 * 1024 * 1024; // 50MB
const ACCEPTED_FILE_TYPES = [
  'image/jpeg',
  'image/jpg', 
  'image/png',
  'image/webp',
  'image/svg+xml',
  'application/pdf',
  'application/zip',
  'text/plain',
  'application/json'
];

const assetCategories = [
  { value: 'graphics', label: 'Graphics & Illustrations' },
  { value: 'icons', label: 'Icons & Symbols' },
  { value: 'templates', label: 'Templates & Layouts' },
  { value: 'patterns', label: 'Patterns & Textures' },
  { value: 'fonts', label: 'Fonts & Typography' },
  { value: 'logos', label: 'Logos & Branding' },
  { value: 'photos', label: 'Photos & Stock Images' },
  { value: 'other', label: 'Other' },
];

const uploadFormSchema = z.object({
  title: z.string().min(1, "Title is required").max(100, "Title must be less than 100 characters"),
  description: z.string().optional(),
  price: z.number().min(CREATOR_MIN_PRICE_CENTS / 100, `Price must be at least $${CREATOR_MIN_PRICE_CENTS / 100}`).max(CREATOR_MAX_PRICE_CENTS / 100, `Price cannot exceed $${CREATOR_MAX_PRICE_CENTS / 100}`),
  category: z.string().min(1, "Category is required"),
  tags: z.string().optional(),
  isExclusive: z.boolean().optional().default(false),
});

type UploadFormData = z.infer<typeof uploadFormSchema>;

interface FileUploadState {
  file: File | null;
  preview: string | null;
  uploading: boolean;
  progress: number;
  error: string | null;
}

export default function CreatorUpload() {
  const [uploadState, setUploadState] = useState<FileUploadState>({
    file: null,
    preview: null,
    uploading: false,
    progress: 0,
    error: null,
  });

  const fileInputRef = useRef<HTMLInputElement>(null);
  const dropZoneRef = useRef<HTMLDivElement>(null);

  const form = useForm<UploadFormData>({
    resolver: zodResolver(uploadFormSchema),
    defaultValues: {
      title: "",
      description: "",
      price: CREATOR_MIN_PRICE_CENTS / 100,
      category: "",
      tags: "",
      isExclusive: false,
    },
  });

  const uploadMutation = useMutation({
    mutationFn: async (data: UploadFormData & { file: File }) => {
      const formData = new FormData();
      formData.append('file', data.file);
      formData.append('title', data.title);
      formData.append('description', data.description || '');
      formData.append('price', (data.price * 100).toString()); // Convert to cents
      formData.append('category', data.category);
      formData.append('tags', data.tags || '');
      formData.append('isExclusive', data.isExclusive?.toString() || 'false');

      // Get proper auth headers (without Content-Type since it's FormData)
      const authHeaders: Record<string, string> = {};
      
      // Add development bypass header in development mode
      if (import.meta.env.DEV) {
        authHeaders["X-Dev-Bypass"] = "development";
      }
      
      // Add Firebase auth token if user is authenticated
      if (auth.currentUser) {
        try {
          const token = await auth.currentUser.getIdToken();
          authHeaders["Authorization"] = `Bearer ${token}`;
        } catch (error) {
          console.warn("Failed to get Firebase ID token:", error);
          // Continue without auth token rather than failing the request
        }
      }

      const response = await fetch('/api/creator-assets/upload', {
        method: 'POST',
        body: formData,
        headers: {
          ...authHeaders,
          // Don't set Content-Type for FormData - browser will set it with boundary
        },
        credentials: 'include',
      });

      if (!response.ok) {
        const error = await response.text();
        throw new Error(error || 'Upload failed');
      }

      return response.json();
    },
    onMutate: () => {
      setUploadState(prev => ({ ...prev, uploading: true, progress: 0, error: null }));
    },
    onSuccess: (data) => {
      toast.success('Asset uploaded successfully! It will be reviewed for approval.');
      setUploadState(prev => ({ ...prev, uploading: false, progress: 100 }));
      form.reset();
      setUploadState({ file: null, preview: null, uploading: false, progress: 0, error: null });
      
      // Invalidate creator assets cache to refresh the list
      queryClient.invalidateQueries({ queryKey: ['/api/creator-assets'] });
      
      // Navigate to assets management after a brief delay
      setTimeout(() => {
        window.location.href = '/creator/assets';
      }, 1500);
    },
    onError: (error: any) => {
      const errorMessage = error.message || 'Upload failed. Please try again.';
      toast.error(errorMessage);
      setUploadState(prev => ({ ...prev, uploading: false, error: errorMessage }));
    },
  });

  const handleFileSelect = (file: File) => {
    // Validate file size
    if (file.size > MAX_FILE_SIZE) {
      setUploadState(prev => ({ ...prev, error: `File size must be less than ${MAX_FILE_SIZE / (1024 * 1024)}MB` }));
      return;
    }

    // Validate file type
    if (!ACCEPTED_FILE_TYPES.includes(file.type)) {
      setUploadState(prev => ({ ...prev, error: `File type ${file.type} is not supported` }));
      return;
    }

    // Create preview for images
    let preview: string | null = null;
    if (file.type.startsWith('image/')) {
      preview = URL.createObjectURL(file);
    }

    setUploadState({
      file,
      preview,
      uploading: false,
      progress: 0,
      error: null,
    });

    // Auto-populate title from filename
    if (!form.getValues('title')) {
      const nameWithoutExt = file.name.replace(/\.[^/.]+$/, "");
      const formattedName = nameWithoutExt
        .replace(/[-_]/g, ' ')
        .replace(/\b\w/g, l => l.toUpperCase());
      form.setValue('title', formattedName);
    }
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    const files = Array.from(e.dataTransfer.files);
    if (files.length > 0) {
      handleFileSelect(files[0]);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
  };

  const removeFile = () => {
    if (uploadState.preview) {
      URL.revokeObjectURL(uploadState.preview);
    }
    setUploadState({ file: null, preview: null, uploading: false, progress: 0, error: null });
  };

  const onSubmit = (data: UploadFormData) => {
    if (!uploadState.file) {
      toast.error('Please select a file to upload');
      return;
    }

    uploadMutation.mutate({ ...data, file: uploadState.file });
  };

  const getFileIcon = (file: File) => {
    if (file.type.startsWith('image/')) return <FileImage className="h-8 w-8" />;
    if (file.type === 'application/pdf') return <FileText className="h-8 w-8" />;
    return <File className="h-8 w-8" />;
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  return (
    <div className="max-w-4xl mx-auto space-y-8" data-testid="creator-upload">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight mb-2" data-testid="text-upload-title">
            Upload New Asset
          </h1>
          <p className="text-muted-foreground">
            Share your creative work with the marketplace
          </p>
        </div>
        <Link href="/creator/dashboard">
          <Button variant="outline" data-testid="button-back-to-dashboard">
            <ArrowLeft className="mr-2 h-4 w-4" />
            Back to Dashboard
          </Button>
        </Link>
      </div>

      <div className="grid gap-8 lg:grid-cols-2">
        {/* File Upload Section */}
        <Card data-testid="upload-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Upload className="h-5 w-5" />
              File Upload
            </CardTitle>
          </CardHeader>
          <CardContent>
            {!uploadState.file ? (
              <div
                ref={dropZoneRef}
                className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center hover:border-muted-foreground/50 transition-colors cursor-pointer"
                onDrop={handleDrop}
                onDragOver={handleDragOver}
                onClick={() => fileInputRef.current?.click()}
                data-testid="drop-zone"
              >
                <Upload className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">Drop your file here</h3>
                <p className="text-muted-foreground mb-4">
                  or click to browse from your computer
                </p>
                <div className="text-sm text-muted-foreground">
                  <p>Supported formats: JPG, PNG, WebP, SVG, PDF, ZIP, TXT, JSON</p>
                  <p>Maximum size: 50MB</p>
                </div>
                <input
                  ref={fileInputRef}
                  type="file"
                  className="hidden"
                  accept={ACCEPTED_FILE_TYPES.join(',')}
                  onChange={(e) => {
                    const file = e.target.files?.[0];
                    if (file) handleFileSelect(file);
                  }}
                  data-testid="file-input"
                />
              </div>
            ) : (
              <div className="space-y-4">
                <div className="flex items-center justify-between p-4 border rounded-lg" data-testid="file-preview">
                  <div className="flex items-center space-x-4">
                    {uploadState.preview ? (
                      <img 
                        src={uploadState.preview} 
                        alt="Preview" 
                        className="h-12 w-12 object-cover rounded"
                      />
                    ) : (
                      <div className="text-muted-foreground">
                        {getFileIcon(uploadState.file)}
                      </div>
                    )}
                    <div>
                      <p className="font-medium">{uploadState.file.name}</p>
                      <p className="text-sm text-muted-foreground">
                        {formatFileSize(uploadState.file.size)}
                      </p>
                    </div>
                  </div>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={removeFile}
                    disabled={uploadState.uploading}
                    data-testid="button-remove-file"
                  >
                    <X className="h-4 w-4" />
                  </Button>
                </div>

                {uploadState.uploading && (
                  <div className="space-y-2">
                    <div className="flex items-center justify-between text-sm">
                      <span>Uploading...</span>
                      <span>{Math.round(uploadState.progress)}%</span>
                    </div>
                    <Progress value={uploadState.progress} data-testid="upload-progress" />
                  </div>
                )}

                {uploadState.error && (
                  <Alert className="border-destructive" data-testid="upload-error">
                    <AlertCircle className="h-4 w-4" />
                    <AlertDescription>{uploadState.error}</AlertDescription>
                  </Alert>
                )}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Asset Details Form */}
        <Card data-testid="details-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Asset Details
            </CardTitle>
          </CardHeader>
          <CardContent>
            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                <FormField
                  control={form.control}
                  name="title"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Title *</FormLabel>
                      <FormControl>
                        <Input 
                          placeholder="Enter a descriptive title for your asset"
                          data-testid="input-title"
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Description</FormLabel>
                      <FormControl>
                        <Textarea 
                          placeholder="Describe your asset, its style, and potential use cases"
                          className="min-h-[100px]"
                          data-testid="textarea-description"
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <div className="grid gap-4 sm:grid-cols-2">
                  <FormField
                    control={form.control}
                    name="price"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className="flex items-center gap-2">
                          <DollarSign className="h-4 w-4" />
                          Price (USD) *
                        </FormLabel>
                        <FormControl>
                          <Input 
                            type="number"
                            min={CREATOR_MIN_PRICE_CENTS / 100}
                            max={CREATOR_MAX_PRICE_CENTS / 100}
                            step="0.01"
                            placeholder="9.99"
                            data-testid="input-price"
                            {...field}
                            onChange={(e) => field.onChange(parseFloat(e.target.value) || 0)}
                          />
                        </FormControl>
                        <div className="text-xs text-muted-foreground">
                          Min: ${CREATOR_MIN_PRICE_CENTS / 100} • Max: ${CREATOR_MAX_PRICE_CENTS / 100}
                        </div>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="category"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Category *</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                          <FormControl>
                            <SelectTrigger data-testid="select-category">
                              <SelectValue placeholder="Select category" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {assetCategories.map((category) => (
                              <SelectItem key={category.value} value={category.value}>
                                {category.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <FormField
                  control={form.control}
                  name="tags"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="flex items-center gap-2">
                        <Tag className="h-4 w-4" />
                        Tags
                      </FormLabel>
                      <FormControl>
                        <Input 
                          placeholder="design, modern, business, logo (comma-separated)"
                          data-testid="input-tags"
                          {...field} 
                        />
                      </FormControl>
                      <div className="text-xs text-muted-foreground">
                        Add relevant tags to help users find your asset
                      </div>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <Separator />

                <Alert data-testid="review-info">
                  <Info className="h-4 w-4" />
                  <AlertDescription>
                    <strong>Review Process:</strong> Your asset will be reviewed for quality and compliance. 
                    This typically takes 1-3 business days. You'll be notified when the review is complete.
                  </AlertDescription>
                </Alert>

                <div className="flex justify-end space-x-4">
                  <Button 
                    type="button" 
                    variant="outline"
                    onClick={() => form.reset()}
                    disabled={uploadState.uploading}
                    data-testid="button-clear-form"
                  >
                    Clear Form
                  </Button>
                  <Button 
                    type="submit" 
                    disabled={!uploadState.file || uploadState.uploading}
                    data-testid="button-upload-asset"
                  >
                    {uploadState.uploading ? (
                      <>
                        <Upload className="mr-2 h-4 w-4 animate-pulse" />
                        Uploading...
                      </>
                    ) : (
                      <>
                        <Upload className="mr-2 h-4 w-4" />
                        Upload Asset
                      </>
                    )}
                  </Button>
                </div>
              </form>
            </Form>
          </CardContent>
        </Card>
      </div>

      {/* Guidelines Card */}
      <Card data-testid="guidelines-card">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CheckCircle className="h-5 w-5" />
            Asset Guidelines
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid gap-6 md:grid-cols-2">
            <div>
              <h4 className="font-semibold mb-3 text-green-600 dark:text-green-400">✓ What We Accept</h4>
              <ul className="text-sm space-y-2 text-muted-foreground">
                <li>• High-quality, original creative work</li>
                <li>• Professional graphics and illustrations</li>
                <li>• Useful templates and design assets</li>
                <li>• Properly formatted files (JPG, PNG, SVG, PDF)</li>
                <li>• Clear, descriptive titles and tags</li>
                <li>• Assets that comply with copyright laws</li>
              </ul>
            </div>
            <div>
              <h4 className="font-semibold mb-3 text-red-600 dark:text-red-400">✗ What We Reject</h4>
              <ul className="text-sm space-y-2 text-muted-foreground">
                <li>• Low-resolution or poor-quality images</li>
                <li>• Copyrighted material without permission</li>
                <li>• Inappropriate or offensive content</li>
                <li>• Heavily compressed or watermarked files</li>
                <li>• Duplicate submissions or spam</li>
                <li>• Assets with misleading descriptions</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}