import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Skeleton } from "@/components/ui/skeleton";
import { useAuth } from "@/contexts/AuthContext";
import { toast } from "sonner";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Link } from "wouter";
import { 
  CheckCircle, 
  Clock, 
  XCircle, 
  RefreshCw, 
  ExternalLink, 
  User, 
  CreditCard, 
  FileCheck,
  AlertTriangle,
  ArrowRight
} from "lucide-react";

interface OnboardingStep {
  id: string;
  title: string;
  description: string;
  icon: React.ComponentType<{ className?: string }>;
  completed: boolean;
  required: boolean;
}

interface StripeAccountStatus {
  charges_enabled: boolean;
  payouts_enabled: boolean;
  currently_due: string[];
  eventually_due: string[];
  disabled_reason?: string;
  details_submitted: boolean;
}

interface CreatorOnboardingData {
  id: string;
  onboardingStatus: string;
  stripeConnectAccountId?: string;
  payoutEnabled: boolean;
  onboardingCompletedAt?: string;
  stripeAccountStatus?: StripeAccountStatus;
  requiresAction: boolean;
}

export default function CreatorOnboarding() {
  const { currentUser } = useAuth();
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Query creator profile and onboarding status
  const { 
    data: onboardingData, 
    isLoading, 
    error,
    refetch 
  } = useQuery<CreatorOnboardingData>({
    queryKey: ['/api/creators/profile'],
    enabled: !!currentUser,
    refetchInterval: (data) => {
      // Auto-refresh if onboarding is in progress
      return data?.onboardingStatus === 'in_progress' ? 5000 : false;
    },
  });

  // Mutation to create creator profile
  const createCreatorMutation = useMutation({
    mutationFn: () => apiRequest('POST', '/api/creators'),
    onSuccess: () => {
      toast.success('Creator profile created successfully');
      queryClient.invalidateQueries({ queryKey: ['/api/creators/profile'] });
    },
    onError: (error: any) => {
      toast.error(error.message || 'Failed to create creator profile');
    },
  });

  // Mutation to start Stripe Connect onboarding
  const startOnboardingMutation = useMutation({
    mutationFn: (refreshUrl?: string) => 
      apiRequest('POST', '/api/creators/connect/onboard', { 
        refresh_url: refreshUrl || window.location.href 
      }),
    onSuccess: (data: { url: string }) => {
      window.location.href = data.url;
    },
    onError: (error: any) => {
      toast.error(error.message || 'Failed to start onboarding');
      setIsSubmitting(false);
    },
  });

  // Mutation to check Stripe Connect status
  const checkStatusMutation = useMutation({
    mutationFn: () => apiRequest('GET', '/api/creators/connect/status'),
    onSuccess: () => {
      toast.success('Status updated successfully');
      queryClient.invalidateQueries({ queryKey: ['/api/creators/profile'] });
    },
    onError: (error: any) => {
      toast.error(error.message || 'Failed to check status');
    },
  });

  const getOnboardingSteps = (data?: CreatorOnboardingData): OnboardingStep[] => {
    const hasStripeAccount = !!data?.stripeConnectAccountId;
    const stripeStatus = data?.stripeAccountStatus;
    const isOnboardingComplete = data?.onboardingStatus === 'completed';

    return [
      {
        id: 'profile',
        title: 'Creator Profile',
        description: 'Set up your creator account',
        icon: User,
        completed: !!data?.id,
        required: true,
      },
      {
        id: 'stripe_connect',
        title: 'Stripe Connect Setup',
        description: 'Connect your bank account for payments',
        icon: CreditCard,
        completed: hasStripeAccount && stripeStatus?.details_submitted,
        required: true,
      },
      {
        id: 'verification',
        title: 'Account Verification',
        description: 'Complete identity verification',
        icon: FileCheck,
        completed: isOnboardingComplete && data?.payoutEnabled,
        required: true,
      },
    ];
  };

  const handleStartOnboarding = async () => {
    setIsSubmitting(true);
    
    try {
      // Create creator profile if it doesn't exist
      if (!onboardingData?.id) {
        await createCreatorMutation.mutateAsync();
        await refetch(); // Refresh data after creation
      }
      
      // Start Stripe Connect onboarding
      startOnboardingMutation.mutate();
    } catch (error) {
      setIsSubmitting(false);
    }
  };

  const handleCheckStatus = () => {
    checkStatusMutation.mutate();
  };

  if (isLoading) {
    return (
      <div className="max-w-4xl mx-auto space-y-8" data-testid="onboarding-loading">
        <div className="text-center">
          <Skeleton className="h-8 w-64 mx-auto mb-2" />
          <Skeleton className="h-4 w-96 mx-auto" />
        </div>
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-48" />
            <Skeleton className="h-4 w-full" />
          </CardHeader>
          <CardContent className="space-y-4">
            {[...Array(3)].map((_, i) => (
              <div key={i} className="flex items-center space-x-4">
                <Skeleton className="h-8 w-8 rounded-full" />
                <div className="flex-1">
                  <Skeleton className="h-4 w-32 mb-1" />
                  <Skeleton className="h-3 w-48" />
                </div>
                <Skeleton className="h-6 w-20" />
              </div>
            ))}
          </CardContent>
        </Card>
      </div>
    );
  }

  if (error) {
    return (
      <div className="max-w-4xl mx-auto text-center py-12" data-testid="onboarding-error">
        <XCircle className="h-16 w-16 text-destructive mx-auto mb-4" />
        <h2 className="text-2xl font-semibold mb-2">Unable to load onboarding</h2>
        <p className="text-muted-foreground mb-4">
          There was an error loading your onboarding status. Please try again.
        </p>
        <Button onClick={() => window.location.reload()} data-testid="button-retry">
          Try Again
        </Button>
      </div>
    );
  }

  const steps = getOnboardingSteps(onboardingData);
  const completedSteps = steps.filter(step => step.completed).length;
  const progress = (completedSteps / steps.length) * 100;
  const isCompleted = onboardingData?.onboardingStatus === 'completed';

  // Show completion screen
  if (isCompleted) {
    return (
      <div className="max-w-4xl mx-auto text-center py-12" data-testid="onboarding-completed">
        <CheckCircle className="h-16 w-16 text-green-600 mx-auto mb-4" />
        <h2 className="text-2xl font-semibold mb-2">Welcome to Creator Marketplace!</h2>
        <p className="text-muted-foreground mb-6 max-w-md mx-auto">
          Your creator account is now set up and ready. You can start uploading and selling your assets.
        </p>
        <div className="flex justify-center space-x-4">
          <Link href="/creator/dashboard">
            <Button data-testid="button-go-to-dashboard">
              Go to Dashboard
            </Button>
          </Link>
          <Link href="/creator/upload">
            <Button variant="outline" data-testid="button-upload-first-asset">
              Upload Your First Asset
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="max-w-4xl mx-auto space-y-8" data-testid="creator-onboarding">
      {/* Header */}
      <div className="text-center">
        <h1 className="text-3xl font-bold tracking-tight mb-2" data-testid="text-onboarding-title">
          Creator Onboarding
        </h1>
        <p className="text-muted-foreground">
          Complete these steps to start selling your assets on the marketplace
        </p>
      </div>

      {/* Progress Indicator */}
      <Card data-testid="progress-card">
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Setup Progress</CardTitle>
            <Badge variant="secondary" data-testid="badge-progress">
              {completedSteps} of {steps.length} completed
            </Badge>
          </div>
        </CardHeader>
        <CardContent>
          <Progress value={progress} className="mb-4" data-testid="progress-bar" />
          <p className="text-sm text-muted-foreground">
            {progress === 100 
              ? "All steps completed! Processing your application..." 
              : `${Math.round(progress)}% complete`
            }
          </p>
        </CardContent>
      </Card>

      {/* Onboarding Steps */}
      <Card data-testid="steps-card">
        <CardHeader>
          <CardTitle>Onboarding Steps</CardTitle>
          <p className="text-sm text-muted-foreground">
            Follow these steps to set up your creator account
          </p>
        </CardHeader>
        <CardContent>
          <div className="space-y-6">
            {steps.map((step, index) => {
              const Icon = step.icon;
              const isActive = !step.completed && steps.slice(0, index).every(s => s.completed);
              
              return (
                <div key={step.id} className="flex items-start space-x-4" data-testid={`step-${step.id}`}>
                  <div className="flex-shrink-0">
                    <div className={`
                      w-10 h-10 rounded-full border-2 flex items-center justify-center
                      ${step.completed 
                        ? 'bg-green-100 border-green-500 text-green-600 dark:bg-green-900 dark:border-green-600' 
                        : isActive 
                        ? 'bg-blue-100 border-blue-500 text-blue-600 dark:bg-blue-900 dark:border-blue-600' 
                        : 'bg-gray-100 border-gray-300 text-gray-400 dark:bg-gray-800 dark:border-gray-600'
                      }
                    `}>
                      {step.completed ? (
                        <CheckCircle className="h-5 w-5" />
                      ) : (
                        <Icon className="h-5 w-5" />
                      )}
                    </div>
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between">
                      <h3 className="font-medium text-foreground">{step.title}</h3>
                      <div className="flex items-center space-x-2">
                        {step.completed ? (
                          <Badge variant="secondary" className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-100">
                            Complete
                          </Badge>
                        ) : isActive ? (
                          <Badge variant="secondary" className="bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-100">
                            In Progress
                          </Badge>
                        ) : (
                          <Badge variant="outline">
                            Pending
                          </Badge>
                        )}
                      </div>
                    </div>
                    <p className="text-sm text-muted-foreground mt-1">
                      {step.description}
                    </p>

                    {/* Step-specific content */}
                    {step.id === 'stripe_connect' && isActive && (
                      <div className="mt-3">
                        <Button 
                          onClick={handleStartOnboarding} 
                          disabled={isSubmitting}
                          data-testid="button-start-stripe-onboarding"
                        >
                          {isSubmitting ? (
                            <>
                              <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                              Setting up...
                            </>
                          ) : (
                            <>
                              Start Stripe Connect Setup
                              <ExternalLink className="ml-2 h-4 w-4" />
                            </>
                          )}
                        </Button>
                      </div>
                    )}

                    {step.id === 'verification' && onboardingData?.requiresAction && (
                      <Alert className="mt-3">
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription>
                          Additional information required. Please check your Stripe dashboard for details.
                          <div className="mt-2">
                            <Button 
                              variant="outline" 
                              size="sm" 
                              onClick={handleCheckStatus}
                              disabled={checkStatusMutation.isPending}
                              data-testid="button-check-status"
                            >
                              {checkStatusMutation.isPending ? (
                                <>
                                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                                  Checking...
                                </>
                              ) : (
                                <>
                                  Check Status
                                  <RefreshCw className="ml-2 h-4 w-4" />
                                </>
                              )}
                            </Button>
                          </div>
                        </AlertDescription>
                      </Alert>
                    )}
                  </div>
                </div>
              );
            })}
          </div>
        </CardContent>
      </Card>

      {/* Additional Information */}
      <Card data-testid="info-card">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FileCheck className="h-5 w-5" />
            What You'll Need
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2">
            <div>
              <h4 className="font-medium mb-2">Personal Information</h4>
              <ul className="text-sm text-muted-foreground space-y-1">
                <li>• Legal name and date of birth</li>
                <li>• Home address</li>
                <li>• Phone number</li>
                <li>• Tax identification number</li>
              </ul>
            </div>
            <div>
              <h4 className="font-medium mb-2">Banking Information</h4>
              <ul className="text-sm text-muted-foreground space-y-1">
                <li>• Bank account details</li>
                <li>• Routing number</li>
                <li>• Government-issued ID</li>
                <li>• Business documents (if applicable)</li>
              </ul>
            </div>
          </div>
          <Separator className="my-4" />
          <div className="flex items-center space-x-2 text-sm text-muted-foreground">
            <FileCheck className="h-4 w-4" />
            <span>
              All information is securely processed by Stripe and encrypted for your protection
            </span>
          </div>
        </CardContent>
      </Card>

      {/* Support Information */}
      <div className="text-center text-sm text-muted-foreground">
        <p>
          Need help with onboarding?{' '}
          <Link href="/support" className="text-primary hover:underline">
            Contact Support
          </Link>
        </p>
      </div>
    </div>
  );
}