import { Outlet } from "wouter";
import { Link, useLocation } from "wouter";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import { Separator } from "@/components/ui/separator";
import { 
  LayoutDashboard, 
  Upload, 
  FileText, 
  DollarSign, 
  Settings,
  User,
  ArrowLeft
} from "lucide-react";

interface CreatorNavItem {
  title: string;
  href: string;
  icon: React.ComponentType<{ className?: string }>;
  description?: string;
}

const creatorNavItems: CreatorNavItem[] = [
  {
    title: "Dashboard",
    href: "/creator/dashboard", 
    icon: LayoutDashboard,
    description: "Overview and stats"
  },
  {
    title: "Upload Asset",
    href: "/creator/upload",
    icon: Upload,
    description: "Add new assets"
  },
  {
    title: "Manage Assets",
    href: "/creator/assets",
    icon: FileText,
    description: "Edit and organize"
  },
  {
    title: "Earnings",
    href: "/creator/earnings",
    icon: DollarSign,
    description: "Revenue and payouts"
  },
];

interface CreatorLayoutProps {
  children: React.ReactNode;
  title?: string;
}

export default function CreatorLayout({ children, title }: CreatorLayoutProps) {
  const [location] = useLocation();

  const isActivePath = (path: string) => {
    if (path === "/creator/dashboard") {
      return location === path;
    }
    return location.startsWith(path);
  };

  return (
    <div className="min-h-screen bg-background" data-testid="creator-layout">
      <div className="border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <Link href="/dashboard">
                <Button variant="ghost" size="sm" data-testid="button-back-to-main">
                  <ArrowLeft className="h-4 w-4 mr-2" />
                  Back to Main Dashboard
                </Button>
              </Link>
              <Separator orientation="vertical" className="h-6" />
              <div>
                <h1 className="text-2xl font-bold tracking-tight" data-testid="text-creator-title">
                  {title || "Creator Marketplace"}
                </h1>
                <p className="text-sm text-muted-foreground">
                  Manage your creative business
                </p>
              </div>
            </div>
            <div className="flex items-center space-x-2">
              <Link href="/profile">
                <Button variant="ghost" size="sm" data-testid="button-profile">
                  <User className="h-4 w-4 mr-2" />
                  Profile
                </Button>
              </Link>
              <Link href="/creator/settings">
                <Button variant="ghost" size="sm" data-testid="button-creator-settings">
                  <Settings className="h-4 w-4 mr-2" />
                  Settings
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-6 py-8">
        <div className="grid gap-8 lg:grid-cols-4">
          {/* Navigation Sidebar */}
          <div className="lg:col-span-1">
            <nav className="space-y-2" data-testid="creator-navigation">
              {creatorNavItems.map((item) => {
                const Icon = item.icon;
                const isActive = isActivePath(item.href);
                
                return (
                  <Link key={item.href} href={item.href}>
                    <div
                      className={cn(
                        "flex items-center space-x-3 rounded-lg px-3 py-3 text-sm font-medium transition-colors hover:bg-accent hover:text-accent-foreground cursor-pointer",
                        isActive 
                          ? "bg-accent text-accent-foreground" 
                          : "text-muted-foreground hover:text-foreground"
                      )}
                      data-testid={`nav-${item.href.split('/').pop()}`}
                    >
                      <Icon className="h-4 w-4" />
                      <div className="flex-1">
                        <div className="font-medium">{item.title}</div>
                        {item.description && (
                          <div className="text-xs text-muted-foreground mt-0.5">
                            {item.description}
                          </div>
                        )}
                      </div>
                    </div>
                  </Link>
                );
              })}
            </nav>

            {/* Creator Status Card */}
            <div className="mt-8 p-4 rounded-lg bg-muted/50 border" data-testid="creator-status-card">
              <div className="flex items-center space-x-2 mb-2">
                <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                <span className="text-sm font-medium">Active Creator</span>
              </div>
              <p className="text-xs text-muted-foreground">
                Your account is set up for selling assets on the marketplace
              </p>
            </div>
          </div>

          {/* Main Content */}
          <div className="lg:col-span-3" data-testid="creator-main-content">
            {children}
          </div>
        </div>
      </div>
    </div>
  );
}