import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useAuth } from "@/contexts/AuthContext";
import { auth } from "@/lib/firebase";
import { Link } from "wouter";
import { 
  DollarSign,
  TrendingUp,
  Download,
  Calendar,
  CreditCard,
  FileText,
  AlertCircle,
  Info,
  ExternalLink,
  ArrowUpRight,
  ArrowDownRight,
  Clock,
  CheckCircle,
  Banknote
} from "lucide-react";
import { CreatorEarning } from "@shared/schema";

interface EarningsStats {
  totalEarningsGross: number;
  totalEarningsNet: number;
  totalPlatformFees: number;
  pendingPayouts: number;
  paidOutEarnings: number;
  thisMonthEarnings: number;
  lastMonthEarnings: number;
  averageOrderValue: number;
  totalSales: number;
}

interface EarningsData {
  stats: EarningsStats;
  earnings: CreatorEarning[];
  recentEarnings: CreatorEarning[];
}

interface StripeAccountInfo {
  accountId: string;
  chargesEnabled: boolean;
  payoutsEnabled: boolean;
  currentlyDue: string[];
  payoutSchedule: {
    interval: string;
    delayDays: number;
  };
}

type DateRange = '7d' | '30d' | '90d' | '1y' | 'all';

async function getAuthHeaders(): Promise<Record<string, string>> {
  const headers: Record<string, string> = {};
  
  // Add development bypass header in development mode
  if (import.meta.env.DEV) {
    headers["X-Dev-Bypass"] = "development";
  }
  
  // Add Firebase auth token if user is authenticated
  if (auth.currentUser) {
    try {
      const token = await auth.currentUser.getIdToken();
      headers["Authorization"] = `Bearer ${token}`;
    } catch (error) {
      console.warn("Failed to get Firebase ID token:", error);
      // Continue without auth token rather than failing the request
    }
  }
  
  return headers;
}

export default function CreatorEarnings() {
  const { currentUser } = useAuth();
  const [dateRange, setDateRange] = useState<DateRange>('30d');
  const [page, setPage] = useState(1);

  // Query earnings data
  const { 
    data: earningsData, 
    isLoading, 
    error 
  } = useQuery<EarningsData>({
    queryKey: ['/api/creators/earnings', dateRange, page],
    queryFn: async () => {
      const params = new URLSearchParams();
      params.append('dateRange', dateRange);
      params.append('page', page.toString());
      
      const response = await fetch(`/api/creators/earnings?${params.toString()}`, {
        headers: await getAuthHeaders(),
        credentials: 'include',
      });
      
      if (!response.ok) {
        throw new Error(`${response.status}: ${response.statusText}`);
      }
      
      return response.json();
    },
    enabled: !!currentUser,
  });

  // Query Stripe account info
  const { 
    data: stripeAccount
  } = useQuery<StripeAccountInfo>({
    queryKey: ['/api/creators/connect/status'],
    enabled: !!currentUser,
  });

  const formatCurrency = (cents: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
    }).format(cents / 100);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const getPayoutStatusBadge = (status: string) => {
    switch (status) {
      case 'pending':
        return (
          <Badge variant="secondary" className="bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-100">
            <Clock className="mr-1 h-3 w-3" />
            Pending
          </Badge>
        );
      case 'processing':
        return (
          <Badge variant="secondary" className="bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-100">
            <CreditCard className="mr-1 h-3 w-3" />
            Processing
          </Badge>
        );
      case 'paid':
        return (
          <Badge variant="secondary" className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-100">
            <CheckCircle className="mr-1 h-3 w-3" />
            Paid
          </Badge>
        );
      case 'failed':
        return (
          <Badge variant="destructive">
            <AlertCircle className="mr-1 h-3 w-3" />
            Failed
          </Badge>
        );
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  const getEarningTypeIcon = (type: string) => {
    switch (type) {
      case 'sale':
        return <Download className="h-4 w-4 text-green-600" />;
      case 'commission':
        return <DollarSign className="h-4 w-4 text-blue-600" />;
      case 'bonus':
        return <TrendingUp className="h-4 w-4 text-purple-600" />;
      case 'payout':
        return <Banknote className="h-4 w-4 text-gray-600" />;
      default:
        return <DollarSign className="h-4 w-4" />;
    }
  };

  if (isLoading) {
    return (
      <div className="space-y-8" data-testid="earnings-loading">
        <div className="flex items-center justify-between">
          <Skeleton className="h-8 w-48" />
          <Skeleton className="h-10 w-32" />
        </div>
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {[...Array(4)].map((_, i) => (
            <Card key={i}>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <Skeleton className="h-4 w-20" />
                <Skeleton className="h-4 w-4" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-8 w-16 mb-2" />
                <Skeleton className="h-3 w-24" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="text-center py-12" data-testid="earnings-error">
        <AlertCircle className="h-16 w-16 text-destructive mx-auto mb-4" />
        <h2 className="text-2xl font-semibold mb-2">Unable to load earnings</h2>
        <p className="text-muted-foreground mb-4">
          There was an error loading your earnings data. Please try again.
        </p>
        <Button onClick={() => window.location.reload()} data-testid="button-retry">
          Try Again
        </Button>
      </div>
    );
  }

  const stats = earningsData?.stats || {
    totalEarningsGross: 0,
    totalEarningsNet: 0,
    totalPlatformFees: 0,
    pendingPayouts: 0,
    paidOutEarnings: 0,
    thisMonthEarnings: 0,
    lastMonthEarnings: 0,
    averageOrderValue: 0,
    totalSales: 0,
  };

  const earnings = earningsData?.earnings || [];
  const recentEarnings = earningsData?.recentEarnings || [];

  const monthlyChange = stats.lastMonthEarnings > 0 
    ? ((stats.thisMonthEarnings - stats.lastMonthEarnings) / stats.lastMonthEarnings) * 100
    : 0;

  return (
    <div className="space-y-8" data-testid="creator-earnings">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight" data-testid="text-earnings-title">
            Earnings Dashboard
          </h1>
          <p className="text-muted-foreground">
            Track your revenue, fees, and payouts
          </p>
        </div>
        <div className="flex items-center space-x-2">
          <Select value={dateRange} onValueChange={(value: DateRange) => setDateRange(value)}>
            <SelectTrigger className="w-[140px]" data-testid="select-date-range">
              <Calendar className="mr-2 h-4 w-4" />
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="7d">Last 7 days</SelectItem>
              <SelectItem value="30d">Last 30 days</SelectItem>
              <SelectItem value="90d">Last 3 months</SelectItem>
              <SelectItem value="1y">Last year</SelectItem>
              <SelectItem value="all">All time</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Earnings Stats Cards */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4" data-testid="earnings-stats">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Earnings (Net)</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-primary" data-testid="text-total-earnings">
              {formatCurrency(stats.totalEarningsNet)}
            </div>
            <p className="text-xs text-muted-foreground">
              After platform fees (30%)
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">This Month</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-monthly-earnings">
              {formatCurrency(stats.thisMonthEarnings)}
            </div>
            <p className={`text-xs flex items-center ${monthlyChange >= 0 ? 'text-green-600' : 'text-red-600'}`}>
              {monthlyChange >= 0 ? (
                <ArrowUpRight className="mr-1 h-3 w-3" />
              ) : (
                <ArrowDownRight className="mr-1 h-3 w-3" />
              )}
              {Math.abs(monthlyChange).toFixed(1)}% vs last month
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Pending Payouts</CardTitle>
            <Clock className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-yellow-600" data-testid="text-pending-payouts">
              {formatCurrency(stats.pendingPayouts)}
            </div>
            <p className="text-xs text-muted-foreground">
              Awaiting next payout cycle
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Sales</CardTitle>
            <Download className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-sales">
              {stats.totalSales}
            </div>
            <p className="text-xs text-muted-foreground">
              Avg: {formatCurrency(stats.averageOrderValue)} per sale
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Stripe Account Status */}
      {stripeAccount && (
        <Card data-testid="stripe-account-card">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CreditCard className="h-5 w-5" />
              Payout Account Status
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <div className="space-y-2">
                <div className="flex items-center space-x-2">
                  <span className="text-sm font-medium">Payouts:</span>
                  {stripeAccount.payoutsEnabled ? (
                    <Badge variant="secondary" className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-100">
                      <CheckCircle className="mr-1 h-3 w-3" />
                      Enabled
                    </Badge>
                  ) : (
                    <Badge variant="destructive">
                      <AlertCircle className="mr-1 h-3 w-3" />
                      Disabled
                    </Badge>
                  )}
                </div>
                <div className="text-sm text-muted-foreground">
                  Payout Schedule: {stripeAccount.payoutSchedule?.interval} 
                  {stripeAccount.payoutSchedule?.delayDays > 0 && 
                    ` (${stripeAccount.payoutSchedule.delayDays} day delay)`
                  }
                </div>
                {stripeAccount.currentlyDue.length > 0 && (
                  <Alert>
                    <Info className="h-4 w-4" />
                    <AlertDescription>
                      Action required: Please complete your Stripe account setup to enable payouts.
                    </AlertDescription>
                  </Alert>
                )}
              </div>
              <Button variant="outline" asChild data-testid="button-manage-payouts">
                <Link href="/creator/onboarding">
                  Manage Payouts
                  <ExternalLink className="ml-2 h-4 w-4" />
                </Link>
              </Button>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Fee Breakdown */}
      <Card data-testid="fee-breakdown-card">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Info className="h-5 w-5" />
            Fee Breakdown
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-3">
            <div className="text-center p-4 border rounded-lg">
              <div className="text-2xl font-bold text-green-600">
                {formatCurrency(stats.totalEarningsGross)}
              </div>
              <div className="text-sm text-muted-foreground">Gross Earnings</div>
              <div className="text-xs text-muted-foreground mt-1">
                Total revenue from sales
              </div>
            </div>
            <div className="text-center p-4 border rounded-lg">
              <div className="text-2xl font-bold text-red-600">
                -{formatCurrency(stats.totalPlatformFees)}
              </div>
              <div className="text-sm text-muted-foreground">Platform Fees (30%)</div>
              <div className="text-xs text-muted-foreground mt-1">
                Marketplace commission
              </div>
            </div>
            <div className="text-center p-4 border rounded-lg bg-primary/5">
              <div className="text-2xl font-bold text-primary">
                {formatCurrency(stats.totalEarningsNet)}
              </div>
              <div className="text-sm text-muted-foreground">Your Earnings</div>
              <div className="text-xs text-muted-foreground mt-1">
                Amount paid to you
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Recent Earnings */}
      <Card data-testid="recent-earnings-card">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5" />
            Recent Earnings
          </CardTitle>
        </CardHeader>
        <CardContent>
          {recentEarnings.length === 0 ? (
            <div className="text-center py-8" data-testid="no-earnings">
              <DollarSign className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">No earnings yet</p>
              <p className="text-sm text-muted-foreground">
                Start selling your assets to see earnings here
              </p>
            </div>
          ) : (
            <div className="space-y-4" data-testid="earnings-list">
              {recentEarnings.map((earning) => (
                <div 
                  key={earning.id} 
                  className="flex items-center justify-between py-3 border-b last:border-0"
                  data-testid={`earning-${earning.id}`}
                >
                  <div className="flex items-center space-x-3">
                    <div className="flex-shrink-0">
                      {getEarningTypeIcon(earning.earningType)}
                    </div>
                    <div>
                      <div className="flex items-center space-x-2">
                        <span className="font-medium capitalize">
                          {earning.earningType}
                        </span>
                        {earning.creatorAssetId && (
                          <Link href={`/creator/assets?id=${earning.creatorAssetId}`}>
                            <Badge variant="outline" className="text-xs cursor-pointer hover:bg-accent">
                              View Asset
                            </Badge>
                          </Link>
                        )}
                      </div>
                      <div className="text-sm text-muted-foreground">
                        {formatDate(earning.occurredAt)}
                      </div>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="font-medium text-green-600">
                      +{formatCurrency(earning.netAmountCents)}
                    </div>
                    <div className="flex items-center space-x-2">
                      <span className="text-xs text-muted-foreground">
                        -{formatCurrency(earning.platformFeeCents)} fee
                      </span>
                      {getPayoutStatusBadge(earning.payoutStatus)}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}

          {recentEarnings.length > 0 && (
            <div className="mt-6 text-center">
              <Button variant="outline" data-testid="button-view-all-earnings">
                View All Earnings History
              </Button>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}