import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Button } from "@/components/ui/button";
import { useAuth } from "@/contexts/AuthContext";
import { TrendingUp, Upload, FileText, DollarSign, Eye, Download, Clock, CheckCircle, XCircle } from "lucide-react";
import { Link } from "wouter";

interface CreatorStats {
  totalAssets: number;
  totalEarnings: number;
  totalSales: number;
  pendingAssets: number;
  approvedAssets: number;
  rejectedAssets: number;
  monthlyEarnings: number;
  pendingPayouts: number;
}

interface RecentActivity {
  id: string;
  type: 'sale' | 'approval' | 'rejection' | 'upload';
  message: string;
  timestamp: string;
  amount?: number;
  assetTitle?: string;
}

interface CreatorProfile {
  id: string;
  onboardingStatus: string;
  stripeConnectAccountId?: string;
  payoutEnabled: boolean;
  totalEarnings: number;
  stats?: CreatorStats;
  recentActivity?: RecentActivity[];
}

export default function CreatorDashboard() {
  const { currentUser } = useAuth();

  const { data: creatorProfile, isLoading, error } = useQuery<CreatorProfile>({
    queryKey: ['/api/creators/profile'],
    enabled: !!currentUser,
  });

  const { data: recentActivity, isLoading: activityLoading } = useQuery<RecentActivity[]>({
    queryKey: ['/api/creators/activity'],
    enabled: !!currentUser,
  });

  if (isLoading) {
    return (
      <div className="space-y-8" data-testid="creator-dashboard-loading">
        <div className="flex items-center justify-between">
          <Skeleton className="h-8 w-48" />
          <Skeleton className="h-10 w-32" />
        </div>
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {[...Array(4)].map((_, i) => (
            <Card key={i}>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <Skeleton className="h-4 w-20" />
                <Skeleton className="h-4 w-4" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-8 w-16 mb-2" />
                <Skeleton className="h-3 w-24" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="text-center py-12" data-testid="creator-dashboard-error">
        <XCircle className="h-16 w-16 text-destructive mx-auto mb-4" />
        <h2 className="text-2xl font-semibold mb-2">Unable to load dashboard</h2>
        <p className="text-muted-foreground mb-4">
          There was an error loading your creator dashboard. Please try again.
        </p>
        <Button onClick={() => window.location.reload()} data-testid="button-retry">
          Try Again
        </Button>
      </div>
    );
  }

  // Check if user needs to complete onboarding
  if (!creatorProfile || creatorProfile.onboardingStatus !== 'completed') {
    return (
      <div className="text-center py-12" data-testid="creator-onboarding-required">
        <Upload className="h-16 w-16 text-primary mx-auto mb-4" />
        <h2 className="text-2xl font-semibold mb-2">Welcome to Creator Marketplace!</h2>
        <p className="text-muted-foreground mb-6 max-w-md mx-auto">
          Complete your creator onboarding to start selling your assets and earning money.
        </p>
        <Link href="/creator/onboarding">
          <Button size="lg" data-testid="button-start-onboarding">
            Complete Onboarding
          </Button>
        </Link>
      </div>
    );
  }

  const stats = creatorProfile.stats || {
    totalAssets: 0,
    totalEarnings: 0,
    totalSales: 0,
    pendingAssets: 0,
    approvedAssets: 0,
    rejectedAssets: 0,
    monthlyEarnings: 0,
    pendingPayouts: 0
  };

  const getActivityIcon = (type: RecentActivity['type']) => {
    switch (type) {
      case 'sale':
        return <DollarSign className="h-4 w-4 text-green-600" />;
      case 'approval':
        return <CheckCircle className="h-4 w-4 text-green-600" />;
      case 'rejection':
        return <XCircle className="h-4 w-4 text-red-600" />;
      case 'upload':
        return <Upload className="h-4 w-4 text-blue-600" />;
      default:
        return <Clock className="h-4 w-4 text-gray-600" />;
    }
  };

  const formatCurrency = (cents: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
    }).format(cents / 100);
  };

  return (
    <div className="space-y-8" data-testid="creator-dashboard">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight" data-testid="text-dashboard-title">
            Creator Dashboard
          </h1>
          <p className="text-muted-foreground">
            Manage your assets, track earnings, and grow your business
          </p>
        </div>
        <Link href="/creator/upload">
          <Button data-testid="button-upload-asset">
            <Upload className="mr-2 h-4 w-4" />
            Upload Asset
          </Button>
        </Link>
      </div>

      {/* Quick Stats Cards */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4" data-testid="stats-cards">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Earnings</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-earnings">
              {formatCurrency(stats.totalEarnings)}
            </div>
            <p className="text-xs text-muted-foreground">
              +{formatCurrency(stats.monthlyEarnings)} this month
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Assets</CardTitle>
            <FileText className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-assets">
              {stats.totalAssets}
            </div>
            <p className="text-xs text-muted-foreground">
              {stats.approvedAssets} approved, {stats.pendingAssets} pending
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Sales</CardTitle>
            <Download className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-total-sales">
              {stats.totalSales}
            </div>
            <p className="text-xs text-muted-foreground">
              Across all assets
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Pending Payouts</CardTitle>
            <Clock className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="text-pending-payouts">
              {formatCurrency(stats.pendingPayouts)}
            </div>
            <p className="text-xs text-muted-foreground">
              Awaiting payment
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Asset Status Overview */}
      <Card data-testid="asset-status-card">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Eye className="h-5 w-5" />
            Asset Status Overview
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-4">
            <Link href="/creator/assets?status=pending">
              <div className="flex items-center space-x-2 cursor-pointer hover:opacity-80" data-testid="link-pending-assets">
                <Badge variant="secondary" className="bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-100">
                  {stats.pendingAssets} Pending Review
                </Badge>
              </div>
            </Link>
            <Link href="/creator/assets?status=approved">
              <div className="flex items-center space-x-2 cursor-pointer hover:opacity-80" data-testid="link-approved-assets">
                <Badge variant="secondary" className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-100">
                  {stats.approvedAssets} Approved
                </Badge>
              </div>
            </Link>
            {stats.rejectedAssets > 0 && (
              <Link href="/creator/assets?status=rejected">
                <div className="flex items-center space-x-2 cursor-pointer hover:opacity-80" data-testid="link-rejected-assets">
                  <Badge variant="secondary" className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-100">
                    {stats.rejectedAssets} Rejected
                  </Badge>
                </div>
              </Link>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Recent Activity Feed */}
      <Card data-testid="recent-activity-card">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5" />
            Recent Activity
          </CardTitle>
        </CardHeader>
        <CardContent>
          {activityLoading ? (
            <div className="space-y-3">
              {[...Array(5)].map((_, i) => (
                <div key={i} className="flex items-center space-x-3">
                  <Skeleton className="h-8 w-8 rounded-full" />
                  <div className="flex-1">
                    <Skeleton className="h-4 w-full mb-1" />
                    <Skeleton className="h-3 w-24" />
                  </div>
                </div>
              ))}
            </div>
          ) : recentActivity && recentActivity.length > 0 ? (
            <div className="space-y-4" data-testid="activity-list">
              {recentActivity.slice(0, 5).map((activity) => (
                <div key={activity.id} className="flex items-start space-x-3" data-testid={`activity-${activity.id}`}>
                  <div className="flex-shrink-0 mt-1">
                    {getActivityIcon(activity.type)}
                  </div>
                  <div className="flex-1 min-w-0">
                    <p className="text-sm font-medium text-gray-900 dark:text-gray-100">
                      {activity.message}
                    </p>
                    <p className="text-xs text-muted-foreground">
                      {new Date(activity.timestamp).toLocaleDateString('en-US', {
                        month: 'short',
                        day: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                      })}
                    </p>
                  </div>
                  {activity.amount && (
                    <div className="text-sm font-medium text-green-600">
                      {formatCurrency(activity.amount)}
                    </div>
                  )}
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-8" data-testid="no-activity">
              <TrendingUp className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">No recent activity</p>
              <p className="text-sm text-muted-foreground">
                Upload your first asset to get started!
              </p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Quick Actions */}
      <div className="grid gap-4 md:grid-cols-3" data-testid="quick-actions">
        <Link href="/creator/upload">
          <Card className="cursor-pointer hover:shadow-md transition-shadow" data-testid="card-upload-asset">
            <CardContent className="pt-6">
              <div className="flex items-center space-x-4">
                <div className="p-2 bg-primary/10 rounded-lg">
                  <Upload className="h-6 w-6 text-primary" />
                </div>
                <div>
                  <h3 className="font-semibold">Upload New Asset</h3>
                  <p className="text-sm text-muted-foreground">
                    Add graphics, icons, or templates
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </Link>

        <Link href="/creator/assets">
          <Card className="cursor-pointer hover:shadow-md transition-shadow" data-testid="card-manage-assets">
            <CardContent className="pt-6">
              <div className="flex items-center space-x-4">
                <div className="p-2 bg-blue-100 dark:bg-blue-900 rounded-lg">
                  <FileText className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                </div>
                <div>
                  <h3 className="font-semibold">Manage Assets</h3>
                  <p className="text-sm text-muted-foreground">
                    Edit, organize, and track sales
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </Link>

        <Link href="/creator/earnings">
          <Card className="cursor-pointer hover:shadow-md transition-shadow" data-testid="card-view-earnings">
            <CardContent className="pt-6">
              <div className="flex items-center space-x-4">
                <div className="p-2 bg-green-100 dark:bg-green-900 rounded-lg">
                  <DollarSign className="h-6 w-6 text-green-600 dark:text-green-400" />
                </div>
                <div>
                  <h3 className="font-semibold">View Earnings</h3>
                  <p className="text-sm text-muted-foreground">
                    Track revenue and payouts
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </Link>
      </div>
    </div>
  );
}