import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from "@/components/ui/alert-dialog";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Skeleton } from "@/components/ui/skeleton";
import { useAuth } from "@/contexts/AuthContext";
import { auth } from "@/lib/firebase";
import { toast } from "sonner";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Link } from "wouter";
import { 
  Search,
  Filter,
  FileImage,
  FileText,
  File,
  Download,
  DollarSign,
  Edit,
  Trash2,
  Eye,
  Calendar,
  TrendingUp,
  AlertCircle,
  CheckCircle,
  Clock,
  XCircle,
  Upload,
  MoreHorizontal,
  ArrowUpDown
} from "lucide-react";
import { CreatorAsset, CREATOR_APPROVAL_STATUS_VALUES, CREATOR_MIN_PRICE_CENTS, CREATOR_MAX_PRICE_CENTS } from "@shared/schema";

type AssetStatus = 'all' | 'pending' | 'approved' | 'rejected';
type SortBy = 'created_at' | 'title' | 'price' | 'sales_count' | 'total_revenue';
type SortOrder = 'asc' | 'desc';

interface AssetFilters {
  status: AssetStatus;
  search: string;
  category: string;
  sortBy: SortBy;
  sortOrder: SortOrder;
}

interface CreatorAssetWithAsset extends CreatorAsset {
  asset: {
    fileName: string;
    fileUrl: string;
    previewUrl?: string;
    fileSize: number;
    mimeType: string;
    assetType: string;
    downloadCount: number;
  };
}

async function getAuthHeaders(): Promise<Record<string, string>> {
  const headers: Record<string, string> = {};
  
  // Add development bypass header in development mode
  if (import.meta.env.DEV) {
    headers["X-Dev-Bypass"] = "development";
  }
  
  // Add Firebase auth token if user is authenticated
  if (auth.currentUser) {
    try {
      const token = await auth.currentUser.getIdToken();
      headers["Authorization"] = `Bearer ${token}`;
    } catch (error) {
      console.warn("Failed to get Firebase ID token:", error);
      // Continue without auth token rather than failing the request
    }
  }
  
  return headers;
}

export default function CreatorAssets() {
  const { currentUser } = useAuth();
  
  const [filters, setFilters] = useState<AssetFilters>({
    status: 'all',
    search: '',
    category: '',
    sortBy: 'created_at',
    sortOrder: 'desc',
  });

  const [editingAsset, setEditingAsset] = useState<CreatorAssetWithAsset | null>(null);
  const [page, setPage] = useState(1);
  const limit = 12;

  // Query creator assets with filters
  const { 
    data: assetsData, 
    isLoading, 
    error,
    refetch 
  } = useQuery<{ assets: CreatorAssetWithAsset[]; total: number; page: number; pages: number }>({
    queryKey: ['/api/creator-assets', filters, page, limit],
    queryFn: async () => {
      const params = new URLSearchParams();
      
      // Add pagination
      params.append('page', page.toString());
      params.append('limit', limit.toString());
      
      // Add filters
      if (filters.status !== 'all') {
        params.append('status', filters.status);
      }
      if (filters.search) {
        params.append('search', filters.search);
      }
      if (filters.category) {
        params.append('category', filters.category);
      }
      
      // Add sorting
      params.append('sortBy', filters.sortBy);
      params.append('sortOrder', filters.sortOrder);
      
      const response = await fetch(`/api/creator-assets?${params.toString()}`, {
        headers: await getAuthHeaders(),
        credentials: 'include',
      });
      
      if (!response.ok) {
        throw new Error(`${response.status}: ${response.statusText}`);
      }
      
      return response.json();
    },
    enabled: !!currentUser,
  });

  // Edit asset mutation
  const editAssetMutation = useMutation({
    mutationFn: async ({ id, data }: { id: string; data: Partial<CreatorAsset> }) => 
      apiRequest('PATCH', `/api/creator-assets/${id}`, data),
    onSuccess: () => {
      toast.success('Asset updated successfully');
      setEditingAsset(null);
      queryClient.invalidateQueries({ queryKey: ['/api/creator-assets'] });
    },
    onError: (error: any) => {
      toast.error(error.message || 'Failed to update asset');
    },
  });

  // Delete asset mutation
  const deleteAssetMutation = useMutation({
    mutationFn: (id: string) => apiRequest('DELETE', `/api/creator-assets/${id}`),
    onSuccess: () => {
      toast.success('Asset deleted successfully');
      queryClient.invalidateQueries({ queryKey: ['/api/creator-assets'] });
    },
    onError: (error: any) => {
      toast.error(error.message || 'Failed to delete asset');
    },
  });

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'pending':
        return (
          <Badge variant="secondary" className="bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-100">
            <Clock className="mr-1 h-3 w-3" />
            Pending
          </Badge>
        );
      case 'approved':
        return (
          <Badge variant="secondary" className="bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-100">
            <CheckCircle className="mr-1 h-3 w-3" />
            Approved
          </Badge>
        );
      case 'rejected':
        return (
          <Badge variant="secondary" className="bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-100">
            <XCircle className="mr-1 h-3 w-3" />
            Rejected
          </Badge>
        );
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  const getFileIcon = (mimeType: string, assetType: string) => {
    if (mimeType.startsWith('image/')) return <FileImage className="h-4 w-4" />;
    if (mimeType === 'application/pdf') return <FileText className="h-4 w-4" />;
    return <File className="h-4 w-4" />;
  };

  const formatCurrency = (cents: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
    }).format(cents / 100);
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const canEditAsset = (asset: CreatorAssetWithAsset) => {
    return asset.approvalStatus === 'pending' || asset.approvalStatus === 'rejected';
  };

  const canDeleteAsset = (asset: CreatorAssetWithAsset) => {
    return asset.approvalStatus !== 'approved' || asset.salesCount === 0;
  };

  const handleStatusFilter = (status: AssetStatus) => {
    setFilters(prev => ({ ...prev, status }));
    setPage(1);
  };

  const handleSearch = (search: string) => {
    setFilters(prev => ({ ...prev, search }));
    setPage(1);
  };

  const handleSort = (sortBy: SortBy) => {
    setFilters(prev => ({
      ...prev,
      sortBy,
      sortOrder: prev.sortBy === sortBy && prev.sortOrder === 'desc' ? 'asc' : 'desc'
    }));
  };

  if (isLoading) {
    return (
      <div className="space-y-8" data-testid="assets-loading">
        <div className="flex items-center justify-between">
          <Skeleton className="h-8 w-48" />
          <Skeleton className="h-10 w-32" />
        </div>
        <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
          {[...Array(6)].map((_, i) => (
            <Card key={i}>
              <CardHeader>
                <Skeleton className="h-4 w-full" />
                <Skeleton className="h-6 w-20" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-32 w-full mb-4" />
                <Skeleton className="h-4 w-full mb-2" />
                <Skeleton className="h-4 w-3/4" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="text-center py-12" data-testid="assets-error">
        <AlertCircle className="h-16 w-16 text-destructive mx-auto mb-4" />
        <h2 className="text-2xl font-semibold mb-2">Unable to load assets</h2>
        <p className="text-muted-foreground mb-4">
          There was an error loading your assets. Please try again.
        </p>
        <Button onClick={() => refetch()} data-testid="button-retry">
          Try Again
        </Button>
      </div>
    );
  }

  const assets = assetsData?.assets || [];
  const totalAssets = assetsData?.total || 0;

  return (
    <div className="space-y-8" data-testid="creator-assets">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight" data-testid="text-assets-title">
            Manage Assets
          </h1>
          <p className="text-muted-foreground">
            {totalAssets} {totalAssets === 1 ? 'asset' : 'assets'} in your portfolio
          </p>
        </div>
        <Link href="/creator/upload">
          <Button data-testid="button-upload-new">
            <Upload className="mr-2 h-4 w-4" />
            Upload New Asset
          </Button>
        </Link>
      </div>

      {/* Filters and Search */}
      <Card data-testid="filters-card">
        <CardContent className="pt-6">
          <div className="flex flex-col space-y-4 lg:flex-row lg:items-center lg:space-y-0 lg:space-x-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Search assets by title, tags, or category..."
                  value={filters.search}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-10"
                  data-testid="input-search"
                />
              </div>
            </div>
            <div className="flex items-center space-x-4">
              <Select value={filters.sortBy} onValueChange={(value: SortBy) => handleSort(value)}>
                <SelectTrigger className="w-[180px]" data-testid="select-sort">
                  <SelectValue placeholder="Sort by" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="created_at">Date Created</SelectItem>
                  <SelectItem value="title">Title</SelectItem>
                  <SelectItem value="price">Price</SelectItem>
                  <SelectItem value="sales_count">Sales</SelectItem>
                  <SelectItem value="total_revenue">Revenue</SelectItem>
                </SelectContent>
              </Select>
              <Button
                variant="outline"
                size="icon"
                onClick={() => setFilters(prev => ({ ...prev, sortOrder: prev.sortOrder === 'asc' ? 'desc' : 'asc' }))}
                data-testid="button-sort-order"
              >
                <ArrowUpDown className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Status Tabs */}
      <Tabs value={filters.status} onValueChange={(value: AssetStatus) => handleStatusFilter(value)}>
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="all" data-testid="tab-all">All Assets</TabsTrigger>
          <TabsTrigger value="pending" data-testid="tab-pending">Pending</TabsTrigger>
          <TabsTrigger value="approved" data-testid="tab-approved">Approved</TabsTrigger>
          <TabsTrigger value="rejected" data-testid="tab-rejected">Rejected</TabsTrigger>
        </TabsList>

        <TabsContent value={filters.status} className="mt-6">
          {assets.length === 0 ? (
            <div className="text-center py-12" data-testid="no-assets">
              <Upload className="h-16 w-16 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">No assets found</h3>
              <p className="text-muted-foreground mb-4">
                {filters.search || filters.status !== 'all'
                  ? "Try adjusting your filters or search terms"
                  : "Upload your first asset to get started"
                }
              </p>
              <Link href="/creator/upload">
                <Button data-testid="button-upload-first">
                  Upload Your First Asset
                </Button>
              </Link>
            </div>
          ) : (
            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3" data-testid="assets-grid">
              {assets.map((asset) => (
                <Card key={asset.id} className="group hover:shadow-md transition-shadow">
                  <CardHeader className="pb-3">
                    <div className="flex items-center justify-between">
                      {getStatusBadge(asset.approvalStatus)}
                      <div className="flex items-center space-x-2">
                        {getFileIcon(asset.asset.mimeType, asset.asset.assetType)}
                        <span className="text-xs text-muted-foreground">
                          {formatFileSize(asset.asset.fileSize)}
                        </span>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent className="pt-0">
                    {/* Asset Preview */}
                    <div className="aspect-video bg-muted rounded-md mb-4 overflow-hidden">
                      {asset.asset.previewUrl || asset.asset.mimeType.startsWith('image/') ? (
                        <img
                          src={asset.asset.previewUrl || asset.asset.fileUrl}
                          alt={asset.title}
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <div className="w-full h-full flex items-center justify-center">
                          {getFileIcon(asset.asset.mimeType, asset.asset.assetType)}
                          <span className="ml-2 text-sm text-muted-foreground">
                            {asset.asset.assetType.toUpperCase()}
                          </span>
                        </div>
                      )}
                    </div>

                    {/* Asset Info */}
                    <div className="space-y-3">
                      <div>
                        <h3 className="font-semibold line-clamp-1" data-testid={`asset-title-${asset.id}`}>
                          {asset.title}
                        </h3>
                        {asset.description && (
                          <p className="text-sm text-muted-foreground line-clamp-2">
                            {asset.description}
                          </p>
                        )}
                      </div>

                      <div className="flex items-center justify-between">
                        <div className="text-lg font-bold text-primary">
                          {formatCurrency(asset.price)}
                        </div>
                        <div className="text-sm text-muted-foreground">
                          {asset.salesCount || 0} sales
                        </div>
                      </div>

                      {asset.totalRevenue > 0 && (
                        <div className="flex items-center justify-between text-sm">
                          <span className="text-muted-foreground">Revenue:</span>
                          <span className="font-medium text-green-600">
                            {formatCurrency(asset.totalRevenue)}
                          </span>
                        </div>
                      )}

                      {asset.rejectionReason && (
                        <div className="p-3 bg-red-50 dark:bg-red-950 border border-red-200 dark:border-red-800 rounded-md">
                          <div className="flex items-center space-x-2 text-red-800 dark:text-red-200">
                            <XCircle className="h-4 w-4" />
                            <span className="text-sm font-medium">Rejection Reason</span>
                          </div>
                          <p className="text-sm text-red-700 dark:text-red-300 mt-1">
                            {asset.rejectionReason}
                          </p>
                        </div>
                      )}

                      {/* Action Buttons */}
                      <div className="flex items-center justify-between pt-2">
                        <div className="text-xs text-muted-foreground">
                          {new Date(asset.createdAt).toLocaleDateString()}
                        </div>
                        <div className="flex items-center space-x-2">
                          {canEditAsset(asset) && (
                            <Dialog>
                              <DialogTrigger asChild>
                                <Button 
                                  variant="outline" 
                                  size="sm"
                                  onClick={() => setEditingAsset(asset)}
                                  data-testid={`button-edit-${asset.id}`}
                                >
                                  <Edit className="h-3 w-3" />
                                </Button>
                              </DialogTrigger>
                              <DialogContent>
                                <DialogHeader>
                                  <DialogTitle>Edit Asset</DialogTitle>
                                </DialogHeader>
                                <EditAssetForm 
                                  asset={asset}
                                  onSave={(data) => editAssetMutation.mutate({ id: asset.id, data })}
                                  onCancel={() => setEditingAsset(null)}
                                  isLoading={editAssetMutation.isPending}
                                />
                              </DialogContent>
                            </Dialog>
                          )}

                          {canDeleteAsset(asset) && (
                            <AlertDialog>
                              <AlertDialogTrigger asChild>
                                <Button 
                                  variant="outline" 
                                  size="sm"
                                  className="text-destructive hover:text-destructive"
                                  data-testid={`button-delete-${asset.id}`}
                                >
                                  <Trash2 className="h-3 w-3" />
                                </Button>
                              </AlertDialogTrigger>
                              <AlertDialogContent>
                                <AlertDialogHeader>
                                  <AlertDialogTitle>Delete Asset</AlertDialogTitle>
                                  <AlertDialogDescription>
                                    Are you sure you want to delete "{asset.title}"? This action cannot be undone.
                                  </AlertDialogDescription>
                                </AlertDialogHeader>
                                <AlertDialogFooter>
                                  <AlertDialogCancel>Cancel</AlertDialogCancel>
                                  <AlertDialogAction 
                                    onClick={() => deleteAssetMutation.mutate(asset.id)}
                                    className="bg-destructive hover:bg-destructive/90"
                                  >
                                    Delete
                                  </AlertDialogAction>
                                </AlertDialogFooter>
                              </AlertDialogContent>
                            </AlertDialog>
                          )}
                        </div>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </TabsContent>
      </Tabs>
    </div>
  );
}

// Edit Asset Form Component
interface EditAssetFormProps {
  asset: CreatorAssetWithAsset;
  onSave: (data: Partial<CreatorAsset>) => void;
  onCancel: () => void;
  isLoading: boolean;
}

function EditAssetForm({ asset, onSave, onCancel, isLoading }: EditAssetFormProps) {
  const [formData, setFormData] = useState({
    title: asset.title,
    description: asset.description || '',
    price: asset.price / 100, // Convert cents to dollars
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSave({
      title: formData.title,
      description: formData.description,
      price: Math.round(formData.price * 100), // Convert dollars to cents
    });
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="title">Title</Label>
        <Input
          id="title"
          value={formData.title}
          onChange={(e) => setFormData(prev => ({ ...prev, title: e.target.value }))}
          required
          data-testid="input-edit-title"
        />
      </div>
      
      <div>
        <Label htmlFor="description">Description</Label>
        <Textarea
          id="description"
          value={formData.description}
          onChange={(e) => setFormData(prev => ({ ...prev, description: e.target.value }))}
          data-testid="textarea-edit-description"
        />
      </div>
      
      <div>
        <Label htmlFor="price">Price (USD)</Label>
        <Input
          id="price"
          type="number"
          min={CREATOR_MIN_PRICE_CENTS / 100}
          max={CREATOR_MAX_PRICE_CENTS / 100}
          step="0.01"
          value={formData.price}
          onChange={(e) => setFormData(prev => ({ ...prev, price: parseFloat(e.target.value) || 0 }))}
          required
          data-testid="input-edit-price"
        />
      </div>

      <div className="flex justify-end space-x-2">
        <Button type="button" variant="outline" onClick={onCancel} disabled={isLoading}>
          Cancel
        </Button>
        <Button type="submit" disabled={isLoading} data-testid="button-save-changes">
          {isLoading ? 'Saving...' : 'Save Changes'}
        </Button>
      </div>
    </form>
  );
}