import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsList, TabsTrigger, TabsContent } from '@/components/ui/tabs';
import { Form, FormField, FormItem, FormLabel, FormControl, FormDescription, FormMessage } from '@/components/ui/form';
import { FileText, Wand2, Copy, CheckCircle2, Loader2, RotateCcw } from 'lucide-react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useToast } from '@/hooks/use-toast';
import { apiRequest } from '@/lib/queryClient';
import { useAuth } from '@/contexts/AuthContext';
import { SurveyModal, hasAnsweredSurvey, type SurveyOption } from "@/components/SurveyModal";
import { useSurveyStore } from "@/stores/useSurveyStore";
import { submitSurvey } from "@/lib/surveys";

// Types
type PlanSection = { 
  id: string; 
  title: string; 
  content: string; 
};

type FinanceRow = { 
  month: string; 
  revenue: number; 
  costs: number; 
  profit: number; 
};

type Plan = { 
  sections: PlanSection[]; 
  finance?: { 
    monthly_table?: FinanceRow[]; 
  }; 
};

// Form schema
const planSchema = z.object({
  businessName: z.string().min(1, "Business name is required"),
  description: z.string().min(1, "Description is required").max(160, "Description must be under 160 characters"),
  industry: z.string().optional(),
  targetMarket: z.string().optional(),
  competitorAnalysis: z.string().optional(),
  monetizationModel: z.string().optional(),
});

type PlanFormData = z.infer<typeof planSchema>;

export default function BusinessPlanPage() {
  const { toast } = useToast();
  const { currentUser, isPaidUser } = useAuth();
  const [activeTab, setActiveTab] = useState<"lean" | "full">("lean");
  const [isGenerating, setIsGenerating] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [plan, setPlan] = useState<Plan | null>(null);
  const [copiedSection, setCopiedSection] = useState<string | null>(null);

  // Survey integration
  const survey = useSurveyStore();
  const SURVEY_ID = "discovery-channel-v1";

  const SURVEY_OPTIONS: SurveyOption[] = [
    { id: "search", label: "Google/Bing Search" },
    { id: "ai", label: "AI Platform (ChatGPT, Claude, Gemini, etc.)" },
    { id: "facebook", label: "Facebook" },
    { id: "reddit", label: "Reddit" },
    { id: "instagram", label: "Instagram" },
    { id: "youtube", label: "YouTube" },
    { id: "blog", label: "Blog / Article / Publication" },
    { id: "review", label: "Review or comparison site" },
    { id: "university", label: "Through my University" },
    { id: "peer", label: "Recommendation from peer/advisor" },
    { id: "store", label: "QuickBooks/Xero App Store" },
    { id: "other", label: "Other" },
  ];

  // Form setup
  const form = useForm<PlanFormData>({
    resolver: zodResolver(planSchema),
    defaultValues: {
      businessName: '',
      description: '',
      industry: '',
      targetMarket: '',
      competitorAnalysis: '',
      monetizationModel: '',
    }
  });

  // Handlers
  const handleCopySection = async (id: string, text: string) => {
    try {
      await navigator.clipboard.writeText(text);
      setCopiedSection(id);
      toast({
        title: "Copied to clipboard",
        description: "Section content has been copied to your clipboard.",
      });
      
      // Reset after 2 seconds
      setTimeout(() => setCopiedSection(null), 2000);
    } catch (error) {
      toast({
        title: "Failed to copy",
        description: "Unable to copy to clipboard. Please try again.",
        variant: "destructive",
      });
    }
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if ((e.metaKey || e.ctrlKey) && e.key === 'Enter') {
      e.preventDefault();
      form.handleSubmit(onSubmit)();
    }
  };

  const onSubmit = async (values: PlanFormData) => {
    setIsGenerating(true);
    setError(null);
    setPlan(null);
    
    try {
      const response = await apiRequest<Plan>('POST', '/api/ai/business-plan', {
        mode: activeTab,
        ...values,
      });
      
      setPlan(response);
      
      // Track usage
      const currentCount = parseInt(localStorage.getItem('businessPlanCount') || '0');
      localStorage.setItem('businessPlanCount', (currentCount + 1).toString());
      
      // 🔔 Trigger survey once per device
      if (!hasAnsweredSurvey(SURVEY_ID)) {
        survey.set({
          open: true,
          surveyId: SURVEY_ID,
          title: "How did you first hear about IBrandBiz?",
          description: "Pick the option that best describes how you found us.",
          options: SURVEY_OPTIONS,
          context: { userId: currentUser?.uid ?? null, planId: undefined },
        });
      }
      
    } catch (error: any) {
      console.error('Failed to generate business plan:', error);
      setError('Failed to generate business plan. Please try again.');
      toast({
        title: "Generation failed",
        description: "Unable to generate business plan. Please check your details and try again.",
        variant: "destructive",
      });
    } finally {
      setIsGenerating(false);
    }
  };

  const handleClear = () => {
    form.reset();
    setPlan(null);
    setError(null);
  };

  return (
    <div className="max-w-6xl mx-auto p-4 md:p-6">
      <div className="mb-8 text-center">
        <h1 className="text-3xl font-bold mb-2">AI Business Plan Developer</h1>
        <p className="text-muted-foreground">
          Generate a comprehensive business plan for your venture
        </p>
      </div>

      <Tabs value={activeTab} onValueChange={(value) => setActiveTab(value as "lean" | "full")}>
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="lean" data-testid="tab-lean">
            Lean Plan
            <Badge variant="outline" className="ml-2">Free</Badge>
          </TabsTrigger>
          <TabsTrigger value="full" data-testid="tab-full">
            Full Plan
            <Badge variant="outline" className="ml-2">Pro</Badge>
          </TabsTrigger>
        </TabsList>

        {/* Lean Tab Content */}
        <TabsContent value="lean" className="space-y-8">
          <div className="grid lg:grid-cols-2 gap-8">
            <div className="space-y-6">
              <div className="flex items-center gap-2 mb-4">
                <Badge className="bg-green-100 text-green-800" data-testid="badge-lean-plan">Lean Plan</Badge>
                <span className="text-sm text-muted-foreground">5 essential sections • Free</span>
              </div>
              
              <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6" onKeyDown={handleKeyDown}>
                  {/* Business Name */}
                  <FormField
                    control={form.control}
                    name="businessName"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Business Name *</FormLabel>
                        <FormControl>
                          <Input
                            placeholder="Enter your business name"
                            {...field}
                            data-testid="input-business-name"
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  {/* Description */}
                  <FormField
                    control={form.control}
                    name="description"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>One-line Description *</FormLabel>
                        <FormControl>
                          <Textarea
                            placeholder="What does your business do?"
                            className="resize-none"
                            rows={2}
                            {...field}
                            data-testid="textarea-description"
                          />
                        </FormControl>
                        <FormDescription>
                          {field.value ? `${field.value.length}/160` : "0/160"} characters
                        </FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <div className="flex gap-3 pt-4">
                    <Button
                      type="submit"
                      disabled={isGenerating || !form.getValues("businessName").trim() || !form.getValues("description").trim()}
                      className="flex-1 md:flex-initial"
                      data-testid="button-generate"
                    >
                      {isGenerating ? (
                        <>
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          Generating...
                        </>
                      ) : (
                        <>
                          <Wand2 className="w-4 h-4 mr-2" />
                          Generate Lean Plan
                        </>
                      )}
                    </Button>
                    <Button
                      type="button"
                      variant="secondary"
                      onClick={handleClear}
                      data-testid="button-clear"
                    >
                      <RotateCcw className="w-4 h-4 mr-2" />
                      Clear
                    </Button>
                  </div>
                </form>
              </Form>
            </div>

            {/* Results column */}
            <div className="space-y-4">
              <h2 className="text-lg font-semibold" data-testid="heading-results">
                Your Lean Business Plan
              </h2>

              {/* Loading State */}
              {isGenerating && (
                <div className="space-y-4" data-testid="loading-results">
                  {[1, 2, 3, 4, 5].map((i) => (
                    <Card key={i} className="animate-pulse">
                      <CardHeader>
                        <div className="h-4 bg-muted rounded w-1/3"></div>
                      </CardHeader>
                      <CardContent>
                        <div className="space-y-2">
                          <div className="h-3 bg-muted rounded w-full"></div>
                          <div className="h-3 bg-muted rounded w-full"></div>
                          <div className="h-3 bg-muted rounded w-2/3"></div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}

              {/* Error State */}
              {error && (
                <div className="p-4 bg-destructive/10 border border-destructive/20 rounded-lg" data-testid="error-message">
                  <p className="text-destructive text-sm">{error}</p>
                </div>
              )}

              {/* Empty State */}
              {!isGenerating && !error && !plan && (
                <div className="p-8 text-center text-muted-foreground" data-testid="empty-state">
                  <FileText className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Fill out your business details to generate a lean business plan.</p>
                  <p className="text-sm mt-2">Includes: Summary • Customer & Problem • Offer • Go-to-Market • 90-Day Objectives</p>
                </div>
              )}

              {/* Results */}
              {plan && (
                <div className="space-y-4" data-testid="results-list">
                  {plan.sections.map((section, index) => (
                    <Card key={section.id} className="group" data-testid={`result-section-${index}`}>
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between">
                          <CardTitle className="text-lg" data-testid={`section-title-${index}`}>
                            {section.title}
                          </CardTitle>
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => handleCopySection(section.id, section.content)}
                            className="opacity-0 group-hover:opacity-100 transition-opacity"
                            data-testid={`button-copy-section-${index}`}
                          >
                            {copiedSection === section.id ? (
                              <>
                                <CheckCircle2 className="w-3 h-3 mr-1 text-green-600" />
                                Copied
                              </>
                            ) : (
                              <>
                                <Copy className="w-3 h-3 mr-1" />
                                Copy
                              </>
                            )}
                          </Button>
                        </div>
                      </CardHeader>
                      <CardContent>
                        <div className="prose prose-sm max-w-none" data-testid={`section-content-${index}`}>
                          {section.content.split('\n').map((paragraph, pIndex) => (
                            paragraph.trim() && (
                              <p key={pIndex} className="mb-2">{paragraph.trim()}</p>
                            )
                          ))}
                        </div>
                      </CardContent>
                    </Card>
                  ))}

                  {/* Finance Table for Full Plans */}
                  {plan.finance && plan.finance.monthly_table && (
                    <Card data-testid="finance-table">
                      <CardHeader>
                        <CardTitle>Financial Projections</CardTitle>
                      </CardHeader>
                      <CardContent>
                        <div className="overflow-x-auto">
                          <table className="w-full border-collapse border border-border">
                            <thead>
                              <tr className="bg-muted">
                                <th className="border border-border p-2 text-left">Month</th>
                                <th className="border border-border p-2 text-left">Revenue</th>
                                <th className="border border-border p-2 text-left">Costs</th>
                                <th className="border border-border p-2 text-left">Profit</th>
                              </tr>
                            </thead>
                            <tbody>
                              {plan.finance.monthly_table.map((row, index) => (
                                <tr key={index}>
                                  <td className="border border-border p-2">{row.month}</td>
                                  <td className="border border-border p-2">${row.revenue.toLocaleString()}</td>
                                  <td className="border border-border p-2">${row.costs.toLocaleString()}</td>
                                  <td className="border border-border p-2">${row.profit.toLocaleString()}</td>
                                </tr>
                              ))}
                            </tbody>
                          </table>
                        </div>
                      </CardContent>
                    </Card>
                  )}
                </div>
              )}
            </div>
          </div>
        </TabsContent>

        {/* Full Tab Content */}
        <TabsContent value="full" className="space-y-8">
          <div className="grid lg:grid-cols-2 gap-8">
            <div className="space-y-6">
              <div className="flex items-center gap-2 mb-4">
                <Badge className="bg-gradient-to-r from-primary to-accent" data-testid="badge-full-plan">Full Plan</Badge>
                <span className="text-sm text-muted-foreground">10+ comprehensive sections • Pro</span>
              </div>
              
              <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6" onKeyDown={handleKeyDown}>
                  {/* Business Name */}
                  <FormField
                    control={form.control}
                    name="businessName"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Business Name *</FormLabel>
                        <FormControl>
                          <Input
                            placeholder="Enter your business name"
                            {...field}
                            data-testid="input-business-name-full"
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  {/* Description */}
                  <FormField
                    control={form.control}
                    name="description"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>One-line Description *</FormLabel>
                        <FormControl>
                          <Textarea
                            placeholder="What does your business do?"
                            className="resize-none"
                            rows={2}
                            {...field}
                            data-testid="textarea-description-full"
                          />
                        </FormControl>
                        <FormDescription>
                          {field.value ? `${field.value.length}/160` : "0/160"} characters
                        </FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  {/* All other form fields similar to lean mode */}
                  <div className="flex gap-3 pt-4">
                    <Button
                      type="submit"
                      disabled={isGenerating || !form.getValues("businessName").trim() || !form.getValues("description").trim()}
                      className="flex-1 md:flex-initial"
                      data-testid="button-generate-full"
                    >
                      {isGenerating ? (
                        <>
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          Generating...
                        </>
                      ) : (
                        <>
                          <FileText className="w-4 h-4 mr-2" />
                          Generate Full Plan
                        </>
                      )}
                    </Button>
                    <Button
                      type="button"
                      variant="secondary"
                      onClick={handleClear}
                      data-testid="button-clear-full"
                    >
                      <RotateCcw className="w-4 h-4 mr-2" />
                      Clear
                    </Button>
                  </div>
                </form>
              </Form>
            </div>

            {/* Results column - same as lean but with full plan results */}
            <div className="space-y-4">
              <h2 className="text-lg font-semibold" data-testid="heading-full-results">
                Your Full Business Plan
              </h2>
              {/* Results will be shown here when plan is generated */}
            </div>
          </div>
        </TabsContent>
      </Tabs>

      {/* Survey Modal */}
      <SurveyModal
        open={survey.open}
        onOpenChange={(v) => survey.set({ open: v })}
        surveyId={survey.surveyId ?? "unknown"}
        title={survey.title}
        description={survey.description}
        options={survey.options}
        userId={survey.context.userId}
        planId={survey.context.planId}
        onSubmit={submitSurvey}
      />
    </div>
  );
}