import { useState, useEffect } from 'react';
import { Textarea } from '@/components/ui/textarea';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { OutlineEditorV2 } from '@/components/business-plan/OutlineEditor.v2';
import { StructuredFormV2 } from '@/components/business-plan/StructuredForm.v2';
import { FinancialsPanel } from '@/components/finance/FinancialsPanel';
import { AIButton } from '@/components/ai/AIButton';
import { DashboardTemplatePage } from '@/components/DashboardTemplatePage';
import { BusinessBrief } from '@/components/business-plan/BusinessBrief';
import { FileText, Eye, Save, Download, Check, FileSearch, RefreshCw, Plus, Minus, Loader2, Info, Sparkles, X, ChevronRight } from 'lucide-react';
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from '@/components/ui/breadcrumb';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from '@/components/ui/alert-dialog';
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle, SheetTrigger } from '@/components/ui/sheet';
import { usePlanStoreV2 } from '@/stores/usePlanStore.v2';
import { useLocation } from 'wouter';
import type { SectionId } from '@shared/plan-types';
import { aiClient } from '@/services/ai/aiClient';
import { useToast } from '@/hooks/use-toast';
import type { AiAction } from '@shared/ai-types';
import { Document, Packer, Paragraph, TextRun, HeadingLevel, AlignmentType } from 'docx';
import { saveAs } from 'file-saver';
import { withPro } from '@/lib/withPro';

function BusinessPlanBuilder() {
  const [selectedSectionId, setSelectedSectionId] = useState<string>('');
  const [showPreview, setShowPreview] = useState(false);
  const [showFullPreview, setShowFullPreview] = useState(false);
  const [isAiLoading, setIsAiLoading] = useState(false);
  const [lastAiAction, setLastAiAction] = useState<AiAction | null>(null);
  const [showUnsavedDialog, setShowUnsavedDialog] = useState(false);
  const [pendingNavigationTarget, setPendingNavigationTarget] = useState<string | null>(null);
  const [originalContent, setOriginalContent] = useState<string>('');
  const [showFrameworksDrawer, setShowFrameworksDrawer] = useState(false);
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const { sections, updateSection, title, setPlanTitle, businessBrief, updateBusinessBrief } = usePlanStoreV2();
  
  const selectedSection = selectedSectionId 
    ? sections.find(s => s.id === selectedSectionId)
    : null;

  // Auto-select first section if none selected and sections exist
  useEffect(() => {
    if (!selectedSectionId && sections.length > 0) {
      setSelectedSectionId(sections[0].id);
    }
  }, [selectedSectionId, sections]);

  // Track original content when section changes
  useEffect(() => {
    if (selectedSection) {
      setOriginalContent(selectedSection.content);
    }
  }, [selectedSectionId]); // Only run when section ID changes, not when content changes

  const handleSectionSelect = (id: SectionId) => {
    // Check for unsaved changes before switching sections
    if (hasUnsavedChanges()) {
      setPendingNavigationTarget(id);
      setShowUnsavedDialog(true);
    } else {
      setSelectedSectionId(id);
      setShowPreview(false);
    }
  };

  const handleContentChange = (content: string) => {
    if (selectedSection) {
      updateSection(selectedSection.id, { content });
    }
  };

  const handleSaveSection = () => {
    if (selectedSection) {
      updateSection(selectedSection.id, { 
        savedAt: new Date().toISOString() 
      });
      // Update original content to current after saving
      setOriginalContent(selectedSection.content);
      toast({
        description: "Section saved",
      });
    }
  };

  const handleClearSection = () => {
    if (selectedSection) {
      updateSection(selectedSection.id, { content: '' });
      toast({
        description: "Section cleared.",
      });
    }
  };

  const hasUnsavedChanges = () => {
    if (!selectedSection) return false;
    // Has unsaved changes if current content differs from original content
    return selectedSection.content !== originalContent;
  };

  const findPreviousSection = (): string | null => {
    const currentIndex = sections.findIndex(s => s.id === selectedSectionId);
    if (currentIndex > 0) {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (sections[i].isEnabled) {
          return sections[i].id;
        }
      }
    }
    return null;
  };

  const findNextSection = (): string | null => {
    const currentIndex = sections.findIndex(s => s.id === selectedSectionId);
    if (currentIndex < sections.length - 1) {
      for (let i = currentIndex + 1; i < sections.length; i++) {
        if (sections[i].isEnabled) {
          return sections[i].id;
        }
      }
    }
    return null;
  };

  const navigateToSection = (targetId: string) => {
    setSelectedSectionId(targetId);
    setShowPreview(false);
    setPendingNavigationTarget(null);
    setShowUnsavedDialog(false);
  };

  const handlePreviousSection = () => {
    const targetId = findPreviousSection();
    if (!targetId) return;
    
    if (hasUnsavedChanges()) {
      setPendingNavigationTarget(targetId);
      setShowUnsavedDialog(true);
    } else {
      navigateToSection(targetId);
    }
  };

  const handleNextSection = () => {
    const targetId = findNextSection();
    if (!targetId) return;
    
    if (hasUnsavedChanges()) {
      setPendingNavigationTarget(targetId);
      setShowUnsavedDialog(true);
    } else {
      navigateToSection(targetId);
    }
  };

  const handleSaveAndContinue = () => {
    handleSaveSection();
    if (pendingNavigationTarget) {
      navigateToSection(pendingNavigationTarget);
    }
  };

  const handleDiscardAndContinue = () => {
    // Discard unsaved changes by restoring original content
    if (selectedSection) {
      updateSection(selectedSection.id, { content: originalContent });
    }
    
    // Navigate to target if set, otherwise just close dialog
    if (pendingNavigationTarget) {
      navigateToSection(pendingNavigationTarget);
    } else {
      setPendingNavigationTarget(null);
      setShowUnsavedDialog(false);
    }
  };

  const isFirstSection = () => {
    const currentIndex = sections.findIndex(s => s.id === selectedSectionId);
    // Check if there's any enabled section before this one
    return !sections.slice(0, currentIndex).some(s => s.isEnabled);
  };

  const isLastSection = () => {
    const currentIndex = sections.findIndex(s => s.id === selectedSectionId);
    // Check if there's any enabled section after this one
    return !sections.slice(currentIndex + 1).some(s => s.isEnabled);
  };

  const formatSaveTime = (savedAt?: string) => {
    if (!savedAt) return null;
    const date = new Date(savedAt);
    return date.toLocaleTimeString('en-US', { 
      hour: 'numeric', 
      minute: '2-digit',
      hour12: true 
    });
  };

  const handleAiAction = async (action: AiAction) => {
    if (!selectedSection || isAiLoading) return;
    
    setIsAiLoading(true);
    setLastAiAction(action);

    try {
      let response;
      
      switch (action) {
        case "rephrase":
          response = await aiClient.rephraseContent(
            selectedSection.kind,
            selectedSection.title,
            selectedSection.content,
            "Professional",
            {
              businessName: businessBrief?.company || title !== "Untitled Business Plan" ? title : undefined,
              businessBrief: businessBrief,
            }
          );
          break;
        case "expand":
          response = await aiClient.expandContent(
            selectedSection.kind,
            selectedSection.title,
            selectedSection.content,
            "Professional",
            {
              businessName: businessBrief?.company || title !== "Untitled Business Plan" ? title : undefined,
              businessBrief: businessBrief,
            }
          );
          break;
        case "summarize":
          response = await aiClient.summarizeContent(
            selectedSection.kind,
            selectedSection.title,
            selectedSection.content,
            "Professional",
            {
              businessName: businessBrief?.company || title !== "Untitled Business Plan" ? title : undefined,
              businessBrief: businessBrief,
            }
          );
          break;
        default:
          throw new Error(`Unknown action: ${action}`);
      }

      if (response.status === "completed" && response.content) {
        handleContentChange(response.content);
        
        // Use SN's toast messages
        const actionMessages = {
          rephrase: "Draft rephrased",
          expand: "Content expanded",
          summarize: "Content summarized"
        };
        
        toast({
          description: actionMessages[action] || "Content updated",
        });
      } else if (response.status === "error") {
        throw new Error(response.error?.message || "AI request failed — please try again");
      }
    } catch (error) {
      console.error("AI action failed:", error);
      toast({
        description: error instanceof Error ? error.message : "AI request failed — please try again",
        variant: "destructive",
      });
    } finally {
      setIsAiLoading(false);
    }
  };

  const getSectionIcon = (kind?: string) => {
    switch (kind) {
      case 'executive-summary':
        return '📊';
      case 'company-overview':
        return '🏢';
      case 'products-services':
        return '💼';
      case 'market-analysis':
        return '📈';
      case 'marketing-sales':
        return '📢';
      case 'operations-plan':
        return '⚙️';
      case 'org-management':
        return '👥';
      case 'financial-plan':
        return '💰';
      case 'milestones-roadmap':
        return '🗺️';
      case 'future-expansion':
        return '🚀';
      case 'partnerships-note':
        return '🤝';
      case 'exit-strategy':
        return '🎯';
      default:
        return '📄';
    }
  };

  const getSectionTip = (kind?: string) => {
    switch (kind) {
      case 'executive-summary':
        return 'Tip: Keep it 200–300 words. Hit the highlights—model, offerings, market, edge, and headline numbers.';
      case 'company-overview':
        return 'Tip: Mission, vision, value prop, origin, legal structure, current stage.';
      case 'products-services':
        return 'Tip: Bulleted offerings (1 line each), then a short value paragraph.';
      case 'market-analysis':
        return 'Tip: ICP, trends, competitors, and a mini-SWOT. Use Structured Templates if helpful.';
      case 'marketing-sales':
        return 'Tip: Positioning, channels, funnel, retention, and 4–6 KPIs.';
      case 'operations-plan':
        return 'Tip: Tech/infra, core processes, vendors/tools, SLAs/security, ops KPIs.';
      case 'org-management':
        return 'Tip: Leadership, roles, hiring plan.';
      case 'financial-plan':
        return 'Tip: Revenue streams, assumptions, Launch/Growth/Scale forecast, risks.';
      case 'milestones-roadmap':
        return 'Tip: 4–8 bullets grouped by phase: Launch, Growth, Scale, Beyond.';
      case 'future-expansion':
        return 'Tip: Integrations, adjacent products, partnerships/channels, long-term vision.';
      case 'partnerships-note':
        return 'Tip: Rationale, types, criteria, model, customer benefit.';
      case 'exit-strategy':
        return 'Tip: For investors—primary path, alternatives, triggers, value story.';
      default:
        return '';
    }
  };

  const renderPreview = (content: string) => {
    if (!content.trim()) {
      return <p className="text-muted-foreground text-sm">No content to preview...</p>;
    }

    // Simple markdown-to-HTML conversion for preview
    const html = content
      .replace(/\n\n/g, '</p><p>')
      .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
      .replace(/\*(.*?)\*/g, '<em>$1</em>')
      .replace(/^### (.*$)/gm, '<h3>$1</h3>')
      .replace(/^## (.*$)/gm, '<h2>$1</h2>')
      .replace(/^# (.*$)/gm, '<h1>$1</h1>');

    return <div dangerouslySetInnerHTML={{ __html: `<p>${html}</p>` }} />;
  };

  const handleDownloadPlan = async () => {
    // Get all enabled sections with content
    const enabledSections = sections.filter(s => s.isEnabled && s.content?.trim());
    
    if (enabledSections.length === 0) {
      toast({
        description: "No content to download yet. Add content to your sections first.",
        variant: "destructive"
      });
      return;
    }

    const planTitle = title || "Business Plan";
    
    try {
      // Create document children (paragraphs)
      const docChildren: Paragraph[] = [];
      
      // Add title
      docChildren.push(
        new Paragraph({
          text: planTitle,
          heading: HeadingLevel.TITLE,
          alignment: AlignmentType.CENTER,
          spacing: { after: 400 }
        })
      );
      
      // Add each section
      enabledSections.forEach((section) => {
        // Section title
        docChildren.push(
          new Paragraph({
            text: section.title,
            heading: HeadingLevel.HEADING_1,
            spacing: { before: 400, after: 200 }
          })
        );
        
        // Section content - split by paragraphs
        const paragraphs = section.content.split('\n\n').filter(p => p.trim());
        paragraphs.forEach((para) => {
          // Handle basic markdown formatting
          const runs: TextRun[] = [];
          let text = para;
          
          // Simple bold handling (**text**)
          const boldRegex = /\*\*(.*?)\*\*/g;
          let lastIndex = 0;
          let match;
          
          while ((match = boldRegex.exec(para)) !== null) {
            // Add text before match
            if (match.index > lastIndex) {
              runs.push(new TextRun(para.substring(lastIndex, match.index)));
            }
            // Add bold text
            runs.push(new TextRun({ text: match[1], bold: true }));
            lastIndex = match.index + match[0].length;
          }
          
          // Add remaining text
          if (lastIndex < para.length) {
            runs.push(new TextRun(para.substring(lastIndex)));
          }
          
          // If no formatting, just add plain text
          if (runs.length === 0) {
            runs.push(new TextRun(text));
          }
          
          docChildren.push(
            new Paragraph({
              children: runs,
              spacing: { after: 200 }
            })
          );
        });
      });
      
      // Create document
      const doc = new Document({
        sections: [{
          properties: {},
          children: docChildren
        }]
      });
      
      // Generate and download
      const blob = await Packer.toBlob(doc);
      
      // Sanitize filename
      const filename = planTitle.replace(/[^a-z0-9]/gi, '_').toLowerCase();
      saveAs(blob, `${filename}.docx`);
      
      toast({
        description: "Business plan downloaded successfully!",
      });
    } catch (error) {
      console.error('Error generating document:', error);
      toast({
        description: "Failed to generate document. Please try again.",
        variant: "destructive"
      });
    }
  };

  return (
    <DashboardTemplatePage title="AI Frameworks Builder">
      {/* Breadcrumbs */}
      <Breadcrumb className="mb-[16px] -mt-1">
        <BreadcrumbList>
          <BreadcrumbItem>
            <BreadcrumbLink href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Business Development</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbPage className="text-[#FF8B00] font-semibold">AI Business Plan Developer</BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>

      {/* Step 1 - Business Brief */}
      <div className="mb-4 p-4 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm bg-white dark:bg-gray-800">
        <div className="flex items-start justify-between mb-2">
          <h3 className="text-base font-semibold text-gray-900 dark:text-gray-100">
            Step 1 – Business Brief
          </h3>
          <Button
            variant="outline"
            size="sm"
            onClick={() => {
              updateBusinessBrief({
                company: '',
                industry: '',
                businessModel: '',
                productsServices: '',
                offeringsFull: '',
                market: '',
                differentiator: '',
                financialHeadline: ''
              });
              setPlanTitle('');
              toast({
                description: "Business Brief cleared successfully",
              });
            }}
            className="flex items-center gap-1"
            data-testid="button-refresh-brief"
          >
            <RefreshCw className="w-3.5 h-3.5" />
            Refresh
          </Button>
        </div>
        <p className="text-sm text-muted-foreground mb-1">
          Add your company's key details once. AI will use this info to personalize every section.
        </p>
        <p className="text-sm text-muted-foreground mb-1">
          <strong>Fill in:</strong> Company Name, Industry, Business Model, Products/Services, Offerings, Target Market, Competitive Edge, and (optional) Financial Headline.
        </p>
        <p className="text-sm text-muted-foreground">
          This sets the context for the entire plan.
        </p>
      </div>

      {/* Business Brief */}
      <BusinessBrief />

      {/* Step 2 - Build Your Plan */}
      <div className="mb-4 mt-6 p-4 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm bg-white dark:bg-gray-800">
        <h3 className="text-base font-semibold text-gray-900 dark:text-gray-100 mb-2">
          Step 2 – Build Your Plan
        </h3>
        <p className="text-sm text-muted-foreground mb-1">
          Use the outline below to add, edit, or generate each section.
        </p>
        <p className="text-sm text-muted-foreground mb-1">
          <strong>+ Add Section</strong> → create a custom section.
        </p>
        <p className="text-sm text-muted-foreground mb-1">
          <strong>Use Template</strong> → insert a structured section (e.g., SWOT, Lean Canvas).
        </p>
        <p className="text-sm text-muted-foreground mb-1">
          <strong>Click into any section</strong> → write, paste, or use AI Assist to draft.
        </p>
        <p className="text-sm text-muted-foreground">
          <strong>Save as you go</strong> → then use Preview Full Document to review your plan anytime.
        </p>
      </div>

      <div className={`grid gap-6 items-start ${
        selectedSection?.kind === 'market-analysis' 
          ? 'lg:grid-cols-[300px_1fr] xl:grid-cols-[300px_1fr_380px]'
          : 'lg:grid-cols-3'
      }`}>
          {/* Left Panel - Outline Editor */}
          <div className="lg:col-span-1">
            <OutlineEditorV2 
              selectedSectionId={selectedSectionId}
              onSectionSelect={handleSectionSelect}
            />
          </div>

          {/* Center Panel - Section Editor or Finance Interface */}
          <div className={selectedSection?.kind === 'market-analysis' ? 'lg:col-span-1' : 'lg:col-span-2'}>
            {selectedSection ? (
              selectedSection.kind === 'financial-plan' ? (
                // Enhanced Financial Tools V2 Interface
                <div className="min-h-[600px]">
                  <FinancialsPanel />
                </div>
              ) : (
                // Regular Section Editor for non-financial sections
                <div className="mt-9">
                <Card className="min-h-[600px] flex flex-col" data-testid="section-editor">
                  <CardHeader className="pb-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <div>
                          <CardTitle className="text-lg">{selectedSection.title}</CardTitle>
                          <div className="flex items-center gap-2 mt-1">
                            <Badge variant="outline" className="text-xs">
                              {selectedSection.kind}
                            </Badge>
                            {selectedSection.content && (
                              <Badge variant="secondary" className="text-xs">
                                {selectedSection.content.length} chars
                              </Badge>
                            )}
                            {selectedSection.savedAt && (
                              <Badge variant="default" className="text-xs bg-green-600 text-white font-bold">
                                <Check className="w-3 h-3 mr-1" />
                                Saved · {formatSaveTime(selectedSection.savedAt)}
                              </Badge>
                            )}
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    {/* AI Quick Actions - Only show when content exists */}
                    {selectedSection.content && selectedSection.content.trim().length > 0 && (
                      <TooltipProvider>
                        <div className="flex gap-2 mt-3">
                          <Tooltip>
                            <TooltipTrigger asChild>
                              <Button
                                size="sm"
                                onClick={() => handleAiAction("rephrase")}
                                disabled={isAiLoading}
                                className="bg-accent hover:bg-accent/90 text-white"
                                data-testid="button-rephrase"
                              >
                                {isAiLoading && lastAiAction === "rephrase" ? (
                                  <Loader2 className="w-3 h-3 mr-1 animate-spin" />
                                ) : (
                                  <RefreshCw className="w-3 h-3 mr-1" />
                                )}
                                Rephrase
                              </Button>
                            </TooltipTrigger>
                            <TooltipContent>
                              <p>Rewrite for clarity (keeps meaning)</p>
                            </TooltipContent>
                          </Tooltip>

                          <Tooltip>
                            <TooltipTrigger asChild>
                              <Button
                                size="sm"
                                onClick={() => handleAiAction("expand")}
                                disabled={isAiLoading}
                                className="bg-accent hover:bg-accent/90 text-white"
                                data-testid="button-expand"
                              >
                                {isAiLoading && lastAiAction === "expand" ? (
                                  <Loader2 className="w-3 h-3 mr-1 animate-spin" />
                                ) : (
                                  <Plus className="w-3 h-3 mr-1" />
                                )}
                                Expand
                              </Button>
                            </TooltipTrigger>
                            <TooltipContent>
                              <p>Add detail or examples</p>
                            </TooltipContent>
                          </Tooltip>

                          <Tooltip>
                            <TooltipTrigger asChild>
                              <Button
                                size="sm"
                                onClick={() => handleAiAction("summarize")}
                                disabled={isAiLoading}
                                className="bg-accent hover:bg-accent/90 text-white"
                                data-testid="button-summarize"
                              >
                                {isAiLoading && lastAiAction === "summarize" ? (
                                  <Loader2 className="w-3 h-3 mr-1 animate-spin" />
                                ) : (
                                  <Minus className="w-3 h-3 mr-1" />
                                )}
                                Summarize
                              </Button>
                            </TooltipTrigger>
                            <TooltipContent>
                              <p>Make it more concise</p>
                            </TooltipContent>
                          </Tooltip>

                          <div className="ml-auto">
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  size="sm"
                                  onClick={handleClearSection}
                                  className="bg-white hover:bg-primary hover:text-white text-black border border-primary/40 font-normal hover:font-bold"
                                  data-testid="button-clear"
                                >
                                  <X className="w-3 h-3 mr-1" />
                                  Clear
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>
                                <p>Clear section content</p>
                              </TooltipContent>
                            </Tooltip>
                          </div>
                        </div>
                      </TooltipProvider>
                    )}
                  </CardHeader>

                  <CardContent className="flex-1 overflow-auto">
                    {!showPreview ? (
                      <div className="h-full flex flex-col" data-testid="content-editor">
                        {/* Section Tip */}
                        {getSectionTip(selectedSection.kind) && (
                          <div className="flex items-start gap-2 mb-3 p-2 bg-muted/30 rounded-md">
                            <Info className="w-4 h-4 mt-0.5 text-muted-foreground flex-shrink-0" />
                            <p className="text-xs text-muted-foreground">{getSectionTip(selectedSection.kind)}</p>
                          </div>
                        )}
                        
                        <Textarea
                          value={selectedSection.content}
                          onChange={(e) => handleContentChange(e.target.value)}
                          placeholder="Write or use AI to generate this section. You can edit freely after generating."
                          className="flex-1 min-h-[400px] resize-none font-mono text-sm"
                          data-testid="textarea-content"
                        />
                        
                        <div className="flex flex-col gap-3 mt-4">
                          {/* AI Assistant Section */}
                          <div className="flex flex-col gap-2 p-3 border rounded-lg bg-muted/20">
                            <div>
                              <h3 className="text-lg font-bold mb-1 flex items-center gap-2">
                                <Sparkles className="w-4 h-4" />
                                AI Assist
                              </h3>
                              <p className="text-xs text-muted-foreground">
                                Uses your Business Brief to draft or refine this section
                              </p>
                            </div>
                            <AIButton
                              sectionKind={selectedSection.kind}
                              sectionTitle={selectedSection.title}
                              existingContent={selectedSection.content}
                              onContentUpdate={handleContentChange}
                              context={{
                                businessName: businessBrief?.company || title !== "Untitled Business Plan" ? title : undefined,
                                businessBrief: businessBrief,
                              }}
                            />
                          </div>

                          {/* Action Buttons */}
                          <TooltipProvider>
                            <div className="flex gap-2">
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    onClick={handleSaveSection}
                                    className="flex-1"
                                    data-testid="button-save-document"
                                  >
                                    <Save className="w-4 h-4 mr-2" />
                                    Save to Document
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>
                                  <p>Save this section to your plan</p>
                                </TooltipContent>
                              </Tooltip>

                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    onClick={() => setShowPreview(true)}
                                    className="flex-1 bg-accent hover:bg-accent/90 text-white"
                                    data-testid="button-section-preview"
                                  >
                                    <Eye className="w-4 h-4 mr-2" />
                                    Section Preview
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>
                                  <p>Preview this section only</p>
                                </TooltipContent>
                              </Tooltip>
                            </div>

                            {/* Navigation Buttons */}
                            <div className="flex gap-2 justify-end">
                              {!isFirstSection() && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      onClick={handlePreviousSection}
                                      variant="outline"
                                      size="sm"
                                      data-testid="button-previous-section"
                                    >
                                      &lt; Previous
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>
                                    <p>Go to previous section</p>
                                  </TooltipContent>
                                </Tooltip>
                              )}

                              {!isLastSection() && (
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      onClick={handleNextSection}
                                      variant="outline"
                                      size="sm"
                                      data-testid="button-next-section"
                                    >
                                      Next &gt;
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>
                                    <p>Go to next section</p>
                                  </TooltipContent>
                                </Tooltip>
                              )}
                            </div>
                          </TooltipProvider>
                        </div>
                      </div>
                    ) : (
                      <div className="h-full" data-testid="content-preview">
                        <div className="prose prose-sm max-w-none h-full p-6 border rounded-lg bg-muted/20 overflow-auto">
                          {renderPreview(selectedSection.content)}
                        </div>
                        
                        <div className="flex gap-2 mt-4">
                          <Button
                            variant="outline"
                            onClick={() => setShowPreview(false)}
                            className="flex-1"
                            data-testid="button-back-to-editor"
                          >
                            Back to Editor
                          </Button>
                        </div>
                      </div>
                    )}
                  </CardContent>
                </Card>
                
                {/* Preview and Download Buttons - Outside card */}
                <div className="flex gap-2 mt-4">
                  <TooltipProvider>
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Button
                          onClick={() => setShowFullPreview(!showFullPreview)}
                          className="flex-1 bg-white hover:bg-gray-50 text-black border-2 border-primary/40 font-normal hover:font-bold"
                          data-testid="button-preview-full"
                        >
                          <FileSearch className="w-4 h-4 mr-2" />
                          Preview Full Document
                        </Button>
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Preview the entire plan</p>
                      </TooltipContent>
                    </Tooltip>
                    
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Button
                          onClick={handleDownloadPlan}
                          className="flex-1 bg-white hover:bg-gray-50 text-black border-2 border-primary/40 font-normal hover:font-bold"
                          data-testid="button-download-plan"
                        >
                          <Download className="w-4 h-4 mr-2" />
                          Download Plan
                        </Button>
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Download your business plan as a file</p>
                      </TooltipContent>
                    </Tooltip>
                  </TooltipProvider>
                </div>
                </div>
              )
            ) : (
              // Empty State
              <Card className="h-full flex items-center justify-center" data-testid="empty-section-state">
                <CardContent className="text-center py-12">
                  <FileText className="w-12 h-12 mx-auto mb-4 text-muted-foreground opacity-50" />
                  <h3 className="text-lg font-medium mb-2">No Section Selected</h3>
                  <p className="text-muted-foreground mb-4">
                    Select a section from the outline to start editing, or create a new section.
                  </p>
                </CardContent>
              </Card>
            )}
          </div>

          {/* Right Panel - AI Frameworks Builder (Market Analysis only, desktop ≥1280px) */}
          {selectedSection?.kind === 'market-analysis' && (
            <div className="hidden xl:block lg:col-span-1 xl:col-span-1">
              <div className="sticky top-6">
                <StructuredFormV2 targetSectionId={selectedSectionId} />
              </div>
            </div>
          )}
        </div>

        {/* Mobile/Tablet Drawer for AI Frameworks Builder (<1280px) */}
        {selectedSection?.kind === 'market-analysis' && (
          <div className="xl:hidden mt-4">
            <Sheet open={showFrameworksDrawer} onOpenChange={setShowFrameworksDrawer}>
              <SheetTrigger asChild>
                <Button 
                  className="w-full bg-accent hover:bg-accent/90 text-white"
                  data-testid="button-open-frameworks"
                >
                  <Sparkles className="w-4 h-4 mr-2" />
                  AI Frameworks Builder
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-full sm:w-[400px] overflow-y-auto">
                <SheetHeader className="mb-4">
                  <SheetTitle className="flex items-center gap-2 text-accent">
                    <Sparkles className="h-5 w-5" />
                    AI Frameworks Builder
                  </SheetTitle>
                  <SheetDescription>
                    AI pre-fills strategic frameworks. Edit or regenerate, then save to your section.
                  </SheetDescription>
                </SheetHeader>
                <StructuredFormV2 targetSectionId={selectedSectionId} />
              </SheetContent>
            </Sheet>
          </div>
        )}

        {/* Full Document Preview Dialog */}
        <Dialog open={showFullPreview} onOpenChange={setShowFullPreview}>
          <DialogContent className="max-w-4xl max-h-[80vh] flex flex-col">
            <DialogHeader className="pb-4 border-b">
              <DialogTitle className="text-2xl font-bold pr-8">{title || "Business Plan"}</DialogTitle>
            </DialogHeader>
            <div className="prose prose-sm max-w-none overflow-y-auto py-4 flex-1">
              {sections
                .filter(s => s.isEnabled && s.content?.trim())
                .map((section) => (
                  <div key={section.id} className="mb-8">
                    <h2 className="text-xl font-bold mb-3">{section.title}</h2>
                    {renderPreview(section.content)}
                  </div>
                ))}
              {sections.filter(s => s.isEnabled && s.content?.trim()).length === 0 && (
                <p className="text-muted-foreground text-center py-8">
                  No content to preview yet. Start adding content to your sections.
                </p>
              )}
            </div>
          </DialogContent>
        </Dialog>

        {/* Unsaved Changes Dialog */}
        <AlertDialog open={showUnsavedDialog} onOpenChange={setShowUnsavedDialog}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Save changes before navigating?</AlertDialogTitle>
              <AlertDialogDescription>
                You have unsaved changes in this section. Would you like to save them before moving to another section?
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel onClick={() => setShowUnsavedDialog(false)}>
                Cancel
              </AlertDialogCancel>
              <AlertDialogAction 
                onClick={handleDiscardAndContinue}
                className="bg-destructive hover:bg-destructive/90"
              >
                Discard
              </AlertDialogAction>
              <AlertDialogAction onClick={handleSaveAndContinue}>
                Save & Continue
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
    </DashboardTemplatePage>
  );
}

export default withPro(BusinessPlanBuilder, {
  feature: "AI Business Plan Builder",
  description: "Create investor-ready business plans with AI-powered content generation and professional templates"
});