import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent } from '@/components/ui/card';
import { Slider } from '@/components/ui/slider';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from '@/components/ui/dialog';
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from '@/components/ui/breadcrumb';
import { Wand2, Check, X, Heart, Loader2, ArrowRight, Palette, ArrowLeft, ExternalLink, Link, ChevronRight, Download } from 'lucide-react';
import {
  generateBusinessNames,
  type GeneratedBusinessName,
} from '@/services/ai/business-names';
import { useLocation } from 'wouter';
import { useAuth } from '@/contexts/AuthContext';
import { LogoPreview } from '@/components/brand/LogoPreview';
import { PalettePicker } from '@/components/brand/PalettePicker';
import { FontPicker } from '@/components/brand/FontPicker';
import { PALETTES } from '@/components/brand/palettes';
import { bulletCubeTeal, bulletCubeWhite } from '@/assets';
import { useUserPlan } from '@/hooks/useUserPlan';
import { getTier } from '@/utils/tiers';
import { PaywallModal } from '@/components/PaywallModal';

type Vibe = 'modern' | 'professional' | 'creative' | 'playful' | 'luxury' | 'techy' | 'organic';

// Attach brand meta to each card selection (persist in localStorage alongside saved names)
type BrandChoice = {
  paletteIndex?: number;
  fontFamily?: string;
};

export default function BusinessName() {
  // Stepper state
  const [step, setStep] = useState<1 | 2 | 3>(1);

  // Form state
  const [industry, setIndustry] = useState('');
  const [keywords, setKeywords] = useState('');
  const [selectedVibe, setSelectedVibe] = useState<Vibe>('modern');
  const [creativity, setCreativity] = useState<number>(60);
  const [description, setDescription] = useState('');
  const [startsWith, setStartsWith] = useState('');
  const [mustInclude, setMustInclude] = useState('');
  const [mustExclude, setMustExclude] = useState('');
  const [lengthRange, setLengthRange] = useState<[number, number]>([6, 10]);

  const [isGenerating, setIsGenerating] = useState(false);
  const [generatedNames, setGeneratedNames] = useState<GeneratedBusinessName[]>([]);
  const [savedNames, setSavedNames] = useState<GeneratedBusinessName[]>([]);
  const [expandedPreview, setExpandedPreview] = useState<number | null>(null);
  // index → BrandChoice (palette/font per generated name card)
  const [cardChoices, setCardChoices] = useState<Record<number, BrandChoice>>({});
  const [editorOpen, setEditorOpen] = useState<{ index: number | null }>({ index: null });

  const [, setLocation] = useLocation();
  const { currentUser } = useAuth();
  
  // Premium gating
  const { isPremium } = useUserPlan() ?? { isPremium: false };
  const tier = getTier(isPremium);
  const [showPaywall, setShowPaywall] = useState(false);

  const SAVED_NAMES_KEY = 'ibrandbiz-saved-names';

  useEffect(() => {
    try {
      const saved = localStorage.getItem(SAVED_NAMES_KEY);
      if (saved) setSavedNames(JSON.parse(saved));
    } catch (error) {
      console.error('Error loading saved names:', error);
    }
  }, []);

  useEffect(() => {
    try {
      localStorage.setItem(SAVED_NAMES_KEY, JSON.stringify(savedNames));
    } catch (error) {
      console.error('Error saving names to localStorage:', error);
    }
  }, [savedNames]);

  // Read business name from URL parameter on component mount
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const businessName = urlParams.get('name');
    if (businessName) {
      // If coming from external source, populate industry field
      setIndustry(decodeURIComponent(businessName));
    }
  }, []);

  const toggleSave = (nameData: GeneratedBusinessName) => {
    setSavedNames(prev => {
      const isAlreadySaved = prev.some(saved => saved.name === nameData.name);
      return isAlreadySaved ? prev.filter(s => s.name !== nameData.name) : [...prev, nameData];
    });
  };
  const isNameSaved = (name: string) => savedNames.some(s => s.name === name);

  // Default palette cycling (classic Canva vibe)
  const defaultPalette = (i: number) => i % PALETTES.length;

  const sendToBrandKit = (businessName: string, index?: number) => {
    const choice = (index != null ? cardChoices[index] : undefined) || {};
    const params = new URLSearchParams({
      name: businessName,
      // pass palette/font if chosen; brand-kit page can read and apply defaults
      palette: choice.paletteIndex != null ? String(choice.paletteIndex) : '',
      font: choice.fontFamily || '',
    });
    setLocation(`/brand-kit?${params.toString()}`);
  };
  
  const goToDomains = (businessName: string) => {
    setLocation(`/domains?search=${encodeURIComponent(businessName.toLowerCase())}`);
  };

  // Step navigation helpers
  const canNextFrom1 = industry.trim().length > 1 || keywords.trim().length > 1;
  const canNextFrom2 = true; // Always can proceed from step 2
  
  const resetCreativity = () => setCreativity(60);

  const handleGenerateNames = async () => {
    if (!industry.trim() && !keywords.trim()) {
      alert('Please enter either industry or keywords to generate names.');
      return;
    }
    setExpandedPreview(null);
    setIsGenerating(true);
    setGeneratedNames([]);

    try {
      const names = await generateBusinessNames({
        industry: industry.trim(),
        keywords: keywords.trim(),
        count: tier.maxResults,
        creativity,
        description: description.trim() || undefined,
        vibe: isPremium ? selectedVibe : undefined,
        length_range: lengthRange,
        starts_with: startsWith.trim() || undefined,
        must_include: mustInclude.trim(),
        must_exclude: mustExclude.trim(),
      });
      setGeneratedNames(names);
    } catch (error) {
      console.error('Failed to generate names:', error);
      alert('Failed to generate business names. Please try again.');
    } finally {
      setIsGenerating(false);
    }
  };
  
  const handleExportAll = () => {
    if (!isPremium) {
      setShowPaywall(true);
      return;
    }
    const text = generatedNames.map(n => `${n.name}${n.description ? ` - ${n.description}` : ''}`).join('\n');
    const blob = new Blob([text], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'business-names.txt';
    a.click();
    URL.revokeObjectURL(url);
  };

  return (
    <div className="p-4 md:p-6">
      {/* Paywall Modal */}
      <PaywallModal
        isOpen={showPaywall}
        onClose={() => setShowPaywall(false)}
        feature="advanced name generation"
        description="Get 10 results, style filters, and export functionality"
      />
      
      {/* Breadcrumbs */}
      <Breadcrumb className="mb-[16px] -mt-1">
        <BreadcrumbList>
          <BreadcrumbItem>
            <BreadcrumbLink href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Business Development</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbPage className="text-[#FF8B00] font-semibold">Brand Name Wizard</BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>

      {/* Header */}
      <h1 className="text-2xl md:text-3xl font-semibold mb-4">
        Brand Name Wizard
      </h1>

      {/* Stepper Header */}
      <div className="flex flex-col gap-2 mb-6">
        <div className="flex items-center gap-3">
          {([1, 2, 3] as const).map((n) => (
            <button
              key={n}
              className={`px-3 py-1.5 rounded-full text-sm border ${
                step === n
                  ? "bg-neutral-900 text-white border-neutral-900"
                  : "bg-white text-neutral-700 hover:bg-neutral-50"
              }`}
              onClick={() => setStep(n)}
              aria-current={step === n ? "step" : undefined}
              data-testid={`button-step-${n}`}
            >
              Step {n}
            </button>
          ))}
        </div>

        {/* Contextual caption */}
        <div className="text-sm text-neutral-500">
          {step === 1 && "Name Style (industry, keywords, vibe)"}
          {step === 2 && "Creativity Level"}
          {step === 3 && "Brand Info & Generate"}
        </div>
        
        {/* Requirements note - visible on all steps */}
        <div className="text-sm text-muted-foreground bg-blue-50 p-3 rounded-lg border border-blue-200">
          <span className="font-medium text-blue-700">💡 Tip:</span> Only one thing is required — enter either Industry/Category or Keywords. Everything else is optional. You can skip to Step 3 and generate names anytime.
        </div>
      </div>

      {/* -------- STEP 1: Name Style -------- */}
      {step === 1 && (
        <section className="space-y-6">
          <Card>
            <CardContent className="p-6 space-y-6">
              {/* Industry and Keywords */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <Label htmlFor="industry-category" className="block text-sm font-medium text-foreground mb-2">
                    Industry/Category
                  </Label>
                  <Input
                    id="industry-category"
                    type="text"
                    placeholder="e.g., Coffee Shop, Tech Startup, Consulting"
                    value={industry}
                    onChange={(e) => setIndustry(e.target.value)}
                    data-testid="input-industry"
                  />
                </div>
                <div>
                  <Label htmlFor="keywords" className="block text-sm font-medium text-foreground mb-2">
                    Keywords
                  </Label>
                  <Input
                    id="keywords"
                    type="text"
                    placeholder="e.g., fast, reliable, premium"
                    value={keywords}
                    onChange={(e) => setKeywords(e.target.value)}
                    data-testid="input-keywords"
                  />
                </div>
              </div>

              {/* Vibe Selection - Premium Only */}
              {tier.filters && (
                <div>
                  <Label className="block text-sm font-medium text-foreground mb-2">Vibe</Label>
                  <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-3">
                    {(['modern','professional','creative','playful','luxury','techy','organic'] as Vibe[]).map(vibe => (
                      <Badge
                        key={vibe}
                        variant={selectedVibe === vibe ? 'default' : 'secondary'}
                        className="cursor-pointer capitalize text-center py-2 px-3 flex items-center justify-center"
                        onClick={() => setSelectedVibe(vibe)}
                        data-testid={`badge-vibe-${vibe}`}
                      >
                        {vibe}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Navigation */}
          <div className="pt-2">
            <button
              disabled={!canNextFrom1}
              onClick={() => setStep(2)}
              className={`px-4 py-2 rounded-lg flex items-center gap-2 ${
                canNextFrom1
                  ? "bg-emerald-600 hover:bg-emerald-700 text-white"
                  : "bg-neutral-200 text-neutral-500 cursor-not-allowed"
              }`}
              data-testid="button-next-to-creativity"
            >
              Next: Creativity <ArrowRight className="h-4 w-4" />
            </button>
          </div>
        </section>
      )}

      {/* -------- STEP 2: Creativity -------- */}
      {step === 2 && (
        <section className="space-y-6">
          <Card>
            <CardContent className="p-6 space-y-6">
              <div>
                <Label className="block text-sm font-medium text-foreground mb-2">
                  Creativity Level
                </Label>
                <div className="space-y-4">
                  <div className="px-1">
                    <Slider
                      min={0}
                      max={100}
                      step={5}
                      value={[creativity]}
                      onValueChange={(v) => setCreativity(v[0])}
                      data-testid="slider-creativity"
                    />
                    <div className="flex justify-between text-xs text-muted-foreground mt-2">
                      <span>Safe (0)</span>
                      <span className="font-medium">Current: {creativity}</span>
                      <span>Wild (100)</span>
                    </div>
                  </div>
                  
                  <div className="text-sm text-muted-foreground bg-neutral-50 p-3 rounded-lg">
                    {creativity <= 30 && "Conservative: Generates traditional, safe business names."}
                    {creativity > 30 && creativity < 70 && "Balanced creativity (60) gives short, brandable names. Lower = safer. Higher = wilder."}
                    {creativity >= 70 && "Creative: Experimental, unique names that stand out."}
                  </div>
                  
                  <button
                    onClick={resetCreativity}
                    className="text-sm text-primary hover:underline"
                    data-testid="button-reset-creativity"
                  >
                    Reset to 60 (recommended)
                  </button>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Navigation */}
          <div className="flex items-center gap-3 pt-2">
            <button
              onClick={() => setStep(1)}
              className="px-4 py-2 rounded-lg bg-white text-neutral-700 border border-neutral-300 hover:bg-neutral-50 flex items-center gap-2"
              data-testid="button-back-to-style"
            >
              <ArrowLeft className="h-4 w-4" /> Back
            </button>
            <button
              onClick={() => setStep(3)}
              className="px-4 py-2 rounded-lg bg-emerald-600 hover:bg-emerald-700 text-white flex items-center gap-2"
              data-testid="button-next-to-brand-info"
            >
              Next: Brand Info <ArrowRight className="h-4 w-4" />
            </button>
          </div>
        </section>
      )}

      {/* -------- STEP 3: Brand Info & Generate -------- */}
      {step === 3 && (
        <section className="space-y-6">
          <Card>
            <CardContent className="p-6 space-y-6">
              {/* Optional Fields */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <Label className="block text-sm font-medium text-foreground mb-2">
                    Business description (optional)
                  </Label>
                  <Input
                    type="text"
                    placeholder="One sentence about what you do or for whom"
                    value={description}
                    onChange={(e) => setDescription(e.target.value)}
                    data-testid="input-description"
                  />
                </div>
                <div>
                  <Label className="block text-sm font-medium text-foreground mb-2">
                    Starts with (optional)
                  </Label>
                  <Input
                    type="text"
                    placeholder="e.g., A, Neo, Omni"
                    value={startsWith}
                    onChange={(e) => setStartsWith(e.target.value)}
                    data-testid="input-starts-with"
                  />
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <Label className="block text-sm font-medium text-foreground mb-2">
                    Must include (comma separated)
                  </Label>
                  <Input
                    type="text"
                    placeholder="e.g., lab, cloud"
                    value={mustInclude}
                    onChange={(e) => setMustInclude(e.target.value)}
                    data-testid="input-must-include"
                  />
                </div>
                <div>
                  <Label className="block text-sm font-medium text-foreground mb-2">
                    Must NOT include (comma separated)
                  </Label>
                  <Input
                    type="text"
                    placeholder="e.g., tech, corp"
                    value={mustExclude}
                    onChange={(e) => setMustExclude(e.target.value)}
                    data-testid="input-must-exclude"
                  />
                </div>
              </div>

              {/* Name Length */}
              <div>
                <Label className="block text-sm font-medium text-foreground mb-2">
                  Name length (characters)
                </Label>
                <div className="px-1">
                  <Slider
                    min={4}
                    max={14}
                    step={1}
                    value={lengthRange}
                    onValueChange={(v) => setLengthRange([v[0], v[1]] as [number, number])}
                    data-testid="slider-name-length"
                  />
                  <div className="text-xs text-muted-foreground mt-2">
                    {lengthRange[0]} – {lengthRange[1]} chars
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Navigation & Generate */}
          <div className="flex items-center gap-3 pt-2">
            <button
              onClick={() => setStep(2)}
              className="px-4 py-2 rounded-lg bg-white text-neutral-700 border border-neutral-300 hover:bg-neutral-50 flex items-center gap-2"
              data-testid="button-back-to-creativity"
            >
              <ArrowLeft className="h-4 w-4" /> Back
            </button>
            <Button
              onClick={handleGenerateNames}
              disabled={isGenerating || !canNextFrom1}
              className="px-6 py-2 bg-primary hover:bg-primary/90 text-white flex items-center gap-2"
              data-testid="button-generate-names"
            >
              {isGenerating ? (
                <>
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Generating...
                </>
              ) : (
                <>
                  <Wand2 className="h-4 w-4" />
                  Generate Names
                </>
              )}
            </Button>
          </div>
          
          {/* Secondary hint - Step 3 only */}
          <div className="text-sm text-muted-foreground text-center mt-3">
            You can leave other fields blank — we'll still generate great names.
          </div>
        </section>
      )}

      {/* Generated Names Results */}
      {generatedNames.length > 0 && (
        <Card className="mt-8">
          <div className="p-6 border-b border-border">
            <div className="flex items-center justify-between mb-2">
              <div>
                <h3 className="text-xl font-semibold text-dark">Generated Names</h3>
                <p className="text-muted-foreground">
                  Found {generatedNames.length} business names. Click any name to explore branding options.
                </p>
              </div>
              <Button
                onClick={handleExportAll}
                variant="outline"
                size="sm"
                disabled={!tier.exportAll}
                data-testid="button-export-all"
              >
                <Download className="h-4 w-4 mr-2" />
                Export All
              </Button>
            </div>
            {!isPremium && (
              <div className="text-xs text-gray-500 mt-2 bg-gray-50 p-2 rounded">
                Showing {tier.maxResults} results. Unlock 10 results, style filters, and export with Premium.
              </div>
            )}
          </div>
          <CardContent className="p-6">
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
              {generatedNames.map((nameData, index) => (
                <div
                  key={index}
                  className="border rounded-lg p-4 hover:border-primary/50 transition-colors bg-white"
                  data-testid={`card-business-name-${index}`}
                >
                  <div className="space-y-3">
                    <div className="flex items-start justify-between">
                      <h4 
                        className="font-semibold text-lg text-dark cursor-pointer hover:text-primary"
                        onClick={() => sendToBrandKit(nameData.name, index)}
                        data-testid={`text-business-name-${index}`}
                      >
                        {nameData.name}
                      </h4>
                      <button
                        onClick={() => toggleSave(nameData)}
                        className={`p-1.5 rounded-full transition-colors ${
                          isNameSaved(nameData.name)
                            ? 'bg-red-100 text-red-600 hover:bg-red-200'
                            : 'bg-gray-100 text-gray-400 hover:bg-gray-200 hover:text-gray-600'
                        }`}
                        data-testid={`button-save-name-${index}`}
                      >
                        <Heart className="h-4 w-4" fill={isNameSaved(nameData.name) ? 'currentColor' : 'none'} />
                      </button>
                    </div>

                    {nameData.description && (
                      <p className="text-sm text-muted-foreground">{nameData.description}</p>
                    )}


                    <div className="flex flex-col gap-2 pt-2">
                      <Button
                        size="sm"
                        onClick={() => sendToBrandKit(nameData.name, index)}
                        className="bg-primary hover:bg-primary/90 text-white"
                        data-testid={`button-brand-kit-${index}`}
                      >
                        <img src={bulletCubeWhite} alt="Brand Kit" className="w-5 h-5 mr-1" />
                        Open in Brand Kit
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => goToDomains(nameData.name)}
                        className="border-gray-300 text-gray-700 hover:bg-gray-50"
                        data-testid={`button-check-domain-${index}`}
                      >
                        <Link className="h-4 w-4 mr-2" />
                        Find a Domain
                      </Button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Saved Names */}
      {savedNames.length > 0 && (
        <Card className="mt-6">
          <div className="p-6 border-b border-border">
            <h3 className="text-xl font-semibold text-dark">Saved Names</h3>
            <p className="text-muted-foreground">Your favorite business names</p>
          </div>
          <CardContent className="p-6">
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
              {savedNames.map((nameData, index) => (
                <div
                  key={index}
                  className="border rounded-lg p-4 hover:border-primary/50 transition-colors bg-white"
                  data-testid={`card-saved-name-${index}`}
                >
                  <div className="space-y-3">
                    <div className="flex items-start justify-between">
                      <h4 
                        className="font-semibold text-lg text-dark cursor-pointer hover:text-primary"
                        onClick={() => sendToBrandKit(nameData.name)}
                        data-testid={`text-saved-name-${index}`}
                      >
                        {nameData.name}
                      </h4>
                      <button
                        onClick={() => toggleSave(nameData)}
                        className="p-1.5 rounded-full transition-colors bg-red-100 text-red-600 hover:bg-red-200"
                        data-testid={`button-unsave-name-${index}`}
                      >
                        <X className="h-4 w-4" />
                      </button>
                    </div>
                    
                    <div className="flex flex-col gap-2 pt-2">
                      <Button
                        size="sm"
                        onClick={() => sendToBrandKit(nameData.name)}
                        className="bg-primary hover:bg-primary/90 text-white"
                        data-testid={`button-saved-brand-kit-${index}`}
                      >
                        <img src={bulletCubeWhite} alt="Brand Kit" className="w-5 h-5 mr-1" />
                        Open in Brand Kit
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => goToDomains(nameData.name)}
                        className="border-gray-300 text-gray-700 hover:bg-gray-50"
                        data-testid={`button-saved-check-domain-${index}`}
                      >
                        <Link className="h-4 w-4 mr-2" />
                        Find a Domain
                      </Button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Brand Chooser Dialog */}
      <Dialog open={editorOpen.index !== null} onOpenChange={() => setEditorOpen({ index: null })}>
        <DialogContent className="sm:max-w-[600px] max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Customize Brand Preview</DialogTitle>
            <DialogDescription>
              Pick colors and fonts for{' '}
              {editorOpen.index !== null 
                ? generatedNames[editorOpen.index]?.name || savedNames[editorOpen.index]?.name
                : ''
              }
            </DialogDescription>
          </DialogHeader>

          {editorOpen.index !== null && (
            <div className="space-y-6 py-4">
              <div>
                <Label className="text-sm font-medium mb-2 block">Color Palette</Label>
                <PalettePicker
                  selectedPalette={cardChoices[editorOpen.index]?.paletteIndex ?? defaultPalette(editorOpen.index)}
                  onPaletteChange={(paletteIndex) =>
                    setCardChoices(prev => ({ ...prev, [editorOpen.index!]: { ...prev[editorOpen.index!], paletteIndex } }))
                  }
                />
              </div>

              <div>
                <Label className="text-sm font-medium mb-2 block">Font Family</Label>
                <FontPicker
                  selectedFont={cardChoices[editorOpen.index]?.fontFamily}
                  onFontChange={(fontFamily) =>
                    setCardChoices(prev => ({ ...prev, [editorOpen.index!]: { ...prev[editorOpen.index!], fontFamily } }))
                  }
                />
              </div>

              <div className="pt-4 border-t">
                <LogoPreview
                  text={generatedNames[editorOpen.index]?.name || savedNames[editorOpen.index]?.name || ''}
                  palette={PALETTES[cardChoices[editorOpen.index]?.paletteIndex ?? defaultPalette(editorOpen.index)]}
                  fontFamily={cardChoices[editorOpen.index]?.fontFamily}
                  size={48}
                  showControls={false}
                />
              </div>
            </div>
          )}

          <DialogFooter>
            <Button variant="outline" onClick={() => setEditorOpen({ index: null })}>
              Close
            </Button>
            <Button
              onClick={() => {
                const name = editorOpen.index !== null 
                  ? generatedNames[editorOpen.index]?.name || savedNames[editorOpen.index]?.name
                  : '';
                if (name) sendToBrandKit(name, editorOpen.index!);
                setEditorOpen({ index: null });
              }}
            >
              <img src={bulletCubeWhite} alt="Brand Kit" className="w-6 h-6 mr-2" />
              Open Brand Kit
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}