import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ResizableHandle, ResizablePanel, ResizablePanelGroup } from "@/components/ui/resizable";
import { ScrollArea } from "@/components/ui/scroll-area";
import { 
  FileText, 
  Layout, 
  BarChart3, 
  Calculator, 
  Settings,
  PanelLeft,
  PanelRight,
  Menu,
  Palette
} from "lucide-react";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { BrandKitManager } from "@/components/BrandKitManager";
import { BrandKitPanel } from "@/components/BrandKitPanel";
import { CoverDesigner } from "@/components/CoverDesigner";
import { MarketSizeFormV2 } from "@/components/visuals/MarketSizeFormV2";
import { PricingTableFormV2 } from "@/components/visuals/PricingTableFormV2";
import { FinanceCharts } from "@/components/finance/FinanceCharts";
import { ExportBar } from "@/components/ExportBar";

export default function BuilderV7() {
  const { sections, title, updateSection, setPlanTitle } = usePlanStoreV2();
  const [activeSection, setActiveSection] = useState<string>(sections[0]?.id || "");
  const [leftPanelCollapsed, setLeftPanelCollapsed] = useState(false);
  const [rightPanelCollapsed, setRightPanelCollapsed] = useState(false);

  const activeSecData = sections.find(s => s.id === activeSection);

  const handleSectionUpdate = (content: string) => {
    if (activeSection) {
      updateSection(activeSection, (prev: any) => ({
        content: `${prev?.content ? prev.content + "\n\n" : ""}${content}`,
      }));
    }
  };

  return (
    <div className="h-screen flex flex-col" data-testid="builder-v7">
      {/* Header */}
      <div className="border-b bg-white px-4 py-3 flex items-center justify-between">
        <div className="flex items-center gap-3">
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setLeftPanelCollapsed(!leftPanelCollapsed)}
            data-testid="toggle-left-panel"
          >
            <PanelLeft className={`w-4 h-4 ${leftPanelCollapsed ? 'opacity-50' : ''}`} />
          </Button>
          <h1 className="text-lg font-semibold">AI Frameworks Builder</h1>
          <Badge variant="secondary">Brand Kit Enhanced</Badge>
        </div>
        
        <div className="flex items-center gap-2">
          <span className="text-sm text-muted-foreground">
            {sections.length} sections
          </span>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setRightPanelCollapsed(!rightPanelCollapsed)}
            data-testid="toggle-right-panel"
          >
            <PanelRight className={`w-4 h-4 ${rightPanelCollapsed ? 'opacity-50' : ''}`} />
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 overflow-hidden">
        <ResizablePanelGroup direction="horizontal" className="h-full">
          {!leftPanelCollapsed && (
            <>
              <ResizablePanel defaultSize={40} minSize={30}>
                <div className="h-full p-6 overflow-auto">
                  <div className="space-y-6">
                    <div>
                      <h2 className="text-xl font-semibold mb-4">Outline & Sections</h2>
                      <div className="space-y-4">
                        {sections.map((section) => (
                          <Card
                            key={section.id}
                            className={`cursor-pointer transition-colors ${
                              activeSection === section.id 
                                ? 'ring-2 ring-purple-500 bg-purple-50' 
                                : 'hover:bg-muted/50'
                            }`}
                            onClick={() => setActiveSection(section.id)}
                            data-testid={`section-${section.id}`}
                          >
                            <CardHeader className="pb-2">
                              <CardTitle className="text-base flex items-center gap-2">
                                <FileText className="w-4 h-4" />
                                {section.title}
                              </CardTitle>
                            </CardHeader>
                            <CardContent>
                              <p className="text-sm text-muted-foreground">
                                {section.content ? 
                                  `${section.content.slice(0, 100)}${section.content.length > 100 ? '...' : ''}` :
                                  'No content yet'
                                }
                              </p>
                            </CardContent>
                          </Card>
                        ))}
                      </div>
                    </div>

                    {/* Section Editor */}
                    {activeSecData && (
                      <div>
                        <h3 className="text-lg font-medium mb-3">
                          Editing: {activeSecData.title}
                        </h3>
                        <textarea
                          className="w-full h-48 p-3 border rounded-lg resize-none"
                          placeholder="Enter section content..."
                          value={activeSecData.content || ""}
                          onChange={(e) => updateSection(activeSection, { content: e.target.value })}
                          data-testid="section-editor"
                        />
                      </div>
                    )}
                  </div>
                </div>
              </ResizablePanel>
              <ResizableHandle withHandle />
            </>
          )}

          {!rightPanelCollapsed && (
            <ResizablePanel defaultSize={60} minSize={40}>
              <div className="h-full">
                <Tabs defaultValue="brand-manager" className="h-full flex flex-col">
                  <div className="border-b px-6 pt-4">
                    <TabsList className="grid w-full grid-cols-5 bg-orange-500">
                      <TabsTrigger value="brand-manager" className="flex items-center gap-1 text-xs">
                        <Palette className="w-3 h-3" />
                        Brand Kit Manager
                      </TabsTrigger>
                      <TabsTrigger value="brand-panel" className="flex items-center gap-1 text-xs">
                        <Settings className="w-3 h-3" />
                        Brand Kit Panel
                      </TabsTrigger>
                      <TabsTrigger value="cover" className="flex items-center gap-1 text-xs">
                        <Layout className="w-3 h-3" />
                        Cover Designer
                      </TabsTrigger>
                      <TabsTrigger value="templates" className="flex items-center gap-1 text-xs">
                        <BarChart3 className="w-3 h-3" />
                        Visual Templates
                      </TabsTrigger>
                      <TabsTrigger value="finance" className="flex items-center gap-1 text-xs">
                        <Calculator className="w-3 h-3" />
                        Financial Tools
                      </TabsTrigger>
                    </TabsList>
                  </div>

                  <div className="flex-1 overflow-auto">
                    <TabsContent value="brand-manager" className="p-6 mt-0">
                      <BrandKitManager />
                    </TabsContent>

                    <TabsContent value="brand-panel" className="p-6 mt-0">
                      <BrandKitPanel />
                    </TabsContent>

                    <TabsContent value="cover" className="p-6 mt-0">
                      <div className="space-y-6">
                        <CoverDesigner />
                        
                        <Card>
                          <CardHeader>
                            <CardTitle className="flex items-center gap-2">
                              <FileText className="w-5 h-5 text-purple-500" />
                              Export Complete Plan
                            </CardTitle>
                          </CardHeader>
                          <CardContent>
                            <ExportBar />
                          </CardContent>
                        </Card>
                      </div>
                    </TabsContent>

                    <TabsContent value="templates" className="p-6 mt-0">
                      <div className="space-y-6">
                        <Card>
                          <CardHeader>
                            <CardTitle className="flex items-center gap-2">
                              <BarChart3 className="w-5 h-5 text-blue-500" />
                              Market Analysis
                            </CardTitle>
                          </CardHeader>
                          <CardContent>
                            <MarketSizeFormV2 targetSectionId={activeSection} />
                          </CardContent>
                        </Card>

                        <Card>
                          <CardHeader>
                            <CardTitle className="flex items-center gap-2">
                              <Calculator className="w-5 h-5 text-green-500" />
                              Pricing Strategy
                            </CardTitle>
                          </CardHeader>
                          <CardContent>
                            <PricingTableFormV2 targetSectionId={activeSection} />
                          </CardContent>
                        </Card>
                      </div>
                    </TabsContent>

                    <TabsContent value="finance" className="p-6 mt-0">
                      <div className="space-y-6">
                        <FinanceCharts />
                      </div>
                    </TabsContent>
                  </div>
                </Tabs>
              </div>
            </ResizablePanel>
          )}
        </ResizablePanelGroup>
      </div>

      {/* Mobile Menu */}
      <div className="md:hidden fixed bottom-4 right-4">
        <Button
          size="sm"
          onClick={() => setRightPanelCollapsed(!rightPanelCollapsed)}
          data-testid="mobile-menu-toggle"
        >
          <Menu className="w-5 h-5" />
        </Button>
      </div>
    </div>
  );
}