import { useState, useRef } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Separator } from "@/components/ui/separator";
import { ResizablePanelGroup, ResizablePanel, ResizableHandle } from "@/components/ui/resizable";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ScrollArea } from "@/components/ui/scroll-area";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { useBrandKit } from "@/stores/useBrandKit";

// Components
import { BrandKitPanel } from "@/components/BrandKitPanel";
import { CoverDesigner } from "@/components/CoverDesigner";
import { FinanceCharts } from "@/components/finance/FinanceCharts";
import { CustomChartBuilder } from "@/components/visuals/CustomChartBuilder";
import { MarketSizeFormV2 } from "@/components/visuals/MarketSizeFormV2";
import { PricingTableFormV2 } from "@/components/visuals/PricingTableFormV2";

// Icons
import { 
  FileText, 
  Palette, 
  BarChart3, 
  TrendingUp, 
  DollarSign, 
  Target, 
  Layout,
  Settings,
  Plus,
  Edit3
} from "lucide-react";

export function BuilderV6() {
  const { sections, addSection, selectSection, selectedSectionId } = usePlanStoreV2();
  const { brand } = useBrandKit();
  const [activeRightTab, setActiveRightTab] = useState("brand");

  const selectedSection = sections.find(s => s.id === selectedSectionId);

  const renderSectionOutline = () => {
    return (
      <div className="space-y-2 p-4" data-testid="section-outline">
        <div className="flex items-center justify-between">
          <h3 className="text-sm font-semibold text-gray-700">Business Plan Sections</h3>
          <Button
            size="sm"
            variant="outline"
            onClick={() => addSection("custom", "New Section")}
            data-testid="add-section-button"
          >
            <Plus className="w-3 h-3 mr-1" />
            Add
          </Button>
        </div>

        <ScrollArea className="h-96">
          <div className="space-y-1">
            {sections.map((section, index) => (
              <div
                key={section.id}
                className={`group flex items-center p-2 rounded-lg cursor-pointer transition-colors ${
                  selectedSectionId === section.id
                    ? 'bg-blue-50 border border-blue-200'
                    : 'hover:bg-gray-50'
                }`}
                onClick={() => selectSection(section.id)}
                data-testid={`section-item-${section.id}`}
              >
                <div className="flex items-center gap-2 flex-1">
                  <span className="text-xs text-gray-400 w-6">{index + 1}.</span>
                  <FileText className="w-4 h-4 text-gray-500" />
                  <span className="text-sm font-medium text-gray-700">{section.title}</span>
                </div>
                <Edit3 className="w-3 h-3 text-gray-400 opacity-0 group-hover:opacity-100 transition-opacity" />
              </div>
            ))}
          </div>
        </ScrollArea>

        <Separator />

        <div className="text-xs text-gray-500 p-2 bg-gray-50 rounded-lg">
          <p className="font-medium mb-1">💡 Quick Tips:</p>
          <ul className="space-y-1">
            <li>• Click any section to select and edit it</li>
            <li>• Use Brand Kit to apply consistent colors</li>
            <li>• Charts will auto-match your brand colors</li>
          </ul>
        </div>
      </div>
    );
  };

  const renderBrandKitTab = () => {
    return (
      <div className="p-4 space-y-4" data-testid="brand-kit-tab">
        <div className="flex items-center gap-2">
          <Palette className="w-5 h-5 text-purple-500" />
          <h3 className="text-lg font-semibold">Brand Kit Configuration</h3>
        </div>
        
        <div className="text-sm text-gray-600 bg-purple-50 p-3 rounded-lg">
          <p className="font-medium text-purple-800">Global Brand Settings</p>
          <p className="mt-1">Configure your brand colors, fonts, and logo. These will be automatically applied to all charts, covers, and visualizations.</p>
        </div>

        <BrandKitPanel />

        {brand.name && (
          <div className="p-3 bg-green-50 border border-green-200 rounded-lg">
            <p className="text-sm font-medium text-green-800">✅ Brand Kit Active</p>
            <p className="text-xs text-green-700 mt-1">
              Your brand "{brand.name}" is applied to all components.
            </p>
          </div>
        )}
      </div>
    );
  };

  const renderCoversTab = () => {
    return (
      <div className="p-4 space-y-4" data-testid="covers-tab">
        <div className="flex items-center gap-2">
          <Layout className="w-5 h-5 text-orange-500" />
          <h3 className="text-lg font-semibold">Cover Design</h3>
        </div>

        <div className="text-sm text-gray-600 bg-orange-50 p-3 rounded-lg">
          <p className="font-medium text-orange-800">Professional Covers</p>
          <p className="mt-1">Create branded business plan covers that automatically match your Brand Kit colors and fonts.</p>
        </div>

        <CoverDesigner />
      </div>
    );
  };

  const renderChartsTab = () => {
    return (
      <div className="p-4 space-y-6" data-testid="charts-tab">
        <div className="flex items-center gap-2">
          <BarChart3 className="w-5 h-5 text-blue-500" />
          <h3 className="text-lg font-semibold">Chart & Visual Tools</h3>
        </div>

        <div className="text-sm text-gray-600 bg-blue-50 p-3 rounded-lg">
          <p className="font-medium text-blue-800">Brand-Aware Charts</p>
          <p className="mt-1">All charts automatically use your Brand Kit colors for consistent visual branding.</p>
        </div>

        {/* Custom Chart Builder */}
        <Card data-testid="custom-chart-builder-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <BarChart3 className="w-4 h-4" />
              Custom Chart Builder
            </CardTitle>
          </CardHeader>
          <CardContent>
            <CustomChartBuilder targetSectionId={selectedSectionId || ""} />
          </CardContent>
        </Card>

        {/* Market Sizing */}
        <Card data-testid="market-sizing-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <Target className="w-4 h-4" />
              Market Sizing (TAM/SAM/SOM)
            </CardTitle>
          </CardHeader>
          <CardContent>
            <MarketSizeFormV2 targetSectionId={selectedSectionId || ""} />
          </CardContent>
        </Card>
      </div>
    );
  };

  const renderFinanceTab = () => {
    return (
      <div className="p-4 space-y-6" data-testid="finance-tab">
        <div className="flex items-center gap-2">
          <DollarSign className="w-5 h-5 text-green-500" />
          <h3 className="text-lg font-semibold">Financial Tools</h3>
        </div>

        <div className="text-sm text-gray-600 bg-green-50 p-3 rounded-lg">
          <p className="font-medium text-green-800">Financial Projections</p>
          <p className="mt-1">Create branded financial charts and projections that match your brand colors.</p>
        </div>

        <Card data-testid="finance-charts-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <TrendingUp className="w-4 h-4" />
              Financial Charts
            </CardTitle>
          </CardHeader>
          <CardContent>
            <FinanceCharts />
          </CardContent>
        </Card>

        {/* Pricing Strategy */}
        <Card data-testid="pricing-strategy-card">
          <CardHeader className="pb-3">
            <CardTitle className="text-base flex items-center gap-2">
              <DollarSign className="w-4 h-4" />
              Pricing Strategy Builder
            </CardTitle>
          </CardHeader>
          <CardContent>
            <PricingTableFormV2 targetSectionId={selectedSectionId || ""} />
          </CardContent>
        </Card>
      </div>
    );
  };

  const renderSettingsTab = () => {
    return (
      <div className="p-4 space-y-4" data-testid="settings-tab">
        <div className="flex items-center gap-2">
          <Settings className="w-5 h-5 text-gray-500" />
          <h3 className="text-lg font-semibold">Settings</h3>
        </div>

        <div className="space-y-4">
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-base">Export Settings</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <div className="flex items-center justify-between">
                <span className="text-sm">Include Brand Colors in Export</span>
                <input type="checkbox" defaultChecked className="rounded" />
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">High-Resolution Charts</span>
                <input type="checkbox" defaultChecked className="rounded" />
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Include Cover Page</span>
                <input type="checkbox" defaultChecked className="rounded" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-base">Brand Kit Status</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span>Primary Color:</span>
                  <div className="flex items-center gap-2">
                    <div 
                      className="w-4 h-4 rounded border"
                      style={{ backgroundColor: brand.primary }}
                    />
                    <span className="text-gray-600">{brand.primary}</span>
                  </div>
                </div>
                <div className="flex justify-between">
                  <span>Secondary Color:</span>
                  <div className="flex items-center gap-2">
                    <div 
                      className="w-4 h-4 rounded border"
                      style={{ backgroundColor: brand.secondary || "#06B6D4" }}
                    />
                    <span className="text-gray-600">{brand.secondary || "#06B6D4"}</span>
                  </div>
                </div>
                <div className="flex justify-between">
                  <span>Accent Color:</span>
                  <div className="flex items-center gap-2">
                    <div 
                      className="w-4 h-4 rounded border"
                      style={{ backgroundColor: brand.accent || "#F59E0B" }}
                    />
                    <span className="text-gray-600">{brand.accent || "#F59E0B"}</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  };

  return (
    <div className="h-screen flex flex-col bg-gray-50" data-testid="builder-v6">
      {/* Header */}
      <div className="bg-white border-b px-6 py-4">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-xl font-semibold text-gray-900">
              IBrandBiz Business Plan Builder V6
            </h1>
            <p className="text-sm text-gray-600">
              Comprehensive business plan creation with integrated brand kit
            </p>
          </div>
          
          <div className="flex items-center gap-3">
            {brand.name && (
              <div className="flex items-center gap-2 px-3 py-1 bg-purple-100 rounded-full">
                <div 
                  className="w-3 h-3 rounded-full"
                  style={{ backgroundColor: brand.primary }}
                />
                <span className="text-sm font-medium text-purple-800">{brand.name}</span>
              </div>
            )}
            <Button variant="outline" size="sm" data-testid="export-button">
              Export Plan
            </Button>
          </div>
        </div>
      </div>

      {/* Main Layout */}
      <div className="flex-1 flex">
        <ResizablePanelGroup direction="horizontal" className="h-full">
          {/* Left Panel - Sections */}
          <ResizablePanel defaultSize={25} minSize={20}>
            <div className="h-full bg-white border-r">
              <div className="h-full flex flex-col">
                <div className="border-b p-4">
                  <h2 className="font-semibold text-gray-800">Plan Structure</h2>
                  <p className="text-xs text-gray-600 mt-1">
                    Select a section to edit its content
                  </p>
                </div>
                <div className="flex-1 overflow-hidden">
                  {renderSectionOutline()}
                </div>
              </div>
            </div>
          </ResizablePanel>

          <ResizableHandle />

          {/* Center Panel - Content Editor */}
          <ResizablePanel defaultSize={50} minSize={30}>
            <div className="h-full bg-white flex flex-col">
              <div className="border-b p-4">
                <h2 className="font-semibold text-gray-800">
                  {selectedSection ? selectedSection.title : "Select a Section"}
                </h2>
                <p className="text-xs text-gray-600 mt-1">
                  {selectedSection 
                    ? "Edit content and add visuals using the tools panel" 
                    : "Choose a section from the left to start editing"}
                </p>
              </div>
              
              <div className="flex-1 p-6 overflow-auto">
                {selectedSection ? (
                  <div className="space-y-4" data-testid="section-editor">
                    <div className="prose max-w-none">
                      <div 
                        className="min-h-32 border rounded-lg p-4 whitespace-pre-wrap"
                        style={{ fontFamily: brand.fontBody }}
                      >
                        {selectedSection.content || "Start writing your content here..."}
                      </div>
                    </div>
                  </div>
                ) : (
                  <div className="flex items-center justify-center h-full text-gray-500">
                    <div className="text-center">
                      <FileText className="w-12 h-12 mx-auto mb-4 text-gray-300" />
                      <p>Select a section to begin editing</p>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </ResizablePanel>

          <ResizableHandle />

          {/* Right Panel - Tools */}
          <ResizablePanel defaultSize={25} minSize={20}>
            <div className="h-full bg-gray-50 border-l">
              <Tabs value={activeRightTab} onValueChange={setActiveRightTab} className="h-full flex flex-col">
                <TabsList className="grid w-full grid-cols-5 rounded-none border-b bg-white">
                  <TabsTrigger value="brand" className="text-xs" data-testid="brand-tab-trigger">
                    <Palette className="w-3 h-3" />
                  </TabsTrigger>
                  <TabsTrigger value="covers" className="text-xs" data-testid="covers-tab-trigger">
                    <Layout className="w-3 h-3" />
                  </TabsTrigger>
                  <TabsTrigger value="charts" className="text-xs" data-testid="charts-tab-trigger">
                    <BarChart3 className="w-3 h-3" />
                  </TabsTrigger>
                  <TabsTrigger value="finance" className="text-xs" data-testid="finance-tab-trigger">
                    <DollarSign className="w-3 h-3" />
                  </TabsTrigger>
                  <TabsTrigger value="settings" className="text-xs" data-testid="settings-tab-trigger">
                    <Settings className="w-3 h-3" />
                  </TabsTrigger>
                </TabsList>

                <div className="flex-1 overflow-hidden">
                  <ScrollArea className="h-full">
                    <TabsContent value="brand" className="mt-0">
                      {renderBrandKitTab()}
                    </TabsContent>

                    <TabsContent value="covers" className="mt-0">
                      {renderCoversTab()}
                    </TabsContent>

                    <TabsContent value="charts" className="mt-0">
                      {renderChartsTab()}
                    </TabsContent>

                    <TabsContent value="finance" className="mt-0">
                      {renderFinanceTab()}
                    </TabsContent>

                    <TabsContent value="settings" className="mt-0">
                      {renderSettingsTab()}
                    </TabsContent>
                  </ScrollArea>
                </div>
              </Tabs>
            </div>
          </ResizablePanel>
        </ResizablePanelGroup>
      </div>
    </div>
  );
}