import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ResizableHandle, ResizablePanel, ResizablePanelGroup } from "@/components/ui/resizable";
import { ScrollArea } from "@/components/ui/scroll-area";
import { 
  FileText, 
  Layout, 
  BarChart3, 
  Calculator, 
  Settings,
  PanelLeft,
  PanelRight,
  Menu
} from "lucide-react";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { CoverDesigner } from "@/components/CoverDesigner";
import { MarketSizeFormV2 } from "@/components/visuals/MarketSizeFormV2";
import { PricingTableFormV2 } from "@/components/visuals/PricingTableFormV2";
import { FinanceCharts } from "@/components/finance/FinanceCharts";

export function BuilderV5() {
  const { sections, title, updateSection, setPlanTitle } = usePlanStoreV2();
  const [activeSection, setActiveSection] = useState<string>(sections[0]?.id || "");
  const [leftPanelCollapsed, setLeftPanelCollapsed] = useState(false);
  const [rightPanelCollapsed, setRightPanelCollapsed] = useState(false);

  const activeSecData = sections.find(s => s.id === activeSection);

  const handleSectionUpdate = (content: string) => {
    if (activeSection) {
      updateSection(activeSection, (prev: any) => ({
        content: `${prev?.content ? prev.content + "\n\n" : ""}${content}`,
      }));
    }
  };

  return (
    <div className="h-screen flex flex-col" data-testid="builder-v5">
      {/* Header */}
      <div className="border-b bg-white px-4 py-3 flex items-center justify-between">
        <div className="flex items-center gap-3">
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setLeftPanelCollapsed(!leftPanelCollapsed)}
            data-testid="toggle-left-panel"
          >
            <PanelLeft className={`w-4 h-4 ${leftPanelCollapsed ? 'opacity-50' : ''}`} />
          </Button>
          <h1 className="text-lg font-semibold">Business Plan Builder V5</h1>
          <Badge variant="secondary">Enhanced</Badge>
        </div>
        
        <div className="flex items-center gap-2">
          <span className="text-sm text-muted-foreground">
            {sections.length} sections
          </span>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setRightPanelCollapsed(!rightPanelCollapsed)}
            data-testid="toggle-right-panel"
          >
            <PanelRight className={`w-4 h-4 ${rightPanelCollapsed ? 'opacity-50' : ''}`} />
          </Button>
        </div>
      </div>

      {/* Main Layout */}
      <div className="flex-1 overflow-hidden">
        <ResizablePanelGroup direction="horizontal" className="h-full">
          {/* Left Panel - Outline/Sections Editor */}
          {!leftPanelCollapsed && (
            <>
              <ResizablePanel defaultSize={25} minSize={20} maxSize={40}>
                <div className="h-full border-r bg-gray-50/50">
                  <div className="p-4 border-b bg-white">
                    <h3 className="font-medium flex items-center gap-2">
                      <FileText className="w-4 h-4" />
                      Plan Outline
                    </h3>
                  </div>
                  
                  <ScrollArea className="h-[calc(100%-60px)]">
                    <div className="p-4 space-y-2">
                      {/* Plan Title */}
                      <div className="mb-4">
                        <input
                          type="text"
                          value={title}
                          onChange={(e) => setPlanTitle(e.target.value)}
                          className="w-full font-medium bg-transparent border-none outline-none text-base"
                          placeholder="Untitled Business Plan"
                          data-testid="plan-title-input"
                        />
                      </div>

                      {/* Sections List */}
                      {sections.map((section) => (
                        <div key={section.id}>
                          <Button
                            variant={activeSection === section.id ? "secondary" : "ghost"}
                            className="w-full justify-start text-left h-auto p-3"
                            onClick={() => setActiveSection(section.id)}
                            data-testid={`section-${section.id}`}
                          >
                            <div className="flex-1 min-w-0">
                              <div className="font-medium truncate">{section.title}</div>
                              <div className="text-xs text-muted-foreground mt-1">
                                {section.content ? `${section.content.length} chars` : 'Empty'}
                              </div>
                            </div>
                          </Button>

                          {/* Child Sections */}
                          {section.children && section.children.length > 0 && (
                            <div className="ml-4 mt-1 space-y-1">
                              {section.children.map((child) => (
                                <Button
                                  key={child.id}
                                  variant="ghost"
                                  size="sm"
                                  className="w-full justify-start text-left"
                                  onClick={() => setActiveSection(child.id)}
                                  data-testid={`child-section-${child.id}`}
                                >
                                  <div className="text-sm truncate">{child.title}</div>
                                </Button>
                              ))}
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  </ScrollArea>
                </div>
              </ResizablePanel>
              <ResizableHandle />
            </>
          )}

          {/* Center Panel - Content Editor */}
          <ResizablePanel defaultSize={leftPanelCollapsed ? 70 : 50} minSize={30}>
            <div className="h-full bg-white">
              {activeSecData ? (
                <div className="h-full flex flex-col">
                  <div className="p-4 border-b">
                    <h2 className="text-xl font-semibold">{activeSecData.title}</h2>
                    <p className="text-sm text-muted-foreground mt-1">
                      {activeSecData.kind} • {activeSecData.content.length} characters
                    </p>
                  </div>
                  
                  <ScrollArea className="flex-1">
                    <div className="p-4">
                      <textarea
                        value={activeSecData.content}
                        onChange={(e) => updateSection(activeSection, { content: e.target.value })}
                        className="w-full h-96 p-4 border rounded-lg resize-none focus:outline-none focus:ring-2 focus:ring-orange-500"
                        placeholder={`Write your ${activeSecData.title.toLowerCase()} content here...`}
                        data-testid="section-content-editor"
                      />
                    </div>
                  </ScrollArea>
                </div>
              ) : (
                <div className="h-full flex items-center justify-center">
                  <div className="text-center">
                    <FileText className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                    <h3 className="text-lg font-medium mb-2">No Section Selected</h3>
                    <p className="text-muted-foreground">Select a section from the left panel to start editing</p>
                  </div>
                </div>
              )}
            </div>
          </ResizablePanel>

          {/* Right Panel - Tools & Templates */}
          {!rightPanelCollapsed && (
            <>
              <ResizableHandle />
              <ResizablePanel defaultSize={25} minSize={20} maxSize={40}>
                <div className="h-full border-l bg-gray-50/50">
                  <Tabs defaultValue="cover" className="h-full">
                    <div className="p-2 border-b bg-white">
                      <TabsList className="grid w-full grid-cols-3">
                        <TabsTrigger value="cover" className="text-xs">
                          <Layout className="w-3 h-3 mr-1" />
                          Cover
                        </TabsTrigger>
                        <TabsTrigger value="visuals" className="text-xs">
                          <BarChart3 className="w-3 h-3 mr-1" />
                          Visuals
                        </TabsTrigger>
                        <TabsTrigger value="finance" className="text-xs">
                          <Calculator className="w-3 h-3 mr-1" />
                          Finance
                        </TabsTrigger>
                      </TabsList>
                    </div>

                    <ScrollArea className="h-[calc(100%-60px)]">
                      {/* Cover & Theme Tab */}
                      <TabsContent value="cover" className="p-4 mt-0">
                        <CoverDesigner
                          onConfigChange={(config) => console.log('Cover config updated:', config)}
                          onExport={(dataUrl) => console.log('Cover exported:', dataUrl)}
                        />
                      </TabsContent>

                      {/* Visual Templates Tab */}
                      <TabsContent value="visuals" className="p-4 mt-0 space-y-6">
                        <Card data-testid="market-size-section">
                          <CardHeader className="pb-3">
                            <CardTitle className="text-base">Market Sizing</CardTitle>
                          </CardHeader>
                          <CardContent>
                            <MarketSizeFormV2
                              onInsert={handleSectionUpdate}
                              targetSectionId={activeSection}
                            />
                          </CardContent>
                        </Card>

                        <Card data-testid="pricing-table-section">
                          <CardHeader className="pb-3">
                            <CardTitle className="text-base">Pricing Table</CardTitle>
                          </CardHeader>
                          <CardContent>
                            <PricingTableFormV2
                              onInsert={handleSectionUpdate}
                              targetSectionId={activeSection}
                            />
                          </CardContent>
                        </Card>
                      </TabsContent>

                      {/* Finance Tools Tab */}
                      <TabsContent value="finance" className="p-4 mt-0">
                        <Card data-testid="finance-charts-section">
                          <CardHeader className="pb-3">
                            <CardTitle className="text-base">Financial Charts</CardTitle>
                          </CardHeader>
                          <CardContent>
                            <FinanceCharts />
                          </CardContent>
                        </Card>
                      </TabsContent>
                    </ScrollArea>
                  </Tabs>
                </div>
              </ResizablePanel>
            </>
          )}
        </ResizablePanelGroup>
      </div>

      {/* Mobile Menu Button */}
      <div className="md:hidden fixed bottom-4 right-4">
        <Button
          size="lg"
          className="rounded-full h-12 w-12 shadow-lg"
          onClick={() => setRightPanelCollapsed(!rightPanelCollapsed)}
          data-testid="mobile-menu-toggle"
        >
          <Menu className="w-5 h-5" />
        </Button>
      </div>
    </div>
  );
}