import { useState, useEffect } from 'react';
import { Textarea } from '@/components/ui/textarea';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { OutlineEditorV2 } from '@/components/business-plan/OutlineEditor.v2';
import { TemplatesPanel } from '@/components/visuals/TemplatesPanel';
import { FinancialsPanel } from '@/components/finance/FinancialsPanel';
import { AIButton } from '@/components/ai/AIButton';
import { DashboardTemplatePage } from '@/components/DashboardTemplatePage';
import { FileText, Edit3, Layout, Eye, Calculator, Sparkles } from 'lucide-react';
import { usePlanStoreV2 } from '@/stores/usePlanStore.v2';
import type { SectionId } from '@shared/plan-types';

export default function BuilderV3() {
  const [selectedSectionId, setSelectedSectionId] = useState<string>('');
  const [activeTab, setActiveTab] = useState<'editor' | 'preview'>('editor');
  const [rightPanelTab, setRightPanelTab] = useState<'templates' | 'finance'>('templates');
  
  const { sections, updateSection, title } = usePlanStoreV2();
  
  const selectedSection = selectedSectionId 
    ? sections.find(s => s.id === selectedSectionId)
    : null;

  // Auto-select first section if none selected and sections exist
  useEffect(() => {
    if (!selectedSectionId && sections.length > 0) {
      setSelectedSectionId(sections[0].id);
    }
  }, [selectedSectionId, sections]);

  const handleSectionSelect = (id: SectionId) => {
    setSelectedSectionId(id);
    setActiveTab('editor'); // Switch to editor when selecting a section
  };

  const handleContentChange = (content: string) => {
    if (selectedSection) {
      updateSection(selectedSection.id, { content });
    }
  };

  const getSectionIcon = (kind?: string) => {
    switch (kind) {
      case 'executive-summary':
        return '📊';
      case 'company-overview':
        return '🏢';
      case 'products-services':
        return '💼';
      case 'market-analysis':
        return '📈';
      case 'marketing-sales':
        return '📢';
      case 'operations':
        return '⚙️';
      case 'org-management':
        return '👥';
      case 'financials':
        return '💰';
      default:
        return '📄';
    }
  };

  const renderPreview = (content: string) => {
    if (!content.trim()) {
      return <p className="text-muted-foreground text-sm">No content to preview...</p>;
    }

    // Simple markdown-to-HTML conversion for preview
    const html = content
      .replace(/\n\n/g, '</p><p>')
      .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
      .replace(/\*(.*?)\*/g, '<em>$1</em>')
      .replace(/^### (.*$)/gm, '<h3 class="text-lg font-semibold mt-4 mb-2">$1</h3>')
      .replace(/^## (.*$)/gm, '<h2 class="text-xl font-semibold mt-6 mb-3">$1</h2>')
      .replace(/^# (.*$)/gm, '<h1 class="text-2xl font-bold mt-8 mb-4">$1</h1>')
      .replace(/^\| (.*) \|$/gm, '<table class="border-collapse border border-gray-300 mt-2 mb-2"><tr><td class="border border-gray-300 px-2 py-1">$1</td></tr></table>')
      .replace(/^- (.*)$/gm, '<ul class="list-disc ml-6 my-2"><li>$1</li></ul>');

    return (
      <div 
        className="prose prose-sm max-w-none"
        dangerouslySetInnerHTML={{ __html: `<div>${html}</div>` }} 
      />
    );
  };

  return (
    <DashboardTemplatePage title="Business Plan Builder V3">
      {/* Description */}
      <div className="mb-6">
        <p className="text-muted-foreground">
          Create comprehensive business plans with enhanced visual templates, structured sections, and integrated financial modeling.
        </p>
      </div>

      <div className="grid lg:grid-cols-5 gap-6 h-[calc(100vh-280px)] max-h-[900px]" data-testid="builder-v3-layout">
        {/* LEFT PANEL - Outline + Section Editor (3 columns) */}
        <div className="lg:col-span-3 grid grid-cols-1 lg:grid-cols-2 gap-4">
          {/* Left Sub-Panel - Outline Editor */}
          <div className="flex flex-col">
            <OutlineEditorV2 
              selectedSectionId={selectedSectionId}
              onSectionSelect={handleSectionSelect}
            />
          </div>

          {/* Right Sub-Panel - Section Editor */}
          <div className="flex flex-col">
            {selectedSection ? (
              selectedSection.kind === 'financials' ? (
                // Financial Section - Show compact version with link to full interface
                <Card className="h-full flex flex-col" data-testid="financial-section-preview">
                  <CardHeader className="pb-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <span className="text-xl" role="img" aria-label="section-icon">
                          {getSectionIcon(selectedSection.kind)}
                        </span>
                        <div>
                          <CardTitle className="text-lg">{selectedSection.title}</CardTitle>
                          <Badge variant="outline" className="text-xs">
                            Financial Planning
                          </Badge>
                        </div>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent className="flex-1 flex flex-col items-center justify-center text-center space-y-4">
                    <div className="text-muted-foreground">
                      <Calculator className="w-12 h-12 mx-auto mb-3 text-orange-500" />
                      <p className="text-sm">
                        Financial planning tools are available in the Finance tab on the right panel.
                      </p>
                    </div>
                    <Button
                      onClick={() => setRightPanelTab('finance')}
                      className="bg-orange-500 hover:bg-orange-600"
                      data-testid="switch-to-finance-button"
                    >
                      <Calculator className="w-4 h-4 mr-2" />
                      Open Finance Tools
                    </Button>
                  </CardContent>
                </Card>
              ) : (
                // Regular Section Editor
                <Card className="h-full flex flex-col" data-testid="section-editor">
                  <CardHeader className="pb-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <span className="text-xl" role="img" aria-label="section-icon">
                          {getSectionIcon(selectedSection.kind)}
                        </span>
                        <div>
                          <CardTitle className="text-lg">{selectedSection.title}</CardTitle>
                          <div className="flex items-center gap-2 mt-1">
                            <Badge variant="outline" className="text-xs">
                              {selectedSection.kind}
                            </Badge>
                            {selectedSection.content && (
                              <Badge variant="secondary" className="text-xs">
                                {selectedSection.content.length} chars
                              </Badge>
                            )}
                          </div>
                        </div>
                      </div>
                    </div>

                    <Tabs value={activeTab} onValueChange={(value) => setActiveTab(value as typeof activeTab)}>
                      <TabsList className="grid w-full grid-cols-2 bg-orange-500 text-white">
                        <TabsTrigger value="editor" data-testid="tab-editor">
                          <Edit3 className="w-4 h-4 mr-2" />
                          Editor
                        </TabsTrigger>
                        <TabsTrigger value="preview" data-testid="tab-preview">
                          <Eye className="w-4 h-4 mr-2" />
                          Preview
                        </TabsTrigger>
                      </TabsList>
                    </Tabs>
                  </CardHeader>

                  <CardContent className="flex-1 overflow-auto">
                    <Tabs value={activeTab} onValueChange={(value) => setActiveTab(value as typeof activeTab)}>
                      {/* Editor Tab */}
                      <TabsContent value="editor" className="h-full">
                        <div className="h-full flex flex-col" data-testid="content-editor">
                          <Textarea
                            value={selectedSection.content}
                            onChange={(e) => handleContentChange(e.target.value)}
                            placeholder={`Write content for ${selectedSection.title}...

You can use Markdown formatting:
**Bold text**
*Italic text*
# Heading 1
## Heading 2
### Heading 3

Or use the Visual Templates on the right to insert structured content.`}
                            className="flex-1 min-h-[300px] resize-none font-mono text-sm"
                            data-testid="textarea-content"
                          />
                          
                          {/* AI Assistant Section */}
                          <div className="flex items-center justify-between p-3 mt-3 border rounded-lg bg-muted/20">
                            <div className="flex-1">
                              <h4 className="text-sm font-medium mb-1 flex items-center gap-2">
                                <Sparkles className="w-4 h-4 text-orange-500" />
                                AI Writing Assistant
                              </h4>
                              <p className="text-xs text-muted-foreground">
                                Generate, rephrase, expand, or summarize content with AI
                              </p>
                            </div>
                            <AIButton
                              sectionKind={selectedSection.kind}
                              sectionTitle={selectedSection.title}
                              existingContent={selectedSection.content}
                              onContentUpdate={handleContentChange}
                              context={{
                                businessName: title !== "Untitled Business Plan" ? title : undefined,
                              }}
                            />
                          </div>
                        </div>
                      </TabsContent>

                      {/* Preview Tab */}
                      <TabsContent value="preview" className="h-full">
                        <div className="h-full overflow-auto p-4 border rounded-lg bg-background" data-testid="content-preview">
                          {renderPreview(selectedSection.content)}
                        </div>
                      </TabsContent>
                    </Tabs>
                  </CardContent>
                </Card>
              )
            ) : (
              // No Section Selected
              <Card className="h-full flex flex-col items-center justify-center text-center" data-testid="no-section-selected">
                <CardContent className="space-y-4">
                  <FileText className="w-16 h-16 mx-auto text-muted-foreground" />
                  <div>
                    <h3 className="text-lg font-semibold">Select a Section</h3>
                    <p className="text-sm text-muted-foreground">
                      Choose a section from the outline to start editing content
                    </p>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        </div>

        {/* RIGHT PANEL - Visual Templates + Finance Tools (2 columns) */}
        <div className="lg:col-span-2">
          <Tabs 
            value={rightPanelTab} 
            onValueChange={(value) => setRightPanelTab(value as typeof rightPanelTab)}
            className="h-full flex flex-col"
            data-testid="right-panel-tabs"
          >
            <TabsList className="grid w-full grid-cols-2 mb-4">
              <TabsTrigger value="templates" className="flex items-center gap-2" data-testid="templates-tab">
                <Layout className="w-4 h-4" />
                Templates
              </TabsTrigger>
              <TabsTrigger value="finance" className="flex items-center gap-2" data-testid="finance-tab">
                <Calculator className="w-4 h-4" />
                Finance
              </TabsTrigger>
            </TabsList>
            
            <div className="flex-1 overflow-auto">
              {/* Visual Templates Panel */}
              <TabsContent value="templates" className="h-full mt-0" data-testid="templates-panel-content">
                <TemplatesPanel 
                  selectedSectionId={selectedSectionId}
                  onSectionChange={handleSectionSelect}
                />
              </TabsContent>

              {/* Financial Tools Panel */}
              <TabsContent value="finance" className="h-full mt-0" data-testid="finance-panel-content">
                <div className="h-full overflow-auto">
                  <FinancialsPanel />
                </div>
              </TabsContent>
            </div>
          </Tabs>
        </div>
      </div>

      {/* Help Text */}
      <Card className="mt-6 border-dashed" data-testid="help-instructions">
        <CardContent className="pt-4">
          <div className="text-sm text-muted-foreground space-y-2">
            <p><strong>Builder V3 Features:</strong></p>
            <ul className="list-disc list-inside space-y-1 ml-2 text-xs">
              <li><strong>Left Panel:</strong> Drag-and-drop outline editor + section content editor with AI assistance</li>
              <li><strong>Right Panel:</strong> Visual templates (SWOT, Personas, Market Sizing, Pricing) + Financial modeling tools</li>
              <li><strong>Visual Templates:</strong> Create structured content with guided forms and insert into any section</li>
              <li><strong>Finance Tools:</strong> Comprehensive financial planning with charts, scenarios, and AI insights</li>
              <li><strong>AI Integration:</strong> Content generation, expansion, and refinement for all sections</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </DashboardTemplatePage>
  );
}