// src/pages/BrandKit/BrandKitPage.tsx - Enhanced LOOKA-style wizard
import React, { useMemo, useState, useEffect, useRef } from "react";
import { withPro } from "@/lib/withPro";
import { ColorSwatch } from "@/components/ColorSwatch";
import {
  rollAll,
  rollOne,
  type Palette,
  type PaletteKey,
  type PalettePreset,
  applyPresetToPalette,
  rollWithinPreset,
} from "@/services/brand/palette";
import { FontPicker } from "@/components/FontPicker";
import RecraftGenerator from "@/components/RecraftGenerator";
import PerLetterColorText from "@/components/PerLetterColorText";
import {
  Download,
  Package,
  Wand2,
  MessageSquare,
  Copy,
  Check,
  ChevronRight,
  Palette as PaletteIcon,
  Monitor,
  CreditCard,
  AppWindow,
} from "lucide-react";
import { generate as generateSlogans, type SloganRequest } from "@/services/ai/slogan";
import { Breadcrumb, BreadcrumbList, BreadcrumbItem, BreadcrumbLink, BreadcrumbPage, BreadcrumbSeparator } from "@/components/ui/breadcrumb";
import { useToast } from "@/hooks/use-toast";
import { featureFlags } from "@/lib/featureFlags";
import JSZip from "jszip";
import BusinessCardMock from "@/components/mockups/BusinessCardMock";
import HeaderMock from "@/components/mockups/HeaderMock";
import AppIconMock from "@/components/mockups/AppIconMock";
import { BrandTokens } from "@/components/mockups/types";
import bulletCubeWhiteIcon from "@/assets/bullet-cube-white.svg";
import { createBrandKitPDF } from "@/services/export/brandKitPdf";
import jsPDF from "jspdf";
import { registerEmbeddedFonts, pdfFontFor } from "@/services/export/embedFonts";
import { composeLogoSvg } from "@/services/export/composeSvg";
import { renderLogoPNG, renderLogoJPG } from "@/services/export/logoRaster";

// ---------- Types ----------
type TextColorChoice = "black" | PaletteKey;
type LayoutOption = "side-by-side" | "top-bottom";
type WizardStep = 1 | 2 | 3 | 4;

// ---------- Seed data ----------
const POPULAR_INDUSTRIES = [
  "Restaurant", "Consulting", "Beauty", "Photography", "Fitness",
  "Real Estate", "Tech Startup", "E-commerce", "Education", "Healthcare",
  "Finance", "Non-profit", "Construction", "Marketing", "Coaching",
];

type StyleCard = { id: string; title: string; subtitle?: string; hex: string; glyph: string };
const STYLE_CARDS: StyleCard[] = [
  { id: "balanced", title: "Balanced", subtitle: "Geometric + Wordmark + Badge", hex: "#111827", glyph: "square" },
  { id: "minimal", title: "Minimal", subtitle: "Clean lines, lots of space", hex: "#1f2937", glyph: "pipe" },
  { id: "bold", title: "Bold", subtitle: "Heavy weight, strong shapes", hex: "#3f38c9", glyph: "square-filled" },
  { id: "monogram", title: "Monogram", subtitle: "Letter-focused marks", hex: "#085f4e", glyph: "A" },
  { id: "badge", title: "Badge", subtitle: "Emblem styles", hex: "#075985", glyph: "circle" },
  { id: "playful", title: "Playful", subtitle: "Rounded, friendly", hex: "#be185d", glyph: "dots" },
  { id: "elegant", title: "Elegant", subtitle: "Serif, luxury vibes", hex: "#2b241f", glyph: "italic-E" },
  { id: "techy", title: "Techy", subtitle: "Futuristic, angular", hex: "#0e5161", glyph: "diamond" },
  { id: "organic", title: "Organic", subtitle: "Nature, hand-drawn", hex: "#0f5f2b", glyph: "oval" },
];

// LOOKA-style quick palette tiles (6 swatches per tile)
type PaletteTile = {
  id: string;
  name: string;
  swatches: [string, string, string, string, string, string]; // primary, secondary, accent, highlight, neutral, surface
  textLight?: string;
  textDark?: string;
};
const PALETTE_TILES: PaletteTile[] = [
  { id: "warm-1", name: "Warm 1", swatches: ["#D94841","#C15C2E","#F29F05","#F2C14E","#F3F4F6","#FFFFFF"], textLight:"#F8FAFC", textDark:"#111827" },
  { id: "cool-1", name: "Cool 1", swatches: ["#2563EB","#0EA5E9","#10B981","#A7F3D0","#F3F4F6","#FFFFFF"], textLight:"#F8FAFC", textDark:"#111827" },
  { id: "earth-1", name: "Earth", swatches: ["#6B7280","#92400E","#A16207","#65A30D","#E5E7EB","#FFFFFF"], textLight:"#F8FAFC", textDark:"#111827" },
  { id: "pastel-1", name: "Pastel", swatches: ["#A78BFA","#F472B6","#60A5FA","#86EFAC","#F3F4F6","#FFFFFF"], textLight:"#FDFDFE", textDark:"#111827" },
  { id: "mono-ink", name: "Mono Ink", swatches: ["#111827","#374151","#2563EB","#9CA3AF","#E5E7EB","#FFFFFF"], textLight:"#F8FAFC", textDark:"#0B1220" },
  { id: "neon-1", name: "Neon", swatches: ["#22D3EE","#22C55E","#EAB308","#A855F7","#111827","#0B1220"], textLight:"#F8FAFC", textDark:"#0B1220" },
];

// ---------- Helpers ----------
const tileToPalette = (tile: PaletteTile): Palette => ({
  primary: tile.swatches[0],
  secondary: tile.swatches[1],
  accent: tile.swatches[2],
  highlight: tile.swatches[3],
  neutral: tile.swatches[4],
  surface: tile.swatches[5],
  textLight: tile.textLight ?? "#F8FAFC",
  textDark: tile.textDark ?? "#1F2937",
});

function TileGlyph({ type }: { type?: string }) {
  return (
    <svg
      aria-hidden
      className="h-8 w-8 opacity-60"
      viewBox="0 0 24 24"
      fill="none"
      stroke="currentColor"
    >
      {type === "square" && <rect x="6" y="6" width="12" height="12" rx="2" strokeWidth="1.5" />}
      {type === "square-filled" && <rect x="7" y="7" width="10" height="10" rx="2" strokeWidth="0" fill="currentColor" />}
      {type === "pipe" && (
        <>
          <line x1="12" y1="6" x2="12" y2="14" strokeWidth="1.5" />
          <line x1="7" y1="17" x2="17" y2="17" strokeWidth="1.5" />
        </>
      )}
      {type === "circle" && <circle cx="12" cy="12" r="3.5" strokeWidth="1.5" />}
      {type === "dots" && (
        <>
          <circle cx="9" cy="12" r="1.2" />
          <circle cx="12" cy="12" r="1.2" />
          <circle cx="15" cy="12" r="1.2" />
        </>
      )}
      {type === "A" && (
        <text x="12" y="14" textAnchor="middle" fontSize="10" fontWeight="700" fill="currentColor">A</text>
      )}
      {type === "italic-E" && (
        <text x="12" y="14" textAnchor="middle" fontSize="10" fontStyle="italic" fill="currentColor">E</text>
      )}
      {type === "diamond" && <polygon points="12,8 16,12 12,16 8,12" strokeWidth="1.5" />}
      {type === "oval" && <ellipse cx="12" cy="12" rx="3.5" ry="5.5" strokeWidth="1.5" />}
    </svg>
  );
}

const StyleTile: React.FC<{
  card: StyleCard;
  active: boolean;
  onToggle: () => void;
}> = ({ card, active, onToggle }) => (
  <button
    onClick={onToggle}
    aria-pressed={active}
    className={`relative w-full text-left rounded-2xl border transition p-5
      ${active ? "border-indigo-500 shadow-[0_0_0_3px_rgba(99,102,241,0.25)]"
               : "border-neutral-200 hover:border-neutral-300"}`}
  >
    <div
      className="relative h-28 rounded-xl mb-4 flex items-center justify-center text-white/80"
      style={{ backgroundColor: card.hex }}
    >
      <TileGlyph type={card.glyph} />
    </div>
    <div className="font-medium">{card.title}</div>
    {card.subtitle && <div className="text-xs text-neutral-500">{card.subtitle}</div>}
    {active && (
      <div className="absolute top-3 right-3 rounded-full bg-white border border-indigo-500 p-1">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
          <path d="M20 6L9 17l-5-5" stroke="#4f46e5" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" />
        </svg>
      </div>
    )}
  </button>
);

// ---------- Page ----------
function BrandKitPage() {
  // Wizard
  const [step, setStep] = useState<WizardStep>(1);

  // Basics
  const [name, setName] = useState("");
  const canNextFrom1 = name.trim().length > 1;

  // Industry + Style inspo
  const [industry, setIndustry] = useState("");
  const [selectedStyles, setSelectedStyles] = useState<string[]>([]);
  const canNextFrom2 = industry.trim().length > 0;

  // Slogans
  const [slogans, setSlogans] = useState<string[]>([]);
  const [isGeneratingSlogans, setIsGeneratingSlogans] = useState(false);
  const [sloganError, setSloganError] = useState<string | null>(null);
  const { toast } = useToast();

  // Colors
  const [palette, setPalette] = useState<Palette>({
    primary: "#4274b9",
    secondary: "#b94c42",
    accent: "#42afb9",
    highlight: "#a78bfa",
    neutral: "#f3f4f6",
    surface: "#ffffff",
    textLight: "#f8fafc",
    textDark: "#1f2937",
  });
  const [locks, setLocks] = useState<Record<PaletteKey, boolean>>({
    primary: false,
    secondary: false,
    accent: false,
    highlight: false,
    neutral: true,
    surface: true,
    textLight: true,
    textDark: true,
  });
  const [preset, setPreset] = useState<PalettePreset>("warm");

  // Font + text color
  const [fontFamily, setFontFamily] = useState<string>(
    '"Inter", system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif'
  );
  const [textColor, setTextColor] = useState<TextColorChoice>("primary");
  const resolveTextColor = () => (textColor === "black" ? "#000000" : palette[textColor]);

  // Logo composition
  const [generatedSvgUrl, setGeneratedSvgUrl] = useState<string | null>(null);
  const [layout, setLayout] = useState<LayoutOption>("side-by-side");

  // Size controls
  const [textSize, setTextSize] = useState<number>(72);
  const [iconSize, setIconSize] = useState<number>(120);

  // Brand kit download
  const [isDownloadingKit, setIsDownloadingKit] = useState(false);

  const perLetterRef = useRef<any>(null);

  // Pre-fill from URL
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const businessName = urlParams.get("name");
    if (businessName) setName(decodeURIComponent(businessName));
    const ind = urlParams.get("industry");
    if (ind) setIndustry(decodeURIComponent(ind));
  }, []);

  // --- Slogans ---
  const handleGenerateSlogans = async () => {
    if (!name.trim()) {
      toast({
        title: "Brand name required",
        description: "Please enter a brand name first",
        variant: "destructive",
      });
      return;
    }
    setIsGeneratingSlogans(true);
    setSloganError(null);
    setSlogans([]);
    try {
      const request: SloganRequest = {
        brandName: name.trim(),
        tone: "Professional",
        maxResults: 3,
      };
      const response = await generateSlogans(request);
      setSlogans(response.suggestions);
    } catch (err) {
      console.error("Failed to generate slogans:", err);
      setSloganError("Failed to generate slogans. Please try again.");
    } finally {
      setIsGeneratingSlogans(false);
    }
  };
  const handleCopySlogan = async (slogan: string) => {
    try {
      await navigator.clipboard.writeText(slogan);
      toast({ title: "Copied!", description: "Slogan copied to clipboard" });
    } catch {
      toast({ title: "Failed to copy", description: "Could not copy to clipboard", variant: "destructive" });
    }
  };

  // --- Palette helpers ---
  const rollAllUnlocked = () => {
    const all = rollAll(palette);
    const next: Palette = { ...palette };
    (Object.keys(next) as PaletteKey[]).forEach((k) => {
      if (!locks[k]) next[k] = (all as any)[k];
    });
    setPalette(next);
  };
  const rollSingle = (key: PaletteKey) => { if (!locks[key]) setPalette((p) => rollOne(p, key)); };
  const toggleLock = (key: PaletteKey) => setLocks((l) => ({ ...l, [key]: !l[key] }));
  const lockAll = () => setLocks({
    primary: true, secondary: true, accent: true, highlight: true,
    neutral: true, surface: true, textLight: true, textDark: true,
  });
  const unlockAll = () => setLocks({
    primary: false, secondary: false, accent: false, highlight: false,
    neutral: false, surface: false, textLight: false, textDark: false,
  });
  const applyPreset = () => setPalette((p) => applyPresetToPalette(preset, p, locks));
  const rollPreset = () => setPalette((p) => rollWithinPreset(preset, p, locks));

  // --- Logo composition + brand kit ---
  const onLogoComplete = (svgUrl: string) => setGeneratedSvgUrl(svgUrl);

  const downloadComposedLogo = async () => {
    if (!generatedSvgUrl || !name.trim()) return;
    const canvas = document.createElement("canvas");
    const ctx = canvas.getContext("2d");
    if (!ctx) return;
    canvas.width = 1200; canvas.height = 800;
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    try {
      const svgImg = new Image();
      svgImg.crossOrigin = "anonymous";
      await new Promise((resolve, reject) => {
        svgImg.onload = resolve;
        svgImg.onerror = reject;
        svgImg.src = generatedSvgUrl;
      });
      const logoSize = iconSize;
      const padding = Math.max(40, Math.min(iconSize, textSize) * 0.5);
      ctx.fillStyle = resolveTextColor();
      ctx.font = `bold ${textSize}px ${fontFamily}`;
      ctx.textBaseline = layout === "side-by-side" ? "bottom" : "middle";
      const textMetrics = ctx.measureText(name);
      const textWidth = textMetrics.width;
      const textHeight = textSize;

      if (layout === "side-by-side") {
        const totalWidth = logoSize + textWidth + padding;
        const startX = Math.max(50, (canvas.width - totalWidth) / 2);
        const centerY = canvas.height / 2;
        ctx.drawImage(svgImg, startX, centerY - logoSize / 2, logoSize, logoSize);
        ctx.fillText(name, startX + logoSize + padding, centerY + logoSize / 2);
      } else {
        const totalHeight = logoSize + textHeight + padding;
        const startY = Math.max(50, (canvas.height - totalHeight) / 2);
        const centerX = canvas.width / 2;
        ctx.drawImage(svgImg, centerX - logoSize / 2, startY, logoSize, logoSize);
        ctx.textAlign = "center";
        ctx.fillText(name, centerX, startY + logoSize + padding + textHeight / 2);
      }

      const link = document.createElement("a");
      link.download = `${name.replace(/[^a-zA-Z0-9]/g, "-")}-composed-logo.png`;
      link.href = canvas.toDataURL("image/png");
      link.click();
    } catch (error) {
      console.error("Failed to create composed logo:", error);
    }
  };

  // Helper to fetch the raw icon SVG text
  async function fetchIconSvgText(svgUrl: string | null): Promise<string | null> {
    if (!svgUrl) return null;
    try {
      const txt = await (await fetch(svgUrl)).text();
      return txt.includes("<svg") ? txt : null;
    } catch { return null; }
  }

  const downloadBrandKit = async () => {
    if (!name.trim()) { alert("Please enter a brand name first"); return; }
    setIsDownloadingKit(true);
    try {
      const zip = new JSZip();
      const brandFolderName = name.replace(/[^a-zA-Z0-9]/g, "-").toLowerCase();
      const brandInfo = {
        brandName: name,
        industry,
        styles: selectedStyles,
        createdAt: new Date().toISOString(),
        font: { family: fontFamily, textSize, textColor: resolveTextColor() },
        palette: { ...palette },
        settings: { preset, layout, iconSize },
        generator: "IBrandBiz Brand Kit",
      };
      zip.file(`${brandFolderName}/brand-info.json`, JSON.stringify(brandInfo, null, 2));

      const colorPalette = `
${name} - Color Palette
==============================
Primary:   ${palette.primary}
Secondary: ${palette.secondary}
Accent:    ${palette.accent}
Highlight: ${palette.highlight}
Neutral:   ${palette.neutral}
Surface:   ${palette.surface}
TextLight: ${palette.textLight}
TextDark:  ${palette.textDark}
`.trim();
      zip.file(`${brandFolderName}/color-palette.txt`, colorPalette);

      if (generatedSvgUrl) {
        try {
          const svgResponse = await fetch(generatedSvgUrl);
          const svgContent = await svgResponse.text();
          zip.file(`${brandFolderName}/icon.svg`, svgContent);
        } catch (e) { console.warn("Could not include SVG icon:", e); }
      }

      // Add composed SVGs to the ZIP
      const logosFolder = zip.folder(`${brandFolderName}/logos`)!;
      const iconSvgText = await fetchIconSvgText(generatedSvgUrl);

      // Normal composed SVG (transparent bg)
      if (iconSvgText) {
        const composed = composeLogoSvg({
          name,
          fontFamily,
          textSize,
          textColor: palette.textDark,
          layout,
          iconSize,
          iconSvgText,
          background: null,
        });
        logosFolder.file(`composed_logo.svg`, composed);

        // Reverse composed SVG (dark background + light text)
        const composedReverse = composeLogoSvg({
          name,
          fontFamily,
          textSize,
          textColor: palette.textDark,       // base (ignored because we override)
          layout,
          iconSize,
          iconSvgText,
          background: palette.textDark,      // fill canvas dark
          textColorOverride: palette.textLight,
        });
        logosFolder.file(`composed_logo_reverse.svg`, composedReverse);
      }

      // High-res rasters (logo, reverse, JPG, text-only, icon, reverse icon)
      const widths = [1000, 2000, 3000];
      for (const w of widths) {
        const scale = w / 1200;
        // PNG logo (transparent)
        logosFolder.file(
          `png_logo_${w}px.png`,
          await renderLogoPNG({
            width: w, bg: null, withIcon: true, withText: true,
            layout, iconSize: Math.round(iconSize * scale), textSize: Math.round(textSize * scale),
            textColor: palette.textDark, fontFamily, name, svgUrl: generatedSvgUrl, pad: 72,
          })
        );
        // PNG reverse logo (on dark)
        logosFolder.file(
          `png_logo_reverse_${w}px.png`,
          await renderLogoPNG({
            width: w, bg: palette.textDark, withIcon: true, withText: true,
            layout, iconSize: Math.round(iconSize * scale), textSize: Math.round(textSize * scale),
            textColor: palette.textLight, fontFamily, name, svgUrl: generatedSvgUrl, pad: 72,
          })
        );
        // JPG (flattened on surface)
        logosFolder.file(
          `jpg_logo_${w}px.jpg`,
          await renderLogoJPG({
            width: w, bg: palette.surface, withIcon: true, withText: true,
            layout, iconSize: Math.round(iconSize * scale), textSize: Math.round(textSize * scale),
            textColor: palette.textDark, fontFamily, name, svgUrl: generatedSvgUrl, pad: 72,
          })
        );
        // Text logo (wordmark only)
        logosFolder.file(
          `text_logo_${w}px.png`,
          await renderLogoPNG({
            width: w, bg: null, withIcon: false, withText: true,
            layout, iconSize: 0, textSize: Math.round(textSize * scale),
            textColor: palette.textDark, fontFamily, name, svgUrl: null, pad: 72,
          })
        );
      }

      // Icon variants (512, 1024, 2048)
      for (const w of [512, 1024, 2048]) {
        // PNG icon (transparent)
        logosFolder.file(
          `png_icon_${w}px.png`,
          await renderLogoPNG({
            width: w, bg: null, withIcon: true, withText: false,
            layout: "top-bottom", iconSize: Math.round(w * 0.66), textSize: 0,
            textColor: palette.textDark, fontFamily, name, svgUrl: generatedSvgUrl, pad: 0,
          })
        );
        // PNG reverse icon (on dark)
        logosFolder.file(
          `png_icon_reverse_${w}px.png`,
          await renderLogoPNG({
            width: w, bg: palette.textDark, withIcon: true, withText: false,
            layout: "top-bottom", iconSize: Math.round(w * 0.66), textSize: 0,
            textColor: palette.textLight, fontFamily, name, svgUrl: generatedSvgUrl, pad: 0,
          })
        );
      }

      // --- 4) Include the Brand Kit PDF
      try {
        const pdfDoc = new jsPDF();
        registerEmbeddedFonts(pdfDoc);
        
        const tokens = {
          name,
          industry,
          styles: selectedStyles,
          palette,
          fontFamily,
          textSize,
          iconUrl: generatedSvgUrl,
          slogans,
        };
        
        // Generate PDF content (using existing PDF generation logic)
        await createBrandKitPDF(tokens, { mockupsNodeId: "brand-mockups" });
        
        // Note: The PDF is generated separately via createBrandKitPDF
        // This could be enhanced to include the PDF in the ZIP if needed
      } catch (pdfError) {
        console.warn("Could not generate PDF for ZIP:", pdfError);
      }

      const zipBlob = await zip.generateAsync({ type: "blob" });
      const link = document.createElement("a");
      link.href = URL.createObjectURL(zipBlob);
      link.download = `${brandFolderName}-brand-kit.zip`;
      link.click();
      URL.revokeObjectURL(link.href);
    } catch (error) {
      console.error("Failed to create brand kit:", error);
      alert("Failed to create brand kit. Please try again.");
    } finally {
      setIsDownloadingKit(false);
    }
  };

  const downloadPDFGuide = async () => {
    if (!name.trim()) { alert("Please enter a brand name first"); return; }
    try {
      const tokens = {
        name,
        industry,
        styles: selectedStyles,
        palette,
        fontFamily,
        textSize,
        iconUrl: generatedSvgUrl,
        slogans,
      };
      await createBrandKitPDF(tokens, { mockupsNodeId: "brand-mockups" });
    } catch (error) {
      console.error("Failed to create PDF guide:", error);
      alert("Failed to create PDF guide. Please try again.");
    }
  };

  // ---------- UI helpers ----------
  const paletteEntries: { key: PaletteKey; label: string }[] = useMemo(
    () => [
      { key: "primary", label: "Primary" },
      { key: "secondary", label: "Secondary" },
      { key: "accent", label: "Accent" },
      { key: "highlight", label: "Highlight" },
      { key: "neutral", label: "Neutral" },
      { key: "surface", label: "Surface" },
      { key: "textLight", label: "Text (Light)" },
      { key: "textDark", label: "Text (Dark)" },
    ],
    []
  );

  const stepTitle = {
    1: "Let’s start with your brand name",
    2: "Pick your industry",
    3: "Pick some styles you like",
    4: "Build your brand kit",
  }[step];

  // ---------- Local components ----------
  const PaletteTileCard: React.FC<{ tile: PaletteTile; onApply: () => void }> = ({ tile, onApply }) => (
    <button
      onClick={onApply}
      className="group relative rounded-2xl border hover:border-neutral-300 transition p-3 text-left"
      title={`Apply ${tile.name}`}
    >
      <div className="grid grid-cols-6 gap-1 mb-2">
        {tile.swatches.map((hex, i) => (
          <div key={i} className="h-8 rounded" style={{ backgroundColor: hex }} />
        ))}
      </div>
      <div className="flex items-center gap-2 text-sm">
        <PaletteIcon className="w-4 h-4 text-neutral-500" />
        <span className="font-medium">{tile.name}</span>
      </div>
      <div className="absolute inset-0 rounded-2xl ring-0 group-hover:ring-1 ring-neutral-300 pointer-events-none" />
    </button>
  );

  const ComposedMark: React.FC<{ size?: "sm" | "md" }> = ({ size = "md" }) => (
    <div
      className={`flex ${layout === "side-by-side" ? "flex-row items-end gap-3" : "flex-col items-center gap-2"}`}
      style={{ color: resolveTextColor(), fontFamily, lineHeight: 1.1 }}
    >
      {generatedSvgUrl && (
        <img
          src={generatedSvgUrl}
          alt="Logo"
          className="object-contain"
          style={{ width: size === "sm" ? 48 : iconSize, height: size === "sm" ? 48 : iconSize }}
        />
      )}
      {name && (
        <span style={{ fontSize: size === "sm" ? 20 : textSize, fontWeight: 800, letterSpacing: "0.04em" }}>
          {name}
        </span>
      )}
    </div>
  );

  const Mockups: React.FC = () => (
    <div className="grid md:grid-cols-3 gap-4">
      {/* Business Card */}
      <div className="rounded-2xl border bg-white overflow-hidden">
        <div className="flex items-center gap-3 p-5" style={{ backgroundColor: palette.surface }}>
          <CreditCard className="w-4 h-4 text-neutral-400" />
          <div className="text-sm font-medium">Business Card</div>
        </div>
        <div className="p-6" style={{ background: `linear-gradient(180deg, ${palette.surface}, ${palette.neutral})` }}>
          <div className="rounded-xl border bg-white p-5 flex items-center justify-center">
            <ComposedMark size="sm" />
          </div>
          <div className="mt-4 grid grid-cols-2 gap-2 text-xs text-neutral-500">
            <div>Email</div><div>hello@brand.com</div>
            <div>Phone</div><div>(555) 555-5555</div>
          </div>
        </div>
      </div>

      {/* Website Header */}
      <div className="rounded-2xl border overflow-hidden">
        <div className="flex items-center gap-3 p-5 bg-white">
          <Monitor className="w-4 h-4 text-neutral-400" />
          <div className="text-sm font-medium">Website Header</div>
        </div>
        <div className="p-0">
          <div
            className="px-6 py-4 flex items-center justify-between"
            style={{ backgroundColor: palette.primary, color: palette.textLight }}
          >
            <div className="flex items-center gap-3">
              {generatedSvgUrl && <img src={generatedSvgUrl} alt="" className="w-7 h-7 object-contain" />}
              <span className="font-semibold">{name || "Brand"}</span>
            </div>
            <div className="flex gap-5 text-sm opacity-90">
              <span>Home</span><span>Products</span><span>Contact</span>
            </div>
          </div>
          <div className="p-6 bg-white text-sm text-neutral-600">
            Hero copy and CTA live here. Your brand colors and mark feel right at home.
          </div>
        </div>
      </div>

      {/* App Icon */}
      <div className="rounded-2xl border bg-white overflow-hidden">
        <div className="flex items-center gap-3 p-5" style={{ backgroundColor: palette.surface }}>
          <AppWindow className="w-4 h-4 text-neutral-400" />
          <div className="text-sm font-medium">App Icon</div>
        </div>
        <div className="p-6 flex items-center justify-center">
          <div
            className="rounded-2xl shadow-inner flex items-center justify-center"
            style={{ width: 96, height: 96, backgroundColor: palette.accent }}
          >
            {generatedSvgUrl ? (
              <img src={generatedSvgUrl} alt="" className="w-12 h-12 object-contain" />
            ) : (
              <span className="text-white font-bold">A</span>
            )}
          </div>
        </div>
      </div>
    </div>
  );

  // ---------- Render ----------
  return (
    <div className="p-4 md:p-6">
      {/* Breadcrumbs */}
      <Breadcrumb className="mb-[16px] -mt-1">
        <BreadcrumbList>
          <BreadcrumbItem>
            <BreadcrumbLink href="/dashboard" className="text-[#FF8B00] hover:text-[#e67d00]">Dashboard</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbLink className="text-[#FF8B00] hover:text-[#e67d00]">Business Development</BreadcrumbLink>
          </BreadcrumbItem>
          <BreadcrumbSeparator>
            <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
          </BreadcrumbSeparator>
          <BreadcrumbItem>
            <BreadcrumbPage className="text-[#FF8B00] font-semibold">Brand Kit</BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>

      <h1 className="text-2xl md:text-3xl font-semibold mb-4">Brand Kit</h1>

      {/* Stepper */}
      <div className="flex items-center gap-3 mb-6">
        {[1, 2, 3, 4].map((n) => (
          <button
            key={n}
            className={`px-3 py-1.5 rounded-full text-sm border transition
              ${step === n ? "bg-neutral-900 text-white border-neutral-900" : "bg-white text-neutral-700 hover:bg-neutral-50"}`}
            onClick={() => setStep(n as WizardStep)}
          >
            Step {n}
          </button>
        ))}
      </div>

      {/* Step header */}
      <div className="flex items-start justify-between mb-2">
        <div>
          <h2 className="text-xl md:text-2xl font-semibold">{stepTitle}</h2>
          <p className="text-sm text-neutral-500 mt-1">
            {step === 1 && "We’ll use this name across your logo, fonts, and brand kit."}
            {step === 2 && "Knowing your industry helps suggest symbols, colors, and styles."}
            {step === 3 && "Choose 2–3 cards as inspiration. This guides the AI."}
            {step === 4 && "Tune colors, pick fonts, generate a logo, and compose your mark."}
          </p>
        </div>

        {/* Skip link for Step 3 */}
        {step === 3 && (
          <button
            onClick={() => setStep(4)}
            className="text-sm text-neutral-500 hover:text-neutral-800 underline"
          >
            Skip →
          </button>
        )}
      </div>

      {/* ---------- STEP 1: NAME ---------- */}
      {step === 1 && (
        <form
          className="space-y-6"
          onSubmit={(e) => { e.preventDefault(); if (canNextFrom1) setStep(2); }}
        >
          <input
            className="w-full px-4 py-3 rounded-md border text-lg"
            value={name}
            onChange={(e) => { setName(e.target.value); if (slogans.length) setSlogans([]); }}
            placeholder="Enter your brand name"
            autoFocus
          />

          {featureFlags.sloganGenerator && name.trim().length > 1 && (
            <div className="rounded-xl border bg-white p-4 space-y-4">
              <div className="flex items-center gap-2">
                <MessageSquare className="h-5 w-5 text-primary" />
                <h3 className="text-lg font-semibold">Generate Slogans</h3>
              </div>
              <p className="text-sm text-neutral-500">Quick ideas to shape your voice. (Saved into your kit.)</p>
              <button
                type="button"
                onClick={handleGenerateSlogans}
                disabled={isGeneratingSlogans || !name.trim()}
                className={`flex items-center gap-2 px-4 py-2 rounded-lg ${
                  isGeneratingSlogans ? "bg-neutral-200 text-neutral-500 cursor-not-allowed" : "bg-primary hover:bg-primary/90 text-white"
                }`}
              >
                {isGeneratingSlogans ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent" />
                    Generating...
                  </>
                ) : (
                  <>
                    <Wand2 className="h-4 w-4" />
                    Generate Slogans
                  </>
                )}
              </button>

              {sloganError && <div className="p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-600">{sloganError}</div>}

              {slogans.length > 0 && (
                <div className="space-y-2">
                  {slogans.map((s, i) => (
                    <div key={i} className="flex items-center justify-between p-3 bg-neutral-50 rounded-lg border">
                      <span className="text-sm">“{s}”</span>
                      <button onClick={() => handleCopySlogan(s)} className="ml-3 p-1.5 text-neutral-500 hover:text-neutral-800">
                        <Copy className="h-4 w-4" />
                      </button>
                    </div>
                  ))}
                </div>
              )}
            </div>
          )}

          <div className="pt-2 flex justify-end">
            <button
              type="submit"
              disabled={!canNextFrom1}
              className={`px-4 py-2 rounded-lg flex items-center gap-2 ${
                canNextFrom1 ? "bg-indigo-600 hover:bg-indigo-700 text-white" : "bg-neutral-200 text-neutral-500 cursor-not-allowed"
              }`}
            >
              Continue <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
            </button>
          </div>
        </form>
      )}

      {/* ---------- STEP 2: INDUSTRY ---------- */}
      {step === 2 && (
        <form
          className="space-y-6"
          onSubmit={(e) => { e.preventDefault(); if (canNextFrom2) setStep(3); }}
        >
          <div>
            <input
              list="industries"
              className="w-full px-4 py-3 rounded-md border text-lg"
              value={industry}
              onChange={(e) => setIndustry(e.target.value)}
              placeholder="e.g., Consulting, Fitness, Photography..."
              autoFocus
            />
            <datalist id="industries">
              {POPULAR_INDUSTRIES.map((opt) => (
                <option key={opt} value={opt} />
              ))}
            </datalist>
            <p className="text-xs text-neutral-500 mt-2">
              Type to search or enter your own industry.
            </p>
          </div>

          <div className="flex items-center justify-between pt-2">
            <button type="button" onClick={() => setStep(1)} className="px-3 py-2 rounded-lg border hover:bg-neutral-50">
              ← Back
            </button>
            <button
              type="submit"
              className={`px-4 py-2 rounded-lg flex items-center gap-2 ${
                canNextFrom2 ? "bg-indigo-600 hover:bg-indigo-700 text-white" : "bg-neutral-200 text-neutral-500 cursor-not-allowed"
              }`}
              disabled={!canNextFrom2}
            >
              Continue <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
            </button>
          </div>
        </form>
      )}

      {/* ---------- STEP 3: STYLE INSPIRATION ---------- */}
      {step === 3 && (
        <div className="space-y-6">
          {/* ---- STYLE PICKER ---- */}
          <div className="space-y-2">
            <h2 className="text-xl font-semibold">Pick some styles you like</h2>
            <p className="text-sm text-neutral-500">Choose 2–3 cards as inspiration. This guides the AI.</p>
            <div className="grid sm:grid-cols-2 md:grid-cols-3 gap-4">
              {STYLE_CARDS.map((card) => {
                const active = selectedStyles.includes(card.id);
                return (
                  <StyleTile
                    key={card.id}
                    card={card}
                    active={active}
                    onToggle={() => {
                      setSelectedStyles((prev) => {
                        if (active) return prev.filter((id) => id !== card.id);
                        // cap at 3 selections
                        if (prev.length >= 3) return prev;
                        return [...prev, card.id];
                      });
                    }}
                  />
                );
              })}
            </div>
            <div className="text-xs text-neutral-500">
              Selected: {selectedStyles.length} / 3
            </div>
          </div>

          <div className="flex items-center justify-between pt-2">
            <button onClick={() => setStep(2)} className="px-3 py-2 rounded-lg border hover:bg-neutral-50">← Back</button>
            <button
              onClick={() => setStep(4)}
              className="px-4 py-2 rounded-lg bg-indigo-600 hover:bg-indigo-700 text-white flex items-center gap-2"
            >
              Continue <ChevronRight className="h-4 w-4 text-[#FF8B00]" />
            </button>
          </div>
        </div>
      )}
      {/* ---------- STEP 4: BUILD (Colors / Fonts / AI Logo / Compose / Mockups) ---------- */}
      {step === 4 && (
        <section className="space-y-6">
          {/* Quick Palettes */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="flex items-center gap-2">
              <PaletteIcon className="w-4 h-4 text-neutral-500" />
              <div className="text-sm font-semibold">Quick Palettes</div>
            </div>
            <div className="grid md:grid-cols-3 gap-3">
              {PALETTE_TILES.map((tile) => (
                <PaletteTileCard
                  key={tile.id}
                  tile={tile}
                  onApply={() => setPalette(tileToPalette(tile))}
                />
              ))}
            </div>
            <div className="text-xs text-neutral-500">Click a tile to apply instantly. You can refine below.</div>
          </div>

          {/* Advanced Palette Controls */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="flex items-center justify-between">
              <div className="text-sm font-semibold">Advanced Colors</div>
              <div className="flex flex-wrap items-center gap-2">
                <label className="text-sm text-neutral-700">Preset</label>
                <select
                  value={preset}
                  onChange={(e) => setPreset(e.target.value as PalettePreset)}
                  className="px-3 py-1.5 rounded-md border"
                >
                  <option value="warm">Warm</option>
                  <option value="cool">Cool</option>
                  <option value="earth">Earth Tones</option>
                  <option value="sepia">Sepia / Vintage</option>
                  <option value="pastel">Pastel</option>
                  <option value="neon">Neon</option>
                </select>
                <button onClick={applyPreset} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">Apply</button>
                <button onClick={rollPreset} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">🎲 Roll within</button>
                <span className="mx-2 h-5 w-px bg-neutral-300" />
                <button onClick={rollAllUnlocked} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">🎲 Roll all</button>
                <button onClick={lockAll} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">📌 Lock all</button>
                <button onClick={unlockAll} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">🔓 Unlock all</button>
              </div>
            </div>

            <div className="grid sm:grid-cols-2 md:grid-cols-3 gap-4">
              {paletteEntries.map(({ key, label }) => (
                <ColorSwatch
                  key={key}
                  label={label}
                  hex={palette[key]}
                  locked={locks[key]}
                  onChange={(v) => setPalette((p) => ({ ...p, [key]: v }))}
                  onRoll={() => rollSingle(key)}
                  onToggleLock={() => toggleLock(key)}
                />
              ))}
            </div>
            <div className="text-xs text-neutral-500">
              Neutral/Surface/Text are locked by default to keep accessibility solid.
            </div>
          </div>

          {/* Fonts */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Fonts</div>
            <FontPicker value={fontFamily} onChange={setFontFamily} sampleText={name || "Sample Text"} />
          </div>

          {/* Per-letter color editor */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Advanced: Per-letter color customization</div>
            <div className="text-xs text-neutral-500">Paint individual letters using your palette and export as SVG.</div>
            {name.trim().length > 0 ? (
              <PerLetterColorText
                ref={perLetterRef}
                key={`${name}-${textSize}`}
                text={name}
                fontFamily={fontFamily.replace(/["\']/g, "").split(",")[0].trim()}
                fontSize={textSize}
                palette={[
                  palette.primary, palette.secondary, palette.accent, palette.highlight,
                  palette.neutral, palette.surface, palette.textLight, palette.textDark,
                ]}
              />
            ) : (
              <div className="text-sm text-neutral-400 italic">Enter your brand name in Step 1 to use this editor.</div>
            )}
          </div>

          {/* AI Logo */}
          <div className="rounded-xl border bg-white p-4 space-y-2">
            <div className="text-sm font-semibold">Generate AI-powered SVG logo</div>
            <div className="text-xs text-neutral-500 mb-2">
              We’ll use your name{industry ? ` and ${industry}` : ""}{selectedStyles.length ? ` + ${selectedStyles.join(", ")}` : ""} as context.
            </div>
            <RecraftGenerator businessName={name || "Sample Business"} className="w-full" onComplete={onLogoComplete} />
          </div>

          {/* Compose */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Compose your logo</div>
            <div className="text-xs text-neutral-500">Combine your icon and wordmark. Adjust sizes below.</div>

            <div className="space-y-2">
              <div className="text-xs font-medium text-neutral-700">Layout</div>
              <div className="flex gap-2">
                <button
                  onClick={() => setLayout("side-by-side")}
                  className={`px-3 py-2 rounded-lg border text-sm ${layout === "side-by-side" ? "border-blue-500 bg-blue-50 text-blue-700" : "border-neutral-300 hover:bg-neutral-50"}`}
                >
                  Side by Side
                </button>
                <button
                  onClick={() => setLayout("top-bottom")}
                  className={`px-3 py-2 rounded-lg border text-sm ${layout === "top-bottom" ? "border-blue-500 bg-blue-50 text-blue-700" : "border-neutral-300 hover:bg-neutral-50"}`}
                >
                  Top/Bottom
                </button>
              </div>
            </div>

            <div
              className={`border rounded-lg p-6 min-h-32 flex justify-center ${layout === "side-by-side" ? "flex-row items-end" : "flex-col gap-4 items-center"}`}
              style={{ backgroundColor: palette.surface, gap: layout === "side-by-side" ? `${textSize * 0.1}px` : undefined }}
            >
              {generatedSvgUrl && (
                <div className="rounded-lg overflow-hidden flex-shrink-0" style={{ width: `${iconSize}px`, height: `${iconSize}px` }}>
                  <img src={generatedSvgUrl} alt="Generated Logo" className="w-full h-full object-contain" />
                </div>
              )}
              {name.trim().length > 0 && (
                <div className="font-bold" style={{ fontFamily, color: resolveTextColor(), fontSize: `${textSize}px`, lineHeight: 1.1, letterSpacing: "0.05em" }}>
                  {name}
                </div>
              )}
              {!generatedSvgUrl && name.trim().length === 0 && (
                <div className="text-neutral-400 text-sm">Generate a logo and enter your business name to see preview</div>
              )}
            </div>

            {/* Sliders */}
            <div className="space-y-4">
              <div>
                <div className="text-xs text-neutral-600 mb-2">Text size: {textSize}px</div>
                <input
                  type="range" min="24" max="200" step="2" value={textSize}
                  onChange={(e) => setTextSize(parseInt(e.target.value))}
                  className="w-full h-2 bg-neutral-200 rounded-lg appearance-none cursor-pointer"
                />
              </div>
              {generatedSvgUrl && (
                <div>
                  <div className="text-xs text-neutral-600 mb-2">Icon size: {iconSize}px</div>
                  <input
                    type="range" min="32" max="300" step="4" value={iconSize}
                    onChange={(e) => setIconSize(parseInt(e.target.value))}
                    className="w-full h-2 bg-neutral-200 rounded-lg appearance-none cursor-pointer"
                  />
                </div>
              )}
            </div>

            {/* Actions */}
            <div className="flex flex-wrap gap-3">
              {(generatedSvgUrl && name.trim().length > 0) && (
                <button
                  onClick={downloadComposedLogo}
                  className="flex items-center gap-2 px-4 py-2 rounded-lg bg-[#05445e] hover:bg-[#033a4a] text-white transition-colors"
                >
                  <Download className="w-4 h-4" />
                  Download Logo
                </button>
              )}
              {name.trim().length > 0 && (
                <button
                  onClick={downloadBrandKit}
                  disabled={isDownloadingKit}
                  className="hidden flex items-center gap-2 px-4 py-2 rounded-lg bg-[#ff8800] hover:bg-[#e67700] text-white transition-colors disabled:opacity-50"
                >
                  <img src={bulletCubeWhiteIcon} alt="" className="w-4 h-4" />
                  {isDownloadingKit ? "Creating Brand Kit..." : "Download Brand Kit"}
                </button>
              )}
            </div>
          </div>

          {/* Mockups */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Brand Mockups</div>
            <div id="brand-mockups" className="grid md:grid-cols-3 gap-4">
              <BusinessCardMock tokens={{ name, palette, fontFamily, textSize, iconUrl: generatedSvgUrl }} layout={layout} />
              <HeaderMock tokens={{ name, palette, fontFamily, textSize, iconUrl: generatedSvgUrl }} />
              <AppIconMock tokens={{ name, palette, fontFamily, textSize, iconUrl: generatedSvgUrl }} />
            </div>
          </div>

          {/* Back */}
          <div className="flex items-center gap-2">
            <button onClick={() => setStep(3)} className="px-3 py-2 rounded-lg border hover:bg-neutral-50">← Back</button>
          </div>

          {/* Download Buttons */}
          {(
            <div className="flex justify-center gap-4">
              <button
                onClick={downloadPDFGuide}
                className="flex items-center gap-2 px-4 py-2 rounded-lg bg-[#05445e] hover:bg-[#033a4a] text-white transition-colors"
              >
                <Download className="w-4 h-4" />
                Download Your Brand Guide
              </button>
              <button
                onClick={downloadBrandKit}
                disabled={isDownloadingKit}
                className="flex items-center gap-2 px-4 py-2 rounded-lg bg-[#ff8800] hover:bg-[#e67700] text-white transition-colors disabled:opacity-50"
              >
                <img src={bulletCubeWhiteIcon} alt="" className="w-4 h-4" />
                {isDownloadingKit ? "Creating Brand Kit..." : "Download Brand Kit"}
              </button>
            </div>
          )}


        </section>
      )}
    </div>
  );
}

export default withPro(BrandKitPage, {
  feature: "brand kit generation",
  description: "Generate complete brand kits with logos, colors, fonts, and slogans",
});
