import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/contexts/AuthContext";
import { useCart } from "@/contexts/CartContext";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Separator } from "@/components/ui/separator";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { toast } from "sonner";
import { Link, useParams, useLocation } from "wouter";
import {
  ArrowLeft,
  Download,
  Heart,
  Share2,
  ShoppingCart,
  User,
  Calendar,
  FileText,
  Eye,
  Star,
  Shield,
  ExternalLink,
  ZoomIn,
  Info,
  DollarSign
} from "lucide-react";

interface Creator {
  id: string;
  name: string;
  email: string;
  profileData?: {
    bio?: string;
    portfolio?: string;
    avatar?: string;
  };
  totalEarnings: number;
  isActive: boolean;
}

interface Asset {
  id: string;
  fileName: string;
  originalFileName: string;
  fileUrl: string;
  previewUrl?: string;
  fileSize: number;
  mimeType: string;
  assetType: string;
  category?: string;
  tags: string[];
  metadata?: any;
  downloadCount: number;
  createdAt: string;
}

interface AssetDetail {
  id: string;
  assetId: string;
  title: string;
  description?: string;
  price: number;
  isExclusive: boolean;
  salesCount: number;
  totalRevenue: number;
  approvalStatus: string;
  approvedAt?: string;
  createdAt: string;
  updatedAt: string;
  asset: Asset;
  creator: Creator;
}

interface RelatedAsset {
  id: string;
  title: string;
  price: number;
  previewUrl?: string;
  creator: {
    name: string;
    email: string;
  };
}

export default function AssetDetailPage() {
  const { id } = useParams();
  const { currentUser } = useAuth();
  const { addItem, hasItem } = useCart();
  const [, setLocation] = useLocation();
  const [imageZoomed, setImageZoomed] = useState(false);

  // Fetch asset details
  const { data: assetDetail, isLoading, error } = useQuery<AssetDetail>({
    queryKey: ['/api/marketplace/assets', id],
    queryFn: async () => {
      const response = await fetch(`/api/marketplace/assets/${id}`, {
        credentials: 'include'
      });

      if (!response.ok) {
        if (response.status === 404) {
          throw new Error('Asset not found');
        }
        throw new Error('Failed to fetch asset details');
      }

      return response.json();
    },
    enabled: !!id,
  });

  // Fetch related assets
  const { data: relatedAssets } = useQuery<RelatedAsset[]>({
    queryKey: ['/api/marketplace/creators', assetDetail?.creator.id, 'assets'],
    queryFn: async () => {
      const response = await fetch(`/api/marketplace/creators/${assetDetail!.creator.id}/assets?limit=6`, {
        credentials: 'include'
      });

      if (!response.ok) {
        return [];
      }

      const data = await response.json();
      return data.assets.filter((asset: RelatedAsset) => asset.id !== id).slice(0, 4);
    },
    enabled: !!assetDetail?.creator.id,
  });

  const formatPrice = (cents: number) => {
    return `$${(cents / 100).toFixed(2)}`;
  };

  const formatFileSize = (bytes: number) => {
    const units = ['B', 'KB', 'MB', 'GB'];
    let size = bytes;
    let unitIndex = 0;
    
    while (size >= 1024 && unitIndex < units.length - 1) {
      size /= 1024;
      unitIndex++;
    }
    
    return `${size.toFixed(1)} ${units[unitIndex]}`;
  };

  const handleAddToCart = () => {
    if (!assetDetail) return;
    
    try {
      const success = addItem({
        itemType: 'creator_asset',
        itemId: assetDetail.id,
        itemName: assetDetail.title,
        itemPrice: assetDetail.price,
        quantity: 1,
        metadata: {
          creatorId: assetDetail.creator.id,
          assetId: assetDetail.assetId,
          previewUrl: assetDetail.asset.previewUrl,
          assetType: assetDetail.asset.assetType,
          category: assetDetail.asset.category,
          creatorName: assetDetail.creator.name
        }
      });
      
      // Only show success toast if item was actually added (cart constraints allow it)
      if (success) {
        toast.success(`"${assetDetail.title}" added to cart!`);
      }
      // Error toasts are handled by the CartProvider
    } catch (error) {
      toast.error("Failed to add item to cart");
      console.error("Add to cart error:", error);
    }
  };

  const handleShare = async () => {
    if (navigator.share) {
      try {
        await navigator.share({
          title: assetDetail?.title,
          text: assetDetail?.description,
          url: window.location.href,
        });
      } catch (error) {
        // User cancelled sharing
      }
    } else {
      // Fallback: copy to clipboard
      await navigator.clipboard.writeText(window.location.href);
      toast.success("Link copied to clipboard!");
    }
  };

  if (error) {
    return (
      <div className="container mx-auto px-4 py-8">
        <Button
          variant="ghost"
          onClick={() => setLocation('/marketplace')}
          className="mb-6"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back to Marketplace
        </Button>
        
        <Alert>
          <AlertDescription>
            {error instanceof Error ? error.message : 'Failed to load asset details'}
          </AlertDescription>
        </Alert>
      </div>
    );
  }

  if (isLoading || !assetDetail) {
    return (
      <div className="container mx-auto px-4 py-8">
        <Button
          variant="ghost"
          onClick={() => setLocation('/marketplace')}
          className="mb-6"
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back to Marketplace
        </Button>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          <Skeleton className="aspect-video w-full" />
          <div className="space-y-4">
            <Skeleton className="h-8 w-3/4" />
            <Skeleton className="h-4 w-full" />
            <Skeleton className="h-4 w-2/3" />
            <Skeleton className="h-12 w-32" />
          </div>
        </div>
      </div>
    );
  }

  const isInCart = hasItem(assetDetail.id, 'creator_asset');

  return (
    <div className="min-h-screen bg-background">
      <div className="container mx-auto px-4 py-8">
        {/* Navigation */}
        <div className="flex items-center justify-between mb-8">
          <Button
            variant="ghost"
            onClick={() => setLocation('/marketplace')}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back to Marketplace
          </Button>

          <div className="flex gap-2">
            <Button variant="outline" size="sm" onClick={handleShare} data-testid="button-share">
              <Share2 className="h-4 w-4" />
            </Button>
            <Button variant="outline" size="sm" data-testid="button-favorite">
              <Heart className="h-4 w-4" />
            </Button>
          </div>
        </div>

        {/* Main Content */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-12">
          {/* Asset Preview */}
          <div className="space-y-4">
            <Card className="overflow-hidden">
              <div 
                className={`aspect-video bg-gray-100 dark:bg-gray-800 overflow-hidden relative cursor-pointer ${
                  imageZoomed ? 'fixed inset-0 z-50 aspect-auto' : ''
                }`}
                onClick={() => setImageZoomed(!imageZoomed)}
              >
                {assetDetail.asset.previewUrl ? (
                  <img
                    src={assetDetail.asset.previewUrl}
                    alt={assetDetail.title}
                    className={`w-full h-full object-contain ${imageZoomed ? 'p-8' : 'object-cover'}`}
                  />
                ) : (
                  <div className="w-full h-full flex items-center justify-center">
                    <FileText className="h-16 w-16 text-muted-foreground" />
                  </div>
                )}
                
                {!imageZoomed && (
                  <div className="absolute inset-0 bg-black/0 hover:bg-black/20 transition-colors duration-300 flex items-center justify-center opacity-0 hover:opacity-100">
                    <div className="bg-white/90 hover:bg-white text-gray-900 p-3 rounded-lg transition-colors duration-200">
                      <ZoomIn className="h-6 w-6" />
                    </div>
                  </div>
                )}

                {imageZoomed && (
                  <Button
                    variant="secondary"
                    className="absolute top-4 right-4"
                    onClick={(e) => {
                      e.stopPropagation();
                      setImageZoomed(false);
                    }}
                  >
                    Close
                  </Button>
                )}
              </div>
            </Card>

            {/* Asset Info */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Info className="h-5 w-5" />
                  Asset Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="grid grid-cols-2 gap-4 text-sm">
                  <div>
                    <span className="font-medium">File Type:</span>
                    <p className="text-muted-foreground">{assetDetail.asset.mimeType}</p>
                  </div>
                  <div>
                    <span className="font-medium">File Size:</span>
                    <p className="text-muted-foreground">{formatFileSize(assetDetail.asset.fileSize)}</p>
                  </div>
                  <div>
                    <span className="font-medium">Downloads:</span>
                    <p className="text-muted-foreground">{assetDetail.asset.downloadCount.toLocaleString()}</p>
                  </div>
                  <div>
                    <span className="font-medium">Sales:</span>
                    <p className="text-muted-foreground">{assetDetail.salesCount.toLocaleString()}</p>
                  </div>
                </div>
                
                {assetDetail.asset.tags.length > 0 && (
                  <div>
                    <span className="font-medium text-sm block mb-2">Tags:</span>
                    <div className="flex flex-wrap gap-1">
                      {assetDetail.asset.tags.map((tag, index) => (
                        <Badge key={index} variant="outline" className="text-xs">
                          {tag}
                        </Badge>
                      ))}
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Asset Details */}
          <div className="space-y-6">
            {/* Header */}
            <div>
              <div className="flex items-start justify-between mb-4">
                <div className="flex-1">
                  <h1 className="text-3xl font-bold mb-2" data-testid="asset-title">
                    {assetDetail.title}
                  </h1>
                  {assetDetail.description && (
                    <p className="text-muted-foreground text-lg leading-relaxed">
                      {assetDetail.description}
                    </p>
                  )}
                </div>
                {assetDetail.isExclusive && (
                  <Badge className="bg-yellow-500 ml-4">
                    Exclusive
                  </Badge>
                )}
              </div>

              <div className="flex items-center gap-4 text-sm text-muted-foreground">
                {assetDetail.asset.category && (
                  <Badge variant="secondary">
                    {assetDetail.asset.category}
                  </Badge>
                )}
                <div className="flex items-center">
                  <Calendar className="h-4 w-4 mr-1" />
                  {new Date(assetDetail.createdAt).toLocaleDateString()}
                </div>
                <div className="flex items-center">
                  <Eye className="h-4 w-4 mr-1" />
                  {assetDetail.asset.downloadCount} views
                </div>
              </div>
            </div>

            {/* Pricing */}
            <Card>
              <CardContent className="pt-6">
                <div className="flex items-center justify-between mb-4">
                  <div>
                    <div className="text-3xl font-bold text-green-600" data-testid="asset-price">
                      {formatPrice(assetDetail.price)}
                    </div>
                    <p className="text-sm text-muted-foreground">One-time purchase</p>
                  </div>
                  <div className="text-right">
                    <div className="text-sm text-muted-foreground">
                      Total Revenue: {formatPrice(assetDetail.totalRevenue)}
                    </div>
                  </div>
                </div>

                <Button
                  onClick={handleAddToCart}
                  disabled={isInCart}
                  className="w-full"
                  size="lg"
                  data-testid="button-add-to-cart"
                >
                  <ShoppingCart className="h-5 w-5 mr-2" />
                  {isInCart ? 'Already in Cart' : 'Add to Cart'}
                </Button>

                <div className="flex items-center justify-center gap-4 mt-4 text-sm text-muted-foreground">
                  <div className="flex items-center">
                    <Shield className="h-4 w-4 mr-1" />
                    Commercial License
                  </div>
                  <div className="flex items-center">
                    <Download className="h-4 w-4 mr-1" />
                    Instant Download
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Creator Info */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="h-5 w-5" />
                  Creator
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex items-start gap-4">
                  <Avatar className="h-12 w-12">
                    <AvatarImage src={assetDetail.creator.profileData?.avatar} />
                    <AvatarFallback>
                      {(assetDetail.creator.name || assetDetail.creator.email).charAt(0).toUpperCase()}
                    </AvatarFallback>
                  </Avatar>
                  
                  <div className="flex-1">
                    <Link href={`/marketplace/creator/${assetDetail.creator.id}`}>
                      <h3 className="font-semibold hover:text-primary cursor-pointer">
                        {assetDetail.creator.name || assetDetail.creator.email}
                      </h3>
                    </Link>
                    {assetDetail.creator.profileData?.bio && (
                      <p className="text-sm text-muted-foreground mt-1 line-clamp-2">
                        {assetDetail.creator.profileData.bio}
                      </p>
                    )}
                    
                    <div className="flex gap-4 mt-3">
                      <Link href={`/marketplace/creator/${assetDetail.creator.id}`}>
                        <Button variant="outline" size="sm">
                          View Profile
                        </Button>
                      </Link>
                      {assetDetail.creator.profileData?.portfolio && (
                        <Button variant="outline" size="sm" asChild>
                          <a href={assetDetail.creator.profileData.portfolio} target="_blank" rel="noopener noreferrer">
                            <ExternalLink className="h-4 w-4 mr-2" />
                            Portfolio
                          </a>
                        </Button>
                      )}
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* License Information */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Shield className="h-5 w-5" />
                  License & Usage Rights
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="space-y-2 text-sm">
                  <div className="flex items-center justify-between">
                    <span>Commercial Use</span>
                    <Badge variant="outline" className="text-green-600">Allowed</Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <span>Resale Rights</span>
                    <Badge variant="outline" className="text-red-600">Not Allowed</Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <span>Attribution Required</span>
                    <Badge variant="outline">No</Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <span>Modification</span>
                    <Badge variant="outline" className="text-green-600">Allowed</Badge>
                  </div>
                </div>
                
                <Separator />
                
                <p className="text-xs text-muted-foreground">
                  This asset comes with a standard commercial license. You can use it in commercial and personal projects, 
                  modify it as needed, but cannot resell or redistribute the original files.
                </p>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Related Assets */}
        {relatedAssets && relatedAssets.length > 0 && (
          <div>
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-2xl font-bold">More from this Creator</h2>
              <Link href={`/marketplace/creator/${assetDetail.creator.id}`}>
                <Button variant="outline">
                  View All Assets
                  <ArrowLeft className="h-4 w-4 ml-2 rotate-180" />
                </Button>
              </Link>
            </div>
            
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
              {relatedAssets.map((asset) => (
                <Card key={asset.id} className="group overflow-hidden hover:shadow-lg transition-all duration-300">
                  <div className="aspect-video bg-gray-100 dark:bg-gray-800 overflow-hidden relative">
                    {asset.previewUrl ? (
                      <img
                        src={asset.previewUrl}
                        alt={asset.title}
                        className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                        loading="lazy"
                      />
                    ) : (
                      <div className="w-full h-full flex items-center justify-center">
                        <FileText className="h-8 w-8 text-muted-foreground" />
                      </div>
                    )}
                    
                    <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
                      <Link href={`/marketplace/asset/${asset.id}`}>
                        <Button variant="secondary" size="sm">
                          <Eye className="h-4 w-4 mr-2" />
                          View
                        </Button>
                      </Link>
                    </div>
                  </div>

                  <CardContent className="p-4">
                    <Link href={`/marketplace/asset/${asset.id}`}>
                      <h3 className="font-semibold text-sm mb-2 line-clamp-1 hover:text-primary cursor-pointer">
                        {asset.title}
                      </h3>
                    </Link>
                    <div className="text-sm font-bold text-green-600">
                      {formatPrice(asset.price)}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}