import React, { useState } from "react";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Trash2, Search, Eye, Download, AlertCircle } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { useAdminAuth } from "@/admin/useAdminAuth";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";

interface IconItem {
  id: string;
  name: string;
  style: string;
  tags: string[];
  previewUrl: string;
  formats: string[];
  createdAt: number;
  svg?: string;
  svgUrl?: string;
}

export default function IconsManager() {
  const { getAuthHeaders, isAuthenticated } = useAdminAuth();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [searchTerm, setSearchTerm] = useState("");
  const [previewItem, setPreviewItem] = useState<IconItem | null>(null);
  const [isPreviewOpen, setIsPreviewOpen] = useState(false);

  // Fetch icons from main library (object storage)
  const { data: iconsData, isLoading: iconsLoading, error: iconsError } = useQuery({
    queryKey: ['/api/icons/list'],
    enabled: isAuthenticated,
    queryFn: async () => {
      const response = await fetch('/api/icons/list');
      if (!response.ok) {
        throw new Error('Failed to fetch icons');
      }
      const data = await response.json();
      return data.icons || [];
    }
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (iconId: string) => {
      const headers = await getAuthHeaders();
      const response = await fetch(`/api/icons/${iconId}`, {
        method: "DELETE",
        headers: headers
      });
      
      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.error || `HTTP ${response.status}: ${response.statusText}`);
      }
      
      return response.json();
    },
    onSuccess: (data, iconId) => {
      toast({
        title: "Delete Successful",
        description: `Successfully deleted icon`,
        variant: "default",
      });
      
      // Invalidate queries to refetch data
      queryClient.invalidateQueries({ queryKey: ['/api/icons/list'] });
    },
    onError: (error: any) => {
      toast({
        title: "Delete Failed",
        description: error.message || "Failed to delete icon",
        variant: "destructive",
      });
    }
  });

  const handleDelete = (item: IconItem) => {
    if (!confirm(`Are you sure you want to delete "${item.name}"? This action cannot be undone.`)) {
      return;
    }
    
    deleteMutation.mutate(item.id);
  };

  const handleDownload = async (item: IconItem, format: string = 'svg') => {
    try {
      const response = await fetch(`/api/icons/${item.id}/download?format=${format}`);
      if (!response.ok) {
        throw new Error('Failed to download icon');
      }
      
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.style.display = 'none';
      a.href = url;
      a.download = `${item.name}.${format}`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
      
      toast({
        title: "Download Successful",
        description: `Downloaded ${item.name}.${format}`,
      });
    } catch (error: any) {
      toast({
        title: "Download Failed",
        description: error.message || "Failed to download icon",
        variant: "destructive",
      });
    }
  };

  if (!isAuthenticated) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
      </div>
    );
  }

  // Filter icons based on search term
  const filteredIcons = iconsData?.filter((icon: IconItem) =>
    icon.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    icon.tags.some(tag => tag.toLowerCase().includes(searchTerm.toLowerCase()))
  ) || [];

  return (
    <DashboardTemplatePage 
      title="Icons Manager"
      description="Manage and delete icons from the main Icons Library"
    >
      <div className="max-w-6xl mx-auto space-y-6">
        {/* Header with Search */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Trash2 className="h-5 w-5 text-red-600" />
              Icons Library Management
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center gap-4">
              <div className="flex-1">
                <Label htmlFor="search" className="sr-only">Search icons</Label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                  <Input
                    id="search"
                    placeholder="Search icons by name or tags..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                    data-testid="input-search-icons"
                  />
                </div>
              </div>
              <div className="text-sm text-muted-foreground">
                {filteredIcons.length} of {iconsData?.length || 0} icons
              </div>
            </div>
            
            {/* Warning Notice */}
            <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
              <div className="flex items-start gap-3">
                <AlertCircle className="h-5 w-5 text-red-600 mt-0.5 flex-shrink-0" />
                <div className="text-sm text-red-800 dark:text-red-200">
                  <p className="font-medium mb-1">⚠️ Danger Zone</p>
                  <p>This page manages the <strong>main Icons Library</strong> that users see. Deleting icons here will permanently remove them from object storage and make them unavailable to all users. This action cannot be undone.</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Icons Grid */}
        <IconsGrid 
          icons={filteredIcons}
          isLoading={iconsLoading}
          hasError={iconsError}
          onDelete={handleDelete}
          onPreview={(item) => { setPreviewItem(item); setIsPreviewOpen(true); }}
          onDownload={handleDownload}
        />

        {/* Preview Modal */}
        <Dialog open={isPreviewOpen} onOpenChange={setIsPreviewOpen}>
          <DialogContent className="max-w-3xl w-[90vw] p-0 bg-transparent border-none shadow-none" data-testid="modal-icon-preview">
            <div className="relative bg-black/90 rounded-lg overflow-hidden">
              {previewItem && (
                <>
                  <div className="flex items-center justify-center p-8">
                    {previewItem.svg ? (
                      <div 
                        className="max-h-[60vh] w-auto mx-auto"
                        dangerouslySetInnerHTML={{ __html: previewItem.svg }}
                      />
                    ) : previewItem.svgUrl ? (
                      <img 
                        src={previewItem.svgUrl} 
                        alt={previewItem.name}
                        className="max-h-[60vh] w-auto mx-auto object-contain"
                      />
                    ) : (
                      <img 
                        src={previewItem.previewUrl} 
                        alt={previewItem.name}
                        className="max-h-[60vh] w-auto mx-auto object-contain"
                      />
                    )}
                  </div>
                  <div className="absolute bottom-0 left-0 right-0 bg-black/70 text-white p-4">
                    <h3 className="font-semibold mb-2">{previewItem.name}</h3>
                    <div className="flex flex-wrap gap-2">
                      <Badge variant="secondary" className="text-xs">
                        {previewItem.style}
                      </Badge>
                      {previewItem.tags.map((tag, index) => (
                        <Badge key={index} variant="outline" className="text-xs text-white border-white">
                          {tag}
                        </Badge>
                      ))}
                    </div>
                  </div>
                </>
              )}
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </DashboardTemplatePage>
  );
}

interface IconsGridProps {
  icons: IconItem[];
  isLoading: boolean;
  hasError: any;
  onDelete: (item: IconItem) => void;
  onPreview: (item: IconItem) => void;
  onDownload: (item: IconItem, format: string) => void;
}

function IconsGrid({ icons, isLoading, hasError, onDelete, onPreview, onDownload }: IconsGridProps) {
  if (isLoading) {
    return (
      <div className="flex items-center justify-center py-12">
        <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
      </div>
    );
  }

  if (hasError) {
    return (
      <div className="text-center py-12">
        <div className="text-red-500 mb-4">
          <svg className="h-16 w-16 mx-auto mb-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
        </div>
        <h3 className="text-lg font-medium mb-2 text-red-600">Error Loading Icons</h3>
        <p className="text-muted-foreground">Failed to load icons from the main library. Please try refreshing the page.</p>
      </div>
    );
  }

  if (icons.length === 0) {
    return (
      <div className="text-center py-12">
        <div className="text-muted-foreground mb-4">
          <svg className="h-16 w-16 mx-auto mb-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
          </svg>
        </div>
        <h3 className="text-lg font-medium mb-2">No Icons Found</h3>
        <p className="text-muted-foreground">No icons match your search criteria or the library is empty.</p>
      </div>
    );
  }

  return (
    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
      {icons.map((icon) => (
        <Card key={icon.id} className="group overflow-hidden hover:shadow-lg transition-all duration-300">
          <div className="aspect-square bg-gray-100 dark:bg-gray-800 overflow-hidden relative cursor-pointer p-4 flex items-center justify-center" onClick={() => onPreview(icon)} data-testid={`icon-preview-${icon.id}`}>
            {icon.svg ? (
              <div 
                className="w-16 h-16 group-hover:scale-110 transition-transform duration-300"
                dangerouslySetInnerHTML={{ __html: icon.svg }}
              />
            ) : icon.svgUrl ? (
              <img 
                src={icon.svgUrl} 
                alt={icon.name} 
                className="w-16 h-16 object-contain group-hover:scale-110 transition-transform duration-300" 
                loading="lazy"
              />
            ) : (
              <img 
                src={icon.previewUrl} 
                alt={icon.name} 
                className="w-16 h-16 object-contain group-hover:scale-110 transition-transform duration-300" 
                loading="lazy"
              />
            )}
          </div>
          <CardContent className="p-4">
            <h3 className="font-medium text-sm mb-2 line-clamp-1" data-testid={`title-${icon.id}`}>
              {icon.name}
            </h3>
            <div className="flex items-center justify-between mb-2">
              <div className="flex flex-wrap gap-1">
                <Badge variant="secondary" className="text-xs">
                  {icon.style}
                </Badge>
              </div>
            </div>
            <div className="flex flex-wrap gap-1 mb-3">
              {icon.tags.slice(0, 2).map((tag, index) => (
                <Badge key={index} variant="outline" className="text-xs">
                  {tag}
                </Badge>
              ))}
              {icon.tags.length > 2 && (
                <Badge variant="outline" className="text-xs">
                  +{icon.tags.length - 2}
                </Badge>
              )}
            </div>
            <div className="flex justify-end">
              <div className="flex gap-1">
                <Button 
                  size="sm" 
                  variant="outline"
                  onClick={() => onPreview(icon)}
                  data-testid={`button-preview-${icon.id}`}
                >
                  <Eye className="h-3 w-3" />
                </Button>
                <Button 
                  size="sm" 
                  variant="outline"
                  onClick={() => onDownload(icon, 'svg')}
                  data-testid={`button-download-${icon.id}`}
                >
                  <Download className="h-3 w-3" />
                </Button>
                <Button 
                  size="sm" 
                  variant="outline"
                  onClick={() => onDelete(icon)}
                  className="text-red-600 hover:text-red-700 hover:bg-red-50"
                  data-testid={`button-delete-${icon.id}`}
                >
                  <Trash2 className="h-3 w-3" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}