import React, { useState } from "react";
import { Tag, Search, Trash2, Save, Eye } from "lucide-react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";

interface ImportedIcon {
  id: string;
  name: string;
  svg: string;
  style: string;
  tags: string[];
  createdAt: string;
}

export default function IconLabeler() {
  const { toast } = useToast();
  const [searchTerm, setSearchTerm] = useState("");
  const [editingIcon, setEditingIcon] = useState<ImportedIcon | null>(null);
  const [editedName, setEditedName] = useState("");
  const [editedTags, setEditedTags] = useState<string[]>([]);
  const [newTag, setNewTag] = useState("");

  // Get admin key from localStorage
  const getAdminKey = () => {
    return localStorage.getItem("adminKey") || "";
  };

  // Fetch imported icons
  const { data: importedIcons = [], isLoading } = useQuery({
    queryKey: ["/api/icons/imported"],
    queryFn: async () => {
      const response = await fetch("/api/icons/imported");
      if (!response.ok) {
        throw new Error("Failed to fetch imported icons");
      }
      const data = await response.json();
      return data.icons || [];
    }
  });

  // Delete icon mutation
  const deleteIcon = useMutation({
    mutationFn: async (iconId: string) => {
      const adminKey = getAdminKey();
      if (!adminKey) {
        throw new Error("Admin key not found");
      }

      const response = await fetch(`/api/icons/imported/${iconId}`, {
        method: "DELETE",
        headers: {
          "x-admin-key": adminKey
        }
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to delete icon");
      }

      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Icon deleted successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/icons/imported"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete icon",
        variant: "destructive",
      });
    }
  });

  const startEditing = (icon: ImportedIcon) => {
    setEditingIcon(icon);
    setEditedName(icon.name);
    setEditedTags([...icon.tags]);
  };

  const cancelEditing = () => {
    setEditingIcon(null);
    setEditedName("");
    setEditedTags([]);
    setNewTag("");
  };

  const addTag = () => {
    if (newTag.trim() && !editedTags.includes(newTag.trim())) {
      setEditedTags([...editedTags, newTag.trim()]);
      setNewTag("");
    }
  };

  const removeTag = (tagToRemove: string) => {
    setEditedTags(editedTags.filter(tag => tag !== tagToRemove));
  };

  const saveChanges = async () => {
    if (!editingIcon) return;

    try {
      const adminKey = getAdminKey();
      if (!adminKey) {
        throw new Error("Admin key not found");
      }

      // For now, we'll show a message that this feature is coming soon
      // In a real implementation, you'd have an update endpoint
      toast({
        title: "Feature Coming Soon",
        description: "Icon editing functionality will be available in a future update",
      });
      
      cancelEditing();
    } catch (error: any) {
      toast({
        title: "Error",
        description: error.message || "Failed to save changes",
        variant: "destructive",
      });
    }
  };

  const filteredIcons = importedIcons.filter((icon: ImportedIcon) =>
    icon.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    icon.tags.some(tag => tag.toLowerCase().includes(searchTerm.toLowerCase()))
  );

  if (isLoading) {
    return (
      <DashboardTemplatePage
        title="Icon Labeler"
        description="Organize and label imported icons"
      >
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
        </div>
      </DashboardTemplatePage>
    );
  }

  return (
    <DashboardTemplatePage
      title="Icon Labeler"
      description="Organize and label imported icons"
    >
      <div className="space-y-6">
        {/* Search Section */}
        <Card data-testid="card-search">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Search className="h-5 w-5" />
              Search Icons
            </CardTitle>
          </CardHeader>
          <CardContent>
            <Input
              type="text"
              placeholder="Search by name or tags..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full"
              data-testid="input-search"
            />
          </CardContent>
        </Card>

        {/* Stats */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="text-2xl font-bold text-primary">{importedIcons.length}</div>
              <div className="text-sm text-muted-foreground">Total Icons</div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="text-2xl font-bold text-primary">{filteredIcons.length}</div>
              <div className="text-sm text-muted-foreground">Filtered Results</div>
            </CardContent>
          </Card>
          <Card>
            <CardContent className="p-4">
              <div className="text-2xl font-bold text-primary">
                {Array.from(new Set(importedIcons.flatMap((icon: ImportedIcon) => icon.tags))).length}
              </div>
              <div className="text-sm text-muted-foreground">Unique Tags</div>
            </CardContent>
          </Card>
        </div>

        {/* Icons Grid */}
        <Card data-testid="card-icons">
          <CardHeader>
            <CardTitle>
              Icons ({filteredIcons.length})
            </CardTitle>
          </CardHeader>
          <CardContent>
            {filteredIcons.length === 0 ? (
              <div className="text-center py-12">
                <Tag className="mx-auto h-12 w-12 text-gray-400" />
                <h3 className="mt-2 text-sm font-medium text-gray-900">No icons found</h3>
                <p className="mt-1 text-sm text-gray-500">
                  {importedIcons.length === 0 
                    ? "Import some icons first using the Icon Importer"
                    : "Try a different search term"
                  }
                </p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredIcons.map((icon: ImportedIcon) => (
                  <div
                    key={icon.id}
                    className="border rounded-lg p-4 space-y-3"
                    data-testid={`icon-card-${icon.id}`}
                  >
                    {/* Icon Preview */}
                    <div className="flex items-center justify-center h-16 bg-gray-50 rounded">
                      <div 
                        className="w-12 h-12"
                        dangerouslySetInnerHTML={{ __html: icon.svg }}
                      />
                    </div>

                    {/* Icon Info */}
                    {editingIcon?.id === icon.id ? (
                      <div className="space-y-3">
                        <Input
                          value={editedName}
                          onChange={(e) => setEditedName(e.target.value)}
                          placeholder="Icon name"
                          data-testid={`input-edit-name-${icon.id}`}
                        />
                        
                        <div className="space-y-2">
                          <div className="flex gap-2">
                            <Input
                              value={newTag}
                              onChange={(e) => setNewTag(e.target.value)}
                              placeholder="Add tag"
                              onKeyPress={(e) => e.key === 'Enter' && addTag()}
                              data-testid={`input-new-tag-${icon.id}`}
                            />
                            <Button onClick={addTag} size="sm">Add</Button>
                          </div>
                          
                          <div className="flex flex-wrap gap-1">
                            {editedTags.map((tag, index) => (
                              <Badge
                                key={index}
                                variant="secondary"
                                className="cursor-pointer"
                                onClick={() => removeTag(tag)}
                                data-testid={`tag-${icon.id}-${index}`}
                              >
                                {tag} ×
                              </Badge>
                            ))}
                          </div>
                        </div>

                        <div className="flex gap-2">
                          <Button onClick={saveChanges} size="sm" className="flex-1">
                            <Save className="w-4 h-4 mr-1" />
                            Save
                          </Button>
                          <Button onClick={cancelEditing} variant="outline" size="sm">
                            Cancel
                          </Button>
                        </div>
                      </div>
                    ) : (
                      <div className="space-y-3">
                        <div>
                          <h3 className="font-medium text-sm">{icon.name}</h3>
                          <p className="text-xs text-gray-500">
                            Style: {icon.style} • {new Date(icon.createdAt).toLocaleDateString()}
                          </p>
                        </div>

                        {icon.tags.length > 0 && (
                          <div className="flex flex-wrap gap-1">
                            {icon.tags.map((tag, index) => (
                              <Badge key={index} variant="outline" className="text-xs">
                                {tag}
                              </Badge>
                            ))}
                          </div>
                        )}

                        <div className="flex gap-2">
                          <Button
                            onClick={() => startEditing(icon)}
                            size="sm"
                            variant="outline"
                            className="flex-1"
                            data-testid={`button-edit-${icon.id}`}
                          >
                            <Tag className="w-4 h-4 mr-1" />
                            Edit
                          </Button>
                          <Button
                            onClick={() => deleteIcon.mutate(icon.id)}
                            size="sm"
                            variant="destructive"
                            disabled={deleteIcon.isPending}
                            data-testid={`button-delete-${icon.id}`}
                          >
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </DashboardTemplatePage>
  );
}