import React, { useState, useCallback } from "react";
import { Upload, AlertCircle, CheckCircle, Download, Trash2, Save, Search, Tag, Eye, FileImage, X, Clock, AlertTriangle, Package } from "lucide-react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Slider } from "@/components/ui/slider";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import { useAdminAuth } from "@/admin/useAdminAuth";

interface ProcessedIcon {
  index: number;
  name: string;
  svg: string;
  selected: boolean;
}

interface ImportedIcon {
  id: string;
  name: string;
  svg: string;
  style: string;
  tags: string[];
  createdAt: string;
}

export default function IconImporter() {
  const { toast } = useToast();
  
  // Tab state
  const [activeTab, setActiveTab] = useState("batch");
  
  // Batch files mode state
  const [batchFiles, setBatchFiles] = useState<File[]>([]);
  const [batchProcessedIcons, setBatchProcessedIcons] = useState<ProcessedIcon[]>([]);
  const [isProcessingBatch, setIsProcessingBatch] = useState(false);
  const [isBatchDragging, setIsBatchDragging] = useState(false);
  
  // Individual files mode state
  const [individualFiles, setIndividualFiles] = useState<File[]>([]);
  const [individualProcessedIcons, setIndividualProcessedIcons] = useState<ProcessedIcon[]>([]);
  const [isProcessingIndividual, setIsProcessingIndividual] = useState(false);
  const [isIndividualDragging, setIsIndividualDragging] = useState(false);
  
  // Common state
  const [searchTerm, setSearchTerm] = useState("");
  const [threshold, setThreshold] = useState([180]);
  const [invert, setInvert] = useState(false);

  // Use dual authentication (owner or admin key)
  const { isAuthenticated, isOwner, getAuthHeaders } = useAdminAuth();

  // Fetch imported icons
  const { data: importedIcons = [], isLoading: isLoadingImported } = useQuery({
    queryKey: ["/api/icons/imported"],
    queryFn: async () => {
      const response = await fetch("/api/icons/imported");
      if (!response.ok) {
        throw new Error("Failed to fetch imported icons");
      }
      const data = await response.json();
      return data.icons || [];
    }
  });

  // Drag & drop handlers - shared between batch and individual
  const createDragHandlers = (setDragging: (dragging: boolean) => void) => ({
    handleDragEnter: useCallback((e: React.DragEvent) => {
      e.preventDefault();
      e.stopPropagation();
      setDragging(true);
    }, [setDragging]),

    handleDragLeave: useCallback((e: React.DragEvent) => {
      e.preventDefault();
      e.stopPropagation();
      setDragging(false);
    }, [setDragging]),

    handleDragOver: useCallback((e: React.DragEvent) => {
      e.preventDefault();
      e.stopPropagation();
    }, [])
  });

  const batchDragHandlers = createDragHandlers(setIsBatchDragging);
  const individualDragHandlers = createDragHandlers(setIsIndividualDragging);

  // File validation utility
  const validateFiles = useCallback((files: File[], currentFileCount: number = 0) => {
    const allowedTypes = ['image/png', 'image/jpg', 'image/jpeg', 'image/svg+xml'];
    const maxFiles = 20;
    const maxFileSize = 10 * 1024 * 1024; // 10MB
    
    const validFiles: File[] = [];
    const errors: string[] = [];
    
    // Check total file count limit
    if (currentFileCount + files.length > maxFiles) {
      errors.push(`Maximum ${maxFiles} files allowed. You tried to add ${files.length} files but already have ${currentFileCount}.`);
      return { validFiles: [], errors };
    }
    
    files.forEach(file => {
      // Check file type
      if (!allowedTypes.includes(file.type)) {
        errors.push(`${file.name}: Only PNG, JPG, JPEG, and SVG files are supported.`);
        return;
      }
      
      // Check file size
      if (file.size > maxFileSize) {
        errors.push(`${file.name}: File size (${(file.size / 1024 / 1024).toFixed(1)}MB) exceeds 10MB limit.`);
        return;
      }
      
      // Check file extension as additional validation
      const allowedExtensions = ['.png', '.jpg', '.jpeg', '.svg'];
      const fileExtension = file.name.toLowerCase().match(/\.[^.]+$/)?.[0];
      if (!fileExtension || !allowedExtensions.includes(fileExtension)) {
        errors.push(`${file.name}: Invalid file extension. Only .png, .jpg, .jpeg, .svg files allowed.`);
        return;
      }
      
      validFiles.push(file);
    });
    
    return { validFiles, errors };
  }, []);

  const handleBatchDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsBatchDragging(false);

    const files = Array.from(e.dataTransfer.files);
    const { validFiles, errors } = validateFiles(files, batchFiles.length);

    if (errors.length > 0) {
      toast({
        title: "File Validation Errors",
        description: errors.length > 1 ? `${errors.length} errors: ${errors[0]} + ${errors.length - 1} more...` : errors[0],
        variant: "destructive",
      });
    }

    if (validFiles.length > 0) {
      setBatchFiles(prev => [...prev, ...validFiles]);
      toast({
        title: "Files Added",
        description: `Successfully added ${validFiles.length} file(s) for batch processing.`,
      });
    }
  }, [toast, validateFiles, batchFiles.length]);

  const handleIndividualDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsIndividualDragging(false);

    const files = Array.from(e.dataTransfer.files);
    const { validFiles, errors } = validateFiles(files, individualFiles.length);

    if (errors.length > 0) {
      toast({
        title: "File Validation Errors",
        description: errors.length > 1 ? `${errors.length} errors: ${errors[0]} + ${errors.length - 1} more...` : errors[0],
        variant: "destructive",
      });
    }

    if (validFiles.length > 0) {
      setIndividualFiles(prev => [...prev, ...validFiles]);
      toast({
        title: "Files Added",
        description: `Successfully added ${validFiles.length} file(s).`,
      });
    }
  }, [toast, validateFiles, individualFiles.length]);

  const handleBatchFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    
    if (files.length === 0) return;
    
    const { validFiles, errors } = validateFiles(files, batchFiles.length);

    if (errors.length > 0) {
      toast({
        title: "File Validation Errors",
        description: errors.length > 1 ? `${errors.length} errors: ${errors[0]} + ${errors.length - 1} more...` : errors[0],
        variant: "destructive",
      });
    }

    if (validFiles.length > 0) {
      setBatchFiles(prev => [...prev, ...validFiles]);
      toast({
        title: "Files Added",
        description: `Successfully added ${validFiles.length} file(s) for batch processing.`,
      });
    }
    
    // Reset the input to allow selecting the same files again if needed
    e.target.value = '';
  };

  const handleIndividualFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = Array.from(e.target.files || []);
    
    if (files.length === 0) return;
    
    const { validFiles, errors } = validateFiles(files, individualFiles.length);

    if (errors.length > 0) {
      toast({
        title: "File Validation Errors",
        description: errors.length > 1 ? `${errors.length} errors: ${errors[0]} + ${errors.length - 1} more...` : errors[0],
        variant: "destructive",
      });
    }

    if (validFiles.length > 0) {
      setIndividualFiles(prev => [...prev, ...validFiles]);
      toast({
        title: "Files Added",
        description: `Successfully added ${validFiles.length} file(s).`,
      });
    }
    
    // Reset the input to allow selecting the same files again if needed
    e.target.value = '';
  };

  const removeBatchFile = (index: number) => {
    setBatchFiles(prev => prev.filter((_, i) => i !== index));
  };

  const removeIndividualFile = (index: number) => {
    setIndividualFiles(prev => prev.filter((_, i) => i !== index));
  };

  const processBatchFiles = async () => {
    if (batchFiles.length === 0) return;

    if (!isAuthenticated) {
      toast({
        title: "Error",
        description: "Admin authentication required. Please authenticate as owner or provide admin key.",
        variant: "destructive",
      });
      return;
    }

    setIsProcessingBatch(true);

    try {
      const authHeaders = await getAuthHeaders();
      
      const formData = new FormData();
      batchFiles.forEach(file => {
        formData.append("files", file);
      });
      formData.append("threshold", threshold[0].toString());
      formData.append("invert", invert.toString());

      const response = await fetch("/api/icons/import-individual-files", {
        method: "POST",
        body: formData,
        headers: authHeaders
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to process batch files");
      }

      const data = await response.json();
      const icons = data.icons.map((icon: any, index: number) => ({
        index,
        name: icon.filename.replace(/\.[^/.]+$/, ""), // Remove file extension
        svg: icon.svg,
        selected: true
      }));
      
      setBatchProcessedIcons(icons);
      
      let message = `Successfully processed ${data.successful} icons`;
      if (data.failed > 0) {
        message += `, ${data.failed} failed`;
      }
      
      toast({
        title: "Success",
        description: message,
      });

      if (data.errors && data.errors.length > 0) {
        console.warn("Batch file processing errors:", data.errors);
      }
    } catch (error: any) {
      console.error("Batch files processing error:", error);
      toast({
        title: "Error",
        description: error.message || "Failed to process batch files",
        variant: "destructive",
      });
    } finally {
      setIsProcessingBatch(false);
    }
  };

  const processIndividualFiles = async () => {
    if (individualFiles.length === 0) return;

    if (!isAuthenticated) {
      toast({
        title: "Error",
        description: "Admin authentication required. Please authenticate as owner or provide admin key.",
        variant: "destructive",
      });
      return;
    }

    setIsProcessingIndividual(true);

    try {
      const authHeaders = await getAuthHeaders();
      
      const formData = new FormData();
      individualFiles.forEach(file => {
        formData.append("files", file);
      });
      formData.append("threshold", threshold[0].toString());
      formData.append("invert", invert.toString());

      const response = await fetch("/api/icons/import-individual-files", {
        method: "POST",
        body: formData,
        headers: authHeaders
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to process individual files");
      }

      const data = await response.json();
      const icons = data.icons.map((icon: any, index: number) => ({
        index,
        name: icon.filename.replace(/\.[^/.]+$/, ""), // Remove file extension
        svg: icon.svg,
        selected: true
      }));
      
      setIndividualProcessedIcons(icons);
      
      let message = `Successfully processed ${data.successful} icons`;
      if (data.failed > 0) {
        message += `, ${data.failed} failed`;
      }
      
      toast({
        title: "Success",
        description: message,
      });

      if (data.errors && data.errors.length > 0) {
        console.warn("Individual file processing errors:", data.errors);
      }
    } catch (error: any) {
      console.error("Individual files processing error:", error);
      toast({
        title: "Error",
        description: error.message || "Failed to process individual files",
        variant: "destructive",
      });
    } finally {
      setIsProcessingIndividual(false);
    }
  };

  const toggleBatchIconSelection = (index: number) => {
    setBatchProcessedIcons(prev => 
      prev.map(icon => 
        icon.index === index 
          ? { ...icon, selected: !icon.selected }
          : icon
      )
    );
  };

  const updateBatchIconName = (index: number, name: string) => {
    setBatchProcessedIcons(prev => 
      prev.map(icon => 
        icon.index === index 
          ? { ...icon, name }
          : icon
      )
    );
  };

  const toggleIndividualIconSelection = (index: number) => {
    setIndividualProcessedIcons(prev => 
      prev.map(icon => 
        icon.index === index 
          ? { ...icon, selected: !icon.selected }
          : icon
      )
    );
  };

  const updateIndividualIconName = (index: number, name: string) => {
    setIndividualProcessedIcons(prev => 
      prev.map(icon => 
        icon.index === index 
          ? { ...icon, name }
          : icon
      )
    );
  };

  const saveBatchIcons = useMutation({
    mutationFn: async () => {
      if (!isAuthenticated) {
        throw new Error("Admin authentication required");
      }

      const authHeaders = await getAuthHeaders();
      
      const selectedIcons = batchProcessedIcons
        .filter(icon => icon.selected)
        .map(icon => ({
          name: icon.name,
          svg: icon.svg,
          style: "flat" as const,
          tags: []
        }));

      if (selectedIcons.length === 0) {
        throw new Error("No icons selected");
      }

      const response = await fetch("/api/icons/save-imported", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          ...authHeaders
        },
        body: JSON.stringify({ icons: selectedIcons })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to save icons");
      }

      return response.json();
    },
    onSuccess: (data) => {
      toast({
        title: "Success",
        description: `Saved ${data.count || data.saved} batch icons to Icons Library`,
      });
      // Reset the form
      setBatchFiles([]);
      setBatchProcessedIcons([]);
      // Refresh the main Icons Library (object storage)
      queryClient.invalidateQueries({ queryKey: ["/api/icons/list"] });
      // Also refresh imported icons for the admin view
      queryClient.invalidateQueries({ queryKey: ["/api/icons/imported"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to save batch icons",
        variant: "destructive",
      });
    }
  });

  const saveIndividualIcons = useMutation({
    mutationFn: async () => {
      if (!isAuthenticated) {
        throw new Error("Admin authentication required");
      }

      const authHeaders = await getAuthHeaders();
      
      const selectedIcons = individualProcessedIcons
        .filter(icon => icon.selected)
        .map(icon => ({
          name: icon.name,
          svg: icon.svg,
          style: "flat" as const,
          tags: []
        }));

      if (selectedIcons.length === 0) {
        throw new Error("No icons selected");
      }

      const response = await fetch("/api/icons/save-imported", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          ...authHeaders
        },
        body: JSON.stringify({ icons: selectedIcons })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to save icons");
      }

      return response.json();
    },
    onSuccess: (data) => {
      toast({
        title: "Success",
        description: `Saved ${data.count || data.saved} individual icons to Icons Library`,
      });
      // Reset the form
      setIndividualFiles([]);
      setIndividualProcessedIcons([]);
      // Refresh the main Icons Library (object storage)
      queryClient.invalidateQueries({ queryKey: ["/api/icons/list"] });
      // Also refresh imported icons for the admin view
      queryClient.invalidateQueries({ queryKey: ["/api/icons/imported"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to save individual icons",
        variant: "destructive",
      });
    }
  });

  const downloadIcon = (icon: ProcessedIcon) => {
    const blob = new Blob([icon.svg], { type: 'image/svg+xml' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${icon.name}.svg`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  // Delete imported icon mutation
  const deleteImportedIcon = useMutation({
    mutationFn: async (iconId: string) => {
      if (!isAuthenticated) {
        throw new Error("Admin authentication required");
      }

      const authHeaders = await getAuthHeaders();
      
      const response = await fetch(`/api/icons/imported/${iconId}`, {
        method: "DELETE",
        headers: authHeaders
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to delete icon");
      }

      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Icon deleted successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/icons/imported"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete icon",
        variant: "destructive",
      });
    }
  });

  const filteredImportedIcons = importedIcons.filter((icon: ImportedIcon) =>
    icon.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    icon.tags.some(tag => tag.toLowerCase().includes(searchTerm.toLowerCase()))
  );

  return (
    <DashboardTemplatePage
      title="Icon Importer"
      description="Upload files to extract icons as clean SVGs with batch and individual processing options"
    >
      <div className="space-y-6">
        {/* Shared Controls */}
        <Card data-testid="card-shared-controls">
          <CardHeader>
            <CardTitle>Conversion Settings</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Threshold Slider */}
              <div className="space-y-3">
                <Label htmlFor="threshold-slider" className="text-sm font-medium">
                  Threshold: {threshold[0]} (PNG/JPG conversion quality)
                </Label>
                <Slider
                  id="threshold-slider"
                  min={0}
                  max={255}
                  step={1}
                  value={threshold}
                  onValueChange={setThreshold}
                  className="w-full"
                  data-testid="slider-threshold"
                />
                <p className="text-xs text-muted-foreground">
                  Higher values create more detailed SVGs from PNG/JPG files
                </p>
              </div>

              {/* Invert Switch */}
              <div className="space-y-3">
                <div className="flex items-center space-x-2">
                  <Switch
                    id="invert-switch"
                    checked={invert}
                    onCheckedChange={setInvert}
                    data-testid="switch-invert"
                  />
                  <Label htmlFor="invert-switch" className="text-sm font-medium">
                    Invert Colors
                  </Label>
                </div>
                <p className="text-xs text-muted-foreground">
                  Converts dark images to light and vice versa during PNG/JPG processing
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Tabs for Upload Methods */}
        <Tabs value={activeTab} onValueChange={setActiveTab} data-testid="tabs-upload-methods">
          <TabsList className="grid w-full grid-cols-2 bg-orange-600 dark:bg-orange-700">
            <TabsTrigger 
              value="batch" 
              data-testid="tab-trigger-batch"
              className="data-[state=active]:bg-white data-[state=active]:text-orange-600 data-[state=inactive]:bg-transparent data-[state=inactive]:text-white"
            >
              <Package className="w-4 h-4 mr-2" />
              Batch Upload
            </TabsTrigger>
            <TabsTrigger 
              value="individual" 
              data-testid="tab-trigger-individual"
              className="data-[state=active]:bg-white data-[state=active]:text-orange-600 data-[state=inactive]:bg-transparent data-[state=inactive]:text-white"
            >
              <FileImage className="w-4 h-4 mr-2" />
              Individual Files
            </TabsTrigger>
          </TabsList>

          {/* Batch Upload Tab */}
          <TabsContent value="batch" className="space-y-6">
            <Card data-testid="card-batch-upload">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  Batch Upload (Multiple Files)
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* Batch Drag & Drop Area */}
                <div
                  onDragEnter={batchDragHandlers.handleDragEnter}
                  onDragLeave={batchDragHandlers.handleDragLeave}
                  onDragOver={batchDragHandlers.handleDragOver}
                  onDrop={handleBatchDrop}
                  className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
                    isBatchDragging 
                      ? 'border-primary bg-primary/5' 
                      : 'border-gray-300 dark:border-gray-600 hover:border-primary/50'
                  }`}
                  data-testid="dropzone-batch-files"
                >
                  <Package className="mx-auto h-16 w-16 text-gray-400 mb-4" />
                  <div className="space-y-2">
                    <p className="text-xl font-medium">
                      Drop multiple files here or{" "}
                      <label className="text-primary cursor-pointer hover:underline">
                        browse
                        <input
                          type="file"
                          multiple
                          accept="image/*,.svg"
                          onChange={handleBatchFileSelect}
                          className="hidden"
                          data-testid="input-batch-files"
                        />
                      </label>
                    </p>
                    <p className="text-sm text-gray-500">
                      PNG, JPG, JPEG, SVG files • Max 20 files • 10MB per file
                    </p>
                    <p className="text-sm text-primary font-medium">
                      Perfect for processing multiple icons at once
                    </p>
                  </div>
                </div>

                {/* Batch Selected Files List */}
                {batchFiles.length > 0 && (
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <h4 className="font-medium">Selected Files ({batchFiles.length})</h4>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setBatchFiles([])}
                        data-testid="button-clear-batch-files"
                      >
                        Clear All
                      </Button>
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-2 max-h-60 overflow-y-auto">
                      {batchFiles.map((file, index) => (
                        <div 
                          key={`${file.name}-${index}`}
                          className="flex items-center justify-between p-2 bg-gray-50 dark:bg-gray-800 rounded text-sm"
                          data-testid={`batch-file-item-${index}`}
                        >
                          <div className="flex-1 truncate">
                            <span className="font-medium">{file.name}</span>
                            <span className="text-gray-500 ml-2">
                              ({(file.size / 1024).toFixed(1)} KB)
                            </span>
                          </div>
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => removeBatchFile(index)}
                            data-testid={`button-remove-batch-file-${index}`}
                          >
                            <X className="h-4 w-4" />
                          </Button>
                        </div>
                      ))}
                    </div>

                    <Button
                      onClick={processBatchFiles}
                      disabled={isProcessingBatch}
                      className="w-full"
                      size="lg"
                      data-testid="button-convert-all-batch"
                    >
                      {isProcessingBatch ? (
                        <>
                          <Clock className="w-5 h-5 mr-2 animate-spin" />
                          Converting All to SVG...
                        </>
                      ) : (
                        <>
                          <Package className="w-5 h-5 mr-2" />
                          Convert All to SVG ({batchFiles.length} files)
                        </>
                      )}
                    </Button>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Batch Results Section */}
            {batchProcessedIcons.length > 0 && (
              <Card data-testid="card-batch-results">
                <CardHeader>
                  <CardTitle className="flex items-center justify-between">
                    <span>Batch Results ({batchProcessedIcons.filter(i => i.selected).length} selected)</span>
                    <Button
                      onClick={() => saveBatchIcons.mutate()}
                      disabled={batchProcessedIcons.filter(i => i.selected).length === 0 || saveBatchIcons.isPending}
                      data-testid="button-save-batch-icons"
                    >
                      <Save className="w-4 h-4 mr-2" />
                      {saveBatchIcons.isPending ? "Saving..." : "Save Selected"}
                    </Button>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
                    {batchProcessedIcons.map((icon) => (
                      <div
                        key={icon.index}
                        className={`border rounded-lg p-3 transition-all ${
                          icon.selected ? 'border-primary bg-primary/5' : 'border-gray-200'
                        }`}
                        data-testid={`batch-icon-item-${icon.index}`}
                      >
                        <div className="flex items-center justify-between mb-2">
                          <input
                            type="checkbox"
                            checked={icon.selected}
                            onChange={() => toggleBatchIconSelection(icon.index)}
                            data-testid={`checkbox-batch-select-${icon.index}`}
                          />
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => downloadIcon(icon)}
                            data-testid={`button-batch-download-${icon.index}`}
                          >
                            <Download className="w-4 h-4" />
                          </Button>
                        </div>
                        
                        <div 
                          className="w-full h-16 mb-2 flex items-center justify-center bg-gray-50 rounded"
                          dangerouslySetInnerHTML={{ __html: icon.svg }}
                        />
                        
                        <input
                          type="text"
                          value={icon.name}
                          onChange={(e) => updateBatchIconName(icon.index, e.target.value)}
                          className="w-full text-xs px-2 py-1 border rounded"
                          placeholder="Icon name"
                          data-testid={`input-batch-name-${icon.index}`}
                        />
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Individual Files Tab */}
          <TabsContent value="individual" className="space-y-6">
            <Card data-testid="card-individual-upload">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <FileImage className="h-5 w-5" />
                  Individual Files (One at a Time)
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {/* Individual Drag & Drop Area */}
                <div
                  onDragEnter={individualDragHandlers.handleDragEnter}
                  onDragLeave={individualDragHandlers.handleDragLeave}
                  onDragOver={individualDragHandlers.handleDragOver}
                  onDrop={handleIndividualDrop}
                  className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${
                    isIndividualDragging 
                      ? 'border-primary bg-primary/5' 
                      : 'border-gray-300 dark:border-gray-600 hover:border-primary/50'
                  }`}
                  data-testid="dropzone-individual-files"
                >
                  <Upload className="mx-auto h-12 w-12 text-gray-400 mb-4" />
                  <div className="space-y-2">
                    <p className="text-lg font-medium">
                      Drop files here or{" "}
                      <label className="text-primary cursor-pointer hover:underline">
                        browse
                        <input
                          type="file"
                          multiple
                          accept="image/*,.svg"
                          onChange={handleIndividualFileSelect}
                          className="hidden"
                          data-testid="input-individual-files"
                        />
                      </label>
                    </p>
                    <p className="text-sm text-gray-500">
                      PNG, JPG, JPEG, SVG files • Max 20 files • 10MB per file
                    </p>
                  </div>
                </div>

                {/* Individual Selected Files List */}
                {individualFiles.length > 0 && (
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <h4 className="font-medium">Selected Files ({individualFiles.length})</h4>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setIndividualFiles([])}
                        data-testid="button-clear-individual-files"
                      >
                        Clear All
                      </Button>
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-2 max-h-40 overflow-y-auto">
                      {individualFiles.map((file, index) => (
                        <div 
                          key={`${file.name}-${index}`}
                          className="flex items-center justify-between p-2 bg-gray-50 dark:bg-gray-800 rounded text-sm"
                          data-testid={`individual-file-item-${index}`}
                        >
                          <div className="flex-1 truncate">
                            <span className="font-medium">{file.name}</span>
                            <span className="text-gray-500 ml-2">
                              ({(file.size / 1024).toFixed(1)} KB)
                            </span>
                          </div>
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => removeIndividualFile(index)}
                            data-testid={`button-remove-individual-file-${index}`}
                          >
                            <X className="h-4 w-4" />
                          </Button>
                        </div>
                      ))}
                    </div>

                    <Button
                      onClick={processIndividualFiles}
                      disabled={isProcessingIndividual}
                      className="w-full"
                      data-testid="button-process-individual-files"
                    >
                      {isProcessingIndividual ? (
                        <>
                          <Clock className="w-4 h-4 mr-2 animate-spin" />
                          Processing...
                        </>
                      ) : (
                        <>
                          <Upload className="w-4 h-4 mr-2" />
                          Process Files ({individualFiles.length})
                        </>
                      )}
                    </Button>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Individual Files Results Section */}
            {individualProcessedIcons.length > 0 && (
              <Card data-testid="card-individual-results">
                <CardHeader>
                  <CardTitle className="flex items-center justify-between">
                    <span>Individual Results ({individualProcessedIcons.filter(i => i.selected).length} selected)</span>
                    <Button
                      onClick={() => saveIndividualIcons.mutate()}
                      disabled={individualProcessedIcons.filter(i => i.selected).length === 0 || saveIndividualIcons.isPending}
                      data-testid="button-save-individual-icons"
                    >
                      <Save className="w-4 h-4 mr-2" />
                      {saveIndividualIcons.isPending ? "Saving..." : "Save Selected"}
                    </Button>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
                    {individualProcessedIcons.map((icon) => (
                      <div
                        key={icon.index}
                        className={`border rounded-lg p-3 transition-all ${
                          icon.selected ? 'border-primary bg-primary/5' : 'border-gray-200'
                        }`}
                        data-testid={`individual-icon-item-${icon.index}`}
                      >
                        <div className="flex items-center justify-between mb-2">
                          <input
                            type="checkbox"
                            checked={icon.selected}
                            onChange={() => toggleIndividualIconSelection(icon.index)}
                            data-testid={`checkbox-individual-select-${icon.index}`}
                          />
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => downloadIcon(icon)}
                            data-testid={`button-individual-download-${icon.index}`}
                          >
                            <Download className="w-4 h-4" />
                          </Button>
                        </div>
                        
                        <div 
                          className="w-full h-16 mb-2 flex items-center justify-center bg-gray-50 rounded"
                          dangerouslySetInnerHTML={{ __html: icon.svg }}
                        />
                        
                        <input
                          type="text"
                          value={icon.name}
                          onChange={(e) => updateIndividualIconName(icon.index, e.target.value)}
                          className="w-full text-xs px-2 py-1 border rounded"
                          placeholder="Icon name"
                          data-testid={`input-individual-name-${icon.index}`}
                        />
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>
        </Tabs>

        {/* Manage Imported Icons Section */}
        <Card data-testid="card-manage-icons">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Tag className="h-5 w-5" />
              Manage Imported Icons
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            {/* Search Section */}
            <div className="flex items-center space-x-4">
              <div className="flex-1">
                <Input
                  type="text"
                  placeholder="Search by name or tags..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full"
                  data-testid="input-search-imported"
                />
              </div>
              <Search className="h-5 w-5 text-muted-foreground" />
            </div>

            {/* Stats */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <Card>
                <CardContent className="p-4">
                  <div className="text-2xl font-bold text-primary">{importedIcons.length}</div>
                  <div className="text-sm text-muted-foreground">Total Icons</div>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4">
                  <div className="text-2xl font-bold text-primary">{filteredImportedIcons.length}</div>
                  <div className="text-sm text-muted-foreground">Filtered Results</div>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4">
                  <div className="text-2xl font-bold text-primary">
                    {Array.from(new Set(importedIcons.flatMap((icon: ImportedIcon) => icon.tags))).length}
                  </div>
                  <div className="text-sm text-muted-foreground">Unique Tags</div>
                </CardContent>
              </Card>
            </div>

            {/* Icons Grid */}
            {isLoadingImported ? (
              <div className="flex items-center justify-center h-32">
                <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div>
              </div>
            ) : filteredImportedIcons.length === 0 ? (
              <div className="text-center py-12">
                <Tag className="mx-auto h-12 w-12 text-gray-400" />
                <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-gray-100">No icons found</h3>
                <p className="mt-1 text-sm text-gray-500">
                  {importedIcons.length === 0 
                    ? "Import some icons first using the upload section above"
                    : "Try a different search term"
                  }
                </p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredImportedIcons.map((icon: ImportedIcon) => (
                  <div
                    key={icon.id}
                    className="border rounded-lg p-4 space-y-3"
                    data-testid={`imported-icon-card-${icon.id}`}
                  >
                    {/* Icon Preview */}
                    <div className="flex items-center justify-center h-16 bg-gray-50 dark:bg-gray-800 rounded">
                      <div 
                        className="w-12 h-12"
                        dangerouslySetInnerHTML={{ __html: icon.svg }}
                      />
                    </div>

                    {/* Icon Info */}
                    <div className="space-y-2">
                      <div className="text-sm font-medium truncate" title={icon.name}>
                        {icon.name}
                      </div>
                      
                      {/* Tags */}
                      <div className="flex flex-wrap gap-1">
                        {icon.tags.map((tag, index) => (
                          <Badge key={index} variant="secondary" className="text-xs">
                            {tag}
                          </Badge>
                        ))}
                      </div>
                      
                      {/* Style */}
                      <div className="text-xs text-muted-foreground">
                        Style: {icon.style}
                      </div>
                    </div>

                    {/* Actions */}
                    <div className="flex gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          const blob = new Blob([icon.svg], { type: 'image/svg+xml' });
                          const url = URL.createObjectURL(blob);
                          const a = document.createElement('a');
                          a.href = url;
                          a.download = `${icon.name}.svg`;
                          document.body.appendChild(a);
                          a.click();
                          document.body.removeChild(a);
                          URL.revokeObjectURL(url);
                        }}
                        data-testid={`button-download-imported-${icon.id}`}
                      >
                        <Download className="w-4 h-4 mr-1" />
                        Download
                      </Button>
                      <Button
                        variant="destructive"
                        size="sm"
                        onClick={() => deleteImportedIcon.mutate(icon.id)}
                        disabled={deleteImportedIcon.isPending}
                        data-testid={`button-delete-imported-${icon.id}`}
                      >
                        <Trash2 className="w-4 h-4 mr-1" />
                        {deleteImportedIcon.isPending ? "Deleting..." : "Delete"}
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </DashboardTemplatePage>
  );
}