import React, { useRef, useState, useEffect } from "react";
import { useLocation } from "wouter";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { useAuth } from "@/contexts/AuthContext";
import { uploadTemplate } from "@/services/templates";
import { buildTemplateId } from "@/utils/slugify";
import { toast } from "sonner";
import { ref, uploadBytes } from "firebase/storage";
import { doc, setDoc } from "firebase/firestore";
import { storage, db } from "@/utils/firebase-templates";

type FormState = {
  name: string;
  tags: string;
};

export default function AdminUploadLogoTemplate() {
  const { currentUser } = useAuth();
  const [location, setLocation] = useLocation();
  const [svgFile, setSvgFile] = useState<File | null>(null);
  const [previewFile, setPreviewFile] = useState<File | null>(null);

  const [svgContent, setSvgContent] = useState<string>("");
  const [previewImageUrl, setPreviewImageUrl] = useState<string>("");

  const [msg, setMsg] = useState<string>("");
  const [busy, setBusy] = useState(false);
  const [previewTitle, setPreviewTitle] = useState<string>("Logo Template");
  const [generatedId, setGeneratedId] = useState<string>("");
  const [uploadSuccess, setUploadSuccess] = useState(false);
  const [uploadFailed, setUploadFailed] = useState(false);
  const [hasAttemptedUpload, setHasAttemptedUpload] = useState(false);

  const [form, setForm] = useState<FormState>({
    name: "",
    tags: "",
  });
  
  // Auto-generate system ID from name
  const slugifyLogoName = (name: string): string => {
    return name
      .toLowerCase()
      .trim()
      .replace(/[^a-z0-9\s-]/g, '') // Remove special chars except spaces and hyphens
      .replace(/\s+/g, '-') // Replace spaces with hyphens
      .replace(/-+/g, '-') // Replace multiple hyphens with single
      .replace(/^-|-$/g, ''); // Remove leading/trailing hyphens
  };

  // Update generated ID when name changes
  useEffect(() => {
    if (form.name) {
      setGeneratedId(buildTemplateId(form.name, 1));
    } else {
      setGeneratedId("");
    }
  }, [form.name]);

  // Test Firebase connectivity
  const testFirebaseConnection = async () => {
    try {
      console.log("🔍 Testing Firebase connection...");
      
      // Test 1: Firestore write
      const testDoc = doc(db, "test", "connectivity");
      await setDoc(testDoc, {
        timestamp: new Date(),
        test: "connection_check"
      });
      console.log("✅ Firestore write successful");
      
      // Test 2: Storage write
      const testFile = new Blob(['test'], { type: 'text/plain' });
      const testRef = ref(storage, 'test/connectivity.txt');
      await uploadBytes(testRef, testFile);
      console.log("✅ Storage write successful");
      
      toast.success("Firebase connection working!");
      return true;
    } catch (error) {
      console.error("❌ Firebase connection failed:", error);
      toast.error(`Firebase error: ${error.code || error.message}`);
      return false;
    }
  };

  // Handle Firebase template upload following SuperNova's spec
  const handleAddTemplate = async () => {
    if ((!svgFile && !previewFile) || !form.name.trim()) {
      toast.error("Please upload at least one file (SVG or PNG/JPG) and enter a logo name");
      return;
    }

    if (!currentUser) {
      toast.error("User not authenticated");
      return;
    }

    setBusy(true);
    setUploadSuccess(false);
    setUploadFailed(false);
    setHasAttemptedUpload(true);
    setMsg("🔄 Starting upload process...");

    try {
      // Build geometry object with minimal canvas data
      const geometry = {
        canvas: { w: 1200, h: 900 }, // Default canvas size
      };

      // Parse tags from comma-separated string
      const tags = form.tags.split(",").map(t => t.trim()).filter(Boolean);

      setMsg("🔄 Uploading to Firebase...");

      // Upload template using SN's dual-file service
      const { templateId } = await uploadTemplate({
        name: form.name,
        description: form.name, // Use name as description for now
        tags,
        version: 1,
        svgFile: svgFile || undefined,        // Pass SVG if available
        rasterFile: previewFile || undefined, // Pass raster if available
        geometry,
        defaults: {
          brandName: "Your Brand",
          tagline: "Your Tagline", 
          estYear: "2025"
        },
        public: true // Make templates public by default
      });

      toast.success(`Template saved successfully: ${templateId}`);
      setMsg(`✅ Template uploaded to Firebase: ${templateId}`);
      setUploadSuccess(true);
      setUploadFailed(false);

      // Refresh preview to show uploaded state
      if (svgFile) {
        await loadSvgForPreview(svgFile);
      }

      // Navigate to Logo Templates page after successful upload
      setTimeout(() => {
        setLocation("/business-assets/logo-templates");
      }, 1500); // Short delay to show success message

    } catch (error) {
      console.error("Failed to upload template:", error);
      console.error("Error details:", {
        message: error?.message || "No message",
        code: error?.code || "No code", 
        stack: error?.stack || "No stack",
        full: JSON.stringify(error, null, 2)
      });
      
      const errorMessage = error instanceof Error ? error.message : 
        error?.code ? `Firebase error: ${error.code}` : 
        "Unknown upload error";
      
      toast.error("Upload failed: " + errorMessage);
      setMsg("❌ Upload failed: " + errorMessage);
      setUploadSuccess(false);
      setUploadFailed(true);
    } finally {
      setBusy(false);
    }
  };

  const svgInputRef = useRef<HTMLInputElement>(null);
  const previewInputRef = useRef<HTMLInputElement>(null);

  // ---------- Utilities ----------
  const readText = (file: File) =>
    new Promise<string>((resolve, reject) => {
      const reader = new FileReader();
      reader.onerror = () => reject(new Error("Failed to read file"));
      reader.onload = () => resolve(String(reader.result || ""));
      reader.readAsText(file);
    });

  const readAsDataUrl = (file: File) =>
    new Promise<string>((resolve, reject) => {
      const reader = new FileReader();
      reader.onerror = () => reject(new Error("Failed to read file"));
      reader.onload = () => resolve(String(reader.result || ""));
      reader.readAsDataURL(file);
    });

  // Drop handling (works for both zones)
  const handleDrop = async (
    e: React.DragEvent<HTMLDivElement>,
    kind: "svg" | "preview"
  ) => {
    e.preventDefault();
    const f = e.dataTransfer.files?.[0];
    if (!f) return;
    setMsg("");

    if (kind === "svg") {
      if (f.type && f.type !== "image/svg+xml" && !f.name.toLowerCase().endsWith(".svg")) {
        return setMsg("SVG drop rejected: not an SVG file.");
      }
      
      // Clear the preview file when dropping SVG
      if (previewFile) {
        setPreviewFile(null);
        if (previewInputRef.current) {
          previewInputRef.current.value = "";
        }
      }
      
      await loadSvgForPreview(f);
      setSvgFile(f);
      if (!form.name) setPreviewTitle(f.name.replace(/\.[^.]+$/, "") || "Logo Template");
    } else {
      if (
        f.type &&
        !/^image\/(png|jpeg|jpg)$/i.test(f.type) &&
        !/\.(png|jpe?g)$/i.test(f.name)
      ) {
        return setMsg("Preview must be PNG or JPG.");
      }
      
      // Clear the SVG file when dropping preview image  
      if (svgFile) {
        setSvgFile(null);
        setSvgContent("");
        if (svgInputRef.current) {
          svgInputRef.current.value = "";
        }
      }
      
      await loadPreviewImage(f);
      setPreviewFile(f);
      if (!form.name && !svgFile)
        setPreviewTitle(f.name.replace(/\.[^.]+$/, "") || "Logo Template");
    }
  };

  // ---------- SVG Preview Pipeline ----------
  const sanitizeSvg = (raw: string) => {
    // strip <script> blocks
    let out = raw.replace(/<script[\s\S]*?>[\s\S]*?<\/script>/gi, "");
    // strip inline on* handlers (onload, onclick, etc.)
    out = out.replace(/\son[a-z]+\s*=\s*(['"]).*?\1/gi, "");
    return out;
  };

  const injectTokensAndVars = (raw: string) => {
    try {
      console.log('Processing SVG for preview...');
      
      let text = raw;
      
      // Add CSS variables by injecting into the opening <svg> tag
      const cssVars = "--primary:#231f20;--secondary:#978752;--accent:#6dc282;";
      text = text.replace(
        /<svg([^>]*?)>/i,
        (match, attributes) => {
          // Check if style attribute already exists
          if (attributes.includes('style=')) {
            return match.replace(
              /style=(["'])([^"']*?)\1/i,
              (styleMatch, quote, existingStyle) => {
                const cleanStyle = existingStyle.trim().replace(/;?$/, ';');
                return `style=${quote}${cleanStyle}${cssVars}${quote}`;
              }
            );
          } else {
            return `<svg${attributes} style="${cssVars}">`;
          }
        }
      );
      
      console.log('CSS variables injected, SVG processed successfully');
      return text;
    } catch (error) {
      console.error('SVG processing failed:', error);
      return raw; // Return original if processing fails
    }
  };

  const convertSvgToImage = async (svgString: string): Promise<string> => {
    return new Promise((resolve, reject) => {
      try {
        // Create a blob from the SVG string
        const blob = new Blob([svgString], { type: 'image/svg+xml' });
        const url = URL.createObjectURL(blob);
        
        // Create an image element to load the SVG
        const img = new Image();
        img.onload = () => {
          try {
            // Create a canvas to render the image
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            
            if (!ctx) {
              reject(new Error('Could not get canvas context'));
              return;
            }
            
            // Set canvas size based on SVG dimensions or default size
            const size = Math.min(img.width || 400, img.height || 400, 400);
            canvas.width = size;
            canvas.height = size;
            
            // Draw the image onto the canvas
            ctx.drawImage(img, 0, 0, size, size);
            
            // Convert canvas to PNG data URL
            const dataUrl = canvas.toDataURL('image/png');
            
            // Clean up
            URL.revokeObjectURL(url);
            resolve(dataUrl);
          } catch (error) {
            URL.revokeObjectURL(url);
            reject(error);
          }
        };
        
        img.onerror = () => {
          URL.revokeObjectURL(url);
          reject(new Error('Failed to load SVG image'));
        };
        
        img.src = url;
      } catch (error) {
        reject(error);
      }
    });
  };

  const loadSvgForPreview = async (file: File) => {
    try {
      const raw = await readText(file);
      
      // More flexible SVG validation - just check if it contains <svg> tags
      if (!/<svg[\s\S]*?<\/svg>/i.test(raw)) {
        setSvgContent("");
        setPreviewImageUrl("");
        setMsg("SVG preview failed: not a valid <svg>…</svg> document.");
        return;
      }
      
      const clean = sanitizeSvg(raw);
      const processedSvg = injectTokensAndVars(clean);
      
      // Store the original SVG content for dangerouslySetInnerHTML fallback
      setSvgContent(processedSvg);
      
      // Convert SVG to PNG for reliable preview
      try {
        const pngDataUrl = await convertSvgToImage(processedSvg);
        setPreviewImageUrl(pngDataUrl);
      } catch (conversionError) {
        // Fallback to direct SVG rendering if conversion fails
        setPreviewImageUrl("");
      }
    } catch (err) {
      setSvgContent("");
      setPreviewImageUrl("");
      setMsg("Failed to load SVG preview: " + (err instanceof Error ? err.message : String(err)));
    }
  };

  // ---------- Image Preview ----------
  const loadPreviewImage = async (file: File) => {
    try {
      const url = await readAsDataUrl(file);
      setPreviewImageUrl(url);
    } catch (err) {
      setPreviewImageUrl("");
      setMsg("Failed to load image preview.");
    }
  };

  // ---------- Effects: react to file changes from file inputs ----------
  React.useEffect(() => {
    if (!svgFile) {
      setSvgContent("");
      return;
    }
    loadSvgForPreview(svgFile);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [svgFile]);

  React.useEffect(() => {
    if (!previewFile) {
      setPreviewImageUrl("");
      return;
    }
    loadPreviewImage(previewFile);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [previewFile]);

  return (
    <DashboardTemplatePage title="Upload Logo Template">
      <form
        className="grid lg:grid-cols-[2fr_1fr] gap-8"
        onSubmit={(e) => {
          e.preventDefault();
          handleAddTemplate();
        }}
        data-testid="form-upload-template"
      >
        {/* Left Side - Preview */}
        <section className="space-y-4">
          <div className="bg-white rounded-xl border border-gray-200 p-6">
            <div className="mb-6 text-lg font-semibold text-gray-900">
              {form.name || previewTitle}
            </div>

            <div className="bg-gray-50 rounded-lg p-6 min-h-[400px] flex items-center justify-center">
              {previewImageUrl ? (
                <div className="relative w-full h-full flex items-center justify-center">
                  <img
                    src={previewImageUrl}
                    alt="Logo Preview"
                    style={{
                      maxWidth: "100%",
                      maxHeight: "350px",
                      objectFit: "contain",
                    }}
                  />
                </div>
              ) : svgContent ? (
                <div className="relative">
                  <div 
                    data-logo-preview 
                    dangerouslySetInnerHTML={{ __html: svgContent }}
                    style={{ maxWidth: "100%", maxHeight: "350px" }}
                  />
                </div>
              ) : (
                <div className="text-gray-400 text-center">
                  <div className="text-lg mb-2">Logo Preview</div>
                  <div className="text-sm">
                    Upload an SVG or Preview file to see the preview
                  </div>
                </div>
              )}
            </div>

            <div className="flex items-center justify-between mt-3">
              <div className="text-xs text-gray-500">
                {previewFile && previewImageUrl
                  ? "Image Preview from uploaded PNG/JPG file"
                  : svgContent
                  ? "SVG preview"
                  : "Preview will show here after uploading an SVG or Preview file"}
              </div>
              
              {/* Upload Status Indicator - Bottom Right of Frame */}
              <div className="flex items-center space-x-2 bg-white/95 backdrop-blur-sm rounded-full px-3 py-1.5 border shadow-lg">
                <span className="text-sm font-medium text-gray-700">Uploaded</span>
                <div className={`w-3 h-3 rounded-full ${
                  uploadSuccess ? 'bg-green-500' : 
                  uploadFailed ? 'bg-red-500' : 
                  busy ? 'bg-yellow-500' :
                  'bg-gray-300'
                }`}></div>
              </div>
            </div>

          </div>
        </section>

        {/* Right Side - Form and Uploads */}
        <section className="space-y-4">
          <FileDrop
            label="SVG Download"
            accept=".svg,image/svg+xml"
            file={svgFile}
            onRemove={() => {
              setSvgFile(null);
              setSvgContent("");
              // If no separate preview file, also clear the preview image (converted SVG)
              if (!previewFile) {
                setPreviewImageUrl("");
              }
            }}
            onChoose={() => svgInputRef.current?.click()}
            onDrop={(e) => handleDrop(e, "svg")}
          />

          <input
            ref={svgInputRef}
            type="file"
            accept=".svg,image/svg+xml"
            className="hidden"
            onChange={async (e) => {
              const f = e.target.files?.[0] || null;
              setMsg("");
              setSvgFile(f);
              
              // Clear the preview file when uploading SVG
              if (f && previewFile) {
                setPreviewFile(null);
                if (previewInputRef.current) {
                  previewInputRef.current.value = "";
                }
              }
              
              if (f) {
                await loadSvgForPreview(f);
                if (!form.name) setPreviewTitle(f.name.replace(/\.[^.]+$/, ""));
              } else {
                setSvgContent("");
              }
            }}
            data-testid="input-svg-file"
          />

          <FileDrop
            label="PNG/JPG Download"
            accept=".png,.jpg,.jpeg,image/png,image/jpeg"
            file={previewFile}
            onRemove={() => {
              setPreviewFile(null);
              // Always clear preview image when removing PNG/JPG file
              setPreviewImageUrl("");
            }}
            onChoose={() => previewInputRef.current?.click()}
            onDrop={(e) => handleDrop(e, "preview")}
          />

          <input
            ref={previewInputRef}
            type="file"
            accept=".png,.jpg,.jpeg,image/png,image/jpeg"
            className="hidden"
            onChange={async (e) => {
              const f = e.target.files?.[0] || null;
              setMsg("");
              setPreviewFile(f);
              
              // Clear the SVG file when uploading preview image
              if (f && svgFile) {
                setSvgFile(null);
                setSvgContent("");
                if (svgInputRef.current) {
                  svgInputRef.current.value = "";
                }
              }
              
              if (f) {
                await loadPreviewImage(f);
                if (!form.name && !svgFile)
                  setPreviewTitle(f.name.replace(/\.[^.]+$/, ""));
              } else {
                setPreviewImageUrl("");
              }
            }}
            data-testid="input-preview-file"
          />

          <Field label="Logo Name / Description">
            <input
              className="rounded-lg border px-3 py-2 w-full"
              placeholder="e.g. Scales of Justice, Vintage Car, Modern Tech"
              value={form.name}
              onChange={(e) => setForm((s) => ({ ...s, name: e.target.value }))}
              data-testid="input-logo-name"
            />
          </Field>
          
          <Field label="System ID (auto-generated)" hint="This technical ID is automatically created from your logo name">
            <input
              className="rounded-lg border px-3 py-2 w-full bg-gray-50"
              value={generatedId || 'will-be-generated-from-name-above'}
              readOnly
              data-testid="display-system-id"
            />
          </Field>

          <Field label="Tags (comma-separated)">
            <input
              className="rounded-lg border px-3 py-2 w-full"
              placeholder="modern, emblem"
              value={form.tags}
              onChange={(e) => setForm((s) => ({ ...s, tags: e.target.value }))}
              data-testid="input-tags"
            />
          </Field>


          {/* Firebase Test and Upload Buttons */}
          <div className="pt-4 space-y-2">
            <button
              type="button"
              className="w-full rounded-lg px-4 py-2 border border-blue-500 text-blue-500 font-medium hover:bg-blue-50 transition-colors"
              onClick={testFirebaseConnection}
              data-testid="button-test-firebase"
            >
              Test Firebase Connection
            </button>
            
            <button
              type="button"
              className="w-full rounded-lg px-4 py-3 text-white font-medium hover:opacity-90 transition-opacity"
              style={{ backgroundColor: "#FF5F00" }}
              onClick={handleAddTemplate}
              disabled={busy || !svgFile || !form.name.trim()}
              data-testid="button-add-logo-name"
            >
              {busy ? "Uploading..." : "Upload"}
            </button>
          </div>

          {msg && (
            <p className="text-sm text-gray-600" data-testid="text-upload-message">
              {msg}
            </p>
          )}

        </section>
      </form>
    </DashboardTemplatePage>
  );
}

// ---------- UI Fragments ----------
function Field({
  label,
  hint,
  children,
}: {
  label: string;
  hint?: string;
  children: React.ReactNode;
}) {
  return (
    <div className="grid gap-1">
      <label className="text-xs text-gray-600">{label}</label>
      {children}
      {hint && <div className="text-[11px] text-gray-400">{hint}</div>}
    </div>
  );
}

function FileDrop({
  label,
  accept,
  file,
  onRemove,
  onChoose,
  onDrop,
}: {
  label: string;
  accept: string;
  file: File | null;
  onRemove: () => void;
  onChoose: () => void;
  onDrop: (e: React.DragEvent<HTMLDivElement>) => void;
}) {
  const idSafe = label.toLowerCase().replace(/[^a-z0-9]/g, "-");
  return (
    <div>
      <label className="text-xs text-gray-600">{label}</label>
      <div
        onDragOver={(e) => e.preventDefault()}
        onDrop={onDrop}
        onClick={onChoose} // make the whole area clickable
        className="mt-1 rounded-xl border-dashed border-2 p-4 text-sm text-gray-600 flex items-center justify-between gap-3 cursor-pointer"
        data-testid={`drop-zone-${idSafe}`}
        title="Click to choose a file or drag & drop"
      >
        <div className="flex-1">
          {file ? (
            <span className="font-medium">{file.name}</span>
          ) : (
            <>Drag & drop, or click to select</>
          )}
          <div className="text-[11px] text-gray-400">Accepted: {accept}</div>
        </div>
        <div className="flex items-center gap-2">
          <button
            type="button"
            className="rounded-lg px-3 py-1 border"
            onClick={(e) => {
              e.stopPropagation();
              onChoose();
            }}
            data-testid={`button-choose-${idSafe}`}
          >
            Choose
          </button>
          {file && (
            <button
              type="button"
              className="rounded-lg px-3 py-1 border"
              onClick={(e) => {
                e.stopPropagation();
                onRemove();
              }}
              data-testid={`button-remove-${idSafe}`}
            >
              Remove
            </button>
          )}
        </div>
      </div>
    </div>
  );
}