import { useState, useRef } from "react";
import { DashboardTemplatePage } from "../../components/DashboardTemplatePage";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Loader2, Upload, CheckCircle, XCircle, Trash2, Search, Edit } from "lucide-react";
import { useAdminAuth } from "@/admin/useAdminAuth";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

export default function AdminUploadBusinessPlanFirebase() {
  const { getAuthHeaders } = useAdminAuth();
  const [form, setForm] = useState({
    title: "",
    category: "",
    industries: "",
    tags: "",
    sections: "Executive Summary\nMarket Analysis\nCompany Description\nProducts & Services\nMarketing Strategy\nSales Strategy\nOperational Plan\nManagement Team\nFinancial Projections\nMilestones & Roadmap\nFuture Expansion\nExit Strategy"
  });

  const [docxFile, setDocxFile] = useState<File | null>(null);
  const [pdfFile, setPdfFile] = useState<File | null>(null);
  const [message, setMessage] = useState<string>("");
  const [status, setStatus] = useState<'idle' | 'uploading' | 'success' | 'error'>('idle');
  const [uploadedId, setUploadedId] = useState<string>("");
  const [isDragging, setIsDragging] = useState(false);

  const docRef = useRef<HTMLInputElement>(null);
  const pdfRef = useRef<HTMLInputElement>(null);

  const handleDocxChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setDocxFile(file);
    }
  };

  const handlePdfChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setPdfFile(file);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(false);
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(false);
    const file = e.dataTransfer.files?.[0];
    if (file && file.name.toLowerCase().endsWith('.docx')) {
      setDocxFile(file);
    } else {
      setMessage("Please drop a .docx file");
      setStatus('error');
    }
  };

  const handleClickUpload = () => {
    docRef.current?.click();
  };

  const handleClickPdfUpload = () => {
    pdfRef.current?.click();
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setMessage("");
    setStatus('idle');

    if (!form.title) {
      setMessage("Title is required");
      setStatus('error');
      return;
    }
    
    if (!form.category) {
      setMessage("Category is required");
      setStatus('error');
      return;
    }
    
    if (!docxFile) {
      setMessage("DOCX file is required");
      setStatus('error');
      return;
    }
    
    if (!pdfFile) {
      setMessage("PDF file is required");
      setStatus('error');
      return;
    }

    try {
      setStatus('uploading');
      
      const authHeaders = await getAuthHeaders();
      
      const fd = new FormData();
      fd.append("title", form.title);
      fd.append("category", form.category);
      fd.append("tags", form.industries || form.tags);
      fd.append("sections", form.sections);
      fd.append("docx", docxFile);
      fd.append("pdf", pdfFile);

      const res = await fetch("/api/bp-templates-firebase/admin/upload", { 
        method: "POST",
        headers: authHeaders,
        body: fd 
      });
      
      const json = await res.json();
      
      if (!res.ok) {
        throw new Error(json?.error || `HTTP ${res.status}`);
      }

      setMessage(json.message || "Upload successful! Processing previews...");
      setStatus('success');
      setUploadedId(json.slug);
      
      // Reset form
      setForm({
        title: "",
        category: "",
        industries: "",
        tags: "",
        sections: form.sections
      });
      setDocxFile(null);
      setPdfFile(null);
      if (docRef.current) {
        docRef.current.value = "";
      }
      if (pdfRef.current) {
        pdfRef.current.value = "";
      }
    } catch (e: any) {
      setMessage("Upload failed: " + (e?.message || "Unknown error"));
      setStatus('error');
    }
  };

  return (
    <DashboardTemplatePage title="Upload Business Plan Template (Firebase)">
      <Card className="mb-6">
        <CardHeader>
          <CardTitle>How It Works</CardTitle>
        </CardHeader>
        <CardContent className="space-y-3 text-sm">
          <div className="flex items-start gap-2">
            <div className="w-6 h-6 rounded-full bg-primary text-primary-foreground flex items-center justify-center flex-shrink-0 text-xs">
              1
            </div>
            <p>Upload your DOCX file with template metadata (title, category, tags)</p>
          </div>
          <div className="flex items-start gap-2">
            <div className="w-6 h-6 rounded-full bg-primary text-primary-foreground flex items-center justify-center flex-shrink-0 text-xs">
              2
            </div>
            <p>DOCX is uploaded to Firebase Storage at: /templates/business-plan/[slug]/v1/</p>
          </div>
          <div className="flex items-start gap-2">
            <div className="w-6 h-6 rounded-full bg-primary text-primary-foreground flex items-center justify-center flex-shrink-0 text-xs">
              3
            </div>
            <p>Preview images are generated automatically (page-1.jpg, page-2.jpg, etc.)</p>
          </div>
          <div className="flex items-start gap-2">
            <div className="w-6 h-6 rounded-full bg-primary text-primary-foreground flex items-center justify-center flex-shrink-0 text-xs">
              4
            </div>
            <p>Metadata is stored in Firestore for easy querying and filtering</p>
          </div>
          <div className="flex items-start gap-2">
            <div className="w-6 h-6 rounded-full bg-primary text-primary-foreground flex items-center justify-center flex-shrink-0 text-xs">
              5
            </div>
            <p>Template appears on the templates page with preview modal</p>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>Upload to Firebase Storage</CardTitle>
          <CardDescription>
            Upload business plan templates to Firebase Storage with automatic preview generation
          </CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-6">
            <div className="grid md:grid-cols-2 gap-6">
              <div className="space-y-4">
                <div>
                  <Label htmlFor="title">Template Title *</Label>
                  <Input
                    id="title"
                    placeholder="e.g., Barber Shop Business Plan"
                    value={form.title}
                    onChange={(e) => setForm({ ...form, title: e.target.value })}
                    data-testid="input-title"
                  />
                </div>

                <div>
                  <Label htmlFor="category">Category *</Label>
                  <Input
                    id="category"
                    placeholder="e.g., Personal Services, Retail, Technology"
                    value={form.category}
                    onChange={(e) => setForm({ ...form, category: e.target.value })}
                    data-testid="input-category"
                  />
                </div>

                <div>
                  <Label htmlFor="industries">Industries (comma-separated) *</Label>
                  <Input
                    id="industries"
                    placeholder="e.g., personal-services, retail, local"
                    value={form.industries}
                    onChange={(e) => setForm({ ...form, industries: e.target.value })}
                    data-testid="input-industries"
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    Used for filtering and tags (e.g., personal-services, retail, technology)
                  </p>
                </div>

                <div>
                  <Label>DOCX File *</Label>
                  <input
                    ref={docRef}
                    type="file"
                    accept=".docx"
                    onChange={handleDocxChange}
                    className="hidden"
                    data-testid="input-docx"
                  />
                  <div
                    onClick={handleClickUpload}
                    onDragOver={handleDragOver}
                    onDragLeave={handleDragLeave}
                    onDrop={handleDrop}
                    className={`
                      border-2 border-dashed rounded-lg p-8 text-center cursor-pointer transition-colors
                      ${isDragging ? 'border-primary bg-primary/5' : 'border-muted-foreground/25 hover:border-primary/50'}
                    `}
                    data-testid="dropzone-docx"
                  >
                    {docxFile ? (
                      <div className="space-y-2">
                        <CheckCircle className="h-10 w-10 mx-auto text-green-500" />
                        <p className="font-medium">{docxFile.name}</p>
                        <p className="text-sm text-muted-foreground">
                          {(docxFile.size / 1024).toFixed(1)} KB
                        </p>
                        <p className="text-xs text-muted-foreground">
                          Click or drag to replace
                        </p>
                      </div>
                    ) : (
                      <div className="space-y-2">
                        <Upload className="h-10 w-10 mx-auto text-muted-foreground" />
                        <p className="font-medium">Click to upload or drag and drop</p>
                        <p className="text-sm text-muted-foreground">
                          DOCX file (max 50MB)
                        </p>
                      </div>
                    )}
                  </div>
                </div>

                <div>
                  <Label>PDF File *</Label>
                  <input
                    ref={pdfRef}
                    type="file"
                    accept=".pdf"
                    onChange={handlePdfChange}
                    className="hidden"
                    data-testid="input-pdf"
                  />
                  <div
                    onClick={handleClickPdfUpload}
                    className="border-2 border-dashed rounded-lg p-8 text-center cursor-pointer transition-colors border-muted-foreground/25 hover:border-primary/50"
                    data-testid="dropzone-pdf"
                  >
                    {pdfFile ? (
                      <div className="space-y-2">
                        <CheckCircle className="h-10 w-10 mx-auto text-green-500" />
                        <p className="font-medium">{pdfFile.name}</p>
                        <p className="text-sm text-muted-foreground">
                          {(pdfFile.size / 1024).toFixed(1)} KB
                        </p>
                        <p className="text-xs text-muted-foreground">
                          Click to replace
                        </p>
                      </div>
                    ) : (
                      <div className="space-y-2">
                        <Upload className="h-10 w-10 mx-auto text-muted-foreground" />
                        <p className="font-medium">Click to upload PDF</p>
                        <p className="text-sm text-muted-foreground">
                          PDF file for preview (max 50MB)
                        </p>
                      </div>
                    )}
                  </div>
                </div>
              </div>

              <div className="space-y-4">
                <div>
                  <Label htmlFor="sections">Sections (one per line)</Label>
                  <Textarea
                    id="sections"
                    rows={12}
                    value={form.sections}
                    onChange={(e) => setForm({ ...form, sections: e.target.value })}
                    data-testid="input-sections"
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    These sections will be used for preview generation
                  </p>
                </div>
              </div>
            </div>

            {message && (
              <div className={`p-4 rounded-lg border ${
                status === 'success' ? 'bg-green-50 border-green-200 text-green-800' :
                status === 'error' ? 'bg-red-50 border-red-200 text-red-800' :
                'bg-blue-50 border-blue-200 text-blue-800'
              }`}>
                <div className="flex items-start gap-2">
                  {status === 'success' && <CheckCircle className="h-5 w-5 flex-shrink-0 mt-0.5" />}
                  {status === 'error' && <XCircle className="h-5 w-5 flex-shrink-0 mt-0.5" />}
                  {status === 'uploading' && <Loader2 className="h-5 w-5 flex-shrink-0 mt-0.5 animate-spin" />}
                  <div className="flex-1">
                    <p className="font-medium">{message}</p>
                    {uploadedId && (
                      <p className="text-sm mt-1">Template slug: {uploadedId}</p>
                    )}
                  </div>
                </div>
              </div>
            )}

            <div className="flex gap-3">
              <Button
                type="submit"
                disabled={status === 'uploading'}
                data-testid="button-upload"
              >
                {status === 'uploading' ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Uploading...
                  </>
                ) : (
                  <>
                    <Upload className="h-4 w-4 mr-2" />
                    Upload to Firebase
                  </>
                )}
              </Button>
            </div>
          </form>
        </CardContent>
      </Card>

      <TemplateManager getAuthHeaders={getAuthHeaders} />
    </DashboardTemplatePage>
  );
}

function TemplateManager({ getAuthHeaders }: { getAuthHeaders: () => Promise<any> }) {
  const { toast } = useToast();
  const [searchQuery, setSearchQuery] = useState('');
  const [editingTemplate, setEditingTemplate] = useState<any>(null);
  const [editForm, setEditForm] = useState({ title: '', category: '', industries: '' });

  const { data: templates, isLoading, refetch } = useQuery({
    queryKey: ['/api/bp-templates-firebase/list'],
    refetchOnMount: true,
    staleTime: 0,
  });

  const updateMutation = useMutation({
    mutationFn: async ({ slug, data }: { slug: string; data: any }) => {
      const authHeaders = await getAuthHeaders();
      const res = await fetch(`/api/bp-templates-firebase/admin/${slug}`, {
        method: 'PATCH',
        headers: {
          ...authHeaders,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(data),
      });
      if (!res.ok) {
        const json = await res.json();
        throw new Error(json?.error || 'Failed to update');
      }
      return res.json();
    },
    onSuccess: async () => {
      await queryClient.invalidateQueries({ queryKey: ['/api/bp-templates-firebase/list'] });
      await refetch();
      setEditingTemplate(null);
      toast({
        title: "Success",
        description: "Template updated successfully",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update template",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (slug: string) => {
      const authHeaders = await getAuthHeaders();
      const res = await fetch(`/api/bp-templates-firebase/admin/${slug}`, {
        method: 'DELETE',
        headers: authHeaders,
      });
      if (!res.ok) {
        const json = await res.json();
        throw new Error(json?.error || 'Failed to delete');
      }
      return res.json();
    },
    onSuccess: async () => {
      await queryClient.invalidateQueries({ queryKey: ['/api/bp-templates-firebase/list'] });
      await refetch();
      toast({
        title: "Success",
        description: "Template deleted successfully",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete template",
        variant: "destructive",
      });
    },
  });

  const handleEdit = (template: any) => {
    setEditingTemplate(template);
    setEditForm({
      title: template.title || '',
      category: template.category || '',
      industries: (template.industries || []).join(', '),
    });
  };

  const handleSaveEdit = () => {
    if (!editingTemplate) return;
    
    updateMutation.mutate({
      slug: editingTemplate.slug,
      data: {
        title: editForm.title,
        category: editForm.category,
        industries: editForm.industries,
      },
    });
  };

  if (isLoading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Manage Templates</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center justify-center py-8">
            <Loader2 className="h-6 w-6 animate-spin" />
          </div>
        </CardContent>
      </Card>
    );
  }

  const items = Array.isArray(templates) ? templates : [];
  const filteredItems = searchQuery.trim() === '' ? items : items.filter((template: any) => {
    const query = searchQuery.toLowerCase().trim();
    return (
      template.title?.toLowerCase().includes(query) ||
      template.slug?.toLowerCase().includes(query) ||
      template.category?.toLowerCase().includes(query) ||
      (template.industries || []).some((ind: string) => ind.toLowerCase().includes(query))
    );
  });

  return (
    <>
      <Card>
        <CardHeader>
          <CardTitle>Manage Templates</CardTitle>
          <CardDescription>
            Search, edit, and delete business plan templates
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {/* Search bar */}
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Search by title, category, or industry..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-9"
                data-testid="input-search"
              />
            </div>

            {/* Template list */}
            {filteredItems.length === 0 ? (
              <p className="text-muted-foreground text-center py-8">
                {searchQuery ? 'No templates found matching your search' : 'No templates uploaded yet'}
              </p>
            ) : (
              <div className="space-y-2">
                {filteredItems.map((template: any) => (
                  <div
                    key={template.slug}
                    className="flex items-center justify-between p-4 border rounded-lg hover:bg-muted/50"
                  >
                    <div className="flex-1">
                      <h4 className="font-medium">{template.title}</h4>
                      <div className="text-sm text-muted-foreground space-x-2">
                        <span>Slug: {template.slug}</span>
                        {template.category && <span>• Category: {template.category}</span>}
                        {template.industries && template.industries.length > 0 && (
                          <span>• Industries: {template.industries.join(', ')}</span>
                        )}
                      </div>
                    </div>
                    <div className="flex gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleEdit(template)}
                        data-testid={`button-edit-${template.slug}`}
                      >
                        <Edit className="h-4 w-4" />
                      </Button>
                      <Button
                        variant="destructive"
                        size="sm"
                        onClick={() => {
                          if (confirm(`Delete template "${template.title}"?\n\nThis will remove all files from Firebase Storage and cannot be undone.`)) {
                            deleteMutation.mutate(template.slug);
                          }
                        }}
                        disabled={deleteMutation.isPending}
                        data-testid={`button-delete-${template.slug}`}
                      >
                        {deleteMutation.isPending ? (
                          <Loader2 className="h-4 w-4 animate-spin" />
                        ) : (
                          <Trash2 className="h-4 w-4" />
                        )}
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Edit Dialog */}
      <Dialog open={!!editingTemplate} onOpenChange={(open) => !open && setEditingTemplate(null)}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edit Template</DialogTitle>
            <DialogDescription>
              Update the template metadata. Note: This does not change the files.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="edit-title">Title</Label>
              <Input
                id="edit-title"
                value={editForm.title}
                onChange={(e) => setEditForm({ ...editForm, title: e.target.value })}
                data-testid="input-edit-title"
              />
            </div>
            <div>
              <Label htmlFor="edit-category">Category</Label>
              <Input
                id="edit-category"
                value={editForm.category}
                onChange={(e) => setEditForm({ ...editForm, category: e.target.value })}
                data-testid="input-edit-category"
              />
            </div>
            <div>
              <Label htmlFor="edit-industries">Industries (comma-separated)</Label>
              <Input
                id="edit-industries"
                value={editForm.industries}
                onChange={(e) => setEditForm({ ...editForm, industries: e.target.value })}
                placeholder="e.g., personal-services, retail, local"
                data-testid="input-edit-industries"
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setEditingTemplate(null)}
              disabled={updateMutation.isPending}
            >
              Cancel
            </Button>
            <Button
              onClick={handleSaveEdit}
              disabled={updateMutation.isPending}
              data-testid="button-save-edit"
            >
              {updateMutation.isPending ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Saving...
                </>
              ) : (
                'Save Changes'
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
