import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { toast } from "sonner";

type Row = {
  id: string; title: string; creatorName: string; price: number; status: string; createdAt: string;
  previewUrl?: string; salesCount?: number;
};

export default function AdminMarketplace() {
  const [search, setSearch] = useState("");
  const qc = useQueryClient();

  const { data, isLoading, refetch } = useQuery({
    queryKey: ["admin-marketplace", search],
    queryFn: async () => {
      const qs = new URLSearchParams(search ? { search } : {}).toString();
      const res = await fetch(`/api/admin/marketplace/overview?${qs}`, { credentials: "include" });
      if (!res.ok) throw new Error("Failed to load");
      return res.json() as Promise<{ pending: Row[]; approved: Row[]; rejected: Row[] }>;
    }
  });

  const approve = useMutation({
    mutationFn: async (id: string) => {
      const r = await fetch(`/api/admin/marketplace/assets/${id}/approve`, { method: "PATCH", credentials: "include" });
      if (!r.ok) throw new Error("fail");
      return r.json();
    },
    onSuccess: () => { toast.success("Approved"); qc.invalidateQueries({ queryKey: ["admin-marketplace"] }); }
  });

  const reject = useMutation({
    mutationFn: async (id: string) => {
      const r = await fetch(`/api/admin/marketplace/assets/${id}/reject`, { method: "PATCH", credentials: "include", headers: { "Content-Type": "application/json" }, body: JSON.stringify({ reason: "Policy" }) });
      if (!r.ok) throw new Error("fail");
      return r.json();
    },
    onSuccess: () => { toast.success("Rejected"); qc.invalidateQueries({ queryKey: ["admin-marketplace"] }); }
  });

  const Section = ({ title, items, actions }: { title: string; items?: Row[]; actions?: boolean }) => (
    <Card>
      <CardHeader><CardTitle>{title} ({items?.length ?? 0})</CardTitle></CardHeader>
      <CardContent className="space-y-4">
        {(items ?? []).map(a => (
          <div key={a.id} className="flex items-center justify-between border rounded-md p-3">
            <div className="min-w-0">
              <div className="font-medium truncate">{a.title}</div>
              <div className="text-xs text-muted-foreground">{a.creatorName} • ${ (a.price/100).toFixed(2) } • {new Date(a.createdAt).toLocaleDateString()}</div>
            </div>
            <div className="flex items-center gap-2">
              <Badge variant="outline">{a.status}</Badge>
              {actions && (
                <>
                  <Button size="sm" onClick={() => approve.mutate(a.id)}>Approve</Button>
                  <Button size="sm" variant="destructive" onClick={() => reject.mutate(a.id)}>Reject</Button>
                </>
              )}
            </div>
          </div>
        ))}
      </CardContent>
    </Card>
  );

  return (
    <div className="space-y-6">
      <Card>
        <CardContent className="p-4 flex gap-2">
          <Input placeholder="Search title or creator…" value={search} onChange={(e) => setSearch(e.target.value)} />
          <Button onClick={() => refetch()} disabled={isLoading}>Search</Button>
        </CardContent>
      </Card>
      <Section title="Pending Review" items={data?.pending} actions />
      <Section title="Approved" items={data?.approved} />
      <Section title="Rejected" items={data?.rejected} />
    </div>
  );
}