import React, { useRef, useState } from "react";
import { DashboardTemplatePage } from "../../components/DashboardTemplatePage";
import { COLORS, btnBase } from "../../ui-tokens";

export default function AdminImportBusinessPlans() {
  const [file, setFile] = useState<File | null>(null);
  const [msg, setMsg] = useState<string>("");
  const [busy, setBusy] = useState(false);
  const ref = useRef<HTMLInputElement>(null);

  const submit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!file) return setMsg("Select a CSV file first.");
    setMsg("");
    try {
      setBusy(true);
      const fd = new FormData();
      fd.append("csv", file);
      const res = await fetch("/api/bp-templates/import-csv", { method: "POST", body: fd });
      const json = await res.json().catch(()=>({}));
      if (!res.ok) throw new Error(json?.error || `HTTP ${res.status}`);
      setMsg(`Imported ${json.imported} items. Library now ${json.total}. ✅`);
    } catch (e:any) {
      setMsg("Import failed: " + (e?.message || "Unknown error"));
    } finally {
      setBusy(false);
    }
  };

  return (
    <DashboardTemplatePage title="Import Business Plans (CSV)">
      <form onSubmit={submit} className="space-y-4">
        <div className="rounded-xl border-dashed border-2 p-4 flex items-center justify-between">
          <div className="text-sm">
            {file ? <span className="font-medium">{file.name}</span> : "Click Choose to select a .csv"}
            <div className="text-[11px] text-gray-500">Columns: id,name,category,tags,previewUrl,docxUrl,updatedAt,sections</div>
          </div>
          <div className="flex gap-2">
            <button type="button" className={btnBase} style={{ backgroundColor: COLORS.green, color: COLORS.white }} onClick={()=>ref.current?.click()}>Choose</button>
            {file && <button type="button" className={btnBase} style={{ backgroundColor: COLORS.darkGray, color: COLORS.white }} onClick={()=>setFile(null)}>Remove</button>}
          </div>
        </div>
        <input ref={ref} type="file" accept=".csv" className="hidden" onChange={(e)=>setFile(e.target.files?.[0] || null)} />
        <button className={btnBase} style={{ backgroundColor: COLORS.green, color: COLORS.white }} disabled={busy} type="submit">
          {busy ? "Importing…" : "Import CSV"}
        </button>
        {msg && <p className="text-sm text-gray-600">{msg}</p>}
        <a
          className={btnBase}
          style={{ backgroundColor: COLORS.orange, color: COLORS.white }}
          href="/site/samples/bp-import-sample.csv"
          download
        >
          Download Sample CSV
        </a>
      </form>
    </DashboardTemplatePage>
  );
}