import { useEffect, useState } from "react";
import { useLocation } from "wouter";
import { requireAdminOrRedirect, adminSignOut } from "@/admin/adminGuard";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Users, Calendar, TrendingUp, AlertTriangle, CheckCircle, XCircle, Database, Shield, Crown, Settings, Pause, Upload, Download, Image, Globe, BarChart3, FileText, ShoppingBag, Palette, Plus, Activity } from "lucide-react";
import { logoComposerIcon } from "@/assets";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { WorldMap } from "@/components/admin/WorldMap";
import { VisitorChart } from "@/components/admin/VisitorChart";

interface AdminStats {
  totalUsers: number;
  freeUsers: number;
  proUsers: number;
  newPayingUsersToday: number;
  newPayingUsersThisWeek: number;
  newPayingUsersThisMonth: number;
  recentUsers: Array<{
    id: string;
    email: string;
    displayName: string;
    createdAt: string;
    isPaid?: boolean;
  }>;
  note?: string;
}

interface User {
  id: string;
  email: string;
  displayName: string;
  createdAt: string;
  isPaid?: boolean;
  role?: string;
}

type Role = 'owner' | 'management' | 'staff' | 'analyst' | 'user' | 'pro';

// StatCard component for KPI display
function StatCard({ label, value, sub, icon: Icon }: {
  label: string; 
  value: string | number; 
  sub?: string;
  icon?: any;
}) {
  return (
    <Card className="h-[120px]">
      <CardContent className="pt-6 h-full">
        <div className="flex items-center h-full">
          {Icon && (
            <div className="p-2 bg-primary/10 rounded-lg">
              <Icon className="h-6 w-6 text-primary" />
            </div>
          )}
          <div className={`${Icon ? "ml-4" : ""} flex flex-col justify-center min-h-[72px]`}>
            <p className="text-sm font-medium text-gray-600">{label}</p>
            <p className="text-2xl font-bold text-gray-900" data-testid={`stat-${label.toLowerCase().replace(/\s+/g, '-')}`}>
              {value}
            </p>
            <div className="h-4 mt-1">
              {sub && <p className="text-xs text-gray-500">{sub}</p>}
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}

interface Metrics {
  mtdRevenue: number;
  ytdRevenue: number;
  activePro: number;
  churnThisMonthCount: number;
  churnThisMonthRate: number;
  newProThisMonth: number;
  freeUsers: number;
  proUsers: number;
  pausedUsers: number; // New: count of paused users
  activeSubs: number;
  unsatisfiedPct: number;
  conversionRate: number;
  totalUsers: number;
  recentPauses: Array<{ // New: recent pauses for owner dashboard
    id: string;
    email: string;
    months: number;
    resumeAt: Date;
    createdAt: Date;
    reason?: string;
    note?: string;
  }>;
  stripeConfigured: boolean;
}

interface SystemStatus {
  database: {
    status: "connected" | "degraded" | "error";
    message: string;
    connectionType: "websocket" | "http" | "failed";
  };
  firebase: {
    status: "connected" | "degraded" | "error";
    message: string;
  };
  auth: {
    status: "connected" | "degraded" | "error";
    message: string;
  };
}

export default function AdminDashboard() {
  // Navigation handler to go back to main app
  const [, setLocation] = useLocation();
  const handleGoHome = () => {
    setLocation('/');
  };
  const [stats, setStats] = useState<AdminStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [systemStatus, setSystemStatus] = useState<SystemStatus | null>(null);
  const [statusLoading, setStatusLoading] = useState(false);
  const [currentUserRole, setCurrentUserRole] = useState<string>("loading");
  const [searchedUser, setSearchedUser] = useState<User | null>(null);
  const [searchQuery, setSearchQuery] = useState<string>("");
  const [searchLoading, setSearchLoading] = useState(false);
  const [roleChanging, setRoleChanging] = useState<string | null>(null);
  const [metrics, setMetrics] = useState<Metrics | null>(null);
  const [metricsLoading, setMetricsLoading] = useState(false);
  const [metricsError, setMetricsError] = useState<string | null>(null);
  const { toast } = useToast();


  // Get current user's admin role
  const fetchCurrentUserRole = async (token: string) => {
    try {
      const response = await fetch("/api/auth/me", {
        headers: { Authorization: `Bearer ${token}` },
        credentials: "include",
      });
      
      if (response.ok) {
        const userData = await response.json();
        // Check if user is owner (from email check)
        // For legacy admin tokens or owner emails, treat as 'owner' role
        const userRole = userData.isOwner ? 'owner' : 'user';
        setCurrentUserRole(userRole);
        return userRole;
      } else {
        setCurrentUserRole('user');
        return 'user';
      }
    } catch (err) {
      console.error("Failed to fetch current user role:", err);
      setCurrentUserRole('user'); // Default to user if error
      return 'user';
    }
  };

  // Fetch admin metrics for owners
  const fetchAdminMetrics = async (token: string) => {
    
    setMetricsLoading(true);
    setMetricsError(null);
    try {
      const response = await fetch("/api/admin/metrics", {
        headers: { Authorization: `Bearer ${token}` },
        credentials: "include",
      });
      
      if (response.ok) {
        const data = await response.json();
        setMetrics(data);
      } else {
        throw new Error('Failed to fetch metrics');
      }
    } catch (err) {
      console.error("Failed to fetch admin metrics:", err);
      setMetricsError(err instanceof Error ? err.message : "Failed to fetch metrics");
    } finally {
      setMetricsLoading(false);
    }
  };

  // Search for a specific user by email
  const searchUser = async (email: string, token: string) => {
    if (!email.trim()) {
      setSearchedUser(null);
      return;
    }
    
    setSearchLoading(true);
    try {
      const response = await fetch("/api/admin/users", {
        headers: { Authorization: `Bearer ${token}` },
        credentials: "include",
      });
      
      if (response.ok) {
        const data = await response.json();
        const foundUser = data.users.find((user: User) => 
          user.email.toLowerCase().includes(email.toLowerCase()) ||
          user.displayName?.toLowerCase().includes(email.toLowerCase())
        );
        
        if (foundUser) {
          // If this is the current logged-in admin user, use their actual role
          const response2 = await fetch("/api/auth/me", {
            headers: { Authorization: `Bearer ${token}` },
            credentials: "include",
          });
          
          let userRole = 'user'; // Default role
          if (response2.ok) {
            const currentUserData = await response2.json();
            // If searching for the current logged-in user, use their admin role
            if (currentUserData.email === foundUser.email) {
              userRole = currentUserRole; // Use the admin role we already detected
            }
          }
          
          setSearchedUser({ ...foundUser, role: userRole });
        } else {
          setSearchedUser(null);
          toast({
            title: "User not found",
            description: `No user found matching "${email}"`,
            variant: "destructive",
          });
        }
      }
    } catch (err) {
      console.error("Failed to search users:", err);
      toast({
        title: "Error",
        description: "Failed to search users",
        variant: "destructive",
      });
    } finally {
      setSearchLoading(false);
    }
  };

  // Handle role change
  const handleRoleChange = async (userId: string, newRole: Role, token: string) => {
    setRoleChanging(userId);
    try {
      const response = await fetch(`/api/admin/users/${userId}/role`, {
        method: 'POST',
        headers: { 
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        credentials: "include",
        body: JSON.stringify({ role: newRole })
      });
      
      if (response.ok) {
        const data = await response.json();
        // Update the searched user's role in the local state
        if (searchedUser && searchedUser.id === userId) {
          setSearchedUser({ ...searchedUser, role: newRole });
        }
        toast({
          title: "Success",
          description: `User role changed to ${newRole}`,
        });
      } else {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to change role');
      }
    } catch (err) {
      console.error("Failed to change role:", err);
      toast({
        title: "Error",
        description: err instanceof Error ? err.message : "Failed to change user role",
        variant: "destructive",
      });
    } finally {
      setRoleChanging(null);
    }
  };


  const checkSystemHealth = async (token: string) => {
    setStatusLoading(true);
    try {
      // Check database connection
      let databaseStatus: SystemStatus["database"];
      try {
        const dbResponse = await fetch("/api/admin/health", {
          headers: { Authorization: `Bearer ${token}` },
          credentials: "include",
        });
        if (dbResponse.ok) {
          databaseStatus = {
            status: "connected",
            message: "Database connection healthy",
            connectionType: "http"
          };
        } else {
          throw new Error("Health check failed");
        }
      } catch {
        databaseStatus = {
          status: "error",
          message: "Database connection failed",
          connectionType: "failed"
        };
      }

      // Check Firebase status (based on stats response)
      const firebaseStatus: SystemStatus["firebase"] = {
        status: "connected",
        message: "Firebase authentication working correctly"
      };

      // Check auth status
      const authStatus: SystemStatus["auth"] = {
        status: "connected",
        message: "Authentication system operational"
      };

      setSystemStatus({
        database: databaseStatus,
        firebase: firebaseStatus,
        auth: authStatus
      });
    } catch (err) {
      console.error("System health check failed:", err);
    } finally {
      setStatusLoading(false);
    }
  };

  useEffect(() => {
    const checkAuthAndFetchStats = async () => {
      const token = await requireAdminOrRedirect(); // redirects to /admin/login if missing
      if (!token) return;

      try {
        const response = await fetch("/api/admin/stats", {
          headers: { Authorization: `Bearer ${token}` },
          credentials: "include",
        });
        
        if (response.status === 403) {
          adminSignOut();
          return;
        }
        
        if (!response.ok) {
          throw new Error("Failed to fetch stats");
        }
        
        const data = await response.json();
        setStats(data);

        // Get current user's role first
        const userRole = await fetchCurrentUserRole(token);
        
        // Also check system health
        await checkSystemHealth(token);
        
        // No need to fetch all users anymore - we use search instead
        
        // Fetch admin metrics for owners only
        if (userRole === 'owner') {
          await fetchAdminMetrics(token);
        }
      } catch (err) {
        setError(err instanceof Error ? err.message : "An error occurred");
      } finally {
        setLoading(false);
      }
    };

    checkAuthAndFetchStats();
  }, []);


  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full mx-auto"></div>
          <p className="mt-2 text-gray-600">Loading admin dashboard...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <Card className="max-w-md">
          <CardContent className="pt-6">
            <p className="text-red-600 text-center">{error}</p>
            <Button className="w-full mt-4" onClick={() => window.location.reload()}>
              Retry
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <DashboardTemplatePage title="Admin Dashboard">
      {/* SubscribedUser Template Wrapper */}
      <div className="space-y-6">
        {/* Admin Level Display */}
        <Card className="mb-6 bg-gradient-to-r from-purple-500 to-purple-600 text-white">
          <CardContent className="pt-6">
            <div className="flex items-center justify-between">
              <div className="flex items-center">
                <div className="p-3 bg-white/20 rounded-lg">
                  <Crown className="h-8 w-8 text-white" />
                </div>
                <div className="ml-4">
                  <p className="text-lg font-semibold">Your Admin Level</p>
                  <p className="text-3xl font-bold capitalize" data-testid="admin-level">
                    {currentUserRole}
                  </p>
                </div>
              </div>
              <div className="text-right">
                <p className="text-sm opacity-90">IBrandBiz Administrator</p>
                <p className="text-sm opacity-75">Full System Access</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Admin Quick Actions Navigation */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Settings className="h-5 w-5" />
              Admin Tools
            </CardTitle>
            <CardDescription>
              Quick access to administrative functions and management tools
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4">
              <Button
                variant="outline"
                className="h-auto p-4 justify-start"
                onClick={() => setLocation('/admin/assets')}
                data-testid="link-asset-reviews"
              >
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-green-100 dark:bg-green-900 rounded-lg">
                    <FileText className="h-5 w-5 text-green-600 dark:text-green-400" />
                  </div>
                  <div className="text-left">
                    <p className="font-medium">Asset Reviews</p>
                    <p className="text-sm text-muted-foreground">Review creator assets</p>
                  </div>
                </div>
              </Button>


              <Button
                variant="outline"
                className="h-auto p-4 justify-start"
                onClick={() => setLocation('/admin/marketplace')}
                data-testid="link-marketplace-management"
              >
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-pink-100 dark:bg-pink-900 rounded-lg">
                    <ShoppingBag className="h-5 w-5 text-pink-600 dark:text-pink-400" />
                  </div>
                  <div className="text-left">
                    <p className="font-medium">Marketplace Management</p>
                    <p className="text-sm text-muted-foreground">Review & manage creator assets</p>
                  </div>
                </div>
              </Button>

              <Button
                variant="outline"
                className="h-auto p-4 justify-start"
                onClick={() => setLocation('/admin/validate-logo-manifest')}
                data-testid="link-logo-templates"
              >
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-purple-100 dark:bg-purple-900 rounded-lg">
                    <img 
                      src={logoComposerIcon} 
                      alt="Logo Creator" 
                      className="h-5 w-5 text-purple-600 dark:text-purple-400"
                    />
                  </div>
                  <div className="text-left">
                    <p className="font-medium">Logo Templates</p>
                    <p className="text-sm text-muted-foreground">Validate logo manifest</p>
                  </div>
                </div>
              </Button>

              <Button
                variant="outline"
                className="h-auto p-4 justify-start"
                onClick={() => setLocation('/admin/upload-logo-template')}
                data-testid="link-upload-logo-template"
              >
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-cyan-100 dark:bg-cyan-900 rounded-lg">
                    <Plus className="h-5 w-5 text-cyan-600 dark:text-cyan-400" />
                  </div>
                  <div className="text-left">
                    <p className="font-medium">Upload Logo Template</p>
                    <p className="text-sm text-muted-foreground">Add new logo template</p>
                  </div>
                </div>
              </Button>

              <Button
                variant="outline"
                className="h-auto p-4 justify-start"
                onClick={() => setLocation('/admin/upload-business-plan-firebase')}
                data-testid="link-upload-business-plan-firebase"
              >
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-blue-100 dark:bg-blue-900 rounded-lg">
                    <FileText className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                  </div>
                  <div className="text-left">
                    <p className="font-medium">Upload Business Plans</p>
                    <p className="text-sm text-muted-foreground">Add new business plan template</p>
                  </div>
                </div>
              </Button>

              {/* Diagnostics - Owner & Management only */}
              {(currentUserRole === 'owner' || currentUserRole === 'management') && (
                <Button
                  variant="outline"
                  className="h-auto p-4 justify-start"
                  onClick={() => setLocation('/__qa/diag')}
                  data-testid="link-diagnostics"
                >
                  <div className="flex items-center gap-3">
                    <div className="p-2 bg-orange-100 dark:bg-orange-900 rounded-lg">
                      <Activity className="h-5 w-5 text-orange-600 dark:text-orange-400" />
                    </div>
                    <div className="text-left">
                      <p className="font-medium">Diagnostics</p>
                      <p className="text-sm text-muted-foreground">System health & telemetry</p>
                    </div>
                  </div>
                </Button>
              )}

            </div>
          </CardContent>
        </Card>

        {/* Owner KPIs vs Role-Based Stats */}
        {currentUserRole === 'owner' ? (
          <>
            {/* Primary Owner KPIs */}
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
              <StatCard 
                label="Sales (MTD)" 
                value={`$${(metrics?.mtdRevenue ?? 0).toLocaleString()}`}
                icon={TrendingUp}
              />
              <StatCard 
                label="Sales (YTD)" 
                value={`$${(metrics?.ytdRevenue ?? 0).toLocaleString()}`}
                icon={TrendingUp}
              />
              <StatCard 
                label="Paying Customers" 
                value={metrics?.activePro ?? 0}
                icon={Crown}
              />
              <StatCard 
                label="Churn (This Month)" 
                value={`${metrics?.churnThisMonthRate ?? 0}%`}
                sub={`${metrics?.churnThisMonthCount ?? 0} cancelled`}
                icon={AlertTriangle}
              />
            </div>

            {/* Secondary Owner KPIs */}
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
              <StatCard 
                label="New Pro (This Month)" 
                value={metrics?.newProThisMonth ?? 0}
                icon={Users}
              />
              <StatCard 
                label="Free → Pro Conversion" 
                value={`${metrics?.conversionRate ?? 0}%`}
                sub="This month"
                icon={TrendingUp}
              />
              <StatCard 
                label="Active Subs" 
                value={metrics?.activeSubs ?? 0}
                icon={Shield}
              />
              <StatCard 
                label="Unsatisfied %" 
                value={`${metrics?.unsatisfiedPct ?? 0}%`}
                sub="Cancellation feedback"
                icon={XCircle}
              />
            </div>

            {/* Basic User Stats */}
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
              <div className="lg:col-start-1">
                <StatCard 
                  label="Total Users" 
                  value={metrics?.totalUsers ?? 0}
                  icon={Users}
                />
              </div>
              <div className="lg:col-start-2">
                <StatCard 
                  label="Free Users" 
                  value={metrics?.freeUsers ?? 0}
                  icon={Users}
                />
              </div>
              <div className="lg:col-start-3">
                <StatCard 
                  label="Pro Users" 
                  value={metrics?.proUsers ?? 0}
                  icon={Crown}
                />
              </div>
              <div className="lg:col-start-4">
                <StatCard 
                  label="Paused" 
                  value={metrics?.pausedUsers ?? 0}
                  sub="Temporarily paused"
                  icon={Pause}
                  data-testid="stat-paused"
                />
              </div>
            </div>

            {/* Visitor Analytics Section */}
            <Card className="mb-8" data-testid="visitor-analytics-plaque">
              <CardHeader className="pb-4">
                <CardTitle className="flex items-center gap-2">
                  <Globe className="h-5 w-5 text-blue-600" />
                  Visitor Analytics
                </CardTitle>
                <CardDescription>
                  Track visitor traffic patterns and geographic distribution over the last 30 days
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                  {/* Left side: Daily visitor chart */}
                  <div className="space-y-2">
                    <div className="flex items-center gap-2 mb-4">
                      <BarChart3 className="h-4 w-4 text-green-600" />
                      <h4 className="font-medium text-gray-900 dark:text-gray-100">
                        Daily Visitor Trends
                      </h4>
                    </div>
                    <VisitorChart className="h-full" />
                  </div>

                  {/* Right side: World map */}
                  <div className="space-y-2">
                    <div className="flex items-center gap-2 mb-4">
                      <Globe className="h-4 w-4 text-blue-600" />
                      <h4 className="font-medium text-gray-900 dark:text-gray-100">
                        Visitor Locations
                      </h4>
                    </div>
                    <WorldMap className="h-full" />
                  </div>
                </div>
              </CardContent>
            </Card>

            {metricsError && (
              <div className="mb-6 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                <p className="text-sm text-yellow-800">
                  <strong>Metrics Warning:</strong> {metricsError}
                </p>
              </div>
            )}

            {/* Recent Pauses Section for Owners */}
            {metrics && metrics.recentPauses && metrics.recentPauses.length > 0 && (
              <Card className="mb-8">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Pause className="h-5 w-5 text-orange-600" />
                    Recent Pauses
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {metrics.recentPauses.slice(0, 10).map((pause, index) => (
                      <div key={pause.id} className="flex items-center justify-between p-3 bg-orange-50 rounded-lg border border-orange-200" data-testid={`recent-pause-${index}`}>
                        <div className="flex-1">
                          <p className="font-medium text-gray-900" data-testid={`pause-email-${index}`}>
                            {pause.email}
                          </p>
                          <p className="text-sm text-gray-600">
                            Paused for {pause.months} month{pause.months !== 1 ? 's' : ''}
                          </p>
                          {pause.reason && (
                            <p className="text-xs text-gray-500 mt-1">
                              Reason: {pause.reason}
                            </p>
                          )}
                        </div>
                        <div className="text-right">
                          <p className="text-sm font-medium text-gray-900" data-testid={`pause-resume-date-${index}`}>
                            Resumes: {new Date(pause.resumeAt).toLocaleDateString()}
                          </p>
                          <p className="text-xs text-gray-500">
                            Paused: {new Date(pause.createdAt).toLocaleDateString()}
                          </p>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </>
        ) : (
          /* Non-owner users see basic stats only */
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <StatCard 
              label="Total Users" 
              value={(stats?.freeUsers ?? 0) + (stats?.proUsers ?? 0)}
              icon={Users}
            />
            <StatCard 
              label="Free Users" 
              value={stats?.freeUsers ?? 0}
              icon={Users}
            />
            <StatCard 
              label="Pro Users" 
              value={stats?.proUsers ?? 0}
              icon={Crown}
            />
            <StatCard 
              label="Active Subs" 
              value={stats?.proUsers ?? 0}
              sub="Estimated active subscriptions"
              icon={Shield}
            />
          </div>
        )}

        {/* System Status */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertTriangle className="h-5 w-5 text-yellow-600" />
              System Status
              {statusLoading && <div className="animate-spin h-4 w-4 border-2 border-gray-300 border-t-transparent rounded-full ml-2"></div>}
            </CardTitle>
          </CardHeader>
          <CardContent>
            {systemStatus ? (
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <Database className="h-5 w-5" />
                    <span className="font-medium">Database</span>
                  </div>
                  <div className="flex items-center gap-2">
                    {systemStatus.database.status === "connected" && <CheckCircle className="h-5 w-5 text-green-600" />}
                    {systemStatus.database.status === "degraded" && <AlertTriangle className="h-5 w-5 text-yellow-600" />}
                    {systemStatus.database.status === "error" && <XCircle className="h-5 w-5 text-red-600" />}
                    <Badge variant={systemStatus.database.status === "connected" ? "default" : systemStatus.database.status === "degraded" ? "secondary" : "destructive"}>
                      {systemStatus.database.status}
                    </Badge>
                  </div>
                </div>
                <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <Shield className="h-5 w-5" />
                    <span className="font-medium">Firebase</span>
                  </div>
                  <div className="flex items-center gap-2">
                    {systemStatus.firebase.status === "connected" && <CheckCircle className="h-5 w-5 text-green-600" />}
                    {systemStatus.firebase.status === "degraded" && <AlertTriangle className="h-5 w-5 text-yellow-600" />}
                    {systemStatus.firebase.status === "error" && <XCircle className="h-5 w-5 text-red-600" />}
                    <Badge variant={systemStatus.firebase.status === "connected" ? "default" : systemStatus.firebase.status === "degraded" ? "secondary" : "destructive"}>
                      {systemStatus.firebase.status}
                    </Badge>
                  </div>
                </div>
                <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center gap-2">
                    <Users className="h-5 w-5" />
                    <span className="font-medium">Auth</span>
                  </div>
                  <div className="flex items-center gap-2">
                    {systemStatus.auth.status === "connected" && <CheckCircle className="h-5 w-5 text-green-600" />}
                    {systemStatus.auth.status === "degraded" && <AlertTriangle className="h-5 w-5 text-yellow-600" />}
                    {systemStatus.auth.status === "error" && <XCircle className="h-5 w-5 text-red-600" />}
                    <Badge variant={systemStatus.auth.status === "connected" ? "default" : systemStatus.auth.status === "degraded" ? "secondary" : "destructive"}>
                      {systemStatus.auth.status}
                    </Badge>
                  </div>
                </div>
              </div>
            ) : (
              <p className="text-gray-500">Loading system status...</p>
            )}
            
            {/* Error Details */}
            {systemStatus && (
              <div className="mt-4 space-y-2">
                {systemStatus.database.status !== "connected" && (
                  <div className="text-sm p-2 bg-yellow-50 border border-yellow-200 rounded">
                    <strong>Database:</strong> {systemStatus.database.message}
                  </div>
                )}
                {systemStatus.firebase.status !== "connected" && (
                  <div className="text-sm p-2 bg-red-50 border border-red-200 rounded">
                    <strong>Firebase:</strong> {systemStatus.firebase.message}
                  </div>
                )}
                {systemStatus.auth.status !== "connected" && (
                  <div className="text-sm p-2 bg-yellow-50 border border-yellow-200 rounded">
                    <strong>Auth:</strong> {systemStatus.auth.message}
                  </div>
                )}
              </div>
            )}

            {/* Debug Information */}
            {stats?.note && (
              <div className="mt-4 p-3 bg-blue-50 border border-blue-200 rounded">
                <p className="text-sm text-blue-800">
                  <strong>Debug Info:</strong> {stats.note}
                </p>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Recent Users */}
        <Card>
          <CardHeader>
            <CardTitle>Recent Users</CardTitle>
          </CardHeader>
          <CardContent>
            {stats?.recentUsers?.length ? (
              <div className="space-y-4">
                {stats.recentUsers.map((user, index) => (
                  <div 
                    key={user.id} 
                    className="flex items-center justify-between p-3 bg-gray-50 rounded-lg"
                    data-testid={`recent-user-${index}`}
                  >
                    <div>
                      <p className="font-medium text-gray-900">{user.displayName}</p>
                      <p className="text-sm text-gray-600">{user.email}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm text-gray-500">
                        {new Date(user.createdAt).toLocaleDateString('en-US')}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-gray-500 text-center py-4">No users yet</p>
            )}
          </CardContent>
        </Card>

        {/* User Role Management */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Settings className="h-5 w-5" />
              User Role Management
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {/* User Search */}
              <div className="flex gap-3">
                <Input
                  placeholder="Search user by email or name..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  onKeyDown={(e) => {
                    if (e.key === 'Enter') {
                      const token = sessionStorage.getItem("adminToken");
                      if (token) {
                        searchUser(searchQuery, token);
                      }
                    }
                  }}
                  className="flex-1"
                  data-testid="user-search-input"
                />
                <Button 
                  onClick={() => {
                    const token = sessionStorage.getItem("adminToken");
                    if (token) {
                      searchUser(searchQuery, token);
                    }
                  }}
                  disabled={searchLoading || !searchQuery.trim()}
                  data-testid="search-button"
                >
                  {searchLoading ? (
                    <div className="animate-spin h-4 w-4 border-2 border-white border-t-transparent rounded-full"></div>
                  ) : (
                    "Search"
                  )}
                </Button>
              </div>

              {/* Search Results */}
              {searchedUser ? (
                <div className="bg-gray-50 rounded-lg border p-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                      <div className="flex-shrink-0">
                        <div className="w-10 h-10 bg-primary/20 rounded-full flex items-center justify-center">
                          <Users className="h-5 w-5 text-primary" />
                        </div>
                      </div>
                      <div>
                        <p className="font-medium text-gray-900" data-testid="searched-user-name">
                          {searchedUser.displayName}
                        </p>
                        <p className="text-sm text-gray-600" data-testid="searched-user-email">
                          {searchedUser.email}
                        </p>
                        <p className="text-xs text-gray-500">
                          Joined: {new Date(searchedUser.createdAt).toLocaleDateString('en-US')}
                        </p>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-3">
                      <span 
                        className="px-3 py-1 rounded-full text-xs font-medium border bg-gray-100 text-gray-800 border-gray-200"
                        data-testid="user-current-role"
                      >
                        {searchedUser.role || 'user'}
                      </span>
                      
                      {currentUserRole === 'owner' && (
                        <Select
                          value={searchedUser.role || 'user'}
                          onValueChange={(newRole: Role) => {
                            const token = sessionStorage.getItem("adminToken");
                            if (token) {
                              handleRoleChange(searchedUser.id, newRole, token);
                            }
                          }}
                          disabled={roleChanging === searchedUser.id}
                        >
                          <SelectTrigger 
                            className="w-32"
                            data-testid="role-change-select"
                          >
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="user">User</SelectItem>
                            <SelectItem value="pro">Pro</SelectItem>
                            <SelectItem value="analyst">Analyst</SelectItem>
                            <SelectItem value="staff">Staff</SelectItem>
                            <SelectItem value="management">Management</SelectItem>
                            <SelectItem value="owner">Owner</SelectItem>
                          </SelectContent>
                        </Select>
                      )}
                      
                      {roleChanging === searchedUser.id && (
                        <div className="flex items-center space-x-2">
                          <div className="animate-spin h-4 w-4 border-2 border-primary border-t-transparent rounded-full"></div>
                          <span className="text-sm text-gray-600">Updating...</span>
                        </div>
                      )}
                      
                      {currentUserRole !== 'owner' && (
                        <span className="text-xs text-gray-500 italic">Owner access required</span>
                      )}
                    </div>
                  </div>
                </div>
              ) : searchQuery && !searchLoading ? (
                <div className="text-center py-8">
                  <p className="text-gray-500">Search for a user by email or name</p>
                </div>
              ) : (
                <div className="text-center py-8">
                  <p className="text-gray-500">Enter a user's email or name to search</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>


      </div>
    </DashboardTemplatePage>
  );
}