import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Switch } from "@/components/ui/switch";
import { toast } from "sonner";
import { Eye, ExternalLink } from "lucide-react";

type CoverTemplate = {
  id: string; 
  title: string; 
  category: string; 
  priceCents: number; 
  previewUrl: string;
  isActive: boolean;
  createdAt: string;
  pptxUrl?: string;
  keynoteUrl?: string;
  gslidesUrl?: string;
  downloadFile?: string;
};

const fmt = (cents: number) => `$${(cents / 100).toFixed(2)}`;

export default function AdminCoverTemplates() {
  const [search, setSearch] = useState("");
  const [showInactive, setShowInactive] = useState(true);
  const qc = useQueryClient();

  const { data, isLoading, refetch } = useQuery({
    queryKey: ["admin-cover-templates", search, showInactive],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (search) params.set("q", search);
      if (showInactive) params.set("includeInactive", "true");
      
      const res = await fetch(`/api/admin/cover-templates?${params.toString()}`, { 
        credentials: "include" 
      });
      if (!res.ok) throw new Error("Failed to load templates");
      return res.json() as Promise<CoverTemplate[]>;
    }
  });

  const toggleActive = useMutation({
    mutationFn: async ({ id, active }: { id: string; active: boolean }) => {
      const res = await fetch(`/api/admin/cover-templates/${id}/activate`, {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ active })
      });
      if (!res.ok) throw new Error("Failed to update template");
      return res.json();
    },
    onSuccess: (data, variables) => {
      toast.success(`Template ${variables.active ? 'activated' : 'deactivated'}`);
      qc.invalidateQueries({ queryKey: ["admin-cover-templates"] });
    },
    onError: () => {
      toast.error("Failed to update template");
    }
  });

  const templates = data || [];
  const activeCount = templates.filter(t => t.isActive).length;
  const inactiveCount = templates.filter(t => !t.isActive).length;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Cover Templates Management</h1>
          <p className="text-muted-foreground">
            Manage cover template submissions and activation status
          </p>
        </div>
        <div className="text-right text-sm text-muted-foreground">
          {activeCount} active • {inactiveCount} inactive
        </div>
      </div>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-4 space-y-4">
          <div className="flex items-center gap-4">
            <div className="flex-1">
              <Input 
                placeholder="Search by title or category..." 
                value={search} 
                onChange={(e) => setSearch(e.target.value)} 
              />
            </div>
            <div className="flex items-center space-x-2">
              <Switch 
                id="show-inactive" 
                checked={showInactive}
                onCheckedChange={setShowInactive}
              />
              <label htmlFor="show-inactive" className="text-sm font-medium">
                Show inactive templates
              </label>
            </div>
            <Button onClick={() => refetch()} disabled={isLoading}>
              Refresh
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Templates List */}
      <Card>
        <CardHeader>
          <CardTitle>Templates ({templates.length})</CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-4">
              {Array.from({length: 5}).map((_, i) => (
                <div key={i} className="animate-pulse flex items-center space-x-4 p-4 border rounded">
                  <div className="w-20 h-15 bg-muted rounded"></div>
                  <div className="flex-1 space-y-2">
                    <div className="h-4 bg-muted rounded w-3/4"></div>
                    <div className="h-3 bg-muted rounded w-1/2"></div>
                  </div>
                  <div className="w-20 h-6 bg-muted rounded"></div>
                </div>
              ))}
            </div>
          ) : templates.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">
              No templates found. Try adjusting your search or filters.
            </div>
          ) : (
            <div className="space-y-4">
              {templates.map((template) => (
                <div key={template.id} className="flex items-center gap-4 p-4 border rounded-lg hover:bg-muted/50 transition-colors">
                  {/* Preview Image */}
                  <div className="w-20 h-15 bg-muted rounded overflow-hidden flex-shrink-0">
                    <img 
                      src={template.previewUrl} 
                      alt={template.title}
                      className="w-full h-full object-cover"
                    />
                  </div>
                  
                  {/* Template Info */}
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2 mb-1">
                      <h3 className="font-medium truncate" data-testid={`text-title-${template.id}`}>
                        {template.title}
                      </h3>
                      <Badge variant="outline" className="capitalize">
                        {template.category}
                      </Badge>
                      <Badge variant={template.isActive ? "default" : "secondary"}>
                        {template.isActive ? "Active" : "Inactive"}
                      </Badge>
                    </div>
                    <div className="text-sm text-muted-foreground">
                      {fmt(template.priceCents)} • Created {new Date(template.createdAt).toLocaleDateString()}
                    </div>
                    <div className="flex items-center gap-1 mt-1">
                      {template.pptxUrl && <Badge variant="outline" className="text-xs">PPTX</Badge>}
                      {template.keynoteUrl && <Badge variant="outline" className="text-xs">Keynote</Badge>}
                      {template.gslidesUrl && <Badge variant="outline" className="text-xs">G.Slides</Badge>}
                      {template.downloadFile && <Badge variant="outline" className="text-xs">Bundle</Badge>}
                    </div>
                  </div>
                  
                  {/* Actions */}
                  <div className="flex items-center gap-2">
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => window.open(template.previewUrl, '_blank')}
                      data-testid={`button-preview-${template.id}`}
                    >
                      <Eye className="h-4 w-4 mr-1" />
                      Preview
                    </Button>
                    
                    <div className="flex items-center space-x-2">
                      <Switch
                        checked={template.isActive}
                        onCheckedChange={(checked) => 
                          toggleActive.mutate({ id: template.id, active: checked })
                        }
                        disabled={toggleActive.isPending}
                        data-testid={`switch-active-${template.id}`}
                      />
                      <span className="text-sm">
                        {template.isActive ? "Active" : "Inactive"}
                      </span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}