import { useState, useEffect } from "react";
import { useLocation } from "wouter";
import { requireAdminOrRedirect } from "@/admin/adminGuard";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Checkbox } from "@/components/ui/checkbox";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { 
  Eye, 
  Check, 
  X, 
  Calendar, 
  User, 
  FileText, 
  Download, 
  Search, 
  Filter,
  CheckSquare,
  Square,
  Clock,
  ShoppingCart,
  Tags,
  DollarSign,
  RefreshCw,
  AlertCircle
} from "lucide-react";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";

// Types for the asset data from the API
interface Creator {
  id: string;
  userId: string;
  name?: string;
  email: string;
  onboardingStatus: string;
}

interface PendingAsset {
  id: string;
  assetId: string;
  title: string;
  description?: string;
  price: number; // Price in cents
  isExclusive: boolean;
  createdAt: string;
  fileName?: string;
  originalFileName?: string;
  fileSize?: number;
  assetType?: string;
  category?: string;
  tags?: string[];
  fileUrl?: string;
  creator: Creator;
}

interface AssetResponse {
  assets: PendingAsset[];
  count: number;
  total?: number;
  page?: number;
  limit?: number;
  totalPages?: number;
  hasNextPage?: boolean;
  hasPrevPage?: boolean;
}

// Asset status filter options
const STATUS_FILTERS = [
  { value: 'pending', label: 'Pending Review' },
  { value: 'approved', label: 'Approved' },
  { value: 'rejected', label: 'Rejected' }
] as const;

type StatusFilter = 'pending' | 'approved' | 'rejected';

export default function AdminAssetsReview() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  
  // State management
  const [selectedAssets, setSelectedAssets] = useState<Set<string>>(new Set());
  const [searchQuery, setSearchQuery] = useState("");
  const [statusFilter, setStatusFilter] = useState<StatusFilter>('pending');
  const [selectedAsset, setSelectedAsset] = useState<PendingAsset | null>(null);
  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [rejectionReason, setRejectionReason] = useState("");
  const [assetToReject, setAssetToReject] = useState<string | null>(null);
  const [showBulkRejectDialog, setShowBulkRejectDialog] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage] = useState(20);

  // Fetch assets with proper status filtering and pagination
  const { data: assetsData, isLoading, error, refetch } = useQuery({
    queryKey: ['/api/admin/assets', statusFilter, currentPage, itemsPerPage],
    queryFn: async () => {
      const token = await requireAdminOrRedirect();
      if (!token) throw new Error('Not authenticated');
      
      // Use correct endpoint based on status filter - THIS WAS THE CRITICAL BUG!
      const endpoint = `/api/admin/assets/${statusFilter}`;
      const url = `${endpoint}?page=${currentPage}&limit=${itemsPerPage}`;
      
      const response = await fetch(url, {
        headers: { Authorization: `Bearer ${token}` },
        credentials: "include",
      });
      
      if (!response.ok) {
        throw new Error(`Failed to fetch ${statusFilter} assets`);
      }
      
      return response.json() as Promise<AssetResponse>;
    },
  });

  // Approve asset mutation with proper cache invalidation and apiRequest
  const approveAssetMutation = useMutation({
    mutationFn: async (creatorAssetId: string) => {
      return await apiRequest('POST', `/api/admin/assets/${creatorAssetId}/approve`);
    },
    onSuccess: () => {
      toast({
        title: "Asset Approved",
        description: "The asset has been approved successfully.",
      });
      // FIXED: Proper cache invalidation with full query key hierarchy
      queryClient.invalidateQueries({ queryKey: ['/api/admin/assets'] });
      setSelectedAssets(new Set());
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error instanceof Error ? error.message : 'Failed to approve asset',
        variant: "destructive",
      });
    },
  });

  // Reject asset mutation with proper cache invalidation and apiRequest
  const rejectAssetMutation = useMutation({
    mutationFn: async ({ creatorAssetId, reason }: { creatorAssetId: string; reason: string }) => {
      return await apiRequest('POST', `/api/admin/assets/${creatorAssetId}/reject`, { reason });
    },
    onSuccess: () => {
      toast({
        title: "Asset Rejected",
        description: "The asset has been rejected successfully.",
      });
      // FIXED: Proper cache invalidation with full query key hierarchy
      queryClient.invalidateQueries({ queryKey: ['/api/admin/assets'] });
      setSelectedAssets(new Set());
      setShowRejectDialog(false);
      setShowBulkRejectDialog(false);
      setRejectionReason("");
      setAssetToReject(null);
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error instanceof Error ? error.message : 'Failed to reject asset',
        variant: "destructive",
      });
    },
  });

  // Filter assets based on search and status
  const filteredAssets = assetsData?.assets?.filter(asset => {
    const matchesSearch = !searchQuery || 
      asset.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
      asset.creator.name?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      asset.creator.email.toLowerCase().includes(searchQuery.toLowerCase()) ||
      asset.tags?.some(tag => tag.toLowerCase().includes(searchQuery.toLowerCase()));
    
    return matchesSearch;
  }) || [];

  // Handle asset selection
  const toggleAssetSelection = (assetId: string) => {
    const newSelected = new Set(selectedAssets);
    if (newSelected.has(assetId)) {
      newSelected.delete(assetId);
    } else {
      newSelected.add(assetId);
    }
    setSelectedAssets(newSelected);
  };

  const selectAllAssets = () => {
    if (selectedAssets.size === filteredAssets.length) {
      setSelectedAssets(new Set());
    } else {
      setSelectedAssets(new Set(filteredAssets.map(asset => asset.id)));
    }
  };

  // Handle bulk approve with proper ID mapping
  const handleBulkApprove = async () => {
    for (const creatorAssetId of selectedAssets) {
      try {
        await approveAssetMutation.mutateAsync(creatorAssetId);
      } catch (error) {
        console.error(`Failed to approve asset ${creatorAssetId}:`, error);
      }
    }
  };

  // Handle bulk reject with proper ID mapping
  const handleBulkReject = async () => {
    if (!rejectionReason.trim()) {
      toast({
        title: "Error",
        description: "Please provide a rejection reason.",
        variant: "destructive",
      });
      return;
    }

    for (const creatorAssetId of selectedAssets) {
      try {
        await rejectAssetMutation.mutateAsync({ creatorAssetId, reason: rejectionReason });
      } catch (error) {
        console.error(`Failed to reject asset ${creatorAssetId}:`, error);
      }
    }
  };

  // Handle single asset actions with proper ID mapping
  const handleApprove = (creatorAssetId: string) => {
    approveAssetMutation.mutate(creatorAssetId);
  };

  const handleReject = (creatorAssetId: string) => {
    setAssetToReject(creatorAssetId);
    setShowRejectDialog(true);
  };

  const confirmReject = () => {
    if (!assetToReject || !rejectionReason.trim()) {
      toast({
        title: "Error",
        description: "Please provide a rejection reason.",
        variant: "destructive",
      });
      return;
    }

    rejectAssetMutation.mutate({ 
      creatorAssetId: assetToReject, 
      reason: rejectionReason 
    });
  };

  // Format file size
  const formatFileSize = (bytes?: number) => {
    if (!bytes) return 'Unknown';
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(1024));
    return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
  };

  // Format price
  const formatPrice = (cents: number) => {
    return `$${(cents / 100).toFixed(2)}`;
  };

  // Reset page when status filter changes
  const handleStatusFilterChange = (newStatus: StatusFilter) => {
    setStatusFilter(newStatus);
    setCurrentPage(1); // Reset to first page
    setSelectedAssets(new Set()); // Clear selections
  };

  if (isLoading) {
    return (
      <DashboardTemplatePage title="Asset Reviews">
        <div className="flex items-center justify-center h-64" data-testid="loading-assets">
          <div className="text-center">
            <div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full mx-auto"></div>
            <p className="mt-2 text-gray-600">Loading {statusFilter} assets...</p>
          </div>
        </div>
      </DashboardTemplatePage>
    );
  };

  if (error) {
    return (
      <DashboardTemplatePage title="Asset Reviews">
        <Card className="max-w-md mx-auto">
          <CardContent className="pt-6">
            <div className="text-center">
              <AlertCircle className="h-12 w-12 text-red-500 mx-auto mb-4" />
              <p className="text-red-600 mb-4">{error.message}</p>
              <Button onClick={() => refetch()}>
                <RefreshCw className="h-4 w-4 mr-2" />
                Retry
              </Button>
            </div>
          </CardContent>
        </Card>
      </DashboardTemplatePage>
    );
  }

  return (
    <DashboardTemplatePage title="Asset Reviews">
      <div className="space-y-6">
        {/* Header with stats and actions */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Creator Asset Reviews
            </CardTitle>
            <CardDescription>
              Review and approve creator marketplace assets
            </CardDescription>
          </CardHeader>
          <CardContent>
            {/* Stats with Pagination Info */}
            <div className="grid grid-cols-1 md:grid-cols-5 gap-4 mb-6">
              <div className="text-center" data-testid="stats-total-assets">
                <div className="text-2xl font-bold text-blue-600">{assetsData?.total || 0}</div>
                <div className="text-sm text-gray-600">Total {statusFilter}</div>
              </div>
              <div className="text-center" data-testid="stats-current-page">
                <div className="text-2xl font-bold text-indigo-600">{assetsData?.count || 0}</div>
                <div className="text-sm text-gray-600">On Page {currentPage}</div>
              </div>
              <div className="text-center" data-testid="stats-selected">
                <div className="text-2xl font-bold text-green-600">{selectedAssets.size}</div>
                <div className="text-sm text-gray-600">Selected</div>
              </div>
              <div className="text-center" data-testid="stats-filtered">
                <div className="text-2xl font-bold text-orange-600">{filteredAssets.length}</div>
                <div className="text-sm text-gray-600">Search Results</div>
              </div>
              <div className="text-center" data-testid="stats-total-value">
                <div className="text-2xl font-bold text-purple-600">
                  ${(filteredAssets.reduce((sum, asset) => sum + asset.price, 0) / 100).toFixed(2)}
                </div>
                <div className="text-sm text-gray-600">Page Value</div>
              </div>
            </div>

            {/* Filters and Search */}
            <div className="flex flex-col sm:flex-row gap-4 mb-6">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    placeholder="Search by title, creator, email, or tags..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-10"
                    data-testid="search-assets"
                  />
                </div>
              </div>
              <Select value={statusFilter} onValueChange={handleStatusFilterChange}>
                <SelectTrigger className="w-[180px]" data-testid="filter-status">
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  {STATUS_FILTERS.map((filter) => (
                    <SelectItem key={filter.value} value={filter.value}>
                      {filter.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Bulk Actions */}
            {selectedAssets.size > 0 && statusFilter === 'pending' && (
              <div className="flex flex-wrap gap-2 mb-6 p-4 bg-blue-50 rounded-lg">
                <span className="text-sm font-medium text-blue-700">
                  {selectedAssets.size} asset{selectedAssets.size > 1 ? 's' : ''} selected:
                </span>
                <Button
                  size="sm"
                  onClick={handleBulkApprove}
                  disabled={approveAssetMutation.isPending}
                  data-testid="bulk-approve"
                >
                  <Check className="h-4 w-4 mr-1" />
                  Approve All
                </Button>
                <Button
                  size="sm"
                  variant="destructive"
                  onClick={() => setShowBulkRejectDialog(true)}
                  disabled={rejectAssetMutation.isPending}
                  data-testid="bulk-reject"
                >
                  <X className="h-4 w-4 mr-1" />
                  Reject All
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setSelectedAssets(new Set())}
                  data-testid="clear-selection"
                >
                  Clear Selection
                </Button>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Assets List */}
        {filteredAssets.length === 0 ? (
          <Card>
            <CardContent className="pt-6">
              <div className="text-center py-8">
                <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No assets found</h3>
                <p className="text-gray-600">
                  {searchQuery
                    ? "Try adjusting your search criteria."
                    : `No ${statusFilter} assets at the moment.`}
                </p>
              </div>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-4">
            {/* Select All Checkbox */}
            {statusFilter === 'pending' && (
              <Card>
                <CardContent className="pt-4">
                  <div className="flex items-center space-x-2">
                    <Checkbox
                      id="select-all"
                      checked={selectedAssets.size === filteredAssets.length && filteredAssets.length > 0}
                      onCheckedChange={selectAllAssets}
                      data-testid="select-all-assets"
                    />
                    <label
                      htmlFor="select-all"
                      className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                    >
                      Select all {filteredAssets.length} assets
                    </label>
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Asset Cards */}
            {filteredAssets.map((asset) => (
              <Card key={asset.id} className="hover:shadow-md transition-shadow">
                <CardContent className="pt-6">
                  <div className="flex items-start justify-between">
                    <div className="flex items-start space-x-4 flex-1">
                      {/* Selection Checkbox */}
                      {statusFilter === 'pending' && (
                        <Checkbox
                          checked={selectedAssets.has(asset.id)}
                          onCheckedChange={() => toggleAssetSelection(asset.id)}
                          data-testid={`select-asset-${asset.id}`}
                        />
                      )}

                      {/* Asset Info */}
                      <div className="flex-1 min-w-0">
                        <div className="flex items-start justify-between mb-2">
                          <div className="flex-1">
                            <h3 className="text-lg font-semibold text-gray-900 truncate">
                              {asset.title}
                            </h3>
                            <p className="text-sm text-gray-600 mb-2 line-clamp-2">
                              {asset.description || 'No description provided'}
                            </p>
                          </div>
                          <div className="ml-4 text-right">
                            <div className="text-lg font-bold text-green-600">
                              {formatPrice(asset.price)}
                            </div>
                            {asset.isExclusive && (
                              <Badge variant="secondary" className="mt-1">
                                Exclusive
                              </Badge>
                            )}
                          </div>
                        </div>

                        {/* Asset Metadata */}
                        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600 mb-4">
                          <div className="flex items-center">
                            <User className="h-4 w-4 mr-2" />
                            <span className="truncate">
                              {asset.creator.name || asset.creator.email}
                            </span>
                          </div>
                          <div className="flex items-center">
                            <Calendar className="h-4 w-4 mr-2" />
                            <span>{new Date(asset.createdAt).toLocaleDateString()}</span>
                          </div>
                          <div className="flex items-center">
                            <FileText className="h-4 w-4 mr-2" />
                            <span>{formatFileSize(asset.fileSize)} • {asset.assetType}</span>
                          </div>
                          {asset.tags && asset.tags.length > 0 && (
                            <div className="flex items-center">
                              <Tags className="h-4 w-4 mr-2" />
                              <span className="truncate">
                                {asset.tags.slice(0, 2).join(', ')}
                                {asset.tags.length > 2 && ` +${asset.tags.length - 2} more`}
                              </span>
                            </div>
                          )}
                        </div>

                        {/* Action Buttons */}
                        <div className="flex flex-wrap gap-2">
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => setSelectedAsset(asset)}
                            data-testid={`view-asset-${asset.id}`}
                          >
                            <Eye className="h-4 w-4 mr-1" />
                            View Details
                          </Button>

                          {statusFilter === 'pending' && (
                            <>
                              <Button
                                size="sm"
                                onClick={() => handleApprove(asset.id)}
                                disabled={approveAssetMutation.isPending}
                                data-testid={`approve-asset-${asset.id}`}
                              >
                                <Check className="h-4 w-4 mr-1" />
                                Approve
                              </Button>
                              <Button
                                size="sm"
                                variant="destructive"
                                onClick={() => handleReject(asset.id)}
                                disabled={rejectAssetMutation.isPending}
                                data-testid={`reject-asset-${asset.id}`}
                              >
                                <X className="h-4 w-4 mr-1" />
                                Reject
                              </Button>
                            </>
                          )}

                          {asset.fileUrl && (
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => window.open(asset.fileUrl, '_blank')}
                              data-testid={`download-asset-${asset.id}`}
                            >
                              <Download className="h-4 w-4 mr-1" />
                              Download
                            </Button>
                          )}
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
        
        {/* Pagination Controls */}
        {assetsData && assetsData.totalPages && assetsData.totalPages > 1 && (
          <Card>
            <CardContent className="pt-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2" data-testid="pagination-info">
                  <span className="text-sm text-gray-600">
                    Showing {((currentPage - 1) * itemsPerPage) + 1} to {Math.min(currentPage * itemsPerPage, assetsData.total || 0)} of {assetsData.total || 0} assets
                  </span>
                </div>
                <div className="flex items-center gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                    disabled={!assetsData.hasPrevPage || isLoading}
                    data-testid="pagination-prev"
                  >
                    Previous
                  </Button>
                  <span className="text-sm font-medium px-3" data-testid="pagination-current">
                    Page {currentPage} of {assetsData.totalPages}
                  </span>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setCurrentPage(prev => prev + 1)}
                    disabled={!assetsData.hasNextPage || isLoading}
                    data-testid="pagination-next"
                  >
                    Next
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Asset Details Modal */}
        {selectedAsset && (
          <Dialog open={!!selectedAsset} onOpenChange={() => setSelectedAsset(null)}>
            <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
              <DialogHeader>
                <DialogTitle>{selectedAsset.title}</DialogTitle>
                <DialogDescription>
                  Asset details and review information
                </DialogDescription>
              </DialogHeader>
              
              <div className="space-y-4">
                {/* Asset Basic Info */}
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-gray-700">Price</label>
                    <p className="text-lg font-bold text-green-600">
                      {formatPrice(selectedAsset.price)}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-700">Type</label>
                    <p className="text-sm text-gray-900">{selectedAsset.assetType}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-700">File Size</label>
                    <p className="text-sm text-gray-900">{formatFileSize(selectedAsset.fileSize)}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-700">Category</label>
                    <p className="text-sm text-gray-900">{selectedAsset.category || 'Uncategorized'}</p>
                  </div>
                </div>

                {/* Description */}
                <div>
                  <label className="text-sm font-medium text-gray-700">Description</label>
                  <p className="text-sm text-gray-900 mt-1">
                    {selectedAsset.description || 'No description provided'}
                  </p>
                </div>

                {/* Tags */}
                {selectedAsset.tags && selectedAsset.tags.length > 0 && (
                  <div>
                    <label className="text-sm font-medium text-gray-700">Tags</label>
                    <div className="flex flex-wrap gap-1 mt-1">
                      {selectedAsset.tags.map((tag, index) => (
                        <Badge key={index} variant="outline">
                          {tag}
                        </Badge>
                      ))}
                    </div>
                  </div>
                )}

                {/* Creator Info */}
                <div className="border-t pt-4">
                  <h4 className="font-medium text-gray-900 mb-2">Creator Information</h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="text-sm font-medium text-gray-700">Name</label>
                      <p className="text-sm text-gray-900">{selectedAsset.creator.name || 'Not provided'}</p>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-gray-700">Email</label>
                      <p className="text-sm text-gray-900">{selectedAsset.creator.email}</p>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-gray-700">Status</label>
                      <Badge variant="outline">{selectedAsset.creator.onboardingStatus}</Badge>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-gray-700">Uploaded</label>
                      <p className="text-sm text-gray-900">
                        {new Date(selectedAsset.createdAt).toLocaleDateString()}
                      </p>
                    </div>
                  </div>
                </div>

                {/* File Info */}
                <div className="border-t pt-4">
                  <h4 className="font-medium text-gray-900 mb-2">File Information</h4>
                  <div className="grid grid-cols-1 gap-2">
                    <div>
                      <label className="text-sm font-medium text-gray-700">Original Filename</label>
                      <p className="text-sm text-gray-900 font-mono">
                        {selectedAsset.originalFileName || selectedAsset.fileName}
                      </p>
                    </div>
                  </div>
                </div>
              </div>

              <DialogFooter className="flex justify-between">
                <div className="flex gap-2">
                  {selectedAsset.fileUrl && (
                    <Button
                      variant="outline"
                      onClick={() => window.open(selectedAsset.fileUrl, '_blank')}
                    >
                      <Download className="h-4 w-4 mr-2" />
                      Download File
                    </Button>
                  )}
                </div>
                
                {statusFilter === 'pending' && (
                  <div className="flex gap-2">
                    <Button
                      onClick={() => {
                        handleApprove(selectedAsset.id);
                        setSelectedAsset(null);
                      }}
                      disabled={approveAssetMutation.isPending}
                    >
                      <Check className="h-4 w-4 mr-2" />
                      Approve
                    </Button>
                    <Button
                      variant="destructive"
                      onClick={() => {
                        handleReject(selectedAsset.id);
                        setSelectedAsset(null);
                      }}
                      disabled={rejectAssetMutation.isPending}
                    >
                      <X className="h-4 w-4 mr-2" />
                      Reject
                    </Button>
                  </div>
                )}
              </DialogFooter>
            </DialogContent>
          </Dialog>
        )}

        {/* Single Asset Rejection Dialog */}
        <Dialog open={showRejectDialog} onOpenChange={setShowRejectDialog}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Reject Asset</DialogTitle>
              <DialogDescription>
                Please provide a reason for rejecting this asset. This will be sent to the creator.
              </DialogDescription>
            </DialogHeader>
            
            <Textarea
              placeholder="Enter rejection reason..."
              value={rejectionReason}
              onChange={(e) => setRejectionReason(e.target.value)}
              className="min-h-[100px]"
              data-testid="rejection-reason"
            />
            
            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => {
                  setShowRejectDialog(false);
                  setRejectionReason("");
                  setAssetToReject(null);
                }}
              >
                Cancel
              </Button>
              <Button
                variant="destructive"
                onClick={confirmReject}
                disabled={!rejectionReason.trim() || rejectAssetMutation.isPending}
                data-testid="confirm-reject"
              >
                {rejectAssetMutation.isPending ? (
                  <>
                    <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                    Rejecting...
                  </>
                ) : (
                  <>
                    <X className="h-4 w-4 mr-2" />
                    Reject Asset
                  </>
                )}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        {/* Bulk Rejection Dialog */}
        <Dialog open={showBulkRejectDialog} onOpenChange={setShowBulkRejectDialog}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Reject {selectedAssets.size} Assets</DialogTitle>
              <DialogDescription>
                Please provide a reason for rejecting these assets. This will be sent to all creators.
              </DialogDescription>
            </DialogHeader>
            
            <Textarea
              placeholder="Enter rejection reason..."
              value={rejectionReason}
              onChange={(e) => setRejectionReason(e.target.value)}
              className="min-h-[100px]"
              data-testid="bulk-rejection-reason"
            />
            
            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => {
                  setShowBulkRejectDialog(false);
                  setRejectionReason("");
                }}
              >
                Cancel
              </Button>
              <Button
                variant="destructive"
                onClick={handleBulkReject}
                disabled={!rejectionReason.trim() || rejectAssetMutation.isPending}
                data-testid="confirm-bulk-reject"
              >
                {rejectAssetMutation.isPending ? (
                  <>
                    <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                    Rejecting...
                  </>
                ) : (
                  <>
                    <X className="h-4 w-4 mr-2" />
                    Reject {selectedAssets.size} Assets
                  </>
                )}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>
    </DashboardTemplatePage>
  );
}