import { useQuery } from "@tanstack/react-query";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

type CoverPurchase = {
  id: string; 
  templateId: string; 
  status: string; 
  downloadUrl?: string; 
  createdAt: string;
};

type InfographicPurchase = {
  id: string;
  selectedIds: string[];
  status: string;
  downloadUrl?: string;
  amountCents: number;
  currency: string;
  createdAt: string;
};

export default function MyPurchases() {
  const { data: coverPurchases, isLoading: isLoadingCovers } = useQuery({
    queryKey: ["my-cover-purchases"],
    queryFn: async () => {
      const r = await fetch("/api/me/cover-purchases", { credentials: "include" });
      if (!r.ok) throw new Error("Failed to load cover purchases");
      return r.json() as Promise<CoverPurchase[]>;
    }
  });

  const { data: infographicPurchases, isLoading: isLoadingInfographics } = useQuery({
    queryKey: ["my-infographic-purchases"],
    queryFn: async () => {
      const r = await fetch("/api/me/infographic-purchases", { credentials: "include" });
      if (!r.ok) throw new Error("Failed to load infographic purchases");
      return r.json() as Promise<InfographicPurchase[]>;
    }
  });

  const isLoading = isLoadingCovers || isLoadingInfographics;

  const formatPrice = (cents: number, currency: string = 'usd') => {
    return `$${(cents / 100).toFixed(2)}`;
  };

  return (
    <DashboardTemplatePage title="My Purchases">
      <Tabs defaultValue="all" className="space-y-6">
        <TabsList>
          <TabsTrigger value="all" data-testid="tab-all">All Purchases</TabsTrigger>
          <TabsTrigger value="covers" data-testid="tab-covers">Cover Templates</TabsTrigger>
          <TabsTrigger value="infographics" data-testid="tab-infographics">Infographics</TabsTrigger>
        </TabsList>

        {isLoading ? (
          <div className="text-center py-8">Loading purchases...</div>
        ) : (
          <>
            <TabsContent value="all" className="space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>Cover Templates</CardTitle>
                </CardHeader>
                <CardContent>
                  {!coverPurchases?.length ? (
                    <div className="text-muted-foreground text-center py-4">No cover template purchases yet.</div>
                  ) : (
                    <div className="space-y-3">
                      {coverPurchases.map(p => (
                        <div key={`cover-${p.id}`} className="flex items-center justify-between border rounded p-3">
                          <div className="text-sm">
                            <div className="font-medium">Cover Template: {p.templateId}</div>
                            <div className="text-muted-foreground text-xs">
                              Purchased {new Date(p.createdAt).toLocaleDateString()}
                            </div>
                            <div className="flex items-center gap-2 mt-1">
                              <Badge variant={p.status === 'paid' ? 'default' : 'secondary'}>
                                {p.status}
                              </Badge>
                            </div>
                          </div>
                          <Button
                            disabled={p.status !== "paid"}
                            onClick={() => window.location.href = `/api/cover-templates/${p.templateId}/download`}
                            data-testid={`download-cover-${p.id}`}
                          >
                            Download
                          </Button>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Infographic Bundles</CardTitle>
                </CardHeader>
                <CardContent>
                  {!infographicPurchases?.length ? (
                    <div className="text-muted-foreground text-center py-4">No infographic purchases yet.</div>
                  ) : (
                    <div className="space-y-3">
                      {infographicPurchases.map(p => (
                        <div key={`infographic-${p.id}`} className="flex items-center justify-between border rounded p-3">
                          <div className="text-sm">
                            <div className="font-medium">
                              Infographic Bundle ({p.selectedIds.length} templates)
                            </div>
                            <div className="text-muted-foreground text-xs">
                              Purchased {new Date(p.createdAt).toLocaleDateString()} • {formatPrice(p.amountCents, p.currency)}
                            </div>
                            <div className="flex items-center gap-2 mt-1">
                              <Badge variant={p.status === 'paid' ? 'default' : 'secondary'}>
                                {p.status}
                              </Badge>
                              <Badge variant="outline" className="text-xs">
                                {p.selectedIds.length} items
                              </Badge>
                            </div>
                          </div>
                          <Button
                            disabled={p.status !== "paid" || !p.downloadUrl}
                            onClick={() => window.location.href = `/api/infographics/purchases/${p.id}/download`}
                            data-testid={`download-infographic-${p.id}`}
                          >
                            {p.status === 'paid' && p.downloadUrl ? 'Download ZIP' : 'Processing...'}
                          </Button>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="covers">
              <Card>
                <CardHeader>
                  <CardTitle>Cover Template Purchases</CardTitle>
                </CardHeader>
                <CardContent>
                  {!coverPurchases?.length ? (
                    <div className="text-muted-foreground text-center py-8">
                      No cover template purchases yet.
                      <div className="text-sm mt-2">
                        Browse our <a href="/business-assets/templates/covers" className="text-primary hover:underline">cover templates</a> to get started.
                      </div>
                    </div>
                  ) : (
                    <div className="space-y-3">
                      {coverPurchases.map(p => (
                        <div key={`cover-tab-${p.id}`} className="flex items-center justify-between border rounded p-3">
                          <div className="text-sm">
                            <div className="font-medium">{p.templateId}</div>
                            <div className="text-muted-foreground text-xs">
                              Purchased {new Date(p.createdAt).toLocaleDateString()}
                            </div>
                            <div className="flex items-center gap-2 mt-1">
                              <Badge variant={p.status === 'paid' ? 'default' : 'secondary'}>
                                {p.status}
                              </Badge>
                            </div>
                          </div>
                          <Button
                            disabled={p.status !== "paid"}
                            onClick={() => window.location.href = `/api/cover-templates/${p.templateId}/download`}
                            data-testid={`download-cover-tab-${p.id}`}
                          >
                            Download
                          </Button>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="infographics">
              <Card>
                <CardHeader>
                  <CardTitle>Infographic Bundle Purchases</CardTitle>
                </CardHeader>
                <CardContent>
                  {!infographicPurchases?.length ? (
                    <div className="text-muted-foreground text-center py-8">
                      No infographic purchases yet.
                      <div className="text-sm mt-2">
                        Browse our <a href="/infographic-library" className="text-primary hover:underline">infographic templates</a> to get started.
                      </div>
                    </div>
                  ) : (
                    <div className="space-y-3">
                      {infographicPurchases.map(p => (
                        <div key={`infographic-tab-${p.id}`} className="flex items-center justify-between border rounded p-3">
                          <div className="text-sm">
                            <div className="font-medium">
                              Bundle Purchase ({p.selectedIds.length} templates)
                            </div>
                            <div className="text-muted-foreground text-xs">
                              Purchased {new Date(p.createdAt).toLocaleDateString()} • {formatPrice(p.amountCents, p.currency)}
                            </div>
                            <div className="flex items-center gap-2 mt-1">
                              <Badge variant={p.status === 'paid' ? 'default' : 'secondary'}>
                                {p.status}
                              </Badge>
                              <Badge variant="outline" className="text-xs">
                                {p.selectedIds.length} items
                              </Badge>
                            </div>
                            {p.selectedIds.length <= 3 && (
                              <div className="text-xs text-muted-foreground mt-1">
                                IDs: {p.selectedIds.join(', ')}
                              </div>
                            )}
                          </div>
                          <Button
                            disabled={p.status !== "paid" || !p.downloadUrl}
                            onClick={() => window.location.href = `/api/infographics/purchases/${p.id}/download`}
                            data-testid={`download-infographic-tab-${p.id}`}
                          >
                            {p.status === 'paid' && p.downloadUrl ? 'Download ZIP' : 'Processing...'}
                          </Button>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </>
        )}
      </Tabs>
    </DashboardTemplatePage>
  );
}